/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.cassandra.sidecar.db;

import java.util.List;
import java.util.concurrent.TimeUnit;

import org.apache.cassandra.sidecar.coordination.ClusterLease;
import org.apache.cassandra.sidecar.db.schema.RestoreJobsSchema;
import org.apache.cassandra.sidecar.db.schema.RestoreRangesSchema;
import org.apache.cassandra.sidecar.db.schema.RestoreSlicesSchema;
import org.apache.cassandra.sidecar.db.schema.SidecarInternalKeyspace;
import org.apache.cassandra.sidecar.db.schema.SidecarLeaseSchema;
import org.apache.cassandra.sidecar.db.schema.SidecarRolePermissionsSchema;
import org.apache.cassandra.sidecar.db.schema.SidecarSchema;
import org.apache.cassandra.sidecar.db.schema.SystemAuthSchema;
import org.apache.cassandra.sidecar.db.schema.TableSchema;
import org.apache.cassandra.sidecar.testing.IntegrationTestBase;
import org.apache.cassandra.testing.CassandraIntegrationTest;

import static org.assertj.core.api.Assertions.assertThat;

class SidecarSchemaIntTest extends IntegrationTestBase
{
    @CassandraIntegrationTest
    void testSidecarSchemaInitializationFromBlank()
    {
        waitForSchemaReady(60, TimeUnit.SECONDS);
        SidecarSchema sidecarSchema = injector.getInstance(SidecarSchema.class);
        assertThat(sidecarSchema.isInitialized())
        .describedAs("SidecarSchema should be initialized")
        .isTrue();

        SidecarInternalKeyspace sidecarInternalKeyspace = sidecarSchema.sidecarInternalKeyspace();
        for (Class<? extends TableSchema> clazz : List.of(RestoreJobsSchema.class,
                                                          RestoreRangesSchema.class,
                                                          RestoreSlicesSchema.class,
                                                          SidecarLeaseSchema.class,
                                                          SidecarRolePermissionsSchema.class,
                                                          SystemAuthSchema.class))
        {
            assertThat(sidecarInternalKeyspace.tableSchema(clazz)).isExactlyInstanceOf(clazz);
        }

        ClusterLease clusterLease = injector.getInstance(ClusterLease.class);
        assertThat(clusterLease.isClaimedByLocalSidecar())
        .describedAs("ClusterLease should be claimed by the local sidecar")
        .isTrue();
    }
}
