/*
 * Decompiled with CFR 0.152.
 */
package org.apache.helix.controller.rebalancer.waged;

import com.google.common.collect.ImmutableSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import org.apache.helix.HelixConstants;
import org.apache.helix.HelixRebalanceException;
import org.apache.helix.controller.changedetector.ResourceChangeDetector;
import org.apache.helix.controller.dataproviders.ResourceControllerDataProvider;
import org.apache.helix.controller.rebalancer.util.WagedRebalanceUtil;
import org.apache.helix.controller.rebalancer.waged.AssignmentManager;
import org.apache.helix.controller.rebalancer.waged.AssignmentMetadataStore;
import org.apache.helix.controller.rebalancer.waged.RebalanceAlgorithm;
import org.apache.helix.controller.rebalancer.waged.model.ClusterModel;
import org.apache.helix.controller.rebalancer.waged.model.ClusterModelProvider;
import org.apache.helix.controller.stages.CurrentStateOutput;
import org.apache.helix.model.Resource;
import org.apache.helix.model.ResourceAssignment;
import org.apache.helix.monitoring.metrics.MetricCollector;
import org.apache.helix.monitoring.metrics.WagedRebalancerMetricCollector;
import org.apache.helix.monitoring.metrics.model.CountMetric;
import org.apache.helix.monitoring.metrics.model.LatencyMetric;
import org.apache.helix.util.RebalanceUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class GlobalRebalanceRunner
implements AutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(GlobalRebalanceRunner.class);
    private static final Set<HelixConstants.ChangeType> GLOBAL_REBALANCE_REQUIRED_CHANGE_TYPES = ImmutableSet.of((Object)((Object)HelixConstants.ChangeType.RESOURCE_CONFIG), (Object)((Object)HelixConstants.ChangeType.IDEAL_STATE), (Object)((Object)HelixConstants.ChangeType.CLUSTER_CONFIG), (Object)((Object)HelixConstants.ChangeType.INSTANCE_CONFIG));
    private final ExecutorService _baselineCalculateExecutor = Executors.newSingleThreadExecutor();
    private final ResourceChangeDetector _changeDetector;
    private final AssignmentManager _assignmentManager;
    private final AssignmentMetadataStore _assignmentMetadataStore;
    private final LatencyMetric _writeLatency;
    private final CountMetric _baselineCalcCounter;
    private final LatencyMetric _baselineCalcLatency;
    private final CountMetric _rebalanceFailureCount;
    private boolean _asyncGlobalRebalanceEnabled;

    public GlobalRebalanceRunner(AssignmentManager assignmentManager, AssignmentMetadataStore assignmentMetadataStore, MetricCollector metricCollector, LatencyMetric writeLatency, CountMetric rebalanceFailureCount, boolean isAsyncGlobalRebalanceEnabled) {
        this._assignmentManager = assignmentManager;
        this._assignmentMetadataStore = assignmentMetadataStore;
        this._changeDetector = new ResourceChangeDetector(true);
        this._writeLatency = writeLatency;
        this._baselineCalcCounter = metricCollector.getMetric(WagedRebalancerMetricCollector.WagedRebalancerMetricNames.GlobalBaselineCalcCounter.name(), CountMetric.class);
        this._baselineCalcLatency = metricCollector.getMetric(WagedRebalancerMetricCollector.WagedRebalancerMetricNames.GlobalBaselineCalcLatencyGauge.name(), LatencyMetric.class);
        this._rebalanceFailureCount = rebalanceFailureCount;
        this._asyncGlobalRebalanceEnabled = isAsyncGlobalRebalanceEnabled;
    }

    public void globalRebalance(ResourceControllerDataProvider clusterData, Map<String, Resource> resourceMap, CurrentStateOutput currentStateOutput, RebalanceAlgorithm algorithm) throws HelixRebalanceException {
        this._changeDetector.updateSnapshots(clusterData);
        Map<HelixConstants.ChangeType, Set<String>> clusterChanges = this._changeDetector.getAllChanges();
        if (clusterChanges.keySet().stream().anyMatch(GLOBAL_REBALANCE_REQUIRED_CHANGE_TYPES::contains)) {
            boolean waitForGlobalRebalance = !this._asyncGlobalRebalanceEnabled;
            Future<Boolean> result = this._baselineCalculateExecutor.submit(() -> {
                try {
                    this.doGlobalRebalance(clusterData, resourceMap, algorithm, currentStateOutput, !waitForGlobalRebalance, clusterChanges);
                }
                catch (HelixRebalanceException e) {
                    if (this._asyncGlobalRebalanceEnabled) {
                        this._rebalanceFailureCount.increment(1L);
                    }
                    LOG.error("Failed to calculate baseline assignment!", (Throwable)e);
                    return false;
                }
                return true;
            });
            if (waitForGlobalRebalance) {
                try {
                    if (!result.get().booleanValue()) {
                        throw new HelixRebalanceException("Failed to calculate for the new Baseline.", HelixRebalanceException.Type.FAILED_TO_CALCULATE);
                    }
                }
                catch (InterruptedException | ExecutionException e) {
                    throw new HelixRebalanceException("Failed to execute new Baseline calculation.", HelixRebalanceException.Type.FAILED_TO_CALCULATE, e);
                }
            }
        }
    }

    private void doGlobalRebalance(ResourceControllerDataProvider clusterData, Map<String, Resource> resourceMap, RebalanceAlgorithm algorithm, CurrentStateOutput currentStateOutput, boolean shouldTriggerMainPipeline, Map<HelixConstants.ChangeType, Set<String>> clusterChanges) throws HelixRebalanceException {
        boolean isBaselineChanged;
        ClusterModel clusterModel;
        LOG.info("Start calculating the new baseline.");
        this._baselineCalcCounter.increment(1L);
        this._baselineCalcLatency.startMeasuringLatency();
        Map<String, ResourceAssignment> currentBaseline = this._assignmentManager.getBaselineAssignment(this._assignmentMetadataStore, currentStateOutput, resourceMap.keySet());
        try {
            clusterModel = ClusterModelProvider.generateClusterModelForBaseline(clusterData, resourceMap, clusterData.getAllInstances(), clusterChanges, currentBaseline);
        }
        catch (Exception ex) {
            throw new HelixRebalanceException("Failed to generate cluster model for global rebalance.", HelixRebalanceException.Type.INVALID_CLUSTER_STATUS, ex);
        }
        Map<String, ResourceAssignment> newBaseline = WagedRebalanceUtil.calculateAssignment(clusterModel, algorithm);
        boolean bl = isBaselineChanged = this._assignmentMetadataStore != null && this._assignmentMetadataStore.isBaselineChanged(newBaseline);
        if (isBaselineChanged) {
            try {
                this._writeLatency.startMeasuringLatency();
                this._assignmentMetadataStore.persistBaseline(newBaseline);
                this._writeLatency.endMeasuringLatency();
            }
            catch (Exception ex) {
                throw new HelixRebalanceException("Failed to persist the new baseline assignment.", HelixRebalanceException.Type.INVALID_REBALANCER_STATUS, ex);
            }
        } else {
            LOG.debug("Assignment Metadata Store is null. Skip persisting the baseline assignment.");
        }
        this._baselineCalcLatency.endMeasuringLatency();
        LOG.info("Global baseline calculation completed and has been persisted into metadata store.");
        if (isBaselineChanged && shouldTriggerMainPipeline) {
            LOG.info("Schedule a new rebalance after the new baseline calculation has finished.");
            RebalanceUtil.scheduleOnDemandPipeline(clusterData.getClusterName(), 0L, false);
        }
    }

    public void setGlobalRebalanceAsyncMode(boolean isAsyncGlobalRebalanceEnabled) {
        this._asyncGlobalRebalanceEnabled = isAsyncGlobalRebalanceEnabled;
    }

    public ResourceChangeDetector getChangeDetector() {
        return this._changeDetector;
    }

    public void resetChangeDetector() {
        this._changeDetector.resetSnapshots();
    }

    @Override
    public void close() {
        if (this._baselineCalculateExecutor != null) {
            this._baselineCalculateExecutor.shutdownNow();
        }
    }
}

