/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.mr.hive;

import java.math.BigDecimal;
import java.sql.Date;
import java.sql.Timestamp;
import java.time.DateTimeException;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.stream.Collectors;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.hadoop.hive.ql.io.sarg.ExpressionTree;
import org.apache.hadoop.hive.ql.io.sarg.PredicateLeaf;
import org.apache.hadoop.hive.ql.io.sarg.SearchArgument;
import org.apache.hadoop.hive.ql.io.sarg.SearchArgumentImpl;
import org.apache.hadoop.hive.ql.parse.TransformSpec;
import org.apache.hadoop.hive.serde2.io.HiveDecimalWritable;
import org.apache.iceberg.common.DynFields;
import org.apache.iceberg.expressions.Expression;
import org.apache.iceberg.expressions.Expressions;
import org.apache.iceberg.expressions.Literal;
import org.apache.iceberg.expressions.UnboundTerm;
import org.apache.iceberg.relocated.com.google.common.base.Preconditions;
import org.apache.iceberg.transforms.Transforms;
import org.apache.iceberg.types.Types;
import org.apache.iceberg.util.DateTimeUtil;
import org.apache.iceberg.util.NaNUtil;

public class HiveIcebergFilterFactory {
    private static final DynFields.UnboundField<?> LITERAL_FIELD = DynFields.builder().hiddenImpl(SearchArgumentImpl.PredicateLeafImpl.class, "literal").build();
    private static final int ICEBERG_EPOCH_YEAR = 1970;
    private static final int ICEBERG_EPOCH_MONTH = 1;

    private HiveIcebergFilterFactory() {
    }

    public static Expression generateFilterExpression(SearchArgument sarg) {
        return HiveIcebergFilterFactory.translate(sarg.getExpression(), sarg.getLeaves());
    }

    private static Expression translate(ExpressionTree tree, List<PredicateLeaf> leaves) {
        List childNodes = tree.getChildren();
        switch (tree.getOperator()) {
            case OR: {
                Expression orResult = Expressions.alwaysFalse();
                for (ExpressionTree child : childNodes) {
                    orResult = Expressions.or(orResult, HiveIcebergFilterFactory.translate(child, leaves));
                }
                return orResult;
            }
            case AND: {
                Expression result = Expressions.alwaysTrue();
                for (ExpressionTree child : childNodes) {
                    result = Expressions.and(result, HiveIcebergFilterFactory.translate(child, leaves));
                }
                return result;
            }
            case NOT: {
                return Expressions.not(HiveIcebergFilterFactory.translate((ExpressionTree)childNodes.get(0), leaves));
            }
            case LEAF: {
                if (tree.getLeaf() >= leaves.size()) {
                    throw new UnsupportedOperationException("No more leaves are available");
                }
                return HiveIcebergFilterFactory.translateLeaf(leaves.get(tree.getLeaf()));
            }
            case CONSTANT: {
                throw new UnsupportedOperationException("CONSTANT operator is not supported");
            }
        }
        throw new UnsupportedOperationException("Unknown operator: " + String.valueOf(tree.getOperator()));
    }

    private static Expression translateLeaf(PredicateLeaf leaf) {
        TransformSpec transformSpec = TransformSpec.fromStringWithColumnName((String)leaf.getColumnName());
        String columnName = transformSpec.getColumnName();
        UnboundTerm column = (UnboundTerm)ObjectUtils.defaultIfNull(HiveIcebergFilterFactory.toTerm(columnName, transformSpec), Expressions.ref(columnName));
        switch (leaf.getOperator()) {
            case EQUALS: {
                Object literal = HiveIcebergFilterFactory.leafToLiteral(leaf, transformSpec);
                return NaNUtil.isNaN(literal) ? Expressions.isNaN(column) : Expressions.equal(column, literal);
            }
            case LESS_THAN: {
                return Expressions.lessThan(column, HiveIcebergFilterFactory.leafToLiteral(leaf, transformSpec));
            }
            case LESS_THAN_EQUALS: {
                return Expressions.lessThanOrEqual(column, HiveIcebergFilterFactory.leafToLiteral(leaf, transformSpec));
            }
            case IN: {
                return Expressions.in(column, HiveIcebergFilterFactory.leafToLiteralList(leaf));
            }
            case BETWEEN: {
                List<Object> icebergLiterals = HiveIcebergFilterFactory.leafToLiteralList(leaf);
                if (icebergLiterals.size() == 2) {
                    return Expressions.and(Expressions.greaterThanOrEqual(column, icebergLiterals.get(0)), Expressions.lessThanOrEqual(column, icebergLiterals.get(1)));
                }
                return Expressions.alwaysTrue();
            }
            case IS_NULL: {
                return Expressions.isNull(column);
            }
        }
        throw new UnsupportedOperationException("Unknown operator: " + String.valueOf(leaf.getOperator()));
    }

    public static UnboundTerm<Object> toTerm(String columnName, TransformSpec transformSpec) {
        if (transformSpec == null) {
            return null;
        }
        switch (transformSpec.getTransformType()) {
            case YEAR: {
                return Expressions.year(columnName);
            }
            case MONTH: {
                return Expressions.month(columnName);
            }
            case DAY: {
                return Expressions.day(columnName);
            }
            case HOUR: {
                return Expressions.hour(columnName);
            }
            case TRUNCATE: {
                return Expressions.truncate(columnName, transformSpec.getTransformParam());
            }
            case BUCKET: {
                return Expressions.bucket(columnName, transformSpec.getTransformParam());
            }
            case IDENTITY: {
                return null;
            }
        }
        throw new UnsupportedOperationException("Unknown transformSpec: " + String.valueOf(transformSpec));
    }

    private static Object leafToLiteral(PredicateLeaf leaf, TransformSpec transform) {
        switch (leaf.getType()) {
            case LONG: 
            case BOOLEAN: 
            case FLOAT: {
                return leaf.getLiteral();
            }
            case STRING: {
                return HiveIcebergFilterFactory.convertLiteral(leaf.getLiteral(), transform);
            }
            case DATE: {
                if (leaf.getLiteral() instanceof Date) {
                    return HiveIcebergFilterFactory.daysFromDate((Date)leaf.getLiteral());
                }
                return HiveIcebergFilterFactory.daysFromTimestamp((Timestamp)leaf.getLiteral());
            }
            case TIMESTAMP: {
                return HiveIcebergFilterFactory.microsFromTimestamp((Timestamp)LITERAL_FIELD.get(leaf));
            }
            case DECIMAL: {
                return HiveIcebergFilterFactory.hiveDecimalToBigDecimal((HiveDecimalWritable)leaf.getLiteral());
            }
        }
        throw new UnsupportedOperationException("Unknown type: " + String.valueOf(leaf.getType()));
    }

    private static Object convertLiteral(Object literal, TransformSpec transform) {
        if (transform == null) {
            return literal;
        }
        try {
            switch (transform.getTransformType()) {
                case YEAR: {
                    return HiveIcebergFilterFactory.parseYearToTransformYear(literal.toString());
                }
                case MONTH: {
                    return HiveIcebergFilterFactory.parseMonthToTransformMonth(literal.toString());
                }
                case DAY: {
                    return HiveIcebergFilterFactory.parseDayToTransformMonth(literal.toString());
                }
                case HOUR: {
                    return HiveIcebergFilterFactory.parseHourToTransformHour(literal.toString());
                }
                case TRUNCATE: {
                    return Transforms.truncate(transform.getTransformParam()).bind(Types.StringType.get()).apply(literal.toString());
                }
                case BUCKET: {
                    return Transforms.bucket(transform.getTransformParam()).bind(Types.StringType.get()).apply((Integer)((Object)literal.toString()));
                }
                case IDENTITY: {
                    return literal;
                }
            }
            throw new UnsupportedOperationException("Unknown transform: " + String.valueOf(transform.getTransformType()));
        }
        catch (IllegalStateException | NumberFormatException | DateTimeException e) {
            throw new RuntimeException(String.format("Unable to parse value '%s' as '%s' transform value", literal.toString(), transform));
        }
    }

    public static Object convertPartitionLiteral(Object literal, TransformSpec transform) {
        if (transform == null) {
            return literal;
        }
        try {
            switch (transform.getTransformType()) {
                case YEAR: {
                    return HiveIcebergFilterFactory.parseYearToTransformYear(literal.toString());
                }
                case MONTH: {
                    return HiveIcebergFilterFactory.parseMonthToTransformMonth(literal.toString());
                }
                case HOUR: {
                    return HiveIcebergFilterFactory.parseHourToTransformHour(literal.toString());
                }
                case DAY: 
                case TRUNCATE: 
                case BUCKET: 
                case IDENTITY: {
                    return literal;
                }
            }
            throw new UnsupportedOperationException("Unknown transform: " + String.valueOf(transform.getTransformType()));
        }
        catch (IllegalStateException | NumberFormatException | DateTimeException e) {
            throw new RuntimeException(String.format("Unable to parse value '%s' as '%s' transform value", literal.toString(), transform));
        }
    }

    private static Integer parseYearToTransformYear(String yearStr) {
        int year = Integer.parseInt(yearStr);
        return year - 1970;
    }

    private static Integer parseMonthToTransformMonth(String monthStr) {
        String[] parts = monthStr.split("-", -1);
        Preconditions.checkState(parts.length == 2);
        int year = Integer.parseInt(parts[0]);
        int month = Integer.parseInt(parts[1]);
        int years = year - 1970;
        int months = month - 1;
        return years * 12 + months;
    }

    private static Integer parseDayToTransformMonth(String monthStr) {
        Literal days = Literal.of(monthStr).to(Types.DateType.get());
        return (Integer)days.value();
    }

    private static Integer parseHourToTransformHour(String hourStr) {
        OffsetDateTime epoch = Instant.ofEpochSecond(0L).atOffset(ZoneOffset.UTC);
        String[] parts = hourStr.split("-", -1);
        Preconditions.checkState(parts.length == 4);
        int year = Integer.parseInt(parts[0]);
        int month = Integer.parseInt(parts[1]);
        int day = Integer.parseInt(parts[2]);
        int hour = Integer.parseInt(parts[3]);
        OffsetDateTime datetime = OffsetDateTime.of(LocalDateTime.of(year, month, day, hour, 0), ZoneOffset.UTC);
        return (int)ChronoUnit.HOURS.between(epoch, datetime);
    }

    private static List<Object> leafToLiteralList(PredicateLeaf leaf) {
        switch (leaf.getType()) {
            case LONG: 
            case BOOLEAN: 
            case FLOAT: 
            case STRING: {
                return leaf.getLiteralList();
            }
            case DATE: {
                return leaf.getLiteralList().stream().map(value -> HiveIcebergFilterFactory.daysFromDate((Date)value)).collect(Collectors.toList());
            }
            case DECIMAL: {
                return leaf.getLiteralList().stream().map(value -> HiveIcebergFilterFactory.hiveDecimalToBigDecimal((HiveDecimalWritable)value)).collect(Collectors.toList());
            }
            case TIMESTAMP: {
                return leaf.getLiteralList().stream().map(value -> HiveIcebergFilterFactory.microsFromTimestamp((Timestamp)value)).collect(Collectors.toList());
            }
        }
        throw new UnsupportedOperationException("Unknown type: " + String.valueOf(leaf.getType()));
    }

    private static BigDecimal hiveDecimalToBigDecimal(HiveDecimalWritable hiveDecimalWritable) {
        return hiveDecimalWritable.getHiveDecimal().bigDecimalValue().setScale(hiveDecimalWritable.scale());
    }

    private static int daysFromDate(Date date) {
        return DateTimeUtil.daysFromDate(date.toLocalDate());
    }

    private static int daysFromTimestamp(Timestamp timestamp) {
        return DateTimeUtil.daysFromDate(timestamp.toLocalDateTime().toLocalDate());
    }

    private static long microsFromTimestamp(Timestamp timestamp) {
        return DateTimeUtil.microsFromInstant(timestamp.toInstant());
    }
}

