/*
 * Decompiled with CFR 0.152.
 */
package org.apache.amoro.server.optimizing;

import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.LinkedTransferQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.apache.amoro.AmoroTable;
import org.apache.amoro.OptimizerProperties;
import org.apache.amoro.ServerTableIdentifier;
import org.apache.amoro.api.BlockableOperation;
import org.apache.amoro.api.OptimizingTaskId;
import org.apache.amoro.exception.OptimizingClosedException;
import org.apache.amoro.exception.PersistenceException;
import org.apache.amoro.optimizing.MetricsSummary;
import org.apache.amoro.optimizing.OptimizingType;
import org.apache.amoro.optimizing.RewriteFilesInput;
import org.apache.amoro.optimizing.RewriteStageTask;
import org.apache.amoro.optimizing.plan.AbstractOptimizingPlanner;
import org.apache.amoro.process.ProcessStatus;
import org.apache.amoro.process.StagedTaskDescriptor;
import org.apache.amoro.resource.ResourceGroup;
import org.apache.amoro.server.catalog.CatalogManager;
import org.apache.amoro.server.manager.MetricManager;
import org.apache.amoro.server.optimizing.KeyedTableCommit;
import org.apache.amoro.server.optimizing.OptimizerGroupMetrics;
import org.apache.amoro.server.optimizing.OptimizingProcess;
import org.apache.amoro.server.optimizing.OptimizingStatus;
import org.apache.amoro.server.optimizing.SchedulingPolicy;
import org.apache.amoro.server.optimizing.TaskRuntime;
import org.apache.amoro.server.optimizing.UnKeyedTableCommit;
import org.apache.amoro.server.persistence.PersistentBase;
import org.apache.amoro.server.persistence.TaskFilesPersistence;
import org.apache.amoro.server.persistence.mapper.OptimizingMapper;
import org.apache.amoro.server.persistence.mapper.TableBlockerMapper;
import org.apache.amoro.server.resource.OptimizerInstance;
import org.apache.amoro.server.resource.QuotaProvider;
import org.apache.amoro.server.table.TableRuntime;
import org.apache.amoro.server.table.blocker.TableBlocker;
import org.apache.amoro.server.utils.IcebergTableUtil;
import org.apache.amoro.shade.guava32.com.google.common.annotations.VisibleForTesting;
import org.apache.amoro.shade.guava32.com.google.common.base.Preconditions;
import org.apache.amoro.shade.guava32.com.google.common.collect.Lists;
import org.apache.amoro.shade.guava32.com.google.common.collect.Maps;
import org.apache.amoro.table.MixedTable;
import org.apache.amoro.table.TableIdentifier;
import org.apache.amoro.utils.CompatiblePropertyUtil;
import org.apache.amoro.utils.ExceptionUtil;
import org.apache.amoro.utils.MixedDataFiles;
import org.apache.amoro.utils.TablePropertyUtil;
import org.apache.iceberg.PartitionSpec;
import org.apache.iceberg.StructLike;
import org.apache.iceberg.data.GenericRecord;
import org.apache.iceberg.types.Types;
import org.apache.iceberg.util.StructLikeMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class OptimizingQueue
extends PersistentBase {
    private static final Logger LOG = LoggerFactory.getLogger(OptimizingQueue.class);
    private final QuotaProvider quotaProvider;
    private final Queue<TableOptimizingProcess> tableQueue = new LinkedTransferQueue<TableOptimizingProcess>();
    private final Queue<TaskRuntime<?>> retryTaskQueue = new LinkedTransferQueue();
    private final SchedulingPolicy scheduler;
    private final CatalogManager catalogManager;
    private final Executor planExecutor;
    private final Set<ServerTableIdentifier> planningTables = new HashSet<ServerTableIdentifier>();
    private final Lock scheduleLock = new ReentrantLock();
    private final Condition planningCompleted = this.scheduleLock.newCondition();
    private final int maxPlanningParallelism;
    private final OptimizerGroupMetrics metrics;
    private ResourceGroup optimizerGroup;

    public OptimizingQueue(CatalogManager catalogManager, ResourceGroup optimizerGroup, QuotaProvider quotaProvider, Executor planExecutor, List<TableRuntime> tableRuntimeList, int maxPlanningParallelism) {
        Preconditions.checkNotNull((Object)optimizerGroup, (Object)"Optimizer group can not be null");
        this.planExecutor = planExecutor;
        this.optimizerGroup = optimizerGroup;
        this.quotaProvider = quotaProvider;
        this.scheduler = new SchedulingPolicy(optimizerGroup);
        this.catalogManager = catalogManager;
        this.maxPlanningParallelism = maxPlanningParallelism;
        this.metrics = new OptimizerGroupMetrics(optimizerGroup.getName(), MetricManager.getInstance().getGlobalRegistry(), this);
        this.metrics.register();
        tableRuntimeList.forEach(this::initTableRuntime);
    }

    private void initTableRuntime(TableRuntime tableRuntime) {
        if (tableRuntime.getOptimizingStatus().isProcessing() && tableRuntime.getProcessId() != 0L) {
            tableRuntime.recover(new TableOptimizingProcess(tableRuntime));
        }
        if (tableRuntime.isOptimizingEnabled()) {
            OptimizingProcess process;
            tableRuntime.resetTaskQuotas(System.currentTimeMillis() - 3600000L);
            if (tableRuntime.getOptimizingStatus() == OptimizingStatus.COMMITTING && (process = tableRuntime.getOptimizingProcess()) != null) {
                LOG.warn("Close the committing process {} on table {}", (Object)process.getProcessId(), (Object)tableRuntime.getTableIdentifier());
                process.close();
            }
            if (!tableRuntime.getOptimizingStatus().isProcessing()) {
                this.scheduler.addTable(tableRuntime);
            } else {
                this.tableQueue.offer(new TableOptimizingProcess(tableRuntime));
            }
        } else {
            OptimizingProcess process = tableRuntime.getOptimizingProcess();
            if (process != null) {
                process.close();
            }
        }
    }

    public String getContainerName() {
        return this.optimizerGroup.getContainer();
    }

    public void refreshTable(TableRuntime tableRuntime) {
        if (tableRuntime.isOptimizingEnabled() && !tableRuntime.getOptimizingStatus().isProcessing()) {
            LOG.info("Bind queue {} success with table {}", (Object)this.optimizerGroup.getName(), (Object)tableRuntime.getTableIdentifier());
            tableRuntime.resetTaskQuotas(System.currentTimeMillis() - 3600000L);
            this.scheduler.addTable(tableRuntime);
        }
    }

    public void releaseTable(TableRuntime tableRuntime) {
        this.scheduler.removeTable(tableRuntime);
        List processList = this.tableQueue.stream().filter(process -> ((TableOptimizingProcess)process).tableRuntime == tableRuntime).collect(Collectors.toList());
        for (OptimizingProcess process2 : processList) {
            process2.close();
            this.clearProcess(process2);
        }
        LOG.info("Release queue {} with table {}", (Object)this.optimizerGroup.getName(), (Object)tableRuntime.getTableIdentifier());
    }

    public boolean containsTable(ServerTableIdentifier identifier) {
        return this.scheduler.getTableRuntime(identifier) != null;
    }

    private void clearProcess(OptimizingProcess optimizingProcess) {
        this.tableQueue.removeIf(process -> process.getProcessId() == optimizingProcess.getProcessId());
        this.retryTaskQueue.removeIf(taskRuntime -> taskRuntime.getTaskId().getProcessId() == optimizingProcess.getProcessId());
    }

    public TaskRuntime<?> pollTask(long maxWaitTime) {
        long deadline = this.calculateDeadline(maxWaitTime);
        TaskRuntime<?> task = this.fetchTask();
        while (task == null && this.waitTask(deadline)) {
            task = this.fetchTask();
        }
        return task;
    }

    private long calculateDeadline(long maxWaitTime) {
        long deadline = System.currentTimeMillis() + maxWaitTime;
        return deadline <= 0L ? Long.MAX_VALUE : deadline;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean waitTask(long waitDeadline) {
        this.scheduleLock.lock();
        try {
            long currentTime = System.currentTimeMillis();
            this.scheduleTableIfNecessary(currentTime);
            boolean bl = waitDeadline > currentTime && this.planningCompleted.await(waitDeadline - currentTime, TimeUnit.MILLISECONDS);
            return bl;
        }
        catch (InterruptedException e) {
            LOG.error("Schedule table interrupted", (Throwable)e);
            boolean bl = false;
            return bl;
        }
        finally {
            this.scheduleLock.unlock();
        }
    }

    private TaskRuntime<?> fetchTask() {
        TaskRuntime<?> task = this.retryTaskQueue.poll();
        return task != null ? task : this.fetchScheduledTask();
    }

    private TaskRuntime<?> fetchScheduledTask() {
        return this.tableQueue.stream().map(TableOptimizingProcess::poll).filter(Objects::nonNull).findFirst().orElse(null);
    }

    private void scheduleTableIfNecessary(long startTime) {
        if (this.planningTables.size() < this.maxPlanningParallelism) {
            HashSet<ServerTableIdentifier> skipTables = new HashSet<ServerTableIdentifier>(this.planningTables);
            this.skipBlockedTables(skipTables);
            Optional.ofNullable(this.scheduler.scheduleTable(skipTables)).ifPresent(tableRuntime -> this.triggerAsyncPlanning((TableRuntime)tableRuntime, (Set<ServerTableIdentifier>)skipTables, startTime));
        }
    }

    private void skipBlockedTables(Set<ServerTableIdentifier> skipTables) {
        List tableBlockerList = this.getAs(TableBlockerMapper.class, mapper -> mapper.selectAllBlockers(System.currentTimeMillis()));
        HashMap identifierMap = Maps.newHashMap();
        for (ServerTableIdentifier identifier : this.scheduler.getTableRuntimeMap().keySet()) {
            identifierMap.put(identifier.getIdentifier(), identifier);
        }
        tableBlockerList.stream().filter(blocker -> TableBlocker.conflict(BlockableOperation.OPTIMIZE, blocker)).map(blocker -> TableIdentifier.of((String)blocker.getCatalog(), (String)blocker.getDatabase(), (String)blocker.getTableName())).map(identifierMap::get).filter(Objects::nonNull).forEach(skipTables::add);
    }

    private void triggerAsyncPlanning(TableRuntime tableRuntime, Set<ServerTableIdentifier> skipTables, long startTime) {
        LOG.info("Trigger planning table {} by policy {}", (Object)tableRuntime.getTableIdentifier(), (Object)this.scheduler.name());
        this.planningTables.add(tableRuntime.getTableIdentifier());
        CompletableFuture.supplyAsync(() -> this.planInternal(tableRuntime), this.planExecutor).whenComplete((process, throwable) -> {
            if (throwable != null) {
                LOG.error("Failed to plan table {}", (Object)tableRuntime.getTableIdentifier(), throwable);
            }
            long currentTime = System.currentTimeMillis();
            this.scheduleLock.lock();
            try {
                tableRuntime.setLastPlanTime(currentTime);
                this.planningTables.remove(tableRuntime.getTableIdentifier());
                if (process != null) {
                    this.tableQueue.offer((TableOptimizingProcess)process);
                    String skipIds = skipTables.stream().map(ServerTableIdentifier::getId).sorted().map(item -> item + "").collect(Collectors.joining(","));
                    LOG.info("Completed planning on table {} with {} tasks with a total cost of {} ms, skipping {} tables.", new Object[]{tableRuntime.getTableIdentifier(), ((TableOptimizingProcess)process).getTaskMap().size(), currentTime - startTime, skipTables.size()});
                    LOG.debug("Skipped planning table IDs:{}", (Object)skipIds);
                } else if (throwable == null) {
                    LOG.info("Skipping planning table {} with a total cost of {} ms.", (Object)tableRuntime.getTableIdentifier(), (Object)(currentTime - startTime));
                }
                this.planningCompleted.signalAll();
            }
            finally {
                this.scheduleLock.unlock();
            }
        });
    }

    private TableOptimizingProcess planInternal(TableRuntime tableRuntime) {
        tableRuntime.beginPlanning();
        try {
            ServerTableIdentifier identifier = tableRuntime.getTableIdentifier();
            AmoroTable<?> table = this.catalogManager.loadTable(identifier.getIdentifier());
            AbstractOptimizingPlanner planner = IcebergTableUtil.createOptimizingPlanner(tableRuntime.refresh(table), (MixedTable)table.originalTable(), this.getAvailableCore(), this.maxInputSizePerThread());
            if (planner.isNecessary()) {
                return new TableOptimizingProcess(planner, tableRuntime);
            }
            tableRuntime.completeEmptyProcess();
            return null;
        }
        catch (Throwable throwable) {
            tableRuntime.planFailed();
            LOG.error("Planning table {} failed", (Object)tableRuntime.getTableIdentifier(), (Object)throwable);
            throw throwable;
        }
    }

    public TaskRuntime<?> getTask(OptimizingTaskId taskId) {
        return this.tableQueue.stream().filter(p -> p.getProcessId() == taskId.getProcessId()).findFirst().map(p -> (TaskRuntime)((TableOptimizingProcess)p).getTaskMap().get(taskId)).orElse(null);
    }

    public List<TaskRuntime<?>> collectTasks() {
        return this.tableQueue.stream().flatMap(p -> ((TableOptimizingProcess)p).getTaskMap().values().stream()).collect(Collectors.toList());
    }

    public List<TaskRuntime<?>> collectTasks(Predicate<TaskRuntime<?>> predicate) {
        return this.tableQueue.stream().flatMap(p -> ((TableOptimizingProcess)p).getTaskMap().values().stream()).filter(predicate).collect(Collectors.toList());
    }

    public void retryTask(TaskRuntime<?> taskRuntime) {
        taskRuntime.reset();
        this.retryTaskQueue.offer(taskRuntime);
    }

    public ResourceGroup getOptimizerGroup() {
        return this.optimizerGroup;
    }

    public void updateOptimizerGroup(ResourceGroup optimizerGroup) {
        Preconditions.checkArgument((boolean)this.optimizerGroup.getName().equals(optimizerGroup.getName()), (Object)"optimizer group name mismatch");
        this.optimizerGroup = optimizerGroup;
        this.scheduler.setTableSorterIfNeeded(optimizerGroup);
    }

    public void addOptimizer(OptimizerInstance optimizerInstance) {
        this.metrics.addOptimizer(optimizerInstance);
    }

    public void removeOptimizer(OptimizerInstance optimizerInstance) {
        this.metrics.removeOptimizer(optimizerInstance);
    }

    public void dispose() {
        this.metrics.unregister();
    }

    private double getAvailableCore() {
        return Math.max(this.quotaProvider.getTotalQuota(this.optimizerGroup.getName()), 1);
    }

    private long maxInputSizePerThread() {
        return CompatiblePropertyUtil.propertyAsLong((Map)this.optimizerGroup.getProperties(), (String)"max-input-file-size-per-thread", (long)OptimizerProperties.MAX_INPUT_FILE_SIZE_PER_THREAD_DEFAULT);
    }

    @VisibleForTesting
    SchedulingPolicy getSchedulingPolicy() {
        return this.scheduler;
    }

    private class TableOptimizingProcess
    implements OptimizingProcess {
        private final Lock lock = new ReentrantLock();
        private final long processId;
        private final OptimizingType optimizingType;
        private final TableRuntime tableRuntime;
        private final long planTime;
        private final long targetSnapshotId;
        private final long targetChangeSnapshotId;
        private final Map<OptimizingTaskId, TaskRuntime<RewriteStageTask>> taskMap = Maps.newHashMap();
        private final Queue<TaskRuntime<RewriteStageTask>> taskQueue = new LinkedList<TaskRuntime<RewriteStageTask>>();
        private volatile ProcessStatus status = ProcessStatus.RUNNING;
        private volatile String failedReason;
        private long endTime = 0L;
        private Map<String, Long> fromSequence = Maps.newHashMap();
        private Map<String, Long> toSequence = Maps.newHashMap();
        private boolean hasCommitted = false;

        public TaskRuntime<?> poll() {
            if (this.lock.tryLock()) {
                try {
                    TaskRuntime<RewriteStageTask> taskRuntime = this.status != ProcessStatus.CLOSED && this.status != ProcessStatus.FAILED ? this.taskQueue.poll() : null;
                    return taskRuntime;
                }
                finally {
                    this.lock.unlock();
                }
            }
            return null;
        }

        public TableOptimizingProcess(AbstractOptimizingPlanner planner, TableRuntime tableRuntime) {
            this.processId = planner.getProcessId();
            this.tableRuntime = tableRuntime;
            this.optimizingType = planner.getOptimizingType();
            this.planTime = planner.getPlanTime();
            this.targetSnapshotId = planner.getTargetSnapshotId();
            this.targetChangeSnapshotId = planner.getTargetChangeSnapshotId();
            this.loadTaskRuntimes(planner.planTasks());
            this.fromSequence = planner.getFromSequence();
            this.toSequence = planner.getToSequence();
            this.beginAndPersistProcess();
        }

        public TableOptimizingProcess(TableRuntime tableRuntime) {
            this.processId = tableRuntime.getProcessId();
            this.tableRuntime = tableRuntime;
            this.optimizingType = tableRuntime.getOptimizingType();
            this.targetSnapshotId = tableRuntime.getTargetSnapshotId();
            this.targetChangeSnapshotId = tableRuntime.getTargetChangeSnapshotId();
            this.planTime = tableRuntime.getLastPlanTime();
            if (tableRuntime.getFromSequence() != null) {
                this.fromSequence = tableRuntime.getFromSequence();
            }
            if (tableRuntime.getToSequence() != null) {
                this.toSequence = tableRuntime.getToSequence();
            }
            if (this.status != ProcessStatus.CLOSED) {
                tableRuntime.recover(this);
            }
            this.loadTaskRuntimes(this);
        }

        @Override
        public long getProcessId() {
            return this.processId;
        }

        @Override
        public OptimizingType getOptimizingType() {
            return this.optimizingType;
        }

        @Override
        public ProcessStatus getStatus() {
            return this.status;
        }

        @Override
        public void close() {
            this.lock.lock();
            try {
                if (this.status != ProcessStatus.RUNNING) {
                    return;
                }
                this.status = ProcessStatus.CLOSED;
                this.endTime = System.currentTimeMillis();
                this.persistAndSetCompleted(false);
            }
            finally {
                this.lock.unlock();
            }
        }

        private void acceptResult(TaskRuntime<?> taskRuntime) {
            this.lock.lock();
            try {
                try {
                    this.tableRuntime.addTaskQuota(taskRuntime.getCurrentQuota());
                }
                catch (Throwable throwable) {
                    LOG.warn("{} failed to add task quota {}, ignore it", new Object[]{this.tableRuntime.getTableIdentifier(), taskRuntime.getTaskId(), throwable});
                }
                if (taskRuntime.getStatus() == TaskRuntime.Status.CANCELED) {
                    return;
                }
                if (this.isClosed()) {
                    throw new OptimizingClosedException(this.processId);
                }
                if (taskRuntime.getStatus() == TaskRuntime.Status.SUCCESS) {
                    if (this.allTasksPrepared() && this.tableRuntime.getOptimizingStatus().isProcessing() && this.tableRuntime.getOptimizingStatus() != OptimizingStatus.COMMITTING) {
                        this.tableRuntime.beginCommitting();
                    }
                } else if (taskRuntime.getStatus() == TaskRuntime.Status.FAILED) {
                    if (taskRuntime.getRetry() < this.tableRuntime.getMaxExecuteRetryCount()) {
                        LOG.info("Put task {} to retry queue, because {}", (Object)taskRuntime.getTaskId(), (Object)taskRuntime.getFailReason());
                        OptimizingQueue.this.retryTask(taskRuntime);
                    } else {
                        LOG.info("Task {} has reached the max execute retry count. Process {} failed.", (Object)taskRuntime.getTaskId(), (Object)this.processId);
                        this.failedReason = taskRuntime.getFailReason();
                        this.status = ProcessStatus.FAILED;
                        this.endTime = taskRuntime.getEndTime();
                        this.persistAndSetCompleted(false);
                    }
                }
            }
            finally {
                this.lock.unlock();
            }
        }

        @Override
        public boolean isClosed() {
            return this.status == ProcessStatus.CLOSED;
        }

        @Override
        public long getPlanTime() {
            return this.planTime;
        }

        @Override
        public long getDuration() {
            long dur = this.endTime == 0L ? System.currentTimeMillis() - this.planTime : this.endTime - this.planTime;
            return Math.max(0L, dur);
        }

        @Override
        public long getTargetSnapshotId() {
            return this.targetSnapshotId;
        }

        @Override
        public long getTargetChangeSnapshotId() {
            return this.targetChangeSnapshotId;
        }

        public String getFailedReason() {
            return this.failedReason;
        }

        private Map<OptimizingTaskId, TaskRuntime<RewriteStageTask>> getTaskMap() {
            return this.taskMap;
        }

        private boolean allTasksPrepared() {
            if (!this.taskMap.isEmpty()) {
                return this.taskMap.values().stream().allMatch(t -> t.getStatus() == TaskRuntime.Status.SUCCESS);
            }
            return false;
        }

        @Override
        public long getRunningQuotaTime(long calculatingStartTime, long calculatingEndTime) {
            return this.taskMap.values().stream().filter(t -> !t.finished()).mapToLong(task -> task.getQuotaTime(calculatingStartTime, calculatingEndTime)).sum();
        }

        @Override
        public void commit() {
            LOG.debug("{} get {} tasks of {} partitions to commit", new Object[]{this.tableRuntime.getTableIdentifier(), this.taskMap.size(), this.taskMap.values()});
            this.lock.lock();
            try {
                if (this.hasCommitted) {
                    LOG.warn("{} has already committed, give up", (Object)this.tableRuntime.getTableIdentifier());
                    try {
                        this.persistAndSetCompleted(this.status == ProcessStatus.SUCCESS);
                    }
                    catch (Exception exception) {
                        // empty catch block
                    }
                    throw new IllegalStateException("repeat commit, and last error " + this.failedReason);
                }
                try {
                    this.hasCommitted = true;
                    this.buildCommit().commit();
                    this.status = ProcessStatus.SUCCESS;
                    this.endTime = System.currentTimeMillis();
                    this.persistAndSetCompleted(true);
                }
                catch (PersistenceException e) {
                    LOG.warn("{} failed to persist process completed, will retry next commit", (Object)this.tableRuntime.getTableIdentifier(), (Object)e);
                }
                catch (Throwable t) {
                    LOG.error("{} Commit optimizing failed ", (Object)this.tableRuntime.getTableIdentifier(), (Object)t);
                    this.status = ProcessStatus.FAILED;
                    this.failedReason = ExceptionUtil.getErrorMessage((Throwable)t, (int)4000);
                    this.endTime = System.currentTimeMillis();
                    this.persistAndSetCompleted(false);
                }
            }
            finally {
                this.lock.unlock();
            }
        }

        @Override
        public MetricsSummary getSummary() {
            List taskSummaries = this.taskMap.values().stream().map(TaskRuntime::getTaskDescriptor).map(StagedTaskDescriptor::getSummary).collect(Collectors.toList());
            return new MetricsSummary(taskSummaries);
        }

        private UnKeyedTableCommit buildCommit() {
            MixedTable table = (MixedTable)OptimizingQueue.this.catalogManager.loadTable(this.tableRuntime.getTableIdentifier().getIdentifier()).originalTable();
            if (table.isUnkeyedTable()) {
                return new UnKeyedTableCommit(this.targetSnapshotId, table, this.taskMap.values());
            }
            return new KeyedTableCommit(table, this.taskMap.values(), this.targetSnapshotId, this.convertPartitionSequence(table, this.fromSequence), this.convertPartitionSequence(table, this.toSequence));
        }

        private StructLikeMap<Long> convertPartitionSequence(MixedTable table, Map<String, Long> partitionSequence) {
            PartitionSpec spec = table.spec();
            StructLikeMap results = StructLikeMap.create((Types.StructType)spec.partitionType());
            partitionSequence.forEach((partition, sequence) -> {
                if (spec.isUnpartitioned()) {
                    results.put(TablePropertyUtil.EMPTY_STRUCT, sequence);
                } else {
                    GenericRecord partitionData = MixedDataFiles.data((PartitionSpec)spec, (String)partition);
                    results.put((StructLike)partitionData, sequence);
                }
            });
            return results;
        }

        private void beginAndPersistProcess() {
            OptimizingQueue.this.doAsTransaction(new Runnable[]{() -> OptimizingQueue.this.doAs(OptimizingMapper.class, mapper -> mapper.insertOptimizingProcess(this.tableRuntime.getTableIdentifier(), this.processId, this.targetSnapshotId, this.targetChangeSnapshotId, this.status, this.optimizingType, this.planTime, this.getSummary(), this.fromSequence, this.toSequence)), () -> OptimizingQueue.this.doAs(OptimizingMapper.class, mapper -> mapper.insertTaskRuntimes(Lists.newArrayList(this.taskMap.values()))), () -> TaskFilesPersistence.persistTaskInputs(this.processId, this.taskMap.values()), () -> this.tableRuntime.beginProcess(this)});
        }

        private void persistAndSetCompleted(boolean success) {
            OptimizingQueue.this.doAsTransaction(new Runnable[]{() -> {
                if (!success) {
                    this.cancelTasks();
                }
            }, () -> OptimizingQueue.this.doAs(OptimizingMapper.class, mapper -> mapper.updateOptimizingProcess(this.tableRuntime.getTableIdentifier().getId(), this.processId, this.status, this.endTime, this.getSummary(), this.getFailedReason())), () -> this.tableRuntime.completeProcess(success), () -> OptimizingQueue.this.clearProcess(this)});
        }

        private void cancelTasks() {
            this.taskMap.values().forEach(TaskRuntime::tryCanceling);
        }

        private void loadTaskRuntimes(OptimizingProcess optimizingProcess) {
            try {
                List taskRuntimes = (List)OptimizingQueue.this.getAs(OptimizingMapper.class, mapper -> mapper.selectTaskRuntimes(this.tableRuntime.getTableIdentifier().getId(), this.processId));
                Map<Integer, RewriteFilesInput> inputs = TaskFilesPersistence.loadTaskInputs(this.processId);
                taskRuntimes.forEach(taskRuntime -> {
                    taskRuntime.getCompletedFuture().whenCompleted(() -> this.acceptResult((TaskRuntime<?>)taskRuntime));
                    ((RewriteStageTask)taskRuntime.getTaskDescriptor()).setInput(inputs.get(taskRuntime.getTaskId().getTaskId()));
                    this.taskMap.put(taskRuntime.getTaskId(), (TaskRuntime<RewriteStageTask>)taskRuntime);
                    if (taskRuntime.getStatus() == TaskRuntime.Status.PLANNED) {
                        this.taskQueue.offer((TaskRuntime<RewriteStageTask>)taskRuntime);
                    } else if (taskRuntime.getStatus() == TaskRuntime.Status.FAILED) {
                        OptimizingQueue.this.retryTask((TaskRuntime<?>)taskRuntime);
                    }
                });
            }
            catch (IllegalArgumentException e) {
                LOG.warn("Load task inputs failed, close the optimizing process : {}", (Object)optimizingProcess.getProcessId(), (Object)e);
                optimizingProcess.close();
            }
        }

        private void loadTaskRuntimes(List<RewriteStageTask> taskDescriptors) {
            int taskId = 1;
            for (RewriteStageTask taskDescriptor : taskDescriptors) {
                TaskRuntime<RewriteStageTask> taskRuntime = new TaskRuntime<RewriteStageTask>(new OptimizingTaskId(this.processId, taskId++), taskDescriptor);
                LOG.info("{} plan new task {}, summary {}", new Object[]{this.tableRuntime.getTableIdentifier(), taskRuntime.getTaskId(), taskRuntime.getSummary()});
                taskRuntime.getCompletedFuture().whenCompleted(() -> this.acceptResult(taskRuntime));
                this.taskMap.put(taskRuntime.getTaskId(), taskRuntime);
                this.taskQueue.offer(taskRuntime);
            }
        }
    }
}

