/*
 * Decompiled with CFR 0.152.
 */
package org.apache.baremaps.pmtiles;

import com.google.common.math.LongMath;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.List;
import org.apache.baremaps.pmtiles.Compression;
import org.apache.baremaps.pmtiles.Directories;
import org.apache.baremaps.pmtiles.Entry;
import org.apache.baremaps.pmtiles.Header;
import org.apache.baremaps.pmtiles.TileType;

class PMTilesUtils {
    private static final int HEADER_SIZE_BYTES = 127;
    private static final long[] tzValues = new long[]{0L, 1L, 5L, 21L, 85L, 341L, 1365L, 5461L, 21845L, 87381L, 349525L, 0x155555L, 0x555555L, 0x1555555L, 0x5555555L, 0x15555555L, 0x55555555L, 0x155555555L, 0x555555555L, 0x1555555555L, 0x5555555555L, 0x15555555555L, 0x55555555555L, 0x155555555555L, 0x555555555555L, 0x1555555555555L, 0x5555555555555L};

    private PMTilesUtils() {
    }

    static long toNum(long low, long high) {
        return high * 0x100000000L + low;
    }

    static long readVarIntRemainder(InputStream input, long l) throws IOException {
        long b = input.read() & 0xFF;
        long h = (b & 0x70L) >> 4;
        if (b < 128L) {
            return PMTilesUtils.toNum(l, h);
        }
        b = input.read() & 0xFF;
        h |= (b & 0x7FL) << 3;
        if (b < 128L) {
            return PMTilesUtils.toNum(l, h);
        }
        b = input.read() & 0xFF;
        h |= (b & 0x7FL) << 10;
        if (b < 128L) {
            return PMTilesUtils.toNum(l, h);
        }
        b = input.read() & 0xFF;
        h |= (b & 0x7FL) << 17;
        if (b < 128L) {
            return PMTilesUtils.toNum(l, h);
        }
        b = input.read() & 0xFF;
        h |= (b & 0x7FL) << 24;
        if (b < 128L) {
            return PMTilesUtils.toNum(l, h);
        }
        b = input.read() & 0xFF;
        h |= (b & 1L) << 31;
        if (b < 128L) {
            return PMTilesUtils.toNum(l, h);
        }
        throw new IllegalArgumentException("Expected varint not more than 10 bytes");
    }

    static int writeVarInt(OutputStream output, long value) throws IOException {
        int n = 1;
        while (value >= 128L) {
            output.write((byte)(value | 0x80L));
            value >>>= 7;
            ++n;
        }
        output.write((byte)value);
        return n;
    }

    static long readVarInt(InputStream input) throws IOException {
        long b = input.read() & 0xFF;
        long val = b & 0x7FL;
        if (b < 128L) {
            return val;
        }
        b = input.read() & 0xFF;
        val |= (b & 0x7FL) << 7;
        if (b < 128L) {
            return val;
        }
        b = input.read() & 0xFF;
        val |= (b & 0x7FL) << 14;
        if (b < 128L) {
            return val;
        }
        b = input.read() & 0xFF;
        val |= (b & 0x7FL) << 21;
        if (b < 128L) {
            return val;
        }
        return PMTilesUtils.readVarIntRemainder(input, val |= (b & 0xFL) << 28);
    }

    static void rotate(long n, long[] xy, long rx, long ry) {
        if (ry == 0L) {
            if (rx == 1L) {
                xy[0] = n - 1L - xy[0];
                xy[1] = n - 1L - xy[1];
            }
            long t = xy[0];
            xy[0] = xy[1];
            xy[1] = t;
        }
    }

    static long[] idOnLevel(int z, long pos) {
        long n = LongMath.pow((long)2L, (int)z);
        long t = pos;
        long[] xy = new long[]{0L, 0L};
        for (long s = 1L; s < n; s *= 2L) {
            long rx = 1L & t / 2L;
            long ry = 1L & (t ^ rx);
            PMTilesUtils.rotate(s, xy, rx, ry);
            xy[0] = xy[0] + s * rx;
            xy[1] = xy[1] + s * ry;
            t /= 4L;
        }
        return new long[]{z, xy[0], xy[1]};
    }

    static long zxyToTileId(int z, long x, long y) {
        if (z > 26) {
            throw new IllegalArgumentException("Tile zoom level exceeds max safe number limit (26)");
        }
        if ((double)x > Math.pow(2.0, z) - 1.0 || (double)y > Math.pow(2.0, z) - 1.0) {
            throw new IllegalArgumentException("tile x/y outside zoom level bounds");
        }
        long acc = tzValues[z];
        long n = LongMath.pow((long)2L, (int)z);
        long rx = 0L;
        long ry = 0L;
        long d = 0L;
        long[] xy = new long[]{x, y};
        for (long s = n / 2L; s > 0L; s /= 2L) {
            rx = (xy[0] & s) > 0L ? 1L : 0L;
            ry = (xy[1] & s) > 0L ? 1L : 0L;
            d += s * s * (3L * rx ^ ry);
            PMTilesUtils.rotate(s, xy, rx, ry);
        }
        return acc + d;
    }

    static long[] tileIdToZxy(long i) {
        long acc = 0L;
        for (int z = 0; z < 27; ++z) {
            long numTiles = (1L << z) * (1L << z);
            if (acc + numTiles > i) {
                return PMTilesUtils.idOnLevel(z, i - acc);
            }
            acc += numTiles;
        }
        throw new IllegalArgumentException("Tile zoom level exceeds max safe number limit (26)");
    }

    static Header deserializeHeader(InputStream input) throws IOException {
        byte[] bytes = new byte[127];
        int num = input.read(bytes);
        if (num != 127) {
            throw new IOException("Invalid header size");
        }
        ByteBuffer buffer = ByteBuffer.wrap(bytes).order(ByteOrder.LITTLE_ENDIAN);
        buffer.position(7);
        return new Header(buffer.get(), buffer.getLong(), buffer.getLong(), buffer.getLong(), buffer.getLong(), buffer.getLong(), buffer.getLong(), buffer.getLong(), buffer.getLong(), buffer.getLong(), buffer.getLong(), buffer.getLong(), buffer.get() == 1, Compression.values()[buffer.get()], Compression.values()[buffer.get()], TileType.values()[buffer.get()], buffer.get(), buffer.get(), (double)buffer.getInt() / 1.0E7, (double)buffer.getInt() / 1.0E7, (double)buffer.getInt() / 1.0E7, (double)buffer.getInt() / 1.0E7, buffer.get(), (double)buffer.getInt() / 1.0E7, (double)buffer.getInt() / 1.0E7);
    }

    static byte[] serializeHeader(Header header) {
        ByteBuffer buffer = ByteBuffer.allocate(127).order(ByteOrder.LITTLE_ENDIAN);
        buffer.put((byte)80);
        buffer.put((byte)77);
        buffer.put((byte)84);
        buffer.put((byte)105);
        buffer.put((byte)108);
        buffer.put((byte)101);
        buffer.put((byte)115);
        buffer.put((byte)header.getSpecVersion());
        buffer.putLong(header.getRootDirectoryOffset());
        buffer.putLong(header.getRootDirectoryLength());
        buffer.putLong(header.getJsonMetadataOffset());
        buffer.putLong(header.getJsonMetadataLength());
        buffer.putLong(header.getLeafDirectoryOffset());
        buffer.putLong(header.getLeafDirectoryLength());
        buffer.putLong(header.getTileDataOffset());
        buffer.putLong(header.getTileDataLength());
        buffer.putLong(header.getNumAddressedTiles());
        buffer.putLong(header.getNumTileEntries());
        buffer.putLong(header.getNumTileContents());
        buffer.put((byte)(header.isClustered() ? 1 : 0));
        buffer.put((byte)header.getInternalCompression().ordinal());
        buffer.put((byte)header.getTileCompression().ordinal());
        buffer.put((byte)header.getTileType().ordinal());
        buffer.put((byte)header.getMinZoom());
        buffer.put((byte)header.getMaxZoom());
        buffer.putInt((int)(header.getMinLon() * 1.0E7));
        buffer.putInt((int)(header.getMinLat() * 1.0E7));
        buffer.putInt((int)(header.getMaxLon() * 1.0E7));
        buffer.putInt((int)(header.getMaxLat() * 1.0E7));
        buffer.put((byte)header.getCenterZoom());
        buffer.putInt((int)(header.getCenterLon() * 1.0E7));
        buffer.putInt((int)(header.getCenterLat() * 1.0E7));
        buffer.flip();
        return buffer.array();
    }

    static void serializeEntries(OutputStream output, List<Entry> entries) throws IOException {
        ByteBuffer buffer = ByteBuffer.allocate(entries.size() * 48);
        PMTilesUtils.writeVarInt(output, entries.size());
        long lastId = 0L;
        for (Entry entry : entries) {
            PMTilesUtils.writeVarInt(output, entry.getTileId() - lastId);
            lastId = entry.getTileId();
        }
        for (Entry entry : entries) {
            PMTilesUtils.writeVarInt(output, entry.getRunLength());
        }
        for (Entry entry : entries) {
            PMTilesUtils.writeVarInt(output, entry.getLength());
        }
        for (int i = 0; i < entries.size(); ++i) {
            Entry entry;
            entry = entries.get(i);
            if (i > 0 && entry.getOffset() == entries.get(i - 1).getOffset() + entries.get(i - 1).getLength()) {
                PMTilesUtils.writeVarInt(output, 0L);
                continue;
            }
            PMTilesUtils.writeVarInt(output, entry.getOffset() + 1L);
        }
        buffer.flip();
        output.write(buffer.array(), 0, buffer.limit());
    }

    static List<Entry> deserializeEntries(InputStream buffer) throws IOException {
        long value;
        long numEntries = PMTilesUtils.readVarInt(buffer);
        ArrayList<Entry> entries = new ArrayList<Entry>((int)numEntries);
        long lastId = 0L;
        int i = 0;
        while ((long)i < numEntries) {
            value = PMTilesUtils.readVarInt(buffer);
            Entry entry = new Entry();
            entry.setTileId(lastId += value);
            entries.add(entry);
            ++i;
        }
        i = 0;
        while ((long)i < numEntries) {
            value = PMTilesUtils.readVarInt(buffer);
            ((Entry)entries.get(i)).setRunLength(value);
            ++i;
        }
        i = 0;
        while ((long)i < numEntries) {
            value = PMTilesUtils.readVarInt(buffer);
            ((Entry)entries.get(i)).setLength(value);
            ++i;
        }
        i = 0;
        while ((long)i < numEntries) {
            value = PMTilesUtils.readVarInt(buffer);
            if (value == 0L && i > 0) {
                Entry prevEntry = (Entry)entries.get(i - 1);
                ((Entry)entries.get(i)).setOffset(prevEntry.getOffset() + prevEntry.getLength());
            } else {
                ((Entry)entries.get(i)).setOffset(value - 1L);
            }
            ++i;
        }
        return entries;
    }

    static Entry findTile(List<Entry> entries, long tileId) {
        int m = 0;
        int n = entries.size() - 1;
        while (m <= n) {
            int k = n + m >> 1;
            long cmp = tileId - entries.get(k).getTileId();
            if (cmp > 0L) {
                m = k + 1;
                continue;
            }
            if (cmp < 0L) {
                n = k - 1;
                continue;
            }
            return entries.get(k);
        }
        if (n >= 0) {
            if (entries.get(n).getRunLength() == 0L) {
                return entries.get(n);
            }
            if (tileId - entries.get(n).getTileId() < entries.get(n).getRunLength()) {
                return entries.get(n);
            }
        }
        return null;
    }

    static Directories buildRootLeaves(List<Entry> entries, int leafSize, Compression compression) throws IOException {
        byte[] rootBytes;
        byte[] leavesBytes;
        ArrayList<Entry> rootEntries = new ArrayList<Entry>();
        int numLeaves = 0;
        try (ByteArrayOutputStream leavesOutput = new ByteArrayOutputStream();){
            for (int i = 0; i < entries.size(); i += leafSize) {
                ++numLeaves;
                int end = i + leafSize;
                if (i + leafSize > entries.size()) {
                    end = entries.size();
                }
                int offset = leavesOutput.size();
                try (ByteArrayOutputStream leafOutput = new ByteArrayOutputStream();){
                    try (OutputStream compressedLeafOutput = compression.compress(leafOutput);){
                        PMTilesUtils.serializeEntries(compressedLeafOutput, entries.subList(i, end));
                    }
                    int length = leafOutput.size();
                    rootEntries.add(new Entry(entries.get(i).getTileId(), offset, length, 0L));
                    leavesOutput.write(leafOutput.toByteArray());
                    continue;
                }
            }
            leavesBytes = leavesOutput.toByteArray();
        }
        try (ByteArrayOutputStream rootOutput = new ByteArrayOutputStream();){
            try (OutputStream compressedRootOutput = compression.compress(rootOutput);){
                PMTilesUtils.serializeEntries(compressedRootOutput, rootEntries);
            }
            rootBytes = rootOutput.toByteArray();
        }
        return new Directories(rootBytes, leavesBytes, numLeaves);
    }

    static Directories optimizeDirectories(List<Entry> entries, int targetRootLength, Compression compression) throws IOException {
        if (entries.size() < 16384) {
            try (ByteArrayOutputStream rootOutput = new ByteArrayOutputStream();){
                try (OutputStream compressedOutput = compression.compress(rootOutput);){
                    PMTilesUtils.serializeEntries(compressedOutput, entries);
                }
                byte[] rootBytes = rootOutput.toByteArray();
                if (rootBytes.length <= targetRootLength) {
                    Directories directories = new Directories(rootBytes, new byte[0], 0);
                    return directories;
                }
            }
        }
        double leafSize = Math.max((double)entries.size() / 3500.0, 4096.0);
        Directories directories;
        while ((directories = PMTilesUtils.buildRootLeaves(entries, (int)leafSize, compression)).getRoot().length > targetRootLength) {
            leafSize *= 1.2;
        }
        return directories;
    }
}

