/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.imapserver.netty;

import com.github.fge.lambdas.Throwing;
import io.netty.buffer.Unpooled;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelHandler;
import io.netty.channel.ChannelHandlerContext;
import io.netty.channel.ChannelInboundHandlerAdapter;
import io.netty.handler.codec.TooLongFrameException;
import io.netty.util.Attribute;
import io.netty.util.concurrent.GenericFutureListener;
import java.io.Closeable;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.time.Duration;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.function.Consumer;
import org.apache.james.imap.api.ImapMessage;
import org.apache.james.imap.api.ImapSessionState;
import org.apache.james.imap.api.display.HumanReadableText;
import org.apache.james.imap.api.message.response.ImapResponseMessage;
import org.apache.james.imap.api.message.response.StatusResponse;
import org.apache.james.imap.api.process.ImapProcessor;
import org.apache.james.imap.api.process.ImapSession;
import org.apache.james.imap.encode.ImapEncoder;
import org.apache.james.imap.encode.ImapResponseComposer;
import org.apache.james.imap.encode.ImapResponseWriter;
import org.apache.james.imap.encode.base.ImapResponseComposerImpl;
import org.apache.james.imap.main.ResponseEncoder;
import org.apache.james.imap.message.request.AbstractImapRequest;
import org.apache.james.imap.message.response.ImmutableStatusResponse;
import org.apache.james.imapserver.netty.ChannelImapResponseWriter;
import org.apache.james.imapserver.netty.IMAPMDCContext;
import org.apache.james.imapserver.netty.IMAPServer;
import org.apache.james.imapserver.netty.ImapHeartbeatHandler;
import org.apache.james.imapserver.netty.ImapMetrics;
import org.apache.james.imapserver.netty.Linearalizer;
import org.apache.james.imapserver.netty.NettyConstants;
import org.apache.james.imapserver.netty.NettyImapSession;
import org.apache.james.imapserver.netty.ReactiveThrottler;
import org.apache.james.metrics.api.Metric;
import org.apache.james.protocols.netty.Encryption;
import org.apache.james.util.MDCBuilder;
import org.apache.james.util.ReactorUtils;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.Disposable;
import reactor.core.publisher.Mono;
import reactor.core.publisher.Signal;
import reactor.util.context.ContextView;

@ChannelHandler.Sharable
public class ImapChannelUpstreamHandler
extends ChannelInboundHandlerAdapter
implements NettyConstants {
    private static final Logger LOGGER = LoggerFactory.getLogger(ImapChannelUpstreamHandler.class);
    public static final String MDC_KEY = "bound_MDC";
    private final String hello;
    private final Encryption secure;
    private final boolean compress;
    private final ImapProcessor processor;
    private final ImapEncoder encoder;
    private final ImapHeartbeatHandler heartbeatHandler;
    private final IMAPServer.AuthenticationConfiguration authenticationConfiguration;
    private final Metric imapConnectionsMetric;
    private final Metric imapCommandsMetric;
    private final boolean ignoreIDLEUponProcessing;
    private final ReactiveThrottler reactiveThrottler;

    public static ImapChannelUpstreamHandlerBuilder builder() {
        return new ImapChannelUpstreamHandlerBuilder();
    }

    public ImapChannelUpstreamHandler(String hello, ImapProcessor processor, ImapEncoder encoder, boolean compress, Encryption secure, ImapMetrics imapMetrics, IMAPServer.AuthenticationConfiguration authenticationConfiguration, boolean ignoreIDLEUponProcessing, int heartbeatIntervalSeconds, ReactiveThrottler reactiveThrottler) {
        this.hello = hello;
        this.processor = processor;
        this.encoder = encoder;
        this.secure = secure;
        this.compress = compress;
        this.authenticationConfiguration = authenticationConfiguration;
        this.imapConnectionsMetric = imapMetrics.getConnectionsMetric();
        this.imapCommandsMetric = imapMetrics.getCommandsMetric();
        this.ignoreIDLEUponProcessing = ignoreIDLEUponProcessing;
        this.heartbeatHandler = new ImapHeartbeatHandler(heartbeatIntervalSeconds, heartbeatIntervalSeconds, heartbeatIntervalSeconds);
        this.reactiveThrottler = reactiveThrottler;
    }

    public void channelActive(ChannelHandlerContext ctx) throws Exception {
        ImapSession.SessionId sessionId = ImapSession.SessionId.generate();
        NettyImapSession imapsession = new NettyImapSession(ctx.channel(), this.secure, this.compress, this.authenticationConfiguration.isSSLRequired(), this.authenticationConfiguration.isPlainAuthEnabled(), sessionId, this.authenticationConfiguration.getOidcSASLConfiguration());
        ctx.channel().attr(IMAP_SESSION_ATTRIBUTE_KEY).set((Object)imapsession);
        ctx.channel().attr(LINEARALIZER_ATTRIBUTE_KEY).set((Object)new Linearalizer());
        MDCBuilder boundMDC = IMAPMDCContext.boundMDC(ctx).addToContext("sessionId", sessionId.asString());
        imapsession.setAttribute(MDC_KEY, boundMDC);
        try (Closeable closeable = this.mdc(imapsession).build();){
            InetSocketAddress address = (InetSocketAddress)ctx.channel().remoteAddress();
            LOGGER.info("Connection established from {}", (Object)address.getAddress().getHostAddress());
            this.imapConnectionsMetric.increment();
            ChannelImapResponseWriter writer = new ChannelImapResponseWriter(ctx.channel());
            ImapResponseComposerImpl response = new ImapResponseComposerImpl((ImapResponseWriter)writer);
            response.untagged().message("OK").message(this.hello).end();
            response.flush();
            super.channelActive(ctx);
        }
    }

    private MDCBuilder mdc(ChannelHandlerContext ctx) {
        ImapSession maybeSession = (ImapSession)ctx.channel().attr(IMAP_SESSION_ATTRIBUTE_KEY).get();
        return this.mdc(maybeSession);
    }

    private MDCBuilder mdc(ImapSession imapSession) {
        return Optional.ofNullable(imapSession).map(session -> {
            MDCBuilder boundMDC = (MDCBuilder)session.getAttribute(MDC_KEY);
            return IMAPMDCContext.from(session).addToContext(boundMDC);
        }).orElseGet(MDCBuilder::create);
    }

    public void channelInactive(ChannelHandlerContext ctx) throws Exception {
        ImapSession imapSession = (ImapSession)ctx.channel().attr(IMAP_SESSION_ATTRIBUTE_KEY).getAndSet(null);
        try (Closeable closeable = this.mdc(imapSession).build();){
            InetSocketAddress address = (InetSocketAddress)ctx.channel().remoteAddress();
            LOGGER.info("Connection closed for {}", (Object)address.getAddress().getHostAddress());
            Disposable disposableAttribute = (Disposable)ctx.channel().attr(REQUEST_IN_FLIGHT_ATTRIBUTE_KEY).getAndSet(null);
            Optional.ofNullable(imapSession).map(ImapSession::logout).orElse(Mono.empty()).doFinally((Consumer)Throwing.consumer(signal -> {
                this.imapConnectionsMetric.decrement();
                super.channelInactive(ctx);
            })).subscribe(any -> {}, arg_0 -> ((ChannelHandlerContext)ctx).fireExceptionCaught(arg_0));
            Optional.ofNullable(disposableAttribute).ifPresent(Disposable::dispose);
        }
    }

    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) throws Exception {
        try (Closeable closeable = this.mdc(ctx).build();){
            LOGGER.warn("Error while processing imap request", cause);
            if (cause instanceof TooLongFrameException) {
                ChannelImapResponseWriter writer = new ChannelImapResponseWriter(ctx.channel());
                ImapResponseComposerImpl response = new ImapResponseComposerImpl((ImapResponseWriter)writer);
                response.untaggedResponse("BAD failed. Maximum command line length exceeded");
                response.flush();
            } else if (cause instanceof ReactiveThrottler.RejectedException) {
                this.manageRejectedException(ctx, (ReactiveThrottler.RejectedException)cause);
            } else {
                this.manageUnknownError(ctx);
            }
        }
    }

    private void manageRejectedException(ChannelHandlerContext ctx, ReactiveThrottler.RejectedException cause) throws IOException {
        if (cause.getImapMessage() instanceof AbstractImapRequest) {
            AbstractImapRequest req = (AbstractImapRequest)cause.getImapMessage();
            ChannelImapResponseWriter writer = new ChannelImapResponseWriter(ctx.channel());
            ImapResponseComposerImpl response = new ImapResponseComposerImpl((ImapResponseWriter)writer);
            new ResponseEncoder(this.encoder, (ImapResponseComposer)response).respond((ImapResponseMessage)new ImmutableStatusResponse(StatusResponse.Type.NO, req.getTag(), req.getCommand(), new HumanReadableText(cause.getClass().getName(), cause.getMessage()), null));
            response.flush();
        } else {
            this.manageUnknownError(ctx);
        }
    }

    private void manageUnknownError(ChannelHandlerContext ctx) {
        ImapSession imapSession = (ImapSession)ctx.channel().attr(IMAP_SESSION_ATTRIBUTE_KEY).get();
        Optional.ofNullable((Disposable)ctx.channel().attr(REQUEST_IN_FLIGHT_ATTRIBUTE_KEY).getAndSet(null)).ifPresent(Disposable::dispose);
        Optional.ofNullable(imapSession).map(ImapSession::logout).orElse(Mono.empty()).doFinally((Consumer)Throwing.consumer(signal -> {
            Channel channel = ctx.channel();
            if (channel.isActive()) {
                channel.writeAndFlush((Object)Unpooled.EMPTY_BUFFER).addListener((GenericFutureListener)ChannelFutureListener.CLOSE);
            }
            super.channelInactive(ctx);
        })).subscribe(any -> {}, e -> {
            LOGGER.error("Exception while handling errors for channel {}", (Object)ctx.channel(), e);
            Channel channel = ctx.channel();
            if (channel.isActive()) {
                channel.writeAndFlush((Object)Unpooled.EMPTY_BUFFER).addListener((GenericFutureListener)ChannelFutureListener.CLOSE);
            }
        });
    }

    public void channelRead(ChannelHandlerContext ctx, Object msg) {
        this.imapCommandsMetric.increment();
        ImapSession session = (ImapSession)ctx.channel().attr(IMAP_SESSION_ATTRIBUTE_KEY).get();
        Linearalizer linearalizer = (Linearalizer)ctx.channel().attr(LINEARALIZER_ATTRIBUTE_KEY).get();
        Attribute disposableAttribute = ctx.channel().attr(REQUEST_IN_FLIGHT_ATTRIBUTE_KEY);
        ChannelImapResponseWriter writer = new ChannelImapResponseWriter(ctx.channel());
        ImapResponseComposerImpl response = new ImapResponseComposerImpl((ImapResponseWriter)writer);
        writer.setFlushCallback(() -> ((ImapResponseComposerImpl)response).flush());
        ImapMessage message = (ImapMessage)msg;
        this.beforeIDLEUponProcessing(ctx);
        ResponseEncoder responseEncoder = new ResponseEncoder(this.encoder, (ImapResponseComposer)response);
        Disposable disposable = this.reactiveThrottler.throttle((Publisher<Void>)linearalizer.execute((Publisher<Void>)this.processor.processReactive(message, (ImapProcessor.Responder)responseEncoder, session)).doOnEach((Consumer)Throwing.consumer(signal -> {
            IOException failure;
            Channel channel;
            if (session.getState() == ImapSessionState.LOGOUT && (channel = ctx.channel()).isActive()) {
                channel.writeAndFlush((Object)Unpooled.EMPTY_BUFFER).addListener((GenericFutureListener)ChannelFutureListener.CLOSE);
            }
            if (signal.isOnComplete() && (failure = responseEncoder.getFailure()) != null) {
                try (Closeable mdc = ReactorUtils.retrieveMDCBuilder((Signal)signal).build();){
                    LOGGER.info(failure.getMessage());
                    LOGGER.debug("Failed to write {}", (Object)message, (Object)failure);
                }
                catch (IOException e) {
                    throw new RuntimeException(e);
                }
                ctx.fireExceptionCaught((Throwable)failure);
            }
            if (signal.isOnComplete() || signal.isOnError()) {
                this.afterIDLEUponProcessing(ctx);
                if (message instanceof Closeable) {
                    ((Closeable)message).close();
                }
            }
            if (signal.hasError()) {
                ctx.fireExceptionCaught(signal.getThrowable());
            }
            disposableAttribute.set(null);
            response.flush();
            ctx.fireChannelReadComplete();
        })).contextWrite((ContextView)ReactorUtils.context((String)"imap", (MDCBuilder)this.mdc(session))), message).doOnError(arg_0 -> ((ChannelHandlerContext)ctx).fireExceptionCaught(arg_0)).subscribe();
        disposableAttribute.set((Object)disposable);
    }

    private void beforeIDLEUponProcessing(ChannelHandlerContext ctx) {
        if (!this.ignoreIDLEUponProcessing) {
            try {
                ctx.pipeline().addBefore("coreHandler", "heartbeatHandler", (ChannelHandler)this.heartbeatHandler);
            }
            catch (IllegalArgumentException e) {
                LOGGER.info("heartbeat handler is already part of this pipeline", (Throwable)e);
            }
        }
    }

    private void afterIDLEUponProcessing(ChannelHandlerContext ctx) {
        if (!this.ignoreIDLEUponProcessing) {
            try {
                ctx.pipeline().remove("heartbeatHandler");
            }
            catch (NoSuchElementException e) {
                LOGGER.info("Heartbeat handler was concurrently removed");
            }
        }
    }

    public static class ImapChannelUpstreamHandlerBuilder {
        private String hello;
        private Encryption secure;
        private boolean compress;
        private ImapProcessor processor;
        private ImapEncoder encoder;
        private IMAPServer.AuthenticationConfiguration authenticationConfiguration;
        private ImapMetrics imapMetrics;
        private boolean ignoreIDLEUponProcessing;
        private Duration heartbeatInterval;
        private ReactiveThrottler reactiveThrottler;

        public ImapChannelUpstreamHandlerBuilder reactiveThrottler(ReactiveThrottler reactiveThrottler) {
            this.reactiveThrottler = reactiveThrottler;
            return this;
        }

        public ImapChannelUpstreamHandlerBuilder hello(String hello) {
            this.hello = hello;
            return this;
        }

        public ImapChannelUpstreamHandlerBuilder secure(Encryption secure) {
            this.secure = secure;
            return this;
        }

        public ImapChannelUpstreamHandlerBuilder compress(boolean compress) {
            this.compress = compress;
            return this;
        }

        public ImapChannelUpstreamHandlerBuilder processor(ImapProcessor processor) {
            this.processor = processor;
            return this;
        }

        public ImapChannelUpstreamHandlerBuilder encoder(ImapEncoder encoder) {
            this.encoder = encoder;
            return this;
        }

        public ImapChannelUpstreamHandlerBuilder authenticationConfiguration(IMAPServer.AuthenticationConfiguration authenticationConfiguration) {
            this.authenticationConfiguration = authenticationConfiguration;
            return this;
        }

        public ImapChannelUpstreamHandlerBuilder imapMetrics(ImapMetrics imapMetrics) {
            this.imapMetrics = imapMetrics;
            return this;
        }

        public ImapChannelUpstreamHandlerBuilder ignoreIDLEUponProcessing(boolean ignoreIDLEUponProcessing) {
            this.ignoreIDLEUponProcessing = ignoreIDLEUponProcessing;
            return this;
        }

        public ImapChannelUpstreamHandlerBuilder heartbeatInterval(Duration heartbeatInterval) {
            this.heartbeatInterval = heartbeatInterval;
            return this;
        }

        public ImapChannelUpstreamHandler build() {
            return new ImapChannelUpstreamHandler(this.hello, this.processor, this.encoder, this.compress, this.secure, this.imapMetrics, this.authenticationConfiguration, this.ignoreIDLEUponProcessing, (int)this.heartbeatInterval.toSeconds(), this.reactiveThrottler);
        }
    }
}

