/*
 * Decompiled with CFR 0.152.
 */
package io.trino.execution.buffer;

import com.google.common.base.Preconditions;
import io.airlift.compress.Compressor;
import io.airlift.compress.Decompressor;
import io.airlift.slice.BasicSliceInput;
import io.airlift.slice.DynamicSliceOutput;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceInput;
import io.airlift.slice.SliceOutput;
import io.airlift.slice.Slices;
import io.trino.execution.buffer.PageCodecMarker;
import io.trino.execution.buffer.PagesSerdeUtil;
import io.trino.spi.Page;
import io.trino.spi.block.BlockEncodingSerde;
import io.trino.spiller.SpillCipher;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.concurrent.NotThreadSafe;

@NotThreadSafe
public class PagesSerde {
    private static final double MINIMUM_COMPRESSION_RATIO = 0.8;
    private static final int SERIALIZED_PAGE_HEADER_SIZE = 13;
    private final BlockEncodingSerde blockEncodingSerde;
    private final Optional<Compressor> compressor;
    private final Optional<Decompressor> decompressor;
    private final Optional<SpillCipher> spillCipher;

    public PagesSerde(BlockEncodingSerde blockEncodingSerde, Optional<Compressor> compressor, Optional<Decompressor> decompressor, Optional<SpillCipher> spillCipher) {
        this.blockEncodingSerde = Objects.requireNonNull(blockEncodingSerde, "blockEncodingSerde is null");
        Preconditions.checkArgument((compressor.isPresent() == decompressor.isPresent() ? 1 : 0) != 0, (Object)"compressor and decompressor must both be present or both be absent");
        this.compressor = Objects.requireNonNull(compressor, "compressor is null");
        this.decompressor = Objects.requireNonNull(decompressor, "decompressor is null");
        this.spillCipher = Objects.requireNonNull(spillCipher, "spillCipher is null");
    }

    public PagesSerdeContext newContext() {
        return new PagesSerdeContext();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Slice serialize(PagesSerdeContext context, Page page) {
        DynamicSliceOutput serializationBuffer = context.acquireSliceOutput(Math.toIntExact(page.getSizeInBytes() + 4L));
        byte[] inUseTempBuffer = null;
        try {
            PagesSerdeUtil.writeRawPage(page, (SliceOutput)serializationBuffer, this.blockEncodingSerde);
            Slice slice = serializationBuffer.slice();
            int uncompressedSize = serializationBuffer.size();
            PageCodecMarker.MarkerSet markers = PageCodecMarker.MarkerSet.empty();
            if (this.compressor.isPresent()) {
                byte[] compressed = context.acquireBuffer(this.compressor.get().maxCompressedLength(uncompressedSize));
                int compressedSize = this.compressor.get().compress(slice.byteArray(), slice.byteArrayOffset(), uncompressedSize, compressed, 0, compressed.length);
                if ((double)compressedSize / (double)uncompressedSize <= 0.8) {
                    slice = Slices.wrappedBuffer((byte[])compressed, (int)0, (int)compressedSize);
                    markers.add(PageCodecMarker.COMPRESSED);
                    inUseTempBuffer = compressed;
                } else {
                    context.releaseBuffer(compressed);
                }
            }
            if (this.spillCipher.isPresent()) {
                byte[] encrypted = context.acquireBuffer(this.spillCipher.get().encryptedMaxLength(slice.length()));
                int encryptedSize = this.spillCipher.get().encrypt(slice.byteArray(), slice.byteArrayOffset(), slice.length(), encrypted, 0);
                slice = Slices.wrappedBuffer((byte[])encrypted, (int)0, (int)encryptedSize);
                markers.add(PageCodecMarker.ENCRYPTED);
                if (inUseTempBuffer != null) {
                    context.releaseBuffer(inUseTempBuffer);
                }
                inUseTempBuffer = encrypted;
            }
            SliceOutput output = Slices.allocate((int)(13 + slice.length())).getOutput();
            output.writeInt(page.getPositionCount());
            output.writeByte((int)markers.byteValue());
            output.writeInt(uncompressedSize);
            output.writeInt(slice.length());
            output.writeBytes(slice);
            Slice slice2 = output.getUnderlyingSlice();
            context.releaseSliceOutput(serializationBuffer);
            if (inUseTempBuffer != null) {
                context.releaseBuffer(inUseTempBuffer);
            }
            return slice2;
        }
        catch (Throwable throwable) {
            context.releaseSliceOutput(serializationBuffer);
            if (inUseTempBuffer != null) {
                context.releaseBuffer(inUseTempBuffer);
            }
            throw throwable;
        }
    }

    public static int getSerializedPagePositionCount(Slice serializedPage) {
        return serializedPage.getInt(0);
    }

    public static boolean isSerializedPageEncrypted(Slice serializedPage) {
        return PagesSerde.getSerializedPageMarkerSet(serializedPage).contains(PageCodecMarker.ENCRYPTED);
    }

    public static boolean isSerializedPageCompressed(Slice serializedPage) {
        return PagesSerde.getSerializedPageMarkerSet(serializedPage).contains(PageCodecMarker.COMPRESSED);
    }

    private static PageCodecMarker.MarkerSet getSerializedPageMarkerSet(Slice serializedPage) {
        return PageCodecMarker.MarkerSet.fromByteValue(serializedPage.getByte(4));
    }

    public Page deserialize(Slice serializedPage) {
        try (PagesSerdeContext context = this.newContext();){
            Page page = this.deserialize(context, serializedPage);
            return page;
        }
    }

    public Page deserialize(PagesSerdeContext context, Slice serializedPage) {
        Preconditions.checkArgument((serializedPage != null ? 1 : 0) != 0, (Object)"serializedPage is null");
        BasicSliceInput input = serializedPage.getInput();
        int positionCount = input.readInt();
        PageCodecMarker.MarkerSet markers = PageCodecMarker.MarkerSet.fromByteValue(input.readByte());
        int uncompressedSize = input.readInt();
        int compressedSize = input.readInt();
        Slice slice = input.readSlice(compressedSize);
        byte[] inUseTempBuffer = null;
        if (markers.contains(PageCodecMarker.ENCRYPTED)) {
            Preconditions.checkState((boolean)this.spillCipher.isPresent(), (Object)"Page is encrypted, but spill cipher is missing");
            byte[] decrypted = context.acquireBuffer(this.spillCipher.get().decryptedMaxLength(slice.length()));
            int decryptedSize = this.spillCipher.get().decrypt(slice.byteArray(), slice.byteArrayOffset(), slice.length(), decrypted, 0);
            slice = Slices.wrappedBuffer((byte[])decrypted, (int)0, (int)decryptedSize);
            inUseTempBuffer = decrypted;
        }
        if (markers.contains(PageCodecMarker.COMPRESSED)) {
            Preconditions.checkState((boolean)this.decompressor.isPresent(), (Object)"Page is compressed, but decompressor is missing");
            byte[] decompressed = context.acquireBuffer(uncompressedSize);
            Preconditions.checkState((this.decompressor.get().decompress(slice.byteArray(), slice.byteArrayOffset(), slice.length(), decompressed, 0, uncompressedSize) == uncompressedSize ? 1 : 0) != 0);
            slice = Slices.wrappedBuffer((byte[])decompressed, (int)0, (int)uncompressedSize);
            if (inUseTempBuffer != null) {
                context.releaseBuffer(inUseTempBuffer);
            }
        }
        return PagesSerdeUtil.readRawPage(positionCount, (SliceInput)slice.getInput(), this.blockEncodingSerde);
    }

    public static Slice readSerializedPage(SliceInput input) {
        int positionCount = input.readInt();
        byte marker = input.readByte();
        int uncompressedSize = input.readInt();
        int compressedSize = input.readInt();
        SliceOutput output = Slices.allocate((int)(13 + compressedSize)).getOutput();
        output.writeInt(positionCount);
        output.writeByte((int)marker);
        output.writeInt(uncompressedSize);
        output.writeInt(compressedSize);
        Slice result = output.getUnderlyingSlice();
        input.readBytes(result, 13, compressedSize);
        return result;
    }

    public static final class PagesSerdeContext
    implements AutoCloseable {
        private static final int MAX_BUFFER_RETAINED_SIZE = 0x400000;
        private DynamicSliceOutput sliceOutput;
        private byte[] largerBuffer;
        private byte[] smallerBuffer;
        private boolean closed;

        private void checkNotClosed() {
            if (this.closed) {
                throw new IllegalStateException("PagesSerdeContext is already closed");
            }
        }

        private DynamicSliceOutput acquireSliceOutput(int estimatedSize) {
            this.checkNotClosed();
            if (this.sliceOutput != null && this.sliceOutput.writableBytes() >= estimatedSize) {
                DynamicSliceOutput result = this.sliceOutput;
                this.sliceOutput = null;
                return result;
            }
            this.sliceOutput = null;
            return new DynamicSliceOutput(estimatedSize);
        }

        private void releaseSliceOutput(DynamicSliceOutput sliceOutput) {
            if (this.closed) {
                return;
            }
            sliceOutput.reset();
            if (sliceOutput.writableBytes() <= 0x400000) {
                this.sliceOutput = sliceOutput;
            }
        }

        private byte[] acquireBuffer(int size) {
            this.checkNotClosed();
            if (this.smallerBuffer != null && this.smallerBuffer.length >= size) {
                byte[] result = this.smallerBuffer;
                this.smallerBuffer = null;
                return result;
            }
            if (this.largerBuffer != null && this.largerBuffer.length >= size) {
                byte[] result = this.largerBuffer;
                this.largerBuffer = this.smallerBuffer;
                this.smallerBuffer = null;
                return result;
            }
            return new byte[size];
        }

        private void releaseBuffer(byte[] buffer) {
            int size = buffer.length;
            if (this.closed || size > 0x400000) {
                return;
            }
            if (this.largerBuffer == null) {
                this.largerBuffer = buffer;
            } else if (size > this.largerBuffer.length) {
                this.smallerBuffer = this.largerBuffer;
                this.largerBuffer = buffer;
            } else if (this.smallerBuffer == null || size >= this.smallerBuffer.length) {
                this.smallerBuffer = buffer;
            }
        }

        @Override
        public void close() {
            this.closed = true;
            this.sliceOutput = null;
            this.smallerBuffer = null;
            this.largerBuffer = null;
        }
    }
}

