/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.authentication.oidc;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.github.benmanes.caffeine.cache.AsyncCacheLoader;
import com.github.benmanes.caffeine.cache.AsyncLoadingCache;
import com.github.benmanes.caffeine.cache.Caffeine;
import io.kubernetes.client.openapi.ApiCallback;
import io.kubernetes.client.openapi.ApiClient;
import io.kubernetes.client.openapi.ApiException;
import io.kubernetes.client.openapi.apis.WellKnownApi;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import javax.naming.AuthenticationException;
import org.apache.pulsar.broker.ServiceConfiguration;
import org.apache.pulsar.broker.authentication.AuthenticationProvider;
import org.apache.pulsar.broker.authentication.oidc.AuthenticationExceptionCode;
import org.apache.pulsar.broker.authentication.oidc.ConfigUtils;
import org.apache.pulsar.broker.authentication.oidc.OpenIDProviderMetadata;
import org.apache.pulsar.common.stats.CacheMetricsCollector;
import org.asynchttpclient.AsyncHttpClient;
import org.jspecify.annotations.NonNull;

class OpenIDProviderMetadataCache {
    private final ObjectReader reader = new ObjectMapper().readerFor(OpenIDProviderMetadata.class);
    private final AuthenticationProvider authenticationProvider;
    private final AsyncHttpClient httpClient;
    private final WellKnownApi wellKnownApi;
    private final AsyncLoadingCache<Optional<String>, OpenIDProviderMetadata> cache;
    private static final String WELL_KNOWN_OPENID_CONFIG = ".well-known/openid-configuration";
    private static final String SLASH_WELL_KNOWN_OPENID_CONFIG = "/.well-known/openid-configuration";

    OpenIDProviderMetadataCache(AuthenticationProvider authenticationProvider, ServiceConfiguration config, AsyncHttpClient httpClient, ApiClient apiClient) {
        this.authenticationProvider = authenticationProvider;
        int maxSize = ConfigUtils.getConfigValueAsInt(config, "openIDCacheSize", 5);
        int refreshAfterWriteSeconds = ConfigUtils.getConfigValueAsInt(config, "openIDCacheRefreshAfterWriteSeconds", 64800);
        int expireAfterSeconds = ConfigUtils.getConfigValueAsInt(config, "openIDCacheExpirationSeconds", 86400);
        this.httpClient = httpClient;
        this.wellKnownApi = apiClient != null ? new WellKnownApi(apiClient) : null;
        AsyncCacheLoader loader = (issuer, executor) -> {
            if (issuer.isPresent()) {
                return this.loadOpenIDProviderMetadataForIssuer((String)issuer.get());
            }
            return this.loadOpenIDProviderMetadataForKubernetesApiServer();
        };
        this.cache = Caffeine.newBuilder().recordStats().maximumSize((long)maxSize).refreshAfterWrite((long)refreshAfterWriteSeconds, TimeUnit.SECONDS).expireAfterWrite((long)expireAfterSeconds, TimeUnit.SECONDS).buildAsync(loader);
        CacheMetricsCollector.CAFFEINE.addCache("open-id-provider-metadata", this.cache);
    }

    CompletableFuture<OpenIDProviderMetadata> getOpenIDProviderMetadataForIssuer(@NonNull String issuer) {
        return this.cache.get(Optional.of(issuer));
    }

    private CompletableFuture<OpenIDProviderMetadata> loadOpenIDProviderMetadataForIssuer(String issuer) {
        String url = issuer.endsWith("/") ? issuer + WELL_KNOWN_OPENID_CONFIG : issuer + SLASH_WELL_KNOWN_OPENID_CONFIG;
        return this.httpClient.prepareGet(url).execute().toCompletableFuture().thenCompose(result -> {
            CompletableFuture<OpenIDProviderMetadata> future = new CompletableFuture<OpenIDProviderMetadata>();
            try {
                OpenIDProviderMetadata openIDProviderMetadata = (OpenIDProviderMetadata)this.reader.readValue(result.getResponseBodyAsBytes());
                this.verifyIssuer(issuer, openIDProviderMetadata, false);
                future.complete(openIDProviderMetadata);
            }
            catch (AuthenticationException e) {
                this.authenticationProvider.incrementFailureMetric((Enum)AuthenticationExceptionCode.ERROR_RETRIEVING_PROVIDER_METADATA);
                future.completeExceptionally(e);
            }
            catch (Exception e) {
                this.authenticationProvider.incrementFailureMetric((Enum)AuthenticationExceptionCode.ERROR_RETRIEVING_PROVIDER_METADATA);
                future.completeExceptionally(new AuthenticationException("Error retrieving OpenID Provider Metadata at " + issuer + ": " + e.getMessage()));
            }
            return future;
        });
    }

    CompletableFuture<OpenIDProviderMetadata> getOpenIDProviderMetadataForKubernetesApiServer(String issClaim) {
        return this.cache.get(Optional.empty()).thenCompose(openIDProviderMetadata -> {
            CompletableFuture<OpenIDProviderMetadata> future = new CompletableFuture<OpenIDProviderMetadata>();
            try {
                this.verifyIssuer(issClaim, (OpenIDProviderMetadata)openIDProviderMetadata, true);
                future.complete((OpenIDProviderMetadata)openIDProviderMetadata);
            }
            catch (AuthenticationException e) {
                this.authenticationProvider.incrementFailureMetric((Enum)AuthenticationExceptionCode.ERROR_RETRIEVING_PROVIDER_METADATA);
                future.completeExceptionally(e);
            }
            return future;
        });
    }

    private CompletableFuture<OpenIDProviderMetadata> loadOpenIDProviderMetadataForKubernetesApiServer() {
        final CompletableFuture<OpenIDProviderMetadata> future = new CompletableFuture<OpenIDProviderMetadata>();
        try {
            this.wellKnownApi.getServiceAccountIssuerOpenIDConfigurationAsync((ApiCallback)new ApiCallback<String>(){

                public void onFailure(ApiException e, int statusCode, Map<String, List<String>> responseHeaders) {
                    OpenIDProviderMetadataCache.this.authenticationProvider.incrementFailureMetric((Enum)AuthenticationExceptionCode.ERROR_RETRIEVING_PROVIDER_METADATA);
                    future.completeExceptionally(new AuthenticationException("Error retrieving OpenID Provider Metadata from Kubernetes API server. Message: " + e.getMessage() + " Response body: " + e.getResponseBody()));
                }

                public void onSuccess(String result, int statusCode, Map<String, List<String>> responseHeaders) {
                    try {
                        OpenIDProviderMetadata openIDProviderMetadata = (OpenIDProviderMetadata)OpenIDProviderMetadataCache.this.reader.readValue(result);
                        future.complete(openIDProviderMetadata);
                    }
                    catch (Exception e) {
                        OpenIDProviderMetadataCache.this.authenticationProvider.incrementFailureMetric((Enum)AuthenticationExceptionCode.ERROR_RETRIEVING_PROVIDER_METADATA);
                        future.completeExceptionally(new AuthenticationException("Error retrieving OpenID Provider Metadata from Kubernetes API Server: " + e.getMessage()));
                    }
                }

                public void onUploadProgress(long bytesWritten, long contentLength, boolean done) {
                }

                public void onDownloadProgress(long bytesRead, long contentLength, boolean done) {
                }
            });
        }
        catch (ApiException e) {
            this.authenticationProvider.incrementFailureMetric((Enum)AuthenticationExceptionCode.ERROR_RETRIEVING_PROVIDER_METADATA);
            future.completeExceptionally(new AuthenticationException("Error retrieving OpenID Provider Metadata from Kubernetes API server: " + e.getMessage()));
        }
        return future;
    }

    private void verifyIssuer(@NonNull String issuer, OpenIDProviderMetadata metadata, boolean isK8s) throws AuthenticationException {
        if (!issuer.equals(metadata.getIssuer())) {
            if (isK8s) {
                this.authenticationProvider.incrementFailureMetric((Enum)AuthenticationExceptionCode.UNSUPPORTED_ISSUER);
                throw new AuthenticationException("Issuer not allowed: " + issuer);
            }
            this.authenticationProvider.incrementFailureMetric((Enum)AuthenticationExceptionCode.ISSUER_MISMATCH);
            throw new AuthenticationException(String.format("Issuer URL mismatch: [%s] should match [%s]", issuer, metadata.getIssuer()));
        }
    }
}

