/*
 * Decompiled with CFR 0.152.
 */
package org.apache.uniffle.common.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.lang.reflect.Constructor;
import java.net.BindException;
import java.net.Inet4Address;
import java.net.InetAddress;
import java.net.InterfaceAddress;
import java.net.NetworkInterface;
import java.nio.ByteBuffer;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.ThreadLocalRandom;
import org.apache.uniffle.common.ShuffleServerInfo;
import org.apache.uniffle.common.config.RssBaseConf;
import org.apache.uniffle.common.config.RssConf;
import org.apache.uniffle.common.exception.RssException;
import org.apache.uniffle.common.rpc.ServerInterface;
import org.apache.uniffle.common.util.BlockIdLayout;
import org.apache.uniffle.common.util.IdHelper;
import org.apache.uniffle.shaded.com.google.common.collect.Lists;
import org.apache.uniffle.shaded.com.google.common.collect.Maps;
import org.apache.uniffle.shaded.com.google.common.collect.Sets;
import org.apache.uniffle.shaded.com.google.common.net.InetAddresses;
import org.apache.uniffle.shaded.io.netty.channel.unix.Errors;
import org.apache.uniffle.shaded.io.netty.util.internal.PlatformDependent;
import org.apache.uniffle.shaded.org.roaringbitmap.longlong.Roaring64NavigableMap;
import org.eclipse.jetty.util.MultiException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RssUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(RssUtils.class);
    public static final String RSS_LOCAL_DIR_KEY = "RSS_LOCAL_DIRS";

    private RssUtils() {
    }

    public static Map<String, String> getPropertiesFromFile(String filename) {
        File file;
        if (filename == null) {
            String rssHome = System.getenv("RSS_HOME");
            if (rssHome == null) {
                LOGGER.error("Both conf file and RSS_HOME env is null");
                return null;
            }
            LOGGER.info("Conf file is null use {}'s server.conf", (Object)rssHome);
            filename = rssHome + "/server.conf";
        }
        if (!(file = new File(filename)).exists()) {
            LOGGER.error("Properties file " + filename + " does not exist");
            return null;
        }
        if (!file.isFile()) {
            LOGGER.error("Properties file " + filename + " is not a normal file");
            return null;
        }
        LOGGER.info("Load config from {}", (Object)filename);
        HashMap<String, String> result = new HashMap<String, String>();
        try (InputStreamReader inReader = new InputStreamReader((InputStream)new FileInputStream(file), StandardCharsets.UTF_8);){
            Properties properties = new Properties();
            properties.load(inReader);
            properties.stringPropertyNames().forEach(k -> result.put((String)k, properties.getProperty((String)k).trim()));
        }
        catch (IOException ignored) {
            LOGGER.error("Failed when loading rss properties from " + filename);
        }
        return result;
    }

    public static String getHostIp() throws Exception {
        String ip = System.getenv("RSS_IP");
        if (ip != null) {
            if (!InetAddresses.isInetAddress(ip)) {
                throw new RssException("Environment RSS_IP: " + ip + " is wrong format");
            }
            return ip;
        }
        boolean isTestMode = Boolean.parseBoolean(System.getProperty("test.mode", "false"));
        Enumeration<NetworkInterface> nif = NetworkInterface.getNetworkInterfaces();
        String siteLocalAddress = null;
        while (nif.hasMoreElements()) {
            NetworkInterface ni = nif.nextElement();
            if (!ni.isUp() || ni.isLoopback() || ni.isPointToPoint() || ni.isVirtual()) continue;
            for (InterfaceAddress ifa : ni.getInterfaceAddresses()) {
                InetAddress ia = ifa.getAddress();
                InetAddress brd = ifa.getBroadcast();
                if ((brd == null || brd.isAnyLocalAddress()) && !isTestMode) {
                    LOGGER.info("ip {} was filtered, because it don't have effective broadcast address", (Object)ia.getHostAddress());
                    continue;
                }
                if (!ia.isLinkLocalAddress() && !ia.isAnyLocalAddress() && !ia.isLoopbackAddress() && ia instanceof Inet4Address && ia.isReachable(5000)) {
                    if (!ia.isSiteLocalAddress()) {
                        return ia.getHostAddress();
                    }
                    if (siteLocalAddress == null) {
                        LOGGER.info("ip {} was candidate, if there is no better choice, we will choose it", (Object)ia.getHostAddress());
                        siteLocalAddress = ia.getHostAddress();
                        continue;
                    }
                    LOGGER.info("ip {} was filtered, because it's not first effect site local address", (Object)ia.getHostAddress());
                    continue;
                }
                if (!(ia instanceof Inet4Address)) {
                    LOGGER.info("ip {} was filtered, because it's just a ipv6 address", (Object)ia.getHostAddress());
                    continue;
                }
                if (ia.isLinkLocalAddress()) {
                    LOGGER.info("ip {} was filtered, because it's just a link local address", (Object)ia.getHostAddress());
                    continue;
                }
                if (ia.isAnyLocalAddress()) {
                    LOGGER.info("ip {} was filtered, because it's just a any local address", (Object)ia.getHostAddress());
                    continue;
                }
                if (ia.isLoopbackAddress()) {
                    LOGGER.info("ip {} was filtered, because it's just a loop back address", (Object)ia.getHostAddress());
                    continue;
                }
                LOGGER.info("ip {} was filtered, because it's just not reachable address", (Object)ia.getHostAddress());
            }
        }
        return siteLocalAddress;
    }

    public static int startServiceOnPort(ServerInterface service, String serviceName, int servicePort, RssBaseConf conf) {
        if (servicePort < 0 || servicePort > 65535) {
            throw new IllegalArgumentException(String.format("Bad service %s on port (%s)", serviceName, servicePort));
        }
        int actualPort = servicePort;
        int maxRetries = conf.get(RssBaseConf.SERVER_PORT_MAX_RETRIES);
        for (int i = 0; i < maxRetries; ++i) {
            try {
                actualPort = servicePort == 0 ? RssUtils.findRandomTcpPort(conf) : (actualPort += i);
                service.startOnPort(actualPort);
                return actualPort;
            }
            catch (Exception e) {
                if (!RssUtils.isServerPortBindCollision(e)) {
                    throw new RssException(String.format("Failed to start service %s on port %s", serviceName, servicePort), e);
                }
                LOGGER.warn(String.format("%s:Service %s failed after %s retries (on a random free port (%s))!", e.getMessage(), serviceName, i + 1, actualPort));
                continue;
            }
        }
        throw new RssException(String.format("Failed to start service %s on port %s", serviceName, servicePort));
    }

    public static boolean isServerPortBindCollision(Throwable e) {
        if (e instanceof BindException) {
            if (e.getMessage() != null) {
                return true;
            }
            return RssUtils.isServerPortBindCollision(e.getCause());
        }
        if (e instanceof MultiException) {
            return !((MultiException)e).getThrowables().stream().noneMatch(throwable -> RssUtils.isServerPortBindCollision(throwable));
        }
        if (e instanceof Errors.NativeIoException) {
            return e.getMessage() != null && e.getMessage().startsWith("bind() failed: ") || RssUtils.isServerPortBindCollision(e.getCause());
        }
        if (e instanceof IOException) {
            return e.getMessage() != null && e.getMessage().startsWith("Failed to bind to address") || RssUtils.isServerPortBindCollision(e.getCause());
        }
        return false;
    }

    public static int findRandomTcpPort(RssBaseConf baseConf) {
        int portRangeMin = baseConf.getInteger(RssBaseConf.RSS_RANDOM_PORT_MIN);
        int portRangeMax = baseConf.getInteger(RssBaseConf.RSS_RANDOM_PORT_MAX);
        int portRange = portRangeMax - portRangeMin;
        return portRangeMin + ThreadLocalRandom.current().nextInt(portRange + 1);
    }

    public static byte[] serializeBitMap(Roaring64NavigableMap bitmap) throws IOException {
        long size = bitmap.serializedSizeInBytes();
        if (size > Integer.MAX_VALUE) {
            throw new RssException("Unsupported serialized size of bitmap: " + size);
        }
        ByteArrayOutputStream arrayOutputStream = new ByteArrayOutputStream((int)size);
        DataOutputStream dataOutputStream = new DataOutputStream(arrayOutputStream);
        bitmap.serialize(dataOutputStream);
        return arrayOutputStream.toByteArray();
    }

    public static Roaring64NavigableMap deserializeBitMap(byte[] bytes) throws IOException {
        Roaring64NavigableMap bitmap = Roaring64NavigableMap.bitmapOf(new long[0]);
        if (bytes.length == 0) {
            return bitmap;
        }
        DataInputStream dataInputStream = new DataInputStream(new ByteArrayInputStream(bytes));
        bitmap.deserialize(dataInputStream);
        return bitmap;
    }

    public static Roaring64NavigableMap cloneBitMap(Roaring64NavigableMap bitmap) {
        Roaring64NavigableMap clone = Roaring64NavigableMap.bitmapOf(new long[0]);
        clone.or(bitmap);
        return clone;
    }

    public static String generateShuffleKey(String appId, int shuffleId) {
        return String.join((CharSequence)"/", appId, String.valueOf(shuffleId));
    }

    public static String generatePartitionKey(String appId, Integer shuffleId, Integer partition) {
        return String.join((CharSequence)"/", appId, String.valueOf(shuffleId), String.valueOf(partition));
    }

    public static <T> List<T> loadExtensions(Class<T> extClass, List<String> classes, Object obj) {
        if (classes == null || classes.isEmpty()) {
            throw new RssException("Empty classes");
        }
        ArrayList<?> extensions = Lists.newArrayList();
        for (String name : classes) {
            name = name.trim();
            try {
                Object instance;
                Class<?> klass = Class.forName(name);
                if (!extClass.isAssignableFrom(klass)) {
                    throw new RssException(name + " is not subclass of " + extClass.getName());
                }
                try {
                    Constructor<?> constructor = klass.getConstructor(obj.getClass());
                    instance = constructor.newInstance(obj);
                }
                catch (Exception e) {
                    LOGGER.error("Fail to new instance.", (Throwable)e);
                    instance = klass.getConstructor(new Class[0]).newInstance(new Object[0]);
                }
                extensions.add(instance);
            }
            catch (Exception e) {
                LOGGER.error("Fail to new instance using default constructor.", (Throwable)e);
                throw new RssException(e);
            }
        }
        return extensions;
    }

    public static void checkQuorumSetting(int replica, int replicaWrite, int replicaRead) {
        if (replica < 1 || replicaWrite > replica || replicaRead > replica) {
            throw new RssException("Replica config is invalid, it cannot be less than 1 or less than replica.write  or less than replica.read. Please check it.");
        }
        if (replicaWrite + replicaRead <= replica) {
            throw new RssException("Replica config is unsafe, recommend replica.write + replica.read > replica");
        }
    }

    public static String getMetricNameForHostName(String hostName) {
        if (hostName == null) {
            return "";
        }
        return hostName.replaceAll("[\\.-]", "_");
    }

    public static Map<Integer, Roaring64NavigableMap> generatePartitionToBitmap(Roaring64NavigableMap shuffleBitmap, int startPartition, int endPartition, BlockIdLayout layout) {
        HashMap<Integer, Roaring64NavigableMap> result = Maps.newHashMap();
        for (int partitionId = startPartition; partitionId < endPartition; ++partitionId) {
            result.computeIfAbsent(partitionId, key -> Roaring64NavigableMap.bitmapOf(new long[0]));
        }
        Iterator<Long> it = shuffleBitmap.iterator();
        while (it.hasNext()) {
            Long blockId = it.next();
            int partitionId = layout.getPartitionId(blockId);
            if (partitionId < startPartition || partitionId >= endPartition) continue;
            ((Roaring64NavigableMap)result.get(partitionId)).add(blockId);
        }
        return result;
    }

    public static Map<ShuffleServerInfo, Set<Integer>> generateServerToPartitions(Map<Integer, List<ShuffleServerInfo>> partitionToServers) {
        HashMap<ShuffleServerInfo, Set<Integer>> serverToPartitions = Maps.newHashMap();
        for (Map.Entry<Integer, List<ShuffleServerInfo>> entry : partitionToServers.entrySet()) {
            int partitionId = entry.getKey();
            for (ShuffleServerInfo serverInfo : entry.getValue()) {
                if (!serverToPartitions.containsKey(serverInfo)) {
                    serverToPartitions.put(serverInfo, Sets.newHashSet(partitionId));
                    continue;
                }
                ((Set)serverToPartitions.get(serverInfo)).add(partitionId);
            }
        }
        return serverToPartitions;
    }

    public static void checkProcessedBlockIds(Roaring64NavigableMap blockIdBitmap, Roaring64NavigableMap processedBlockIds) {
        if (!blockIdBitmap.equals(processedBlockIds)) {
            Roaring64NavigableMap cloneBitmap = RssUtils.cloneBitMap(blockIdBitmap);
            cloneBitmap.and(processedBlockIds);
            if (!blockIdBitmap.equals(cloneBitmap)) {
                throw new RssException("Blocks read inconsistent: expected " + blockIdBitmap.getLongCardinality() + " blocks, actual " + cloneBitmap.getLongCardinality() + " blocks");
            }
        }
    }

    public static Roaring64NavigableMap generateTaskIdBitMap(Roaring64NavigableMap blockIdBitmap, IdHelper idHelper) {
        Iterator<Long> iterator = blockIdBitmap.iterator();
        Roaring64NavigableMap taskIdBitmap = Roaring64NavigableMap.bitmapOf(new long[0]);
        while (iterator.hasNext()) {
            taskIdBitmap.addLong(idHelper.getTaskAttemptId(iterator.next()));
        }
        return taskIdBitmap;
    }

    public static List<String> getConfiguredLocalDirs(RssConf conf) {
        if (conf.getEnv(RSS_LOCAL_DIR_KEY) != null) {
            return Arrays.asList(conf.getEnv(RSS_LOCAL_DIR_KEY).split(","));
        }
        return conf.get(RssBaseConf.RSS_STORAGE_BASE_PATH);
    }

    public static void releaseByteBuffer(ByteBuffer byteBuffer) {
        if (byteBuffer == null || !byteBuffer.isDirect()) {
            return;
        }
        PlatformDependent.freeDirectBuffer(byteBuffer);
    }

    public static Constructor<?> getConstructor(String className, Class<?> ... parameterTypes) throws ClassNotFoundException, NoSuchMethodException {
        Class<?> klass = Class.forName(className);
        Constructor<?> constructor = klass.getConstructor(parameterTypes);
        return constructor;
    }
}

