#!/usr/bin/env perl

use strict;
use warnings;

use Test::More;
use Test::Exception;
use MooseX::Params::Validate;

use Crypt::Rijndael::PP;

use Readonly;

Readonly my $TEST_VALUES => {
    128 => {
        cipher_text => [ 0x69, 0xc4, 0xe0, 0xd8, 0x6a, 0x7b, 0x04, 0x30, 0xd8, 0xcd, 0xb7, 0x80, 0x70, 0xb4, 0xc5, 0x5a, ],
        key         => [ 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, ],
        plain_text  => [ 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff, ],
    },
    192 => {
        cipher_text => [ 0xdd, 0xa9, 0x7c, 0xa4, 0x86, 0x4c, 0xdf, 0xe0, 0x6e, 0xaf, 0x70, 0xa0, 0xec, 0x0d, 0x71, 0x91, ],
        key         => [ 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
                         0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, ],
        plain_text  => [ 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff, ],
    },
    256 => {
        cipher_text => [ 0x8e, 0xa2, 0xb7, 0xca, 0x51, 0x67, 0x45, 0xbf, 0xea, 0xfc, 0x49, 0x90, 0x4b, 0x49, 0x60, 0x89, ],
        key         => [ 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
                         0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, ],
        plain_text  => [ 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff, ],
    },
};

subtest "Decryption with 128 Bit Key" => sub {
    test_decryption( $TEST_VALUES->{128} );
};

subtest "Decryption with 192 Bit Key" => sub {
    test_decryption( $TEST_VALUES->{192} );
};

subtest "Decryption with 256 Bit Key" => sub {
    test_decryption( $TEST_VALUES->{256} );
};

done_testing;

sub test_decryption {
    my ( $case ) = pos_validated_list( \@_, { isa => 'HashRef' } );

    my $packed_cipher_text = pack( "C*", @{ $case->{cipher_text} } );
    my $packed_cipher_key  = pack( "C*", @{ $case->{key} } );
    my $packed_expected_plain_text = pack( "C*", @{ $case->{plain_text} } );

    my $plain_text;
    lives_ok {
        $plain_text = Crypt::Rijndael::PP->decrypt_block(
            $packed_cipher_text, $packed_cipher_key
        );
    } "Lives through decryption of a block";

    cmp_ok( unpack( "H*", $plain_text ), 'eq',
        unpack( "H*", $packed_expected_plain_text ), "Correct Cipher Text" );

    return;
}
