package Net::Async::Redis::Multi;

use strict;
use warnings;

our $VERSION = '2.002_002'; # TRIAL VERSION

=head1 NAME

Net::Async::Redis::Multi - represents multiple operations in a single Redis transaction

=head1 DESCRIPTION

Instances are generated by L<Net::Async::Redis/multi>.

=cut

use Syntax::Keyword::Try;
use Future::AsyncAwait;
use Scalar::Util qw(weaken blessed);
use Log::Any qw($log);

sub new {
    my ($class, %args) = @_;
    weaken($args{redis} // die 'Must be provided a Net::Async::Redis instance');
    bless \%args, $class;
}

async sub exec {
    my ($self, $code) = @_;
    my $f = $self->$code;
    $f->retain if blessed($f) and $f->isa('Future');

    try {
        my ($exec_result) = await $self->redis->exec;
        my @reply = $exec_result->@*;
        my $success = 0;
        my $failure = 0;
        while(@reply and my ($queued) = shift @{$self->{queued_requests}}) {
            try {
                my $reply = shift @reply;
                $queued->done($reply) unless $queued->is_ready;
                ++$success
            } catch {
                $log->warnf("Failure during transaction: %s", $@);
                ++$failure
            }
        }
        return Future->done(
            $success, $failure
        )
    } catch {
        my $err = $@;
        for my $queued (splice @{$self->{queued_requests}}) {
            try {
                $queued->fail("Transaction failed", redis => 'transaction_failure') unless $queued->is_ready;
            } catch {
                $log->warnf("Failure during transaction: %s", $@);
            }
        }
        return Future->fail($err);
    }
}

=head2 redis

Accessor for the L<Net::Async::Redis> instance.

=cut

sub redis { shift->{redis} }

sub AUTOLOAD {
    my ($method) = our $AUTOLOAD =~ m{::([^:]+)$};
    die "Unknown method $method" unless Net::Async::Redis::Commands->can($method);

    my $code = async sub {
        my ($self, @args) = @_;
        my ($resp) = await do {
            local $self->redis->{_is_multi} = 1;
            $self->redis->$method(@args);
        };
        die $resp unless $resp eq 'QUEUED';
        my $f = $self->redis->future->set_label($method);
        push @{$self->{queued_requests}}, $f;
        return await $f;
    };
    { no strict 'refs'; *$method = $code }
    $code->(@_)->retain;
}

sub DESTROY {
    my ($self) = @_;
    return if ${^GLOBAL_PHASE} eq 'DESTRUCT' or not $self->{queued_requests};

    for my $queued (splice @{$self->{queued_requests}}) {
        try {
            $queued->cancel;
        } catch {
            $log->warnf("Failure during cleanup: %s", $@);
        }
    }
}

1;

__END__

=head1 AUTHOR

Tom Molesworth <TEAM@cpan.org>

=head1 LICENSE

Copyright Tom Molesworth 2015-2020. Licensed under the same terms as Perl itself.

