/**
 *	@file    algorithmTest.c
 *	@version 33ef80f (HEAD, tag: MATRIXSSL-3-7-2-OPEN, tag: MATRIXSSL-3-7-2-COMM, origin/master, origin/HEAD, master)
 *
 *	Crypto harness to check algorithms against known inputs/outputs.
 */
/*
 *	Copyright (c) 2013-2015 INSIDE Secure Corporation
 *	Copyright (c) PeerSec Networks, 2002-2011
 *	All Rights Reserved
 *
 *	The latest version of this code is available at http://www.matrixssl.org
 *
 *	This software is open source; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This General Public License does NOT permit incorporating this software
 *	into proprietary programs.  If you are unable to comply with the GPL, a
 *	commercial license for this software may be purchased from INSIDE at
 *	http://www.insidesecure.com/eng/Company/Locations
 *
 *	This program is distributed in WITHOUT ANY WARRANTY; without even the
 *	implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *	See the GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *	http://www.gnu.org/copyleft/gpl.html
 */
/******************************************************************************/

#include "crypto/cryptoApi.h"

/*
	Sample keys and certs in memory for X.509 testing
*/
#include "../../sampleCerts/RSA/1024_RSA_CA.h"
#include "../../sampleCerts/RSA/2048_RSA_CA.h"
#include "../../sampleCerts/RSA/1024_RSA.h"
#include "../../sampleCerts/RSA/2048_RSA.h"
#include "../../sampleCerts/RSA/1024_RSA_KEY.h"
#include "../../sampleCerts/RSA/2048_RSA_KEY.h"

/******************************************************************************/
#ifdef USE_AES
#define AES_ITER	1000	/* For AES Block mode test */
static int32 psAesTestBlock(void)
{
	int32 err, i, y;
	psAesKey_t ekey, dkey;
	unsigned char tmp[2][16];

	static const struct {
		int32 keylen;
		unsigned char key[32], pt[16], ct[16];
	} tests[] = {
	{ 16,
		{ 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f },
		{ 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },
		{ 0x69, 0xc4, 0xe0, 0xd8, 0x6a, 0x7b, 0x04, 0x30,
			0xd8, 0xcd, 0xb7, 0x80, 0x70, 0xb4, 0xc5, 0x5a }
		}, {
#ifndef USE_AESNI_CRYPTO
		24,
		{ 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17 },
		{ 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },
		{ 0xdd, 0xa9, 0x7c, 0xa4, 0x86, 0x4c, 0xdf, 0xe0,
			0x6e, 0xaf, 0x70, 0xa0, 0xec, 0x0d, 0x71, 0x91 }
		}, {
#endif
		32,
		{ 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
			0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
			0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
			0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f },
		{ 0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
			0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff },
		{ 0x8e, 0xa2, 0xb7, 0xca, 0x51, 0x67, 0x45, 0xbf,
			0xea, 0xfc, 0x49, 0x90, 0x4b, 0x49, 0x60, 0x89 }
		}
	};

	for (i = 0; i < (int32)(sizeof(tests)/sizeof(tests[0])); i++) {
		_psTraceInt("	AES-%d known vector test... ", tests[i].keylen * 8);
		memset(&ekey, 0x0, sizeof(ekey));
		if ((err = psAesInitKey(tests[i].key, tests[i].keylen, &ekey)) !=
				PS_SUCCESS) {
			_psTraceInt("FAILED:  psAesInitKey returned %d\n", err);
			return err;
		}
		memset(&dkey, 0x0, sizeof(dkey));
		if ((err = psAesInitKey(tests[i].key, tests[i].keylen, &dkey)) !=
				PS_SUCCESS) {
			_psTraceInt("FAILED:  psAesInitKey returned %d\n", err);
			return err;
		}
		psAesEncryptBlock(tests[i].pt, tmp[0], &ekey);
		psAesDecryptBlock(tmp[0], tmp[1], &dkey);
		if (memcmp(tmp[0], tests[i].ct, 16) || memcmp(tmp[1], tests[i].pt, 16))
		{
			_psTraceInt("FAILED: mem compare failed\n", i);
			if (memcmp(tmp[0], tests[i].ct, 16)) {
				psTraceCrypto("CT: ");
				for (i = 0; i < 16; i++) {
					_psTraceInt("%02x ", tmp[0][i]);
				}
				_psTrace("\n");
			} else {
				_psTrace("PT: ");
				for (i = 0; i < 16; i++) {
					_psTraceInt("%02x ", tmp[1][i]);
				}
				_psTrace("\n");
			}
			return -1;
		}
		_psTrace("PASSED\n");
/*
		now see if we can encrypt all zero bytes 1000 times,
		decrypt and come back where we started
*/
		_psTraceInt("	AES-%d wind/unwind test... ", tests[i].keylen * 8);
		for (y = 0; y < 16; y++) tmp[0][y] = 0;
		for (y = 0; y < AES_ITER; y++) psAesEncryptBlock(tmp[0], tmp[0], &ekey);
		for (y = 0; y < AES_ITER; y++) psAesDecryptBlock(tmp[0], tmp[0], &dkey);
		for (y = 0; y < 16; y++) {
			if (tmp[0][y] != 0) {
				_psTraceInt("FAILED:  unable to unwind %d to zero\n", y);
				return -1;
			}
		}
		_psTrace("PASSED\n");
	}
	return 0;
}

/*
	Known vector test - AES-CBC
*/
static int32 psAesTestCBC(void)
{
	int32	err, i;
	psCipherContext_t	eCtx, dCtx;
	unsigned char tmp[2][16];

	static struct {
		int32 keylen;
		unsigned char key[32], iv[16], pt[16], ct[16];
	} tests[] = {
	{ 16,
		{ 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7,
			0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c },
		{ 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
			0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F },
		{ 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d,
			0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a },
		{ 0x76, 0x49, 0xab, 0xac, 0x81, 0x19, 0xb2, 0x46, 0xce, 0xe9,
			0x8e, 0x9b, 0x12, 0xe9, 0x19, 0x7d }
		}, {
#ifndef USE_AESNI_CRYPTO
		24,
		{ 0x8e, 0x73, 0xb0, 0xf7, 0xda, 0x0e, 0x64, 0x52, 0xc8, 0x10,
			0xf3, 0x2b, 0x80, 0x90, 0x79, 0xe5, 0x62, 0xf8, 0xea, 0xd2,
			0x52, 0x2c, 0x6b, 0x7b },
		{ 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
			0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F },
		{ 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d,
			0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a },
		{ 0x4f, 0x02, 0x1d, 0xb2, 0x43, 0xbc, 0x63, 0x3d, 0x71, 0x78,
			0x18, 0x3a, 0x9f, 0xa0, 0x71, 0xe8 }
		}, {
#endif
		32,
		{ 0x60, 0x3d, 0xeb, 0x10, 0x15, 0xca, 0x71, 0xbe, 0x2b, 0x73, 0xae,
			0xf0, 0x85, 0x7d, 0x77, 0x81, 0x1f, 0x35, 0x2c, 0x07, 0x3b, 0x61,
			0x08, 0xd7, 0x2d, 0x98, 0x10, 0xa3, 0x09, 0x14, 0xdf, 0xf4 },
		{ 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
			0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F },
		{ 0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96, 0xe9, 0x3d,
			0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a },
		{ 0xf5, 0x8c, 0x4c, 0x04, 0xd6, 0xe5, 0xf1, 0xba, 0x77, 0x9e,
			0xab, 0xfb, 0x5f, 0x7b, 0xfb, 0xd6 }
		}
	};


	for (i = 0; i < (int32)(sizeof(tests)/sizeof(tests[0])); i++) {
		_psTraceInt("	AES-CBC-%d known vector test... ", tests[i].keylen * 8);
		if ((err = psAesInit(&eCtx, tests[i].iv, tests[i].key, tests[i].keylen))
				!=	PS_SUCCESS) {
			_psTraceInt("FAILED:  psAesInit returned %d\n", err);
			return err;
		}
		if ((err = psAesInit(&dCtx, tests[i].iv, tests[i].key, tests[i].keylen))
				!= PS_SUCCESS) {
			_psTraceInt("FAILED:  psAesInit returned %d\n", err);
			return err;
		}
		psAesEncrypt(&eCtx, tests[i].pt, tmp[0], 16);
		psAesDecrypt(&dCtx, tmp[0], tmp[1], 16);
		if (memcmp(tmp[0], tests[i].ct, 16) || memcmp(tmp[1], tests[i].pt, 16))
		{
			_psTraceInt("FAILED: mem compare failed\n", i);
			if (memcmp(tmp[0], tests[i].ct, 16)) {
				psTraceCrypto("CT: ");
				for (i = 0; i < 16; i++) {
					_psTraceInt("%02x ", tmp[0][i]);
				}
				_psTrace("\n");
			} else {
				_psTrace("PT: ");
				for (i = 0; i < 16; i++) {
					_psTraceInt("%02x ", tmp[1][i]);
				}
				_psTrace("\n");
			}
			return -1;
		}
		_psTrace("PASSED\n");
	}
	return 0;
}


#ifdef USE_AES_GCM
int32 psAesTestGCM(void)
{
	int32				i;
	psCipherContext_t	eCtx, dCtx;
	unsigned char		ciphertext[32];
	unsigned char		plaintext[32];
	unsigned char		tag[16];

	static struct {
		int32 keylen, ptlen, aadlen;
		unsigned char key[32], iv[12], pt[32], aad[32], ct[32], tag[16];
	} tests[] = {
	{ 16, 16, 16,
		{	0xc9, 0x39, 0xcc, 0x13, 0x39, 0x7c, 0x1d, 0x37,
			0xde, 0x6a, 0xe0, 0xe1, 0xcb, 0x7c, 0x42, 0x3c },
		{	0xb3, 0xd8, 0xcc, 0x01, 0x7c, 0xbb, 0x89, 0xb3,
			0x9e, 0x0f, 0x67, 0xe2 },
		{	0xc3, 0xb3, 0xc4, 0x1f, 0x11, 0x3a, 0x31, 0xb7,
			0x3d, 0x9a, 0x5c, 0xd4, 0x32, 0x10, 0x30, 0x69},
		{	0x24, 0x82, 0x56, 0x02, 0xbd, 0x12, 0xa9, 0x84,
			0xe0, 0x09, 0x2d, 0x3e, 0x44, 0x8e, 0xda, 0x5f},
		{	0x93, 0xfe, 0x7d, 0x9e, 0x9b, 0xfd, 0x10, 0x34,
			0x8a, 0x56, 0x06, 0xe5, 0xca, 0xfa, 0x73, 0x54},
		{	0x00, 0x32, 0xa1, 0xdc, 0x85, 0xf1, 0xc9, 0x78,
			0x69, 0x25, 0xa2, 0xe7, 0x1d, 0x82, 0x72, 0xdd}
		},
#ifndef USE_AESNI_CRYPTO
	{ 24, 16, 20,
		{	0x48, 0xb8, 0x2b, 0x72, 0xfc, 0x81, 0xbe, 0x86,
			0x86, 0x0f, 0x72, 0x06, 0x5e, 0xfa, 0x62, 0x18,
			0x6f, 0x9d, 0xa5, 0x96, 0xbd, 0xf9, 0x51, 0x58 },
		{	0x24, 0xda, 0x67, 0xae, 0xf7, 0x4b, 0xe3, 0x33,
			0x85, 0x89, 0xcc, 0xb6 },
		{	0xf8, 0x6a, 0x57, 0xa3, 0x0c, 0x06, 0xef, 0xb1,
			0x0e, 0xcb, 0xeb, 0x3d, 0x5a, 0x9d, 0x97, 0xb5},
		{	0xc1, 0xf7, 0xf5, 0xc8, 0xf8, 0xcc, 0x51, 0x37,
			0xa5, 0x39, 0x91, 0x94, 0x1f, 0x38, 0x8f, 0xfd,
			0x2b, 0x27, 0xf0, 0xdd},
		{	0x4b, 0xe5, 0x8c, 0xd0, 0x63, 0x78, 0xc0, 0x38,
			0x78, 0xd4, 0xf1, 0x65, 0x94, 0x43, 0xc5, 0xd1},
		{	0x84, 0xdd, 0x00, 0xb8, 0xb0, 0x3a, 0xcb, 0x92,
			0x19, 0xec, 0x51, 0x16, 0xe3, 0xde, 0x73, 0x62}
		},
#endif
	{ 32, 32, 16,
		{	0x27, 0x2d, 0x16, 0x49, 0xa3, 0xdd, 0x80, 0x4d,
			0xe0, 0x96, 0x2d, 0x3e, 0x07, 0x06, 0x4a, 0x70,
			0x54, 0xc0, 0x0a, 0x62, 0x34, 0xab, 0x1b, 0x0c,
			0xdc, 0xf6, 0x85, 0xab, 0x39, 0x48, 0x37, 0xe5},
		{	0x95, 0x5b, 0x58, 0x97, 0xf6, 0xb9, 0x80, 0x6b,
			0xbe, 0xc5, 0xc3, 0x3e},
		{	0x36, 0xe5, 0x7c, 0x29, 0xc0, 0x8c, 0x51, 0xad,
			0x7f, 0xa9, 0x1c, 0x04, 0x16, 0xf9, 0x76, 0xcf,
			0xd0, 0x11, 0x78, 0x0e, 0xb4, 0x4c, 0xc5, 0xab,
			0xd3, 0x4c, 0x7b, 0x43, 0x1b, 0x09, 0x3b, 0x8d},
		{	0x33, 0xe6, 0x18, 0xec, 0xbb, 0xe5, 0xeb, 0x05,
			0x66, 0xdf, 0x21, 0xc3, 0xc3, 0x4b, 0x7e, 0x25},
		{	0xcd, 0x6a, 0xeb, 0x34, 0x50, 0x81, 0xdc, 0x0b,
			0xb2, 0xc8, 0xb4, 0xd1, 0x9b, 0x28, 0x06, 0x58,
			0xfb, 0x87, 0xc0, 0xf2, 0xbd, 0x0f, 0x4c, 0x9d,
			0xa6, 0x94, 0xdc, 0x1f, 0xee, 0xb3, 0x2f, 0x4e},
		{	0xdd, 0x37, 0xea, 0xc6, 0xbd, 0x6a, 0x4d, 0x36,
			0x18, 0x24, 0x17, 0x38, 0x77, 0x97, 0x35, 0xd7}
		}
	};
/*
	Test GCM mode AES
*/
	for (i = 0; i < (int32)(sizeof(tests)/sizeof(tests[0])); i++) {
		_psTraceInt("	AES-GCM-%d known vector encrypt test... ", tests[i].keylen * 8);
		psAesInitGCM(&eCtx, tests[i].key, tests[i].keylen);
		psAesReadyGCM(&eCtx, tests[i].iv, tests[i].aad, tests[i].aadlen);
		psAesEncryptGCM(&eCtx, tests[i].pt, ciphertext, tests[i].ptlen);
		psAesGetGCMTag(&eCtx, 16, tag);
		if ((memcmp(ciphertext, tests[i].ct, tests[i].ptlen) != 0) ||
				(memcmp(tag, tests[i].tag, 16) != 0)) {
			printf("FAILED: memcmp mismatch\n");
		} else {
			printf("PASSED\n");
		}

		_psTraceInt("	AES-GCM-%d known vector decrypt test... ", tests[i].keylen * 8);
		psAesInitGCM(&dCtx, tests[i].key, tests[i].keylen);
		psAesReadyGCM(&dCtx, tests[i].iv, tests[i].aad, tests[i].aadlen);
		psAesDecryptGCMtagless(&dCtx, tests[i].ct, plaintext, tests[i].ptlen);
		memset(tag, 0x0, 16);
		psAesGetGCMTag(&dCtx, 16, tag);
		if ((memcmp(plaintext, tests[i].pt, tests[i].ptlen) != 0) ||
				(memcmp(tag, tests[i].tag, 16) != 0)) {
			printf("FAILED: memcmp mismatch\n");
		} else {
			printf("PASSED\n");
		}
	}
	return PS_SUCCESS;
}
#endif /* USE_AES_GCM */






#endif /* USE_AES */

#if 0
static psAesHmacPerf(void)
{
/*
	 Time AES-CBC encryption and decryption of AES_KB of data
 */
	useHmac = 0;
	aesTag = "AES-CBC";
L_CBC_TIMING:
#ifdef USE_AESNI_CRYPTO
	printf("Timing %s with %d KiB of data (aes-ni)\n", aesTag, AES_KB);
#else
	printf("Timing %s with %d KiB of data\n", aesTag, AES_KB);
#endif
	buf = psMalloc(pool, 1024 + 20);
	for (keysize = 16; keysize <= 32; keysize += 16) {
	_psTraceInt("AES-%d Key\n", keysize == 16 ? 128 : 256);
	for (i = 0; i < 1024; i++) {
		buf[i] = (unsigned char)(i & 0xFF);
	}
	memset(iv, 0x1, 16);
	memset(cbckey, 0x2, keysize);
	memset(plaintext, 0x3, 16);
	memset(hmackey, 0x4, 20);
	hmackeylen = 20;
	psAesInit(&encryptCtx, iv, cbckey, keysize);
	psGetTime(&start, NULL);
	for (k = 0; k < AES_KB; k++) {
		if (useHmac) psHmacSha1(hmackey, hmackeylen, buf, 1024, buf + 1024,
			hmackey, &hmackeylen);
		if (psAesEncrypt(&encryptCtx, buf, buf, 1024) != 1024) {
printf("ERROR LINE %d\n", __LINE__);
			return PS_FAILURE;
		}
	}
	psGetTime(&end, NULL);
	printf("%s: encrypt %u msecs\n", aesTag, psDiffMsecs(start, end, NULL));
	if (buf[0] == 0 && buf[1] == 1 && buf[2] == 2 && buf[3] == 3) {
		return PS_FAILURE;
	}
	psAesInit(&decryptCtx, iv, cbckey, keysize);
	psGetTime(&start, NULL);
	for (k = 0; k < AES_KB; k++) {
		if (useHmac) psHmacSha1(hmackey, hmackeylen, buf, 1024, buf + 1024,
			hmackey, &hmackeylen);
		if (psAesDecrypt(&decryptCtx, buf, buf, 1024) != 1024) {
			return PS_FAILURE;
		}
	}
	psGetTime(&end, NULL);
	printf("%s: decrypt %u msecs\n", aesTag, psDiffMsecs(start, end, NULL));
	}
	psFree(buf, pool);
	if (useHmac == 0) {
		useHmac = 1;
		aesTag = "AES-CBC-SHA-HMAC";
		goto L_CBC_TIMING;
	}

	return PS_SUCCESS;
}
#endif /* 0 */

/******************************************************************************/
#ifdef USE_DES
#define DES_ITER	1000
static int32 psDesTest(void)
{
	int32 err;
	static const struct des_test_case {
		int32 num, mode; /* mode 1 = encrypt */
		unsigned char key[8], txt[8], out[8];
	} cases[] = {
		{ 1, 1,     { 0x10, 0x31, 0x6E, 0x02, 0x8C, 0x8F, 0x3B, 0x4A },
					{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
					{ 0x82, 0xDC, 0xBA, 0xFB, 0xDE, 0xAB, 0x66, 0x02 } },
		{ 2, 1,     { 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01 },
					{ 0x95, 0xF8, 0xA5, 0xE5, 0xDD, 0x31, 0xD9, 0x00 },
					{ 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 } },
		{ 3, 1,     { 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01 },
					{ 0xDD, 0x7F, 0x12, 0x1C, 0xA5, 0x01, 0x56, 0x19 },
					{ 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 } },
		{ 4, 1,     { 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01 },
					{ 0x2E, 0x86, 0x53, 0x10, 0x4F, 0x38, 0x34, 0xEA },
					{ 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 } },
		{ 5, 1,     { 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01 },
					{ 0x4B, 0xD3, 0x88, 0xFF, 0x6C, 0xD8, 0x1D, 0x4F },
					{ 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 } },
		{ 6, 1,     { 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01 },
					{ 0x20, 0xB9, 0xE7, 0x67, 0xB2, 0xFB, 0x14, 0x56 },
					{ 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 } },
		{ 7, 1,     { 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01 },
					{ 0x55, 0x57, 0x93, 0x80, 0xD7, 0x71, 0x38, 0xEF },
					{ 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 } },
		{ 8, 1,     { 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01 },
					{ 0x6C, 0xC5, 0xDE, 0xFA, 0xAF, 0x04, 0x51, 0x2F },
					{ 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 } },
		{ 9, 1,     { 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01 },
					{ 0x0D, 0x9F, 0x27, 0x9B, 0xA5, 0xD8, 0x72, 0x60 },
					{ 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 } },
		{10, 1,     { 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01 },
					{ 0xD9, 0x03, 0x1B, 0x02, 0x71, 0xBD, 0x5A, 0x0A },
					{ 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 } },

		{ 1, 0,     { 0x10, 0x31, 0x6E, 0x02, 0x8C, 0x8F, 0x3B, 0x4A },
					{ 0x82, 0xDC, 0xBA, 0xFB, 0xDE, 0xAB, 0x66, 0x02 },
					{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 } },
		{ 2, 0,     { 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01 },
					{ 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
					{ 0x95, 0xF8, 0xA5, 0xE5, 0xDD, 0x31, 0xD9, 0x00 } },
		{ 3, 0,     { 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01 },
					{ 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
					{ 0xDD, 0x7F, 0x12, 0x1C, 0xA5, 0x01, 0x56, 0x19 } },
		{ 4, 0,     { 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01 },
					{ 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
					{ 0x2E, 0x86, 0x53, 0x10, 0x4F, 0x38, 0x34, 0xEA } },
		{ 5, 0,     { 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01 },
					{ 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
					{ 0x4B, 0xD3, 0x88, 0xFF, 0x6C, 0xD8, 0x1D, 0x4F } },
		{ 6, 0,     { 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01 },
					{ 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
					{ 0x20, 0xB9, 0xE7, 0x67, 0xB2, 0xFB, 0x14, 0x56 } },
		{ 7, 0,     { 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01 },
					{ 0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
					{ 0x55, 0x57, 0x93, 0x80, 0xD7, 0x71, 0x38, 0xEF } },
		{ 8, 0,     { 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01 },
					{ 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
					{ 0x6C, 0xC5, 0xDE, 0xFA, 0xAF, 0x04, 0x51, 0x2F } },
		{ 9, 0,     { 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01 },
					{ 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
					{ 0x0D, 0x9F, 0x27, 0x9B, 0xA5, 0xD8, 0x72, 0x60 } },
		{10, 0,     { 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01 },
					{ 0x00, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
					{ 0xD9, 0x03, 0x1B, 0x02, 0x71, 0xBD, 0x5A, 0x0A } }

	};

	int32 i, y;
	unsigned char tmp[8];
	psDes3Key_t des;

	for(i=0; i < (int)(sizeof(cases)/sizeof(cases[0])); i++)
	{
		if ((err = psDesInitKey(cases[i].key, 8, &des)) != PS_SUCCESS) {
		   return err;
		}
		if (cases[i].mode != 0) {
			_psTraceInt("	DES known vector encrypt test %d... ", cases[i].num);
		   psDesEncryptBlock(cases[i].txt, tmp, &des);
		} else {
			_psTraceInt("	DES known vector encrypt test %d... ", cases[i].num);
		   psDesDecryptBlock(cases[i].txt, tmp, &des);
		}

		if (memcmp(cases[i].out, tmp, sizeof(tmp)) != 0) {
			_psTrace("FAILED:  memcmp\n");
		   return -1;
		} else {
			_psTrace("PASSED\n");
		}

	  /* now see if we can encrypt all zero bytes DES_ITER times,
	   decrypt and come back where we started */
		_psTrace("	DES wind/unwind test... ");
		for (y = 0; y < 8; y++) tmp[y] = 0;
		for (y = 0; y < DES_ITER; y++) psDesEncryptBlock(tmp, tmp, &des);
		for (y = 0; y < DES_ITER; y++) psDesDecryptBlock(tmp, tmp, &des);
		for (y = 0; y < 8; y++) {
			if (tmp[y] != 0) {
				_psTrace("FAILED:  memcmp\n");
				return -1;
			}
		}
		_psTrace("PASSED\n");
	}

	return PS_SUCCESS;
}
#endif /* DES */

#ifdef USE_PKCS5
int32 psPBKDF2(void)
{
	int32	i;
	unsigned char	key[32];
	static struct {
		int32 rounds, dkLen;
		unsigned char *pass, *salt;
		unsigned char output[32];
	} tests[] = {
	{ 1, 20, (unsigned char *)"password", (unsigned char *)"salt",
		{ 0x0c, 0x60, 0xc8, 0x0f, 0x96, 0x1f, 0x0e, 0x71, 0xf3, 0xa9, 0xb5,
			0x24, 0xaf, 0x60, 0x12, 0x06, 0x2f, 0xe0, 0x37, 0xa6}
		},
	{ 4096, 20, (unsigned char *)"password", (unsigned char *)"salt",
		{ 0x4b, 0x00, 0x79, 0x01, 0xb7, 0x65, 0x48, 0x9a, 0xbe, 0xad, 0x49,
			0xd9, 0x26, 0xf7, 0x21, 0xd0, 0x65, 0xa4, 0x29, 0xc1}
		},
	{ 4096, 25, (unsigned char *)"passwordPASSWORDpassword",
		(unsigned char *)"saltSALTsaltSALTsaltSALTsaltSALTsalt",
		{ 0x3d, 0x2e, 0xec, 0x4f, 0xe4, 0x1c, 0x84, 0x9b, 0x80, 0xc8, 0xd8,
			0x36, 0x62, 0xc0, 0xe4, 0x4a, 0x8b, 0x29, 0x1a, 0x96, 0x4c, 0xf2,
			0xf0, 0x70, 0x38}
		}
	};

	for (i = 0; i < (int32)(sizeof(tests)/sizeof(tests[0])); i++) {
		_psTraceInt("	PBKDF2 known vector test %d... ", i + 1);
		pkcs5pbkdf2(tests[i].pass, (uint32)strlen((char*)tests[i].pass),
			tests[i].salt, (uint32)strlen((char*)tests[i].salt),
			tests[i].rounds, key, tests[i].dkLen);
		if (memcmp(key, tests[i].output, tests[i].dkLen) != 0) {
			_psTrace("FAILED\n");
		} else {
			_psTrace("PASSED\n");
		}
	}
	return 0;
}
#endif /* PKCS5 */

/******************************************************************************/
#ifdef USE_3DES
int32 psDes3Test(void)
{
	static struct {
		unsigned char key[24], iv[8], pt[8], ct[8];
	} tests[] = {
	{
		{ 0x80, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x80, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x80, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01},
		{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		{ 0x95, 0xa8, 0xd7, 0x28, 0x13, 0xda, 0xa9, 0x4d }
		},

	{
		{ 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
			0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01 },
		{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		{ 0x95, 0xf8, 0xa5, 0xe5, 0xdd, 0x31, 0xd9, 0x00 },
		{ 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 }
		},
	{
		{ 0x10, 0x46, 0x91, 0x34, 0x89, 0x98, 0x01, 0x31,
			0x10, 0x46, 0x91, 0x34, 0x89, 0x98, 0x01, 0x31,
			0x10, 0x46, 0x91, 0x34, 0x89, 0x98, 0x01, 0x31},
		{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		{ 0x88, 0xd5, 0x5e, 0x54, 0xf5, 0x4c, 0x97, 0xb4 }
		}
	};
	psCipherContext_t	skey;
	unsigned char		ct[32];
	int32				i;

	for (i = 0; i < (int32)(sizeof(tests)/sizeof(tests[0])); i++) {
		_psTraceInt("	3DES CBC known vector test %d... ", i + 1);
		psDes3Init(&skey, tests[i].iv, tests[i].key, 24);
		if (psDes3Encrypt(&skey, tests[i].pt, ct, 8) != 8) {
			return PS_FAILURE;
		}
		if (memcmp(ct, tests[i].ct, 8) != 0) {
			_psTrace("FAILED\n");
		} else {
			_psTrace("PASSED\n");
		}
	}
	return 0;
}
#endif /* USE_3DES */
/******************************************************************************/

#ifdef USE_ARC4
int32 psArc4Test(void)
{
	static struct {
		int32	keylen, ptlen;
		unsigned char key[24], pt[16], ct[16];
	} tests[] = {
	{ 5, 16,
		{ 0x01, 0x02, 0x03, 0x04, 0x05 },
		{ 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		{ 0xb2, 0x39, 0x63, 0x05, 0xf0, 0x3d, 0xc0, 0x27,
			0xcc, 0xc3, 0x52, 0x4a, 0x0a, 0x11, 0x18, 0xa8 }
		},
	{ 8, 8,
		{ 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		{ 0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef },
		{ 0x75, 0xb7, 0x87, 0x80, 0x99, 0xe0, 0xc5, 0x96 }
		}
	};
	psCipherContext_t	skey;
	unsigned char		ct[32];
	int32				i;

	for (i = 0; i < (int32)(sizeof(tests)/sizeof(tests[0])); i++) {
		_psTraceInt("	RC4 known vector test %d... ", i + 1);
		psArc4Init(&skey, tests[i].key, tests[i].keylen);
		if (psArc4(&skey, tests[i].pt, ct, tests[i].ptlen) != tests[i].ptlen) {
			return PS_FAILURE;
		}
		if (memcmp(ct, tests[i].ct, tests[i].ptlen) != 0) {
			_psTrace("FAILED\n");
		} else {
			_psTrace("PASSED\n");
		}
	}
	return 0;
}
#endif /* USE_ARC4 */

#ifdef USE_IDEA
int32 psIdeaTest(void)
{
	int32	err, i;
	psCipherContext_t	eCtx, dCtx;
	unsigned char tmp[2][16];

	static struct {
		unsigned char key[16], iv[8], pt[16], ct[16];
	} tests[] = {
	{
		{ 0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6, 0xab, 0xf7,
			0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c },
		{ 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07 },
		{ 0x68, 0x65, 0x6c, 0x6c, 0x6f, 0x20, 0x77, 0x6f, 0x72, 0x6c,
			0x64, 0x21, 0x21, 0x21, 0x21, 0x0a },
		{ 0x57, 0x1c, 0xce, 0x5a, 0xea, 0xee, 0x81, 0x04, 0x87, 0x2a,
			0x5a, 0xa9, 0xe2, 0xdf, 0x64, 0xa9 }
		}
	};


	for (i = 0; i < (int32)(sizeof(tests)/sizeof(tests[0])); i++) {
		_psTrace("	IDEA-CBC known vector test... ");
		if ((err = psIdeaInit(&eCtx, tests[i].iv, tests[i].key, 16))
				!=	PS_SUCCESS) {
			_psTraceInt("FAILED:  psAesInit returned %d\n", err);
			return err;
		}
		if ((err = psIdeaInit(&dCtx, tests[i].iv, tests[i].key, 16))
				!=	PS_SUCCESS) {
			_psTraceInt("FAILED:  psAesInit returned %d\n", err);
			return err;
		}
		psIdeaEncrypt(&eCtx, tests[i].pt, tmp[0], 16);
		//psTraceBytes("IDEA CT", tmp[0], 16);
		psIdeaDecrypt(&dCtx, tmp[0], tmp[1], 16);
		//psTraceBytes("IDEA PT", tmp[1], 16);
	}
	return 0;
}
#endif

/******************************************************************************/
#ifdef USE_SEED
/*
  Performs a self-test of the SEED block cipher
  @return CRYPT_OK if functional, CRYPT_NOP if self-test has been disabled
*/
int32 psSeedTest(void)
{
	static const struct test {
		unsigned char pt[16], ct[16], key[16];
	} tests[] = {
		{
			{ 0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,0x0F },
			{ 0x5E,0xBA,0xC6,0xE0,0x05,0x4E,0x16,0x68,0x19,0xAF,0xF1,0xCC,0x6D,0x34,0x6C,0xDB },
			{ 0 },
		},

		{
			{ 0 },
			{ 0xC1,0x1F,0x22,0xF2,0x01,0x40,0x50,0x50,0x84,0x48,0x35,0x97,0xE4,0x37,0x0F,0x43 },
			{ 0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,0x0F },
		},

		{
			{ 0x83,0xA2,0xF8,0xA2,0x88,0x64,0x1F,0xB9,0xA4,0xE9,0xA5,0xCC,0x2F,0x13,0x1C,0x7D },
			{ 0xEE,0x54,0xD1,0x3E,0xBC,0xAE,0x70,0x6D,0x22,0x6B,0xC3,0x14,0x2C,0xD4,0x0D,0x4A },
			{ 0x47,0x06,0x48,0x08,0x51,0xE6,0x1B,0xE8,0x5D,0x74,0xBF,0xB3,0xFD,0x95,0x61,0x85 },
		},

		{
			{ 0xB4,0x1E,0x6B,0xE2,0xEB,0xA8,0x4A,0x14,0x8E,0x2E,0xED,0x84,0x59,0x3C,0x5E,0xC7 },
			{ 0x9B,0x9B,0x7B,0xFC,0xD1,0x81,0x3C,0xB9,0x5D,0x0B,0x36,0x18,0xF4,0x0F,0x51,0x22 },
			{ 0x28,0xDB,0xC3,0xBC,0x49,0xFF,0xD8,0x7D,0xCF,0xA5,0x09,0xB1,0x1D,0x42,0x2B,0xE7 },
		}
	};

	int32			x;
	unsigned char	buf[2][16];
	psSeedKey_t		skey;

	for (x = 0; x < (int)(sizeof(tests)/sizeof(tests[0])); x++) {
		psSeedInitKey(tests[x].key, 16, &skey);
		psSeedEncryptBlock(tests[x].pt, buf[0], &skey);
		psSeedDecryptBlock(buf[0], buf[1], &skey);
		if (memcmp(buf[0], tests[x].ct, 16) || memcmp(buf[1], tests[x].pt, 16)){
			_psTraceInt("SEED failure: test %d failed\n", x);
			return -1;
		}
	}
	return PS_SUCCESS;
}
#endif /* USE_SEED */
/******************************************************************************/

/******************************************************************************/
#ifdef USE_SHA1
int32  psSha1Test(void)
{
	static const struct {
		char *msg;
		unsigned char hash[20];
	} tests[] = {
		{ "abc",
		{ 0xa9, 0x99, 0x3e, 0x36, 0x47, 0x06, 0x81, 0x6a,
			0xba, 0x3e, 0x25, 0x71, 0x78, 0x50, 0xc2, 0x6c,
			0x9c, 0xd0, 0xd8, 0x9d }
		},
		{ "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		{ 0x84, 0x98, 0x3E, 0x44, 0x1C, 0x3B, 0xD2, 0x6E,
			0xBA, 0xAE, 0x4A, 0xA1, 0xF9, 0x51, 0x29, 0xE5,
			0xE5, 0x46, 0x70, 0xF1 }
		},
		{ "The quick brown fox jumps over the lazy dog",
		{ 0x2f, 0xd4, 0xe1, 0xc6, 0x7a, 0x2d, 0x28, 0xfc, 0xed, 0x84, 0x9e,
			0xe1, 0xbb, 0x76, 0xe7, 0x39, 0x1b, 0x93, 0xeb, 0x12 }
		},
		{ "adfkadsofijadlkjoaijeoadaoijealfkajeflakepoieuproeirupqoeiruadslfasdlfadfowifjadfaleifafoeafwdflaheadfalefaahfaefefaeofadflafefaefeaeorea",
		{ 0xb5, 0x6a, 0x92, 0x83, 0x2a, 0xb8, 0x98, 0xdf, 0x37, 0xf6, 0x4b,
			0x5b, 0x95, 0x9b, 0xfe, 0x9a, 0xb5, 0x9f, 0x36, 0x86}
		}
	};

	int32 i;
	unsigned char tmp[20];
	psDigestContext_t md;

	for (i = 0; i < (int32)(sizeof(tests) / sizeof(tests[0]));  i++) {
		_psTraceInt("	SHA-1 known vector test %d... ", i + 1);
		psSha1Init(&md);
		psSha1Update(&md, (unsigned char*)tests[i].msg, (uint32)strlen(tests[i].msg));
		psSha1Final(&md, tmp);
		if (memcmp(tmp, tests[i].hash, 20) != 0) {
			_psTrace("FAILED: mem compare failure\n");
			return -1;
		} else {
			_psTrace("PASSED\n");
		}
	}
	return PS_SUCCESS;
}

#endif /* USE_SHA1 */
/******************************************************************************/

/******************************************************************************/
#ifdef USE_SHA256
int32 psSha256Test(void)
{
	static const struct {
		char *msg;
		char *moreMsg;
		unsigned char hash[32];
	} tests[] = {
		{ "abc", NULL,
		{ 0xba, 0x78, 0x16, 0xbf, 0x8f, 0x01, 0xcf, 0xea,
			0x41, 0x41, 0x40, 0xde, 0x5d, 0xae, 0x22, 0x23,
			0xb0, 0x03, 0x61, 0xa3, 0x96, 0x17, 0x7a, 0x9c,
			0xb4, 0x10, 0xff, 0x61, 0xf2, 0x00, 0x15, 0xad }
		},
		{ "ab", "c",
		{ 0xba, 0x78, 0x16, 0xbf, 0x8f, 0x01, 0xcf, 0xea,
			0x41, 0x41, 0x40, 0xde, 0x5d, 0xae, 0x22, 0x23,
			0xb0, 0x03, 0x61, 0xa3, 0x96, 0x17, 0x7a, 0x9c,
			0xb4, 0x10, 0xff, 0x61, 0xf2, 0x00, 0x15, 0xad }
		},
		{ "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq", NULL,
		{ 0x24, 0x8d, 0x6a, 0x61, 0xd2, 0x06, 0x38, 0xb8,
			0xe5, 0xc0, 0x26, 0x93, 0x0c, 0x3e, 0x60, 0x39,
			0xa3, 0x3c, 0xe4, 0x59, 0x64, 0xff, 0x21, 0x67,
			0xf6, 0xec, 0xed, 0xd4, 0x19, 0xdb, 0x06, 0xc1 }
		},
		{ "abcdbcdecd", "efdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
		{ 0x24, 0x8d, 0x6a, 0x61, 0xd2, 0x06, 0x38, 0xb8,
			0xe5, 0xc0, 0x26, 0x93, 0x0c, 0x3e, 0x60, 0x39,
			0xa3, 0x3c, 0xe4, 0x59, 0x64, 0xff, 0x21, 0x67,
			0xf6, 0xec, 0xed, 0xd4, 0x19, 0xdb, 0x06, 0xc1 }
		},
		{ "The quick brown fox jumps over the lazy dog", NULL,
		{ 0xd7, 0xa8, 0xfb, 0xb3, 0x07, 0xd7, 0x80, 0x94,
			0x69, 0xca, 0x9a, 0xbc, 0xb0, 0x08, 0x2e, 0x4f,
			0x8d, 0x56, 0x51, 0xe4, 0x6d, 0x3c,	0xdb, 0x76,
			0x2d, 0x02, 0xd0, 0xbf, 0x37, 0xc9, 0xe5, 0x92 }
		},
		{ "The quick brow", "n fox jumps over the lazy dog",
		{ 0xd7, 0xa8, 0xfb, 0xb3, 0x07, 0xd7, 0x80, 0x94,
			0x69, 0xca, 0x9a, 0xbc, 0xb0, 0x08, 0x2e, 0x4f,
			0x8d, 0x56, 0x51, 0xe4, 0x6d, 0x3c,	0xdb, 0x76,
			0x2d, 0x02, 0xd0, 0xbf, 0x37, 0xc9, 0xe5, 0x92 }
		},
		{ "adfkadsofijadlkjoaijeoadaoijealfkajeflakepoieuproeirupqoeiruadslfasdlfadfowifjadfaleifafoeafwdflaheadfalefaahfaefefaeofadflafefaefeaeorea", NULL,
		{ 0xd9, 0xe0, 0xb1, 0xfb, 0x08, 0x8a, 0xe4, 0xb9,
			0xfe, 0xf0, 0xb2, 0xb8, 0x33, 0x9e, 0x0f, 0xea,
			0x59, 0xa7, 0x35, 0x7f, 0x5b, 0x65, 0xde, 0x42,
			0x7b, 0xaa, 0x28, 0x89, 0xfd, 0x5f, 0x49, 0xd0}
		},
		{ "adfkadsofijadlkjoaijeoadaoijealfkajeflakepoieuproeirupqoeiruadslfasdlfadfowifjadfaleifafoeafwdflaheadfalefaahfaefef", "aeofadflafefaefeaeorea",
		{ 0xd9, 0xe0, 0xb1, 0xfb, 0x08, 0x8a, 0xe4, 0xb9,
			0xfe, 0xf0, 0xb2, 0xb8, 0x33, 0x9e, 0x0f, 0xea,
			0x59, 0xa7, 0x35, 0x7f, 0x5b, 0x65, 0xde, 0x42,
			0x7b, 0xaa, 0x28, 0x89, 0xfd, 0x5f, 0x49, 0xd0}
		}
	};

	int32 i;
	unsigned char tmp[32];
	psDigestContext_t md;


	for (i = 0; i < (int32)(sizeof(tests) / sizeof(tests[0])); i++) {
		_psTraceInt("	SHA-256 known vector test %d... ", i + 1);
		psSha256Init(&md);
		psSha256Update(&md, (unsigned char*)tests[i].msg,
			(uint32)strlen(tests[i].msg));
		if (tests[i].moreMsg != NULL) {
			psSha256Update(&md, (unsigned char*)tests[i].moreMsg,
				(uint32)strlen(tests[i].moreMsg));
		}
		psSha256Final(&md, tmp);
		if (memcmp(tmp, tests[i].hash, 32) != 0) {
			_psTrace("FAILED: memcmp\n");
			return -1;
		} else {
			_psTrace("PASSED\n");
		}
	}
	return PS_SUCCESS;
}
#endif /* USE_SHA256 */
/******************************************************************************/

#ifdef USE_SHA224
/**
  Self-test the hash
  @return CRYPT_OK if successful, CRYPT_NOP if self-tests have been disabled
*/
int32 psSha224Test(void)
{
	static const struct {
		char *msg;
		unsigned char hash[28];
	} tests[] = {
		{ "abc",
			{ 0x23, 0x09, 0x7d, 0x22, 0x34, 0x05, 0xd8,
			0x22, 0x86, 0x42, 0xa4, 0x77, 0xbd, 0xa2,
			0x55, 0xb3, 0x2a, 0xad, 0xbc, 0xe4, 0xbd,
			0xa0, 0xb3, 0xf7, 0xe3, 0x6c, 0x9d, 0xa7 }
		},
		{ "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq",
			{ 0x75, 0x38, 0x8b, 0x16, 0x51, 0x27, 0x76,
			0xcc, 0x5d, 0xba, 0x5d, 0xa1, 0xfd, 0x89,
			0x01, 0x50, 0xb0, 0xc6, 0x45, 0x5c, 0xb4,
			0xf5, 0x8b, 0x19, 0x52, 0x52, 0x25, 0x25 }
		},
	};

	int i;
	unsigned char tmp[28];
	psDigestContext_t md;

	for (i = 0; i < (int)(sizeof(tests) / sizeof(tests[0])); i++) {
		_psTraceInt("	SHA-224 known vector test %d... ", i + 1);
		psSha224Init(&md);
		psSha224Update(&md, (unsigned char*)tests[i].msg, (unsigned long)strlen(tests[i].msg));
		psSha224Final(&md, tmp);
		if (memcmp(tmp, tests[i].hash, 28) != 0) {
			_psTrace("FAILED: memcmp\n");
			return -1;
		} else {
			_psTrace("PASSED\n");
		}
	}
	return PS_SUCCESS;
}
#endif /* USE_SHA224 */


#ifdef USE_SHA512
int32  psSha512Test(void)
{
	static const struct {
		char *msg;
		unsigned char hash[64];
	} tests[] = {
		{ "abc",
		{ 0xdd, 0xaf, 0x35, 0xa1, 0x93, 0x61, 0x7a, 0xba,
		0xcc, 0x41, 0x73, 0x49, 0xae, 0x20, 0x41, 0x31,
		0x12, 0xe6, 0xfa, 0x4e, 0x89, 0xa9, 0x7e, 0xa2,
		0x0a, 0x9e, 0xee, 0xe6, 0x4b, 0x55, 0xd3, 0x9a,
		0x21, 0x92, 0x99, 0x2a, 0x27, 0x4f, 0xc1, 0xa8,
		0x36, 0xba, 0x3c, 0x23, 0xa3, 0xfe, 0xeb, 0xbd,
		0x45, 0x4d, 0x44, 0x23, 0x64, 0x3c, 0xe8, 0x0e,
		0x2a, 0x9a, 0xc9, 0x4f, 0xa5, 0x4c, 0xa4, 0x9f }
		},
		{ "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu",
		{ 0x8e, 0x95, 0x9b, 0x75, 0xda, 0xe3, 0x13, 0xda,
		0x8c, 0xf4, 0xf7, 0x28, 0x14, 0xfc, 0x14, 0x3f,
		0x8f, 0x77, 0x79, 0xc6, 0xeb, 0x9f, 0x7f, 0xa1,
		0x72, 0x99, 0xae, 0xad, 0xb6, 0x88, 0x90, 0x18,
		0x50, 0x1d, 0x28, 0x9e, 0x49, 0x00, 0xf7, 0xe4,
		0x33, 0x1b, 0x99, 0xde, 0xc4, 0xb5, 0x43, 0x3a,
		0xc7, 0xd3, 0x29, 0xee, 0xb6, 0xdd, 0x26, 0x54,
		0x5e, 0x96, 0xe5, 0x5b, 0x87, 0x4b, 0xe9, 0x09 }
		},
	};

	int i;
	unsigned char tmp[64];
	psDigestContext_t md;

	for (i = 0; i < (int)(sizeof(tests) / sizeof(tests[0])); i++) {
		_psTraceInt("	SHA-512 known vector test %d... ", i + 1);
		psSha512Init(&md);
		psSha512Update(&md, (unsigned char *)tests[i].msg, (uint32)strlen(tests[i].msg));
		psSha512Final(&md, tmp);
		if (memcmp(tmp, tests[i].hash, 64) != 0) {
			_psTrace("FAILED: memcmp\n");
			return -1;
		}
		_psTrace("PASSED\n");
	}
	return PS_SUCCESS;
}
#endif /* USE_SHA512 */

#ifdef USE_SHA384
int32 psSha384Test(void)
{
	static const struct {
		char *msg;
		unsigned char hash[48];
	} tests[] = {
	{ "abc",
	  { 0xcb, 0x00, 0x75, 0x3f, 0x45, 0xa3, 0x5e, 0x8b,
		0xb5, 0xa0, 0x3d, 0x69, 0x9a, 0xc6, 0x50, 0x07,
		0x27, 0x2c, 0x32, 0xab, 0x0e, 0xde, 0xd1, 0x63,
		0x1a, 0x8b, 0x60, 0x5a, 0x43, 0xff, 0x5b, 0xed,
		0x80, 0x86, 0x07, 0x2b, 0xa1, 0xe7, 0xcc, 0x23,
		0x58, 0xba, 0xec, 0xa1, 0x34, 0xc8, 0x25, 0xa7 }
	},
	{ "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu",
	  { 0x09, 0x33, 0x0c, 0x33, 0xf7, 0x11, 0x47, 0xe8,
		0x3d, 0x19, 0x2f, 0xc7, 0x82, 0xcd, 0x1b, 0x47,
		0x53, 0x11, 0x1b, 0x17, 0x3b, 0x3b, 0x05, 0xd2,
		0x2f, 0xa0, 0x80, 0x86, 0xe3, 0xb0, 0xf7, 0x12,
		0xfc, 0xc7, 0xc7, 0x1a, 0x55, 0x7e, 0x2d, 0xb9,
		0x66, 0xc3, 0xe9, 0xfa, 0x91, 0x74, 0x60, 0x39 }
	},
	};

	int i;
	unsigned char tmp[48];
	psDigestContext_t md;

	for (i = 0; i < (int)(sizeof(tests) / sizeof(tests[0])); i++) {
		_psTraceInt("	SHA-512 known vector test %d... ", i + 1);
		psSha384Init(&md);
		psSha384Update(&md, (unsigned char*)tests[i].msg, (uint32)strlen(tests[i].msg));
		psSha384Final(&md, tmp);
		if (memcmp(tmp, tests[i].hash, 48) != 0) {
			_psTrace("FAILED: memcmp\n");
			return -1;
		}
		_psTrace("PASSED\n");
	}
	return PS_SUCCESS;
}
#endif /* USE_SHA384 */

/******************************************************************************/
#ifdef USE_MD5
int32 psMd5Test(void)
{
	static const struct {
		char *msg;
		unsigned char hash[16];
	} tests[] = {
		{ "",
		{ 0xd4, 0x1d, 0x8c, 0xd9, 0x8f, 0x00, 0xb2, 0x04,
			0xe9, 0x80, 0x09, 0x98, 0xec, 0xf8, 0x42, 0x7e } },
		{ "a",
		{0x0c, 0xc1, 0x75, 0xb9, 0xc0, 0xf1, 0xb6, 0xa8,
			0x31, 0xc3, 0x99, 0xe2, 0x69, 0x77, 0x26, 0x61 } },
		{ "abc",
		{ 0x90, 0x01, 0x50, 0x98, 0x3c, 0xd2, 0x4f, 0xb0,
			0xd6, 0x96, 0x3f, 0x7d, 0x28, 0xe1, 0x7f, 0x72 } },
		{ "message digest",
		{ 0xf9, 0x6b, 0x69, 0x7d, 0x7c, 0xb7, 0x93, 0x8d,
			0x52, 0x5a, 0x2f, 0x31, 0xaa, 0xf1, 0x61, 0xd0 } },
		{ "abcdefghijklmnopqrstuvwxyz",
		{ 0xc3, 0xfc, 0xd3, 0xd7, 0x61, 0x92, 0xe4, 0x00,
			0x7d, 0xfb, 0x49, 0x6c, 0xca, 0x67, 0xe1, 0x3b } },
		{ "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789",
		{ 0xd1, 0x74, 0xab, 0x98, 0xd2, 0x77, 0xd9, 0xf5,
			0xa5, 0x61, 0x1c, 0x2c, 0x9f, 0x41, 0x9d, 0x9f } },
		{ "12345678901234567890123456789012345678901234567890123456789012345678901234567890",
		{ 0x57, 0xed, 0xf4, 0xa2, 0x2b, 0xe3, 0xc9, 0x55,
			0xac, 0x49, 0xda, 0x2e, 0x21, 0x07, 0xb6, 0x7a } },
		{ NULL, { 0 } }
	};

	int32 i;
	unsigned char tmp[16];
	psDigestContext_t md;

	for (i = 0; tests[i].msg != NULL; i++) {
	_psTraceInt("	MD5 known vector test %d... ", i + 1);
		psMd5Init(&md);
		psMd5Update(&md, (unsigned char *)tests[i].msg,
			(uint32)strlen(tests[i].msg));
		psMd5Final(&md, tmp);
		if (memcmp(tmp, tests[i].hash, 16) != 0) {
			_psTrace("FAILED: memcmp\n");
			return -1;
		}
		_psTrace("PASSED\n");
	}
	return PS_SUCCESS;
}
#endif /* USE_MD5 */
/******************************************************************************/

/******************************************************************************/
#ifdef  USE_MD4
int32 psMd4Test(void)
{
	static const struct md4_test_case {
		char *input;
		unsigned char digest[16];
	} cases[] = {
		{ "",
		  {0x31, 0xd6, 0xcf, 0xe0, 0xd1, 0x6a, 0xe9, 0x31,
		   0xb7, 0x3c, 0x59, 0xd7, 0xe0, 0xc0, 0x89, 0xc0} },
		{ "a",
		  {0xbd, 0xe5, 0x2c, 0xb3, 0x1d, 0xe3, 0x3e, 0x46,
		   0x24, 0x5e, 0x05, 0xfb, 0xdb, 0xd6, 0xfb, 0x24} },
		{ "abc",
		  {0xa4, 0x48, 0x01, 0x7a, 0xaf, 0x21, 0xd8, 0x52,
		   0x5f, 0xc1, 0x0a, 0xe8, 0x7a, 0xa6, 0x72, 0x9d} },
		{ "message digest",
		  {0xd9, 0x13, 0x0a, 0x81, 0x64, 0x54, 0x9f, 0xe8,
		   0x18, 0x87, 0x48, 0x06, 0xe1, 0xc7, 0x01, 0x4b} },
		{ "abcdefghijklmnopqrstuvwxyz",
		  {0xd7, 0x9e, 0x1c, 0x30, 0x8a, 0xa5, 0xbb, 0xcd,
		   0xee, 0xa8, 0xed, 0x63, 0xdf, 0x41, 0x2d, 0xa9} },
		{ "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789",
		  {0x04, 0x3f, 0x85, 0x82, 0xf2, 0x41, 0xdb, 0x35,
		   0x1c, 0xe6, 0x27, 0xe1, 0x53, 0xe7, 0xf0, 0xe4} },
		{ "12345678901234567890123456789012345678901234567890123456789012345678901234567890",
		  {0xe3, 0x3b, 0x4d, 0xdc, 0x9c, 0x38, 0xf2, 0x19,
		   0x9c, 0x3e, 0x7b, 0x16, 0x4f, 0xcc, 0x05, 0x36} },
	};
	int32 i;
	psDigestContext_t md;
	unsigned char digest[16];

	for(i = 0; i < (int32)(sizeof(cases) / sizeof(cases[0])); i++) {
		_psTraceInt("	MD4 known vector test %d... ", i + 1);
		psMd4Init(&md);
		psMd4Update(&md, (unsigned char *)cases[i].input,
			(uint32)strlen(cases[i].input));
		psMd4Final(&md, digest);
		if (memcmp(digest, cases[i].digest, 16) != 0) {
			_psTrace("FAILED: memcmp\n");
			return -1;
		}
		_psTrace("PASSED\n");
	}
	return PS_SUCCESS;
}
#endif /* USE_MD4 */
/******************************************************************************/

/******************************************************************************/
#ifdef USE_MD2
int32 psMd2Test(void)
{
	static const struct {
		char *msg;
		unsigned char md[16];
	} tests[] = {
		{ "",
			{0x83,0x50,0xe5,0xa3,0xe2,0x4c,0x15,0x3d,
			0xf2,0x27,0x5c,0x9f,0x80,0x69,0x27,0x73}
		},
		{ "a",
			{0x32,0xec,0x01,0xec,0x4a,0x6d,0xac,0x72,
			0xc0,0xab,0x96,0xfb,0x34,0xc0,0xb5,0xd1}
		},
		{ "message digest",
			{0xab,0x4f,0x49,0x6b,0xfb,0x2a,0x53,0x0b,
			0x21,0x9f,0xf3,0x30,0x31,0xfe,0x06,0xb0}
		},
		{ "abcdefghijklmnopqrstuvwxyz",
			{0x4e,0x8d,0xdf,0xf3,0x65,0x02,0x92,0xab,
			0x5a,0x41,0x08,0xc3,0xaa,0x47,0x94,0x0b}
		},
		{ "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789",
			{0xda,0x33,0xde,0xf2,0xa4,0x2d,0xf1,0x39,
			0x75,0x35,0x28,0x46,0xc3,0x03,0x38,0xcd}
		},
		{	"1234567890123456789012345678901234567890"
			"1234567890123456789012345678901234567890",
			{0xd5,0x97,0x6f,0x79,0xd8,0x3d,0x3a,0x0d,
			0xc9,0x80,0x6c,0x3c,0x66,0xf3,0xef,0xd8}
		}
	};
	int32 i;
	psDigestContext_t md;
	unsigned char buf[16];

	for (i = 0; i < (int32)(sizeof(tests) / sizeof(tests[0])); i++) {
		_psTraceInt("	MD2 known vector test %d... ", i + 1);
		psMd2Init(&md);
		psMd2Update(&md, (unsigned char*)tests[i].msg,
			(uint32)strlen(tests[i].msg));
		psMd2Final(&md, buf);
		if (memcmp(buf, tests[i].md, 16) != 0) {
			_psTrace("FAILED: memcmp\n");
			return -1;
		}
		_psTrace("PASSED\n");
	}
	return PS_SUCCESS;
}
#endif /* USE_MD2 */
/******************************************************************************/


/******************************************************************************/
#ifdef USE_RSA
static int32 psRsaEncryptTest(void)
{
	psPool_t		*pool = NULL;
	unsigned char   in[] = "hello";
	unsigned char   out[128];
	unsigned char   decrypted[6];
#ifdef USE_CERT_PARSE
	psX509Cert_t	*cert;
#endif /* USE_CERT_PARSE */
	psPubKey_t		*privkey;

/*
	Start with getting both key halfs from the same source
*/
	pkcs1ParsePrivBin(pool, (unsigned char *)RSA1024KEY, sizeof(RSA1024KEY), &privkey);
	psRsaEncryptPub(pool, &privkey->key->rsa, in, sizeof(in), out, 128, NULL);
	psRsaDecryptPriv(pool, &privkey->key->rsa, out, 128, decrypted,
		sizeof(decrypted), NULL);
	if (memcmp(decrypted, "hello", 5) != 0) {
		_psTrace("RSA failure: mem compare failure\n");
		psFreePubKey(privkey);
		return -1;
	}
	psFreePubKey(privkey);
	memset(decrypted, 0x0, sizeof(decrypted));
	memset(out, 0x0, sizeof(out));

#ifdef USE_CERT_PARSE
/*
	Get the public key from the cert
*/
	pkcs1ParsePrivBin(pool, (unsigned char *)RSA1024KEY, sizeof(RSA1024KEY), &privkey);
	psX509ParseCert(pool, (unsigned char *)RSA1024, sizeof(RSA1024), &cert, 0);
	psRsaEncryptPub(pool, &cert->publicKey.key->rsa, in, sizeof(in), out, 128, NULL);
	psRsaDecryptPriv(pool, &privkey->key->rsa, out, 128, decrypted,
		sizeof(decrypted), NULL);
	if (memcmp(decrypted, "hello", 5) != 0) {
		psFreePubKey(privkey);
		psX509FreeCert(cert);
		_psTrace("RSA failure: mem compare failure\n");
		return -1;
	}
	psFreePubKey(privkey);
	psX509FreeCert(cert);
	memset(decrypted, 0x0, sizeof(decrypted));
	memset(out, 0x0, sizeof(out));
#endif /* USE_CERT_PARSE */

	_psTrace("	PASSED\n");
	return PS_SUCCESS;
}

static int32 psRsaSignTest(void)
{
	psPool_t		*pool = NULL;
	unsigned char   in[] = "hello";
	unsigned char   out[128];
	unsigned char   decrypted[6];
	psPubKey_t		*privkey;

	pkcs1ParsePrivBin(pool, (unsigned char *)RSA1024KEY, sizeof(RSA1024KEY), &privkey);
	if (psRsaEncryptPriv(pool, &privkey->key->rsa, in, sizeof(in), out, 128, NULL)
			< 0) {
		return PS_FAILURE;
	}

	if (psRsaDecryptPub(pool, &privkey->key->rsa, out, 128, decrypted,
						sizeof(decrypted), NULL) < 0) {
		return PS_FAILURE;
	}
	if (memcmp(decrypted, "hello", 5) != 0) {
		_psTrace("RSA failure: mem compare failure\n");
		psFreePubKey(privkey);
		return PS_FAILURE;
	}

	psFreePubKey(privkey);
	_psTrace("	PASSED\n");
	return PS_SUCCESS;
}


/******************************************************************************/
#ifdef USE_PKCS1_OAEP
/* OAEP-VEC.TXT from RSA PKCS#1 web page */
static unsigned char key1N[] = {
	0xa8, 0xb3, 0xb2, 0x84, 0xaf, 0x8e, 0xb5, 0x0b, 0x38, 0x70, 0x34, 0xa8,
	0x60, 0xf1, 0x46, 0xc4, 0x91, 0x9f, 0x31, 0x87, 0x63, 0xcd, 0x6c, 0x55,
	0x98, 0xc8, 0xae, 0x48, 0x11, 0xa1, 0xe0, 0xab, 0xc4, 0xc7, 0xe0, 0xb0,
	0x82, 0xd6, 0x93, 0xa5, 0xe7, 0xfc, 0xed, 0x67, 0x5c, 0xf4, 0x66, 0x85,
	0x12, 0x77, 0x2c, 0x0c, 0xbc, 0x64, 0xa7, 0x42, 0xc6, 0xc6, 0x30, 0xf5,
	0x33, 0xc8, 0xcc, 0x72, 0xf6, 0x2a, 0xe8, 0x33, 0xc4, 0x0b, 0xf2, 0x58,
	0x42, 0xe9, 0x84, 0xbb, 0x78, 0xbd, 0xbf, 0x97, 0xc0, 0x10, 0x7d, 0x55,
	0xbd, 0xb6, 0x62, 0xf5, 0xc4, 0xe0, 0xfa, 0xb9, 0x84, 0x5c, 0xb5, 0x14,
	0x8e, 0xf7, 0x39, 0x2d, 0xd3, 0xaa, 0xff, 0x93, 0xae, 0x1e, 0x6b, 0x66,
	0x7b, 0xb3, 0xd4, 0x24, 0x76, 0x16, 0xd4, 0xf5, 0xba, 0x10, 0xd4, 0xcf,
	0xd2, 0x26, 0xde, 0x88, 0xd3, 0x9f, 0x16, 0xfb
};
static unsigned char key1e[] = {0x01, 0x00, 0x01};

static unsigned char key1d[] = {
	0x53, 0x33, 0x9c, 0xfd, 0xb7, 0x9f, 0xc8, 0x46, 0x6a, 0x65, 0x5c, 0x73,
	0x16, 0xac, 0xa8, 0x5c, 0x55, 0xfd, 0x8f, 0x6d, 0xd8, 0x98, 0xfd, 0xaf,
	0x11, 0x95, 0x17, 0xef, 0x4f, 0x52, 0xe8, 0xfd, 0x8e, 0x25, 0x8d, 0xf9,
	0x3f, 0xee, 0x18, 0x0f, 0xa0, 0xe4, 0xab, 0x29, 0x69, 0x3c, 0xd8, 0x3b,
	0x15, 0x2a, 0x55, 0x3d, 0x4a, 0xc4, 0xd1, 0x81, 0x2b, 0x8b, 0x9f, 0xa5,
	0xaf, 0x0e, 0x7f, 0x55, 0xfe, 0x73, 0x04, 0xdf, 0x41, 0x57, 0x09, 0x26,
	0xf3, 0x31, 0x1f, 0x15, 0xc4, 0xd6, 0x5a, 0x73, 0x2c, 0x48, 0x31, 0x16,
	0xee, 0x3d, 0x3d, 0x2d, 0x0a, 0xf3, 0x54, 0x9a, 0xd9, 0xbf, 0x7c, 0xbf,
	0xb7, 0x8a, 0xd8, 0x84, 0xf8, 0x4d, 0x5b, 0xeb, 0x04, 0x72, 0x4d, 0xc7,
	0x36, 0x9b, 0x31, 0xde, 0xf3, 0x7d, 0x0c, 0xf5, 0x39, 0xe9, 0xcf, 0xcd,
	0xd3, 0xde, 0x65, 0x37, 0x29, 0xea, 0xd5, 0xd1
};

static unsigned char key1p[] = {
	0xd3, 0x27, 0x37, 0xe7, 0x26, 0x7f, 0xfe, 0x13, 0x41, 0xb2, 0xd5, 0xc0,
	0xd1, 0x50, 0xa8, 0x1b, 0x58, 0x6f, 0xb3, 0x13, 0x2b, 0xed, 0x2f, 0x8d,
	0x52, 0x62, 0x86, 0x4a, 0x9c, 0xb9, 0xf3, 0x0a, 0xf3, 0x8b, 0xe4, 0x48,
	0x59, 0x8d, 0x41, 0x3a, 0x17, 0x2e, 0xfb, 0x80, 0x2c, 0x21, 0xac, 0xf1,
	0xc1, 0x1c, 0x52, 0x0c, 0x2f, 0x26, 0xa4, 0x71, 0xdc, 0xad, 0x21, 0x2e,
	0xac, 0x7c, 0xa3, 0x9d
};

static unsigned char key1q[] = {
	0xcc, 0x88, 0x53, 0xd1, 0xd5, 0x4d, 0xa6, 0x30, 0xfa, 0xc0, 0x04, 0xf4,
	0x71, 0xf2, 0x81, 0xc7, 0xb8, 0x98, 0x2d, 0x82, 0x24, 0xa4, 0x90, 0xed,
	0xbe, 0xb3, 0x3d, 0x3e, 0x3d, 0x5c, 0xc9, 0x3c, 0x47, 0x65, 0x70, 0x3d,
	0x1d, 0xd7, 0x91, 0x64, 0x2f, 0x1f, 0x11, 0x6a, 0x0d, 0xd8, 0x52, 0xbe,
	0x24, 0x19, 0xb2, 0xaf, 0x72, 0xbf, 0xe9, 0xa0, 0x30, 0xe8, 0x60, 0xb0,
	0x28, 0x8b, 0x5d, 0x77
};

static unsigned char key1dP[] = {
	0x0e, 0x12, 0xbf, 0x17, 0x18, 0xe9, 0xce, 0xf5, 0x59, 0x9b, 0xa1, 0xc3,
	0x88, 0x2f, 0xe8, 0x04, 0x6a, 0x90, 0x87, 0x4e, 0xef, 0xce, 0x8f, 0x2c,
	0xcc, 0x20, 0xe4, 0xf2, 0x74, 0x1f, 0xb0, 0xa3, 0x3a, 0x38, 0x48, 0xae,
	0xc9, 0xc9, 0x30, 0x5f, 0xbe, 0xcb, 0xd2, 0xd7, 0x68, 0x19, 0x96, 0x7d,
	0x46, 0x71, 0xac, 0xc6, 0x43, 0x1e, 0x40, 0x37, 0x96, 0x8d, 0xb3, 0x78,
	0x78, 0xe6, 0x95, 0xc1
};

static unsigned char key1dQ[] = {
	0x95, 0x29, 0x7b, 0x0f, 0x95, 0xa2, 0xfa, 0x67, 0xd0, 0x07, 0x07, 0xd6,
	0x09, 0xdf, 0xd4, 0xfc, 0x05, 0xc8, 0x9d, 0xaf, 0xc2, 0xef, 0x6d, 0x6e,
	0xa5, 0x5b, 0xec, 0x77, 0x1e, 0xa3, 0x33, 0x73, 0x4d, 0x92, 0x51, 0xe7,
	0x90, 0x82, 0xec, 0xda, 0x86, 0x6e, 0xfe, 0xf1, 0x3c, 0x45, 0x9e, 0x1a,
	0x63, 0x13, 0x86, 0xb7, 0xe3, 0x54, 0xc8, 0x99, 0xf5, 0xf1, 0x12, 0xca,
	0x85, 0xd7, 0x15, 0x83
};

static unsigned char key1qP[] = {
	0x4f, 0x45, 0x6c, 0x50, 0x24, 0x93, 0xbd, 0xc0, 0xed, 0x2a, 0xb7, 0x56,
	0xa3, 0xa6, 0xed, 0x4d, 0x67, 0x35, 0x2a, 0x69, 0x7d, 0x42, 0x16, 0xe9,
	0x32, 0x12, 0xb1, 0x27, 0xa6, 0x3d, 0x54, 0x11, 0xce, 0x6f, 0xa9, 0x8d,
	0x5d, 0xbe, 0xfd, 0x73, 0x26, 0x3e, 0x37, 0x28, 0x14, 0x27, 0x43, 0x81,
	0x81, 0x66, 0xed, 0x7d, 0xd6, 0x36, 0x87, 0xdd, 0x2a, 0x8c, 0xa1, 0xd2,
	0xf4, 0xfb, 0xd8, 0xe1
};

static unsigned char key1TestMsg[] = {
	0x66, 0x28, 0x19, 0x4e, 0x12, 0x07, 0x3d, 0xb0, 0x3b, 0xa9, 0x4c, 0xda,
	0x9e, 0xf9, 0x53, 0x23, 0x97, 0xd5, 0x0d, 0xba, 0x79, 0xb9, 0x87, 0x00,
	0x4a, 0xfe, 0xfe, 0x34
};

static unsigned char key1SeedPad[] = {
	0x18, 0xb7, 0x76, 0xea, 0x21, 0x06, 0x9d, 0x69, 0x77, 0x6a, 0x33, 0xe9,
	0x6b, 0xad, 0x48, 0xe1, 0xdd, 0xa0, 0xa5, 0xef
};

static unsigned char key1EncryptedResult[] = {
	0x35, 0x4f, 0xe6, 0x7b, 0x4a, 0x12, 0x6d, 0x5d, 0x35, 0xfe, 0x36, 0xc7,
	0x77, 0x79, 0x1a, 0x3f, 0x7b, 0xa1, 0x3d, 0xef, 0x48, 0x4e, 0x2d, 0x39,
	0x08, 0xaf, 0xf7, 0x22, 0xfa, 0xd4, 0x68, 0xfb, 0x21, 0x69, 0x6d, 0xe9,
	0x5d, 0x0b, 0xe9, 0x11, 0xc2, 0xd3, 0x17, 0x4f, 0x8a, 0xfc, 0xc2, 0x01,
	0x03, 0x5f, 0x7b, 0x6d, 0x8e, 0x69, 0x40, 0x2d, 0xe5, 0x45, 0x16, 0x18,
	0xc2, 0x1a, 0x53, 0x5f, 0xa9, 0xd7, 0xbf, 0xc5, 0xb8, 0xdd, 0x9f, 0xc2,
	0x43, 0xf8, 0xcf, 0x92, 0x7d, 0xb3, 0x13, 0x22, 0xd6, 0xe8, 0x81, 0xea,
	0xa9, 0x1a, 0x99, 0x61, 0x70, 0xe6, 0x57, 0xa0, 0x5a, 0x26, 0x64, 0x26,
	0xd9, 0x8c, 0x88, 0x00, 0x3f, 0x84, 0x77, 0xc1, 0x22, 0x70, 0x94, 0xa0,
	0xd9, 0xfa, 0x1e, 0x8c, 0x40, 0x24, 0x30, 0x9c, 0xe1, 0xec, 0xcc, 0xb5,
	0x21, 0x00, 0x35, 0xd4, 0x7a, 0xc7, 0x2e, 0x8a
};

static int32 psRsaOaepVectorTest(void)
{
	psPool_t		*pool = NULL;
	psRsaKey_t		key1;
	pstm_int		mpN, mpe, mpd, mpp, mpq, mpdP, mpdQ, mpqP;
	uint32			digSize, outLen;
	unsigned char	*outOaep, *outRsaE, *outRsaD;
	int32			ret;

	outOaep = outRsaE = outRsaD = NULL;
	digSize = sizeof(pstm_digit);

	if (pstm_init_for_read_unsigned_bin(pool, &mpN, sizeof(key1N) + digSize)
			!= PS_SUCCESS) {
		return PS_FAILURE;
	}
	pstm_read_unsigned_bin(&mpN, key1N, sizeof(key1N));

	if (pstm_init_for_read_unsigned_bin(pool, &mpe, sizeof(key1e) + digSize)
			!= PS_SUCCESS) {
		return PS_FAILURE;
	}
	pstm_read_unsigned_bin(&mpe, key1e, sizeof(key1e));

	if (pstm_init_for_read_unsigned_bin(pool, &mpd, sizeof(key1d) + digSize)
			!= PS_SUCCESS) {
		return PS_FAILURE;
	}
	pstm_read_unsigned_bin(&mpd, key1d, sizeof(key1d));

	if (pstm_init_for_read_unsigned_bin(pool, &mpp, sizeof(key1p) + digSize)
			!= PS_SUCCESS) {
		return PS_FAILURE;
	}
	pstm_read_unsigned_bin(&mpp, key1p, sizeof(key1p));

	if (pstm_init_for_read_unsigned_bin(NULL, &mpq, sizeof(key1q) + digSize)
			!= PS_SUCCESS) {
		return PS_FAILURE;
	}
	pstm_read_unsigned_bin(&mpq, key1q, sizeof(key1q));

	if (pstm_init_for_read_unsigned_bin(pool, &mpdP, sizeof(key1dP) + digSize)
			!= PS_SUCCESS) {
		return PS_FAILURE;
	}
	pstm_read_unsigned_bin(&mpdP, key1dP, sizeof(key1dP));

	if (pstm_init_for_read_unsigned_bin(pool, &mpdQ, sizeof(key1dQ) + digSize)
			!= PS_SUCCESS) {
		return PS_FAILURE;
	}
	pstm_read_unsigned_bin(&mpdQ, key1dQ, sizeof(key1dQ));

	if (pstm_init_for_read_unsigned_bin(pool, &mpqP, sizeof(key1qP) + digSize)
			!= PS_SUCCESS) {
		return PS_FAILURE;
	}
	pstm_read_unsigned_bin(&mpqP, key1qP, sizeof(key1qP));

	key1.e = mpe;
	key1.N = mpN;
	key1.d = mpd;
	key1.p = mpp;
	key1.q = mpq;
	key1.dP = mpdP;
	key1.dQ = mpdQ;
	key1.qP = mpqP;
	key1.optimized = 1;
	outLen = key1.size = pstm_unsigned_bin_size(&mpN);

	outOaep = psMalloc(pool, outLen);
	pkcs1OaepEncode(pool, key1TestMsg, sizeof(key1TestMsg), NULL, 0,
		key1SeedPad, sizeof(key1SeedPad), key1.size * CHAR_BIT, 0, outOaep,
		&outLen);

	outRsaE = psMalloc(pool, outLen);
	psRsaCrypt(pool, outOaep, outLen, outRsaE, &outLen, &key1, PUBKEY_TYPE, NULL);

	if (memcmp(outRsaE, key1EncryptedResult, outLen) != 0) {
		_psTrace("ERROR in encrypting key1 RSAES_OAEP vector test\n");
		ret = PS_FAILURE;
		goto LBL_ERR;
	}

	outRsaD = psMalloc(pool, outLen);
	psRsaCrypt(pool, outRsaE, outLen, outRsaD, &outLen, &key1, PRIVKEY_TYPE, NULL);

	memset(outOaep, 0x0, key1.size);
	pkcs1OaepDecode(pool, outRsaD, outLen, NULL, 0, key1.size * CHAR_BIT, 0,
		outOaep, &outLen);

	if ((outLen != sizeof(key1TestMsg)) || (memcmp(outOaep, key1TestMsg, outLen)
			!= 0)) {
		_psTrace("ERROR in decrypting key1 RSAES_OAEP vector test\n");
		ret = PS_FAILURE;
		goto LBL_ERR;
	}

/*
	Now just run a plaintext test of our choosing with the MD5 hash and label
*/
	outLen = key1.size;
	if (pkcs1OaepEncode(pool, (const unsigned char*)"Hello", 5,
			(const unsigned char*)"mylabel", 7,	NULL, 0,
			key1.size * CHAR_BIT, 1, outOaep, &outLen) < PS_SUCCESS) {
		_psTrace("ERROR encoding standard RSAES_OAEP vector test\n");
		ret = PS_FAILURE;
		goto LBL_ERR;
	}
	psRsaCrypt(pool, outOaep, outLen, outRsaE, &outLen, &key1, PUBKEY_TYPE, NULL);

	psRsaCrypt(pool, outRsaE, outLen, outRsaD, &outLen, &key1, PRIVKEY_TYPE, NULL);
	pkcs1OaepDecode(pool, outRsaD, outLen, (const unsigned char*)"mylabel", 7,
		key1.size * CHAR_BIT, 1, outOaep, &outLen);

	if (memcmp(outOaep, "Hello", outLen) != 0) {
		_psTrace("ERROR in standard RSAES_OAEP vector test\n");
		ret = PS_FAILURE;
		goto LBL_ERR;
	}

	_psTrace("	PASSED\n");
	ret = PS_SUCCESS;
LBL_ERR:
	pstm_clear(&key1.N);
	pstm_clear(&key1.e);
	pstm_clear(&key1.d);
	pstm_clear(&key1.p);
	pstm_clear(&key1.q);
	pstm_clear(&key1.dP);
	pstm_clear(&key1.dQ);
	pstm_clear(&key1.qP);
	psFree(outOaep, pool);
	psFree(outRsaE, pool);
	psFree(outRsaD, pool);
	return ret;
}
#endif /* USE_PKCS1_OAEP */

/******************************************************************************/
#ifdef USE_PKCS1_PSS
/* PSS-VEC.TXT from RSA PKCS#1 web page */
static unsigned char key2N[] = {
	0xa5, 0x6e, 0x4a, 0x0e, 0x70, 0x10, 0x17, 0x58, 0x9a, 0x51, 0x87, 0xdc,
	0x7e, 0xa8, 0x41, 0xd1, 0x56, 0xf2, 0xec, 0x0e, 0x36, 0xad, 0x52, 0xa4,
	0x4d, 0xfe, 0xb1, 0xe6, 0x1f, 0x7a, 0xd9, 0x91, 0xd8, 0xc5, 0x10, 0x56,
	0xff, 0xed, 0xb1, 0x62, 0xb4, 0xc0, 0xf2, 0x83, 0xa1, 0x2a, 0x88, 0xa3,
	0x94, 0xdf, 0xf5, 0x26, 0xab, 0x72, 0x91, 0xcb, 0xb3, 0x07, 0xce, 0xab,
	0xfc, 0xe0, 0xb1, 0xdf, 0xd5, 0xcd, 0x95, 0x08, 0x09, 0x6d, 0x5b, 0x2b,
	0x8b, 0x6d, 0xf5, 0xd6, 0x71, 0xef, 0x63, 0x77, 0xc0, 0x92, 0x1c, 0xb2,
	0x3c, 0x27, 0x0a, 0x70, 0xe2, 0x59, 0x8e, 0x6f, 0xf8, 0x9d, 0x19, 0xf1,
	0x05, 0xac, 0xc2, 0xd3, 0xf0, 0xcb, 0x35, 0xf2, 0x92, 0x80, 0xe1, 0x38,
	0x6b, 0x6f, 0x64, 0xc4, 0xef, 0x22, 0xe1, 0xe1, 0xf2, 0x0d, 0x0c, 0xe8,
	0xcf, 0xfb, 0x22, 0x49, 0xbd, 0x9a, 0x21, 0x37
};
static unsigned char key2e[] = {0x01, 0x00, 0x01};

static unsigned char key2d[] = {
	0x33, 0xa5, 0x04, 0x2a, 0x90, 0xb2, 0x7d, 0x4f, 0x54, 0x51, 0xca, 0x9b,
	0xbb, 0xd0, 0xb4, 0x47, 0x71, 0xa1, 0x01, 0xaf, 0x88, 0x43, 0x40, 0xae,
	0xf9, 0x88, 0x5f, 0x2a, 0x4b, 0xbe, 0x92, 0xe8, 0x94, 0xa7, 0x24, 0xac,
	0x3c, 0x56, 0x8c, 0x8f, 0x97, 0x85, 0x3a, 0xd0, 0x7c, 0x02, 0x66, 0xc8,
	0xc6, 0xa3, 0xca, 0x09, 0x29, 0xf1, 0xe8, 0xf1, 0x12, 0x31, 0x88, 0x44,
	0x29, 0xfc, 0x4d, 0x9a, 0xe5, 0x5f, 0xee, 0x89, 0x6a, 0x10, 0xce, 0x70,
	0x7c, 0x3e, 0xd7, 0xe7, 0x34, 0xe4, 0x47, 0x27, 0xa3, 0x95, 0x74, 0x50,
	0x1a, 0x53, 0x26, 0x83, 0x10, 0x9c, 0x2a, 0xba, 0xca, 0xba, 0x28, 0x3c,
	0x31, 0xb4, 0xbd, 0x2f, 0x53, 0xc3, 0xee, 0x37, 0xe3, 0x52, 0xce, 0xe3,
	0x4f, 0x9e, 0x50, 0x3b, 0xd8, 0x0c, 0x06, 0x22, 0xad, 0x79, 0xc6, 0xdc,
	0xee, 0x88, 0x35, 0x47, 0xc6, 0xa3, 0xb3, 0x25
};

static unsigned char key2p[] = {
	0xe7, 0xe8, 0x94, 0x27, 0x20, 0xa8, 0x77, 0x51, 0x72, 0x73, 0xa3, 0x56,
	0x05, 0x3e, 0xa2, 0xa1, 0xbc, 0x0c, 0x94, 0xaa, 0x72, 0xd5, 0x5c, 0x6e,
	0x86, 0x29, 0x6b, 0x2d, 0xfc, 0x96, 0x79, 0x48, 0xc0, 0xa7, 0x2c, 0xbc,
	0xcc, 0xa7, 0xea, 0xcb, 0x35, 0x70, 0x6e, 0x09, 0xa1, 0xdf, 0x55, 0xa1,
	0x53, 0x5b, 0xd9, 0xb3, 0xcc, 0x34, 0x16, 0x0b, 0x3b, 0x6d, 0xcd, 0x3e,
	0xda, 0x8e, 0x64, 0x43
};

static unsigned char key2q[] = {
	0xb6, 0x9d, 0xca, 0x1c, 0xf7, 0xd4, 0xd7, 0xec, 0x81, 0xe7, 0x5b, 0x90,
	0xfc, 0xca, 0x87, 0x4a, 0xbc, 0xde, 0x12, 0x3f, 0xd2, 0x70, 0x01, 0x80,
	0xaa, 0x90, 0x47, 0x9b, 0x6e, 0x48, 0xde, 0x8d, 0x67, 0xed, 0x24, 0xf9,
	0xf1, 0x9d, 0x85, 0xba, 0x27, 0x58, 0x74, 0xf5, 0x42, 0xcd, 0x20, 0xdc,
	0x72, 0x3e, 0x69, 0x63, 0x36, 0x4a, 0x1f, 0x94, 0x25, 0x45, 0x2b, 0x26,
	0x9a, 0x67, 0x99, 0xfd
};

static unsigned char key2dP[] = {
	0x28, 0xfa, 0x13, 0x93, 0x86, 0x55, 0xbe, 0x1f, 0x8a, 0x15, 0x9c, 0xba,
	0xca, 0x5a, 0x72, 0xea, 0x19, 0x0c, 0x30, 0x08, 0x9e, 0x19, 0xcd, 0x27,
	0x4a, 0x55, 0x6f, 0x36, 0xc4, 0xf6, 0xe1, 0x9f, 0x55, 0x4b, 0x34, 0xc0,
	0x77, 0x79, 0x04, 0x27, 0xbb, 0xdd, 0x8d, 0xd3, 0xed, 0xe2, 0x44, 0x83,
	0x28, 0xf3, 0x85, 0xd8, 0x1b, 0x30, 0xe8, 0xe4, 0x3b, 0x2f, 0xff, 0xa0,
	0x27, 0x86, 0x19, 0x79
};

static unsigned char key2dQ[] = {
	0x1a, 0x8b, 0x38, 0xf3, 0x98, 0xfa, 0x71, 0x20, 0x49, 0x89, 0x8d, 0x7f,
	0xb7, 0x9e, 0xe0, 0xa7, 0x76, 0x68, 0x79, 0x12, 0x99, 0xcd, 0xfa, 0x09,
	0xef, 0xc0, 0xe5, 0x07, 0xac, 0xb2, 0x1e, 0xd7, 0x43, 0x01, 0xef, 0x5b,
	0xfd, 0x48, 0xbe, 0x45, 0x5e, 0xae, 0xb6, 0xe1, 0x67, 0x82, 0x55, 0x82,
	0x75, 0x80, 0xa8, 0xe4, 0xe8, 0xe1, 0x41, 0x51, 0xd1, 0x51, 0x0a, 0x82,
	0xa3, 0xf2, 0xe7, 0x29
};

static unsigned char key2qP[] = {
	0x27, 0x15, 0x6a, 0xba, 0x41, 0x26, 0xd2, 0x4a, 0x81, 0xf3, 0xa5, 0x28,
	0xcb, 0xfb, 0x27, 0xf5, 0x68, 0x86, 0xf8, 0x40, 0xa9, 0xf6, 0xe8, 0x6e,
	0x17, 0xa4, 0x4b, 0x94, 0xfe, 0x93, 0x19, 0x58, 0x4b, 0x8e, 0x22, 0xfd,
	0xde, 0x1e, 0x5a, 0x2e, 0x3b, 0xd8, 0xaa, 0x5b, 0xa8, 0xd8, 0x58, 0x41,
	0x94, 0xeb, 0x21, 0x90, 0xac, 0xf8, 0x32, 0xb8, 0x47, 0xf1, 0x3a, 0x3d,
	0x24, 0xa7, 0x9f, 0x4d
};

static unsigned char key2TestMsg[] = {
	0xcd, 0xc8, 0x7d, 0xa2, 0x23, 0xd7, 0x86, 0xdf, 0x3b, 0x45, 0xe0, 0xbb,
	0xbc, 0x72, 0x13, 0x26, 0xd1, 0xee, 0x2a, 0xf8, 0x06, 0xcc, 0x31, 0x54,
	0x75, 0xcc, 0x6f, 0x0d, 0x9c, 0x66, 0xe1, 0xb6, 0x23, 0x71, 0xd4, 0x5c,
	0xe2, 0x39, 0x2e, 0x1a, 0xc9, 0x28, 0x44, 0xc3, 0x10, 0x10, 0x2f, 0x15,
	0x6a, 0x0d, 0x8d, 0x52, 0xc1, 0xf4, 0xc4, 0x0b, 0xa3, 0xaa, 0x65, 0x09,
	0x57, 0x86, 0xcb, 0x76, 0x97, 0x57, 0xa6, 0x56, 0x3b, 0xa9, 0x58, 0xfe,
	0xd0, 0xbc, 0xc9, 0x84, 0xe8, 0xb5, 0x17, 0xa3, 0xd5, 0xf5, 0x15, 0xb2,
	0x3b, 0x8a, 0x41, 0xe7, 0x4a, 0xa8, 0x67, 0x69, 0x3f, 0x90, 0xdf, 0xb0,
	0x61, 0xa6, 0xe8, 0x6d, 0xfa, 0xae, 0xe6, 0x44, 0x72, 0xc0, 0x0e, 0x5f,
	0x20, 0x94, 0x57, 0x29, 0xcb, 0xeb, 0xe7, 0x7f, 0x06, 0xce, 0x78, 0xe0,
	0x8f, 0x40, 0x98, 0xfb, 0xa4, 0x1f, 0x9d, 0x61, 0x93, 0xc0, 0x31, 0x7e,
	0x8b, 0x60, 0xd4, 0xb6, 0x08, 0x4a, 0xcb, 0x42, 0xd2, 0x9e, 0x38, 0x08,
	0xa3, 0xbc, 0x37, 0x2d, 0x85, 0xe3, 0x31, 0x17, 0x0f, 0xcb, 0xf7, 0xcc,
	0x72, 0xd0, 0xb7, 0x1c, 0x29, 0x66, 0x48, 0xb3, 0xa4, 0xd1, 0x0f, 0x41,
	0x62, 0x95, 0xd0, 0x80, 0x7a, 0xa6, 0x25, 0xca, 0xb2, 0x74, 0x4f, 0xd9,
	0xea, 0x8f, 0xd2, 0x23, 0xc4, 0x25, 0x37, 0x02, 0x98, 0x28, 0xbd, 0x16,
	0xbe, 0x02, 0x54, 0x6f, 0x13, 0x0f, 0xd2, 0xe3, 0x3b, 0x93, 0x6d, 0x26,
	0x76, 0xe0, 0x8a, 0xed, 0x1b, 0x73, 0x31, 0x8b, 0x75, 0x0a, 0x01, 0x67,
	0xd0
};

static unsigned char key2salt[] = {
	0xde, 0xe9, 0x59, 0xc7, 0xe0, 0x64, 0x11, 0x36, 0x14, 0x20, 0xff, 0x80,
	0x18, 0x5e, 0xd5, 0x7f, 0x3e, 0x67, 0x76, 0xaf
};

static unsigned char key2sig[] = {
	0x90, 0x74, 0x30, 0x8f, 0xb5, 0x98, 0xe9, 0x70, 0x1b, 0x22, 0x94, 0x38,
	0x8e, 0x52, 0xf9, 0x71, 0xfa, 0xac, 0x2b, 0x60, 0xa5, 0x14, 0x5a, 0xf1,
	0x85, 0xdf, 0x52, 0x87, 0xb5, 0xed, 0x28, 0x87, 0xe5, 0x7c, 0xe7, 0xfd,
	0x44, 0xdc, 0x86, 0x34, 0xe4, 0x07, 0xc8, 0xe0, 0xe4, 0x36, 0x0b, 0xc2,
	0x26, 0xf3, 0xec, 0x22, 0x7f, 0x9d, 0x9e, 0x54, 0x63, 0x8e, 0x8d, 0x31,
	0xf5, 0x05, 0x12, 0x15, 0xdf, 0x6e, 0xbb, 0x9c, 0x2f, 0x95, 0x79, 0xaa,
	0x77, 0x59, 0x8a, 0x38, 0xf9, 0x14, 0xb5, 0xb9, 0xc1, 0xbd, 0x83, 0xc4,
	0xe2, 0xf9, 0xf3, 0x82, 0xa0, 0xd0, 0xaa, 0x35, 0x42, 0xff, 0xee, 0x65,
	0x98, 0x4a, 0x60, 0x1b, 0xc6, 0x9e, 0xb2, 0x8d, 0xeb, 0x27, 0xdc, 0xa1,
	0x2c, 0x82, 0xc2, 0xd4, 0xc3, 0xf6, 0x6c, 0xd5, 0x00, 0xf1, 0xff, 0x2b,
	0x99, 0x4d, 0x8a, 0x4e, 0x30, 0xcb, 0xb3, 0x3c
};

static int32 psRsaPssVectorTest(void)
{
	psPool_t		*pool = NULL;
	psRsaKey_t		key1;
	pstm_int		mpN, mpe, mpd, mpp, mpq, mpdP, mpdQ, mpqP;
	uint32			digSize, outLen;
	unsigned char	*outPss, *outRsaE, *outRsaD;
	psDigestContext_t md;
	unsigned char	hash[SHA1_HASH_SIZE];
	int32			result, ret;

	outPss = outRsaE = outRsaD = NULL;

	digSize = sizeof(pstm_digit);

	if (pstm_init_for_read_unsigned_bin(pool, &mpN, sizeof(key2N) + digSize)
			!= PS_SUCCESS) {
		return PS_FAILURE;
	}
	pstm_read_unsigned_bin(&mpN, key2N, sizeof(key2N));

	if (pstm_init_for_read_unsigned_bin(pool, &mpe, sizeof(key2e) + digSize)
			!= PS_SUCCESS) {
		return PS_FAILURE;
	}
	pstm_read_unsigned_bin(&mpe, key2e, sizeof(key2e));

	if (pstm_init_for_read_unsigned_bin(pool, &mpd, sizeof(key2d) + digSize)
			!= PS_SUCCESS) {
		return PS_FAILURE;
	}
	pstm_read_unsigned_bin(&mpd, key2d, sizeof(key2d));

	if (pstm_init_for_read_unsigned_bin(pool, &mpp, sizeof(key2p) + digSize)
			!= PS_SUCCESS) {
		return PS_FAILURE;
	}
	pstm_read_unsigned_bin(&mpp, key2p, sizeof(key2p));

	if (pstm_init_for_read_unsigned_bin(pool, &mpq, sizeof(key2q) + digSize)
			!= PS_SUCCESS) {
		return PS_FAILURE;
	}
	pstm_read_unsigned_bin(&mpq, key2q, sizeof(key2q));

	if (pstm_init_for_read_unsigned_bin(pool, &mpdP, sizeof(key2dP) + digSize)
			!= PS_SUCCESS) {
		return PS_FAILURE;
	}
	pstm_read_unsigned_bin(&mpdP, key2dP, sizeof(key2dP));

	if (pstm_init_for_read_unsigned_bin(pool, &mpdQ, sizeof(key2dQ) + digSize)
			!= PS_SUCCESS) {
		return PS_FAILURE;
	}
	pstm_read_unsigned_bin(&mpdQ, key2dQ, sizeof(key2dQ));

	if (pstm_init_for_read_unsigned_bin(pool, &mpqP, sizeof(key2qP) + digSize)
			!= PS_SUCCESS) {
		return PS_FAILURE;
	}
	pstm_read_unsigned_bin(&mpqP, key2qP, sizeof(key2qP));

	key1.e = mpe;
	key1.N = mpN;
	key1.d = mpd;
	key1.p = mpp;
	key1.q = mpq;
	key1.dP = mpdP;
	key1.dQ = mpdQ;
	key1.qP = mpqP;
	key1.optimized = 1;
	outLen = key1.size = pstm_unsigned_bin_size(&mpN);

/*
	First take SHA-1 hash of test message
*/
	psSha1Init(&md);
	psSha1Update(&md, key2TestMsg, sizeof(key2TestMsg));
	psSha1Final(&md, hash);

/*
	Then PSS
*/
	outPss = psMalloc(pool, outLen);
	pkcs1PssEncode(pool, hash, SHA1_HASH_SIZE, key2salt, sizeof(key2salt),
			0,	key1.size * CHAR_BIT, outPss, &outLen);

	outRsaE = psMalloc(pool, outLen);
/*
	Then sign
*/
	psRsaCrypt(pool, outPss, outLen, outRsaE, &outLen, &key1, PRIVKEY_TYPE, NULL);

/*
	Check against vector
*/
	if (memcmp(outRsaE, key2sig, outLen) != 0) {
		_psTrace("ERROR in encrypting key1 RSASSA_PSS vector test\n");
		ret = PS_FAILURE;
		goto LBL_ERR;
	}
/*
	Verify
*/
	outRsaD = psMalloc(pool, outLen);
/*
	Decrypt
*/
	psRsaCrypt(pool, outRsaE, outLen, outRsaD, &outLen, &key1, PUBKEY_TYPE, NULL);
/*
	PSS decode
*/
	pkcs1PssDecode(pool, hash, SHA1_HASH_SIZE, outRsaD, outLen,
		sizeof(key2salt), 0, key1.size * CHAR_BIT, &result);

	if (result != 1) {
		_psTrace("ERROR in decrypting key1 RSASSA_PSS vector test\n");
		ret = PS_FAILURE;
		goto LBL_ERR;
	}

	_psTrace("	PASSED\n");
	ret = PS_SUCCESS;

LBL_ERR:
	pstm_clear(&key1.N);
	pstm_clear(&key1.e);
	pstm_clear(&key1.d);
	pstm_clear(&key1.p);
	pstm_clear(&key1.q);
	pstm_clear(&key1.dP);
	pstm_clear(&key1.dQ);
	pstm_clear(&key1.qP);
	psFree(outRsaD, pool);
	psFree(outPss, pool);
	psFree(outRsaE, pool);
	return ret;
}

#endif /* USE_PKCS1_PSS */

#endif /* USE_RSA */
/******************************************************************************/

typedef struct {
	int32	(*fn)(void);
	char	name[64];
} test_t;

static test_t tests[] = {
#ifdef USE_AES
{psAesTestBlock, "***** AES BLOCK TESTS *****"},
{psAesTestCBC, "***** AES-CBC TESTS *****"},
#ifdef USE_AES_GCM
{psAesTestGCM, "***** AES-GCM TESTS *****"},
#endif
#else
{NULL, "AES"},
#endif

#ifdef USE_PKCS5
{psPBKDF2, "***** PBKDF2 TESTS *****"},
#endif

#ifdef USE_DES
{psDesTest
#else
{NULL
#endif
, "***** DES BLOCK TESTS *****"},

#ifdef USE_3DES
{psDes3Test
#else
{NULL
#endif
, "***** 3DES TESTS *****"},

#ifdef USE_SEED
{psSeedTest
#else
{NULL
#endif
, "***** SEED TESTS *****"},

#ifdef USE_ARC4
{psArc4Test
#else
{NULL
#endif
, "***** RC4 TESTS *****"},

#ifdef USE_IDEA
{psIdeaTest
#else
{NULL
#endif
, "***** IDEA TESTS *****"},


#ifdef USE_SHA1
{psSha1Test
#else
{NULL
#endif
, "***** SHA1 TESTS *****"},

#ifdef USE_SHA256
{psSha256Test
#else
{NULL
#endif
, "***** SHA256 TESTS *****"},

#ifdef USE_SHA224
{psSha224Test
#else
{NULL
#endif
, "***** SHA224 TESTS *****"},

#ifdef USE_SHA384
{psSha384Test
#else
{NULL
#endif
, "***** SHA384 TESTS *****"},

#ifdef USE_SHA512
{psSha512Test
#else
{NULL
#endif
, "***** SHA512 TESTS *****"},

#ifdef USE_MD5
{psMd5Test
#else
{NULL
#endif
, "***** MD5 TESTS *****"},

#ifdef USE_MD4
{psMd4Test
#else
{NULL
#endif
, "***** MD4 TESTS *****"},

#ifdef USE_MD2
{psMd2Test
#else
{NULL
#endif
, "***** MD2 TESTS *****"},

#ifdef USE_RSA
{psRsaEncryptTest
#else
{NULL
#endif
, "***** RSA ENCRYPT TESTS *****"},

#ifdef USE_RSA
{psRsaSignTest
#else
{NULL
#endif
, "***** RSA SIGN TESTS *****"},

#ifdef USE_PKCS1_OAEP
{psRsaOaepVectorTest
#else
{NULL
#endif
, "***** RSA RSAES_OAEP TESTS *****"},

#ifdef USE_PKCS1_PSS
{psRsaPssVectorTest
#else
{NULL
#endif
, "***** RSA RSASSA_PSS TESTS *****"},

{NULL, ""}
};

/******************************************************************************/
/*
	Main
*/

int main(int argc, char **argv)
{
	int32		i;

	if (psCryptoOpen(PSCRYPTO_CONFIG) < PS_SUCCESS) {
		_psTrace("Failed to initialize library:  psCryptoOpen failed\n");
		return -1;
	}

	for (i = 0; *tests[i].name; i++) {
		if (tests[i].fn) {
			_psTraceStr("%s\n", tests[i].name);
			tests[i].fn();
		} else {
			_psTraceStr("%s: SKIPPED\n", tests[i].name);
		}
	}
	psCryptoClose();

#ifdef WIN32
	_psTrace("Press any key to close");
	getchar();
#endif

	return 0;
}
