#!/usr/bin/python3
"""
Compiler ID  handling for dhfortran

Copyright (C) 2025 Alastair McKinstry <mckinstry@debian.org>
Released under the GPL-3 Gnu Public License.
"""

import os
import logging

log = logging.getLogger("dhfortran")

from subprocess import check_output

_multiarch = check_output(["dpkg-architecture", "-qDEB_HOST_MULTIARCH"]).strip()

# compiler exe name
compiler_list = {
    "gfortran-13": "gfortran13",
    "gfortran-14": "gfortran14",
    "gfortran-15": "gfortran15",
    "flang-new-17": "flang17",
    "flang-new-18": "flang18",
    "flang-new-19": "flang19",
    "flang-new-20": "flang20",
    "flang-17": "flang17",
    "flang-18": "flang18",
    "flang-19": "flang19",
    "flang-20": "flang20",
    "flang-to-external-fc-17": "flangext17",
    "flang-to-external-fc-18": "flangext18",
    "lfortran": "lfortran",
    # Commercial compilers. More work needed
    "ifort": "intel",
    "ifc": "intel",
    "efc": "intel",
    "ifx": "intel",
    "pgf95": "pgi",
    "pgf90": "pgi",
    "pgf77": "pgi",
    "pgfortran": "pgi",
    # TODO add more
}

#


def _basename(fc: str) -> str:
    return fc.split("/")[-1]


def get_fc_flavor(fc: str) -> str:
    """Given a compiler name, work out flavor"""
    base = _basename(fc)
    if base in compiler_list:
        return compiler_list[base]

    parts = base.split("-")
    if "gfortran" in parts:
        return f"gfortran{parts[-1]}"

    # TODO: mpifort , h5fc

    raise Exception("Can't recognize compiler {fc}")


def get_fc_default() -> str:
    """Return the default Fortran compiler"""
    if "FC" in os.environ():
        return get_fc_flavor(os.envion["FC"])

    f95 = os.readlink("/etc/alternatives/f95")
    # TODO add handling for broken symlinks, etc
    return get_fc_flavor(f95)


def get_fc_optional_compilers() -> str:
    """Return the list of other compilers present on the system"""
    compilers = []
    for f in compiler_list.keys():
        if os.exists(f"/usr/bin/{f}"):
            compilers.append(compiler_list[f])
    return compilers


def get_fort_root(fc: str, flags=None) -> str:
    """Return the  $FLIBDIR root directory. Depends on target architecture"""
    # TODO
    return f"/usr/lib/{_multiarch}/fortran"


def fc_target_arch_flang(fcflags: str) -> str | None:
    """Return target triple from flang ,
    or None for default
    """
    flags = fcflags.split()

    if "-target" in flags:
        try:
            target = flags[flags.index("-target") + 1]
        except Exception as ex:
            raise Exception("Parse error finding -target in flang flags list")
        try:
            return triplet[target]
        except:
            raise Exception(f"Unknown arch target {target} in flang flags")
    return None


def fc_target_arch_lfortran(fcflags: str) -> str:
    """Return the target triple for an lfortran compilation
    TODO: Needs more work
    """

    flags = fcflags.split()

    if "-target" in flags:
        try:
            target = flags[flags.index("-target") + 1]
        except Exception as ex:
            raise Exception("Parse error finding -target in lfortran flags list")

        # Translate target archs to GNU triplet
        archs = {
            "aarch64": "aarch64-linux-gnu",  #   - AArch64 (little endian)
            "aarch64_32": None,  #  - AArch64 (little endian ILP32) unsupported
            "aarch64_be": "aarch64_be-linux-gnu",  #  AArch64 (big endian) unsupported on Debian
            "arm64": "aarch64-linux-gnu",  # ARM64 (little endian)
            "arm64_32": None,  #   - ARM64 (little endian ILP32)
            "x86": "i386-linux-gnu",  #     - 32-bit X86: Pentium-Pro and above
            "x86-64": "x86_64-linux-gnu",  #     - 64-bit X86: EM64T and AMD64
        }
        return archs[target]
    return None


def get_fc_exe(flavor=None) -> str:
    """Return compiler name,  suitable for -DCMAKE_Fortran_COMPILER=$(call get_fc_exe,XXX)"""

    exes = {
        "gfortran": "/usr/bin/gfortran-14",
        "gfortran14": ";/usr/bin/gfortran-14",
    }
    if flavor in exes:
        return exes[flavor]
    return get_fc_default()


def get_abi_vendor(flavor: str) -> str:
    """Return vendor name"""

    # set(_Fortran_COMPILER_NAMES_LCC       lfortran gfortran)
    # set(_Fortran_COMPILER_NAMES_GNU       gfortran)
    # set(_Fortran_COMPILER_NAMES_Intel     ifort ifc efc ifx)
    # set(_Fortran_COMPILER_NAMES_Absoft    af95 af90 af77)
    # set(_Fortran_COMPILER_NAMES_PGI       pgf95 pgfortran pgf90 pgf77)
    # set(_Fortran_COMPILER_NAMES_Flang     flang)
    # set(_Fortran_COMPILER_NAMES_LLVMFlang flang)
    # set(_Fortran_COMPILER_NAMES_PathScale pathf2003 pathf95 pathf90)
    # set(_Fortran_COMPILER_NAMES_XL        xlf)
    # set(_Fortran_COMPILER_NAMES_VisualAge xlf95 xlf90 xlf)
    # set(_Fortran_COMPILER_NAMES_NAG       nagfor)
    # set(_Fortran_COMPILER_NAMES_NVHPC     nvfortran)

    if flavor.startswith("gfortran") or flavor.startswith("flangext"):
        return "gfortran"
    if flavor.startswith("flang"):
        return "flang"
    if flavor.startswith("lfortran"):
        # Not presumed to be ABI-compatible with gfortran, but likely
        return "lfortran"
    # Commercial compilers; names from CMake 3.11 Fortran ID
    if flavor.startswith("path"):
        return "pathscale"

    raise Exception(f"ABI Vendor unknown for flavor {flavor}")
