from typing import List

from xsdata.codegen.container import ClassContainer
from xsdata.codegen.models import Class
from xsdata.codegen.validator import ClassValidator
from xsdata.exceptions import AnalyzerValueError


class ClassAnalyzer:
    """Validate, analyze, sanitize and select the final class list to be
    generated."""

    @classmethod
    def process(cls, container: ClassContainer) -> List[Class]:
        """Run all the processes."""

        # Run validation checks for duplicate, invalid and redefined types.
        ClassValidator(container).process()

        # Run analyzer handlers
        container.process()

        classes = list(container)
        cls.validate_references(classes)

        return classes

    @classmethod
    def class_references(cls, target: Class) -> List:
        """Produce a list of instance references for the given class."""
        result = [id(target)]
        for attr in target.attrs:
            result.append(id(attr))
            result.extend(id(attr_type) for attr_type in attr.types)

        for extension in target.extensions:
            result.append(id(extension))
            result.append(id(extension.type))

        for inner in target.inner:
            result.extend(cls.class_references(inner))

        return result

    @classmethod
    def validate_references(cls, classes: List[Class]):
        """Validate all code gen objects are not cross referenced."""
        references = [ref for obj in classes for ref in cls.class_references(obj)]
        if len(references) != len(set(references)):
            raise AnalyzerValueError("Cross references detected!")
