<?php
/**
 * Class for importing via the webinterface, extends 'general' Import class.
 *
 * Zoph is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Zoph is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with Zoph; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 * @author Jeroen Roos
 * @package Zoph
 */

namespace import;

use DomDocument;
use conf\conf;
use file;
use log;
use photo;
use template\template;
use xmp\decoder as xmpdecoder;
use xmp\reader as xmpreader;

/**
 * This class holds all the functions for uploading and importing images
 * to Zoph via the web interface.
 *
 * @author Jeroen Roos
 * @package Zoph
 */
class web extends base {

    /** @param Name of the root node in XML responses */
    const XMLROOT="importprogress";
    /** @param Name of the leaf nodes in XML responses */
    const XMLNODE="import";

    /**
     * Create object, used to track progress of upload
     * @return import\web The created object
     * @param string generated upload id
     */
    public function __construct(private string $uploadId) {
    }

    /**
     * Import photos
     *
     * Takes an array of files and an array of vars and imports them in Zoph
     * @param array Files to be imported
     * @param array Vars to be applied to the photos.
     */
    public static function photos(array $files, array $vars) : array {
        // thumbnails have already been created, no need to repeat...
        conf::set("import.cli.thumbs", false);
        conf::set("import.cli.exif", true);
        conf::set("import.cli.size", true);
        return parent::photos($files, $vars);
    }

    /**
     * Return a translated, textual error message from a PHP upload error
     *
     * @param int PHP upload error
     */
    public static function handleUploadErrors(int $error) : string {
        $errortext=translate("File upload failed") . "<br>";
        switch ($error) {
        case UPLOAD_ERR_INI_SIZE:
            $errortext.=sprintf(translate("The uploaded file exceeds the " .
                "upload_max_filesize directive (%s) in php.ini."),
                ini_get("upload_max_filesize"));
            $errortext.=" " . sprintf(translate("This may also be caused by " .
                "the post_max_size (%s) in php.ini."), ini_get("post_max_size"));
            break;
        case UPLOAD_ERR_FORM_SIZE:
            $errortext.=sprintf(translate("The uploaded file exceeds the maximum " .
                "filesize setting in config.inc.php (%s)."), conf::get("import.maxupload"));
            break;
        case UPLOAD_ERR_PARTIAL:
            $errortext.=translate("The uploaded file was only partially uploaded.");
            break;
        case UPLOAD_ERR_NO_FILE:
            $errortext.=translate("No file was uploaded.");
            break;
        case UPLOAD_ERR_NO_TMP_DIR:
            $errortext.=translate("Missing a temporary folder.");
            break;
        case UPLOAD_ERR_CANT_WRITE:
            $errortext.=translate("Failed to write to disk");
            break;
        case UPLOAD_ERR_EXTENSION:
            $errortext.=translate("A PHP extension stopped the upload. Don't ask me why.");
            break;
        default:
            $errortext.=translate("An unknown file upload error occurred.");
        }
        return $errortext;
    }

    /**
     * Process uploaded file
     *
     * Catches the uploaded file, runs some checks and moves it into the
     * upload directory.
     * @param array PHP _FILE var with data about the uploaded file
     */
    public static function processUpload(array $file) : bool {
        $filename=$file["name"];
        $tmpName=$file["tmp_name"];

        $error=$file["error"];

        if ($error) {
            // should do some nicer printing to this error some time
            log::msg(static::handleUploadErrors($error), log::FATAL, log::IMPORT);
            return false;
        }

        $file=new file($tmpName);
        $mime=$file->getMime();

        if (!$file->type) {
            log::msg("Illegal filetype: $mime", log::FATAL, log::IMPORT);
            return false;
        }

        $dir=conf::get("path.images") . DIRECTORY_SEPARATOR . conf::get("path.upload");
        $realDir=realpath($dir);
        if ($realDir === false) {
            log::msg($dir . " does not exist, creating...", log::WARN, log::IMPORT);
            try {
                file::createDirRecursive($dir);
            } catch (\fileDirCreationFailedException $e) {
                log::msg($dir . " does not exist, and I can not create it. (" .
                    $e->getMessage() . ")", log::FATAL, log::IMPORT);
                die();
            }
            // doublecheck if path really has been correctly created.
            $realDir=realpath($dir);
            if ($realDir === false) {
                log::msg($dir . " does not exist, and I can not create it.", log::WARN, log::FATAL);
            }
        }
        $dir=$realDir;
        $dest=$dir . "/" . basename($filename);
        if (is_writable($dir)) {
            if (!file_exists($dest)) {
                move_uploaded_file($tmpName, $dest);
            } else {
                log::msg("A file named <b>" . $filename .
                    "</b> already exists in <b>" . $dir . "</b>", log::FATAL, log::IMPORT);
            }
        } else {
            log::msg("Directory <b>" . $dir . "</b> is not writable",
                log::FATAL, log::IMPORT);
            return false;
        }
        return true;
    }

    /**
     * Processes a file
     *
     * Depending on file type it will either launch a resize or an unpack
     * function.
     * This function is called from a javascript call
     * @param string MD5 hash of the file <b>name</b>.
     */
    public static function processFile(string $md5) : ?bool {
        // continue when hitting fatal error.
        log::$stopOnFatal=false;

        $dir=conf::get("path.images") . "/" . conf::get("path.upload") . "/";
        $file=file::getFromMD5($dir, $md5);

        if ($file instanceof file) {
            $mime=$file->getMime();
            $type=$file->type;
        } else {
            $type="unknown (file not found)";
        }

        switch ($type) {
        case "image":
            if ($mime=="image/jpeg" && conf::get("import.rotate")) {
                static::autorotate($file);
            }
            static::resizeImage($file);
            $return=null;
            break;
        case "archive":
            $return=static::unpackArchive($file);
            break;
        case "gpx":
            static::XMLimport($file);
            $return=null;
            break;
        default:
            log::msg("Unknown filetype " . $type .
                 " for file" . $file, log::FATAL, log::IMPORT);
            $return=false;
            break;
        }
        return $return;
    }

    /**
     * Automatically rotate images based on EXIF tag.
     * @param string filename
     */
    protected static function autorotate(string $file) : void {
        try {
            parent::autorotate($file);
        } catch (\importAutorotException $e) {
            touch($file . ".zophignore");
            log::msg($e->getMessage(), log::FATAL, log::IMPORT);
            die;
        }
    }

    /**
     * Unpack archive of different types
     * *WARNING* this function is *not* safe to run on unchecked user-input
     * use processFile() as a wrapper for this function
     * @see processFile
     * @param string full path to file
     */
    private static function unpackArchive(file $file) : bool {
        $dir = conf::get("path.images") . "/" . conf::get("path.upload");
        $mime=$file->getMime();
        switch ($mime) {
        case "application/zip":
            $extr = conf::get("path.unzip");
            $msg = "Unzip command";
            break;
        case "application/x-tar":
            $extr = conf::get("path.untar");
            $msg = "Untar command";
            break;
        case "application/gzip":
        case "application/x-gzip":
            $extr = conf::get("path.ungz");
            $msg = "Ungzip command";
            break;
        case "application/x-bzip2":
            $extr = conf::get("path.unbz");
            $msg = "Unbzip command";
            break;
        default:
            touch($file . ".zophignore");
            throw new importFileNotImportableException(basename($file) . " has an unknown archive format, " . $mime . ".");
            break;
        }
        if (empty($extr)) {
            log::msg("To be able to process an archive of type " . $mime .
                ", you need to set \"" . $msg . "\" in the configuration screen " .
                " to a program that can unpack this file.", log::FATAL, log::IMPORT);
            touch($file . ".zophignore");
            return false;
        }
        $uploadId=uniqid("zoph_");
        $unpackDir=$dir . "/" . $uploadId;
        $unpackFile=$unpackDir . "/" . basename($file);
        ob_start();
            mkdir($unpackDir);
            rename($file, $unpackFile);

            $cmd = "cd " . escapeshellarg($unpackDir) . " && " .
                $extr . " " .  escapeshellarg($unpackFile) . " 2>&1";
            system($cmd);
            if (file_exists($unpackFile)) {
                unlink($unpackFile);
            }
        $output=ob_end_clean();
        log::msg($output, log::NOTIFY, log::IMPORT);
        $files=file::getFromDir($unpackDir, true);
        foreach ($files as $importFile) {
            $type=$importFile->type;
            if ($type == "image" || $type == "archive" || $type == "xml") {
                $importFile->setDestination($dir);
                try {
                    $importFile->move();
                } catch (\fileException $e) {
                    echo $e->getMessage() . "<br>\n";
                }
            }
        }
        static::removeDir($unpackDir);
        return true;
    }

    /**
     * Remove dirs
     * Remove temporary directories that were created when unpacking an archive
     * Only removes emptry directories, a warning will be displayed when there are still files
     * left in the directory. This could happen when something went wrong during import or
     * non-image files were present in the archive.
     * @param string directory to traverse.
     */
    private static function removeDir(string $dir) : void {
        foreach (glob($dir . "/*", GLOB_ONLYDIR) as $subdir) {
            static::removeDir($subdir);
        }
        rmdir($dir);
    }

    /**
     * Resize an image before import
     *
     * @param string filename
     */
    private static function resizeImage(string $file) {
        log::msg("resizing" . $file, log::DEBUG, log::IMPORT);
        $photo = new photo();

        $photo->set("path", conf::get("path.upload"));
        $photo->set("name", basename($file));

        ob_start();
            $dir=conf::get("path.images") . "/" . conf::get("path.upload");
            $thumbDir=$dir. "/" . THUMB_PREFIX;
            $midDir=$dir . "/" . MID_PREFIX;
            if (!file_exists($thumbDir)) {
                mkdir($thumbDir);
            } else if (!is_dir($thumbDir)) {
                log::msg("Cannot create " . $thumbDir . ", file exists.", log::FATAL, log::IMPORT);
            }
            if (!file_exists($midDir)) {
                mkdir($midDir);
            } else if (!is_dir($midDir)) {
                log::msg("Cannot create " . $midDir . ", file exists.", log::FATAL, log::IMPORT);
            }
            try {
                $photo->thumbnail();
            } catch (\Exception $e) {
                echo "Thumb could not be made: " . $e->getMessage();
                touch($file . ".zophignore");
            }
            log::msg("Thumb made succesfully.", log::DEBUG, log::IMPORT);
        $log=ob_get_contents();
        ob_end_clean();
        echo $log;
    }

    /**
     * Get XML for Import
     */
    public static function getXML(string $search) : DomDocument {
        if ($search=="thumbs") {
            return static::getThumbsXML();
        }
    }

    /**
     * Generate an XML file with thumbs in the import dir
     */
    public static function getThumbsXML() : DomDocument {
        $xml=new DOMDocument('1.0', 'UTF-8');
        $root=$xml->createElement("files");

        $dir=conf::get("path.images") . DIRECTORY_SEPARATOR . conf::get("path.upload");
        $files = file::getFromDir($dir);

        foreach ($files as $file) {
            unset($icon);
            unset($status);
            unset($rating);
            $subject = null;

            $md5=$file->getMD5();

            $type=$file->type;

            switch ($type) {
            case "image":
                $thumb=THUMB_PREFIX . DIRECTORY_SEPARATOR . THUMB_PREFIX . "_" . $file->getName();
                $mid=MID_PREFIX . DIRECTORY_SEPARATOR . MID_PREFIX . "_" . $file->getName();
                if (file_exists($dir . DIRECTORY_SEPARATOR . $thumb) &&
                  file_exists($dir . DIRECTORY_SEPARATOR . $mid)) {
                    $status="done";
                    $xmp = new xmpreader($file);
                    $data=new xmpdecoder($xmp->getXMP());
                    if (sizeof($data) === 0) {
                        $data = new xmpdecoder($xmp->getXMPfromSidecar());
                        if ($data) {
                            $sidecar = $xmp->sidecar;
                        }
                    }
                    $subjects = $data->getSubjects();
                    $rating = $data->getRating();
                    $people = $data->getPeople();
                } else {
                    $icon=template::getImage("icons/pause.png");
                    $status="waiting";
                }
                break;
            case "archive":
                $icon=template::getImage("icons/archive.png");
                $status="waiting";
                break;
            case "gpx":
                $icon=template::getImage("icons/tracks.png");
                $status="done";
                break;
            case "ignore":
                $icon=template::getImage("icons/error.png");
                $status="ignore";
                break;
            case "xmp":
                // Don't show XMP files
                // Maybe at some point we can create an icon with the photo showing
                // an sidecar file was found for this photo
                continue 2;
                break;
            default:
                throw new importException("Unknown type: " . e(type));
            }

            $xmlfile=$xml->createElement("file");
            $xmlfile->setAttribute("name", $file->getName());
            $xmlfile->setAttribute("type", $type);
            $xmlmd5=$xml->createElement("md5", $md5);
            $xmlfile->appendChild($xmlmd5);
            if (!empty($icon)) {
                $xmlicon=$xml->createElement("icon", $icon);
                $xmlfile->appendChild($xmlicon);
            }
            if (!empty($status)) {
                $xmlstatus=$xml->createElement("status", $status);
                $xmlfile->appendChild($xmlstatus);
            }

            if (!empty($sidecar)) {
                $xmlSidecar=$xml->createElement("sidecar", $sidecar);
                $xmlfile->appendChild($xmlSidecar);
            }
            if (isset($subjects) && !empty($subjects)) {
                $xmlsubjects=$xml->createElement("subjects");
                foreach ($subjects as $subject) {
                    $xmlsubjects->appendChild($xml->createElement("subject", $subject));
                }
                $xmlfile->appendChild($xmlsubjects);
            }

            if (isset($rating)) {
                $xmlfile->appendChild($xml->createElement("rating", $rating));
            }

            if (isset($people)) {
                $xmlpeople=$xml->createElement("people");
                foreach ($people as $person) {
                    $xmlpeople->appendChild($xml->createElement("person", $person));
                }
                $xmlfile->appendChild($xmlpeople);
            }

            $root->appendChild($xmlfile);
        }
        $xml->appendChild($root);
        return $xml;
    }

    /**
     * Retry making of thumbnails
     *
     * This function reacts to a click on the "retry" link in the thumbnail
     * list on the import page. It looks up which file is referenced by the
     * supplied MD5 and deleted thumbnail, mid and 'ignore" files, this will
     * cause the webinterface to retry making thumbnail and midsize images
     *
     * @param string md5 hash of the filename
     */

    public static function retryFile(string $md5) : void {
        $dir=conf::get("path.images") . "/" . conf::get("path.upload");

        $file=file::getFromMD5($dir, $md5);
        // only delete "related files", not the referenced file.
        $file->delete(true, true);
    }

    /**
     * Delete a file
     *
     * Deletes a file referenced by the MD5 hash of the filename and all
     * related files, such as thumbnail, midsize images and "ignore" files.
     * @param string md5 hash of the filename
     */
    public static function deleteFile(string $md5) : void {
        $dir=conf::get("path.images") . "/" . conf::get("path.upload");

        $file=file::getFromMD5($dir, $md5);
        $file->delete(true);
    }

    /**
     * Get a file list from a list of MD5 hashes.
     *
     * Take a list of MD5 hashes (in $vars["_import_image"]) and return an
     * array of @see file objects
     * @param array $vars
     */
    public static function getFileList(array $import) : ?array {
        foreach ($import as $md5) {
            $file=file::getFromMD5(conf::get("path.images") . "/" . conf::get("path.upload"), $md5);
            if (!empty($file)) {
                $files[$md5]=$file;
            }
        }
        if (is_array($files)) {
            return $files;
        } else {
            log::msg("No files specified", log::FATAL, log::IMPORT);
            return false;
        }
    }

    public static function progress(int $cur, int $total) : void {
        /* progress is not currently shown in web interface */
    }
}

?>
