// Copyright Earl Warren <contact@earl-warren.org>
// Copyright Loïc Dachary <loic@dachary.org>
// SPDX-License-Identifier: MIT

package f3

import "time"

const (
	IssueStateOpen   = "open"
	IssueStateClosed = "closed"
)

type Issue struct {
	Common
	PosterID    *Reference    `json:"poster_id"`
	Assignees   []*Reference  `json:"assignees"`
	Labels      []*Reference  `json:"labels"`
	Title       string        `json:"title"`
	Content     string        `json:"content"`
	Attachments []*Attachment `json:"attachments"`
	Milestone   *Reference    `json:"milestone"`
	State       string        `json:"state"` // open, closed
	IsLocked    bool          `json:"is_locked"`
	Created     time.Time     `json:"created"`
	Updated     time.Time     `json:"updated"`
	Closed      *time.Time    `json:"closed"`
}

func (o *Issue) GetReferences() References {
	references := o.Common.GetReferences()
	for _, assignee := range o.Assignees {
		references = append(references, assignee)
	}
	for _, label := range o.Labels {
		references = append(references, label)
	}
	if !o.Milestone.IsNil() {
		references = append(references, o.Milestone)
	}
	return append(references, o.PosterID)
}

func (o Issue) Equal(other Issue) bool {
	return o.Common.Equal(other.Common) &&
		nilOrEqual(o.PosterID, other.PosterID) &&
		arrayEqual(o.Assignees, other.Assignees) &&
		arrayEqual(o.Labels, other.Labels) &&
		o.Title == other.Title &&
		arrayEqual(o.Attachments, other.Attachments) &&
		nilOrEqual(o.Milestone, other.Milestone) &&
		o.State == other.State &&
		o.IsLocked == other.IsLocked
}

func (o *Issue) Clone() Interface {
	clone := &Issue{}
	*clone = *o
	return clone
}
