/*
 *  $Id: terracefit.c 22777 2020-03-05 13:42:30Z yeti-dn $
 *  Copyright (C) 2019-2020 David Necas (Yeti).
 *  E-mail: yeti@gwyddion.net.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 */

/**
 * TODO:
 * - Add an option to exclude terraces which do not form full rings.  Petr
 *   thinks people will want it.  I don't.
 **/

#include "config.h"
#include <gtk/gtk.h>
#include <libgwyddion/gwymacros.h>
#include <libgwyddion/gwymath.h>
#include <libgwyddion/gwythreads.h>
#include <libprocess/gwyprocess.h>
#include <libgwydgets/gwystock.h>
#include <libgwydgets/gwycombobox.h>
#include <libgwydgets/gwycheckboxes.h>
#include <libgwydgets/gwynullstore.h>
#include <libgwymodule/gwymodule-process.h>
#include <app/gwymoduleutils.h>
#include <app/gwyapp.h>
#include "libgwyddion/gwyomp.h"
#include "preview.h"

#define TERRACE_RUN_MODES (GWY_RUN_INTERACTIVE)

#define FIT_GRADIENT_NAME "__GwyFitDiffGradient"

/* Lower symmetric part indexing */
/* i MUST be greater or equal than j */
#define SLi(a, i, j) a[(i)*((i) + 1)/2 + (j)]

#define coord_index(a,i) g_array_index(a,TerraceCoords,i)
#define info_index(a,i) g_array_index(a,TerraceInfo,i)

#define MAX_BROADEN 128.0
#define pwr 0.65

enum {
    MAX_DEGREE = 18,
};

typedef enum {
    PREVIEW_DATA       = 0,
    PREVIEW_SEGMENTED  = 1,
    PREVIEW_FITTED     = 2,
    PREVIEW_RESIDUUM   = 3,
    PREVIEW_TERRACES   = 4,
    PREVIEW_LEVELLED   = 5,
    PREVIEW_BACKGROUND = 6,
    PREVIEW_NTYPES
} PreviewMode;

typedef enum {
    OUTPUT_SEGMENTED  = (1 << 0),
    OUTPUT_FITTED     = (1 << 1),
    OUTPUT_RESIDUUM   = (1 << 2),
    OUTPUT_TERRACES   = (1 << 3),
    OUTPUT_LEVELLED   = (1 << 4),
    OUTPUT_BACKGROUND = (1 << 5),
    OUTPUT_ALL        = (1 << 6) - 1,
} OutputFlags;

typedef struct {
    gint poly_degree;
    gdouble edge_kernel_size;
    gdouble edge_threshold;
    GwyMaskingType masking;
    GwyResultsReportType report_style;
    gdouble min_area_frac;
    gdouble edge_broadening;
    gboolean independent;
    gboolean use_only_mask;
    guint output_flags;

    gboolean survey_poly;
    gint poly_degree_min;
    gint poly_degree_max;
    gboolean survey_broadening;
    gint broadening_min;
    gint broadening_max;

    PreviewMode preview_mode;
} TerraceArgs;

static const gchar *guivalues[] = {
    "step", "resid", "discrep", "nterraces",
};

typedef struct {
    GwyXYZ *xyz;
    guint *pixels;
    guint ncoords;
    gint level;
    /* Quantities gathered for terrace info. */
    gdouble msq;
    gdouble off;
} TerraceCoords;

typedef struct {
    TerraceArgs *args;
    GtkWidget *dialogue;
    GtkObject *edge_kernel_size;
    GtkObject *edge_threshold;
    GtkObject *edge_broadening;
    GtkObject *poly_degree;
    GtkObject *min_area_frac;
    GtkWidget *masking;
    GtkWidget *preview_mode;
    GtkWidget *independent;
    GtkWidget *use_only_mask;
    GtkWidget *color_button;
    GwyResults *results;
    GtkWidget *guivalues[G_N_ELEMENTS(guivalues)];
    GtkWidget *rexport_result;
    GtkWidget *message;
    GtkWidget *view;
    GtkWidget *terracelist;
    GtkWidget *rexport_list;
    GSList *output_flags;
    GtkWidget *survey_table;
    GtkWidget *survey_poly;
    GtkObject *poly_degree_min;
    GtkObject *poly_degree_max;
    GtkWidget *survey_broadening;
    GtkObject *broadening_min;
    GtkObject *broadening_max;
    GtkWidget *run_survey;
    GtkWidget *survey_message;
    GArray *terraceinfo;
    GwyGradient *diff_gradient;
    GwyContainer *mydata;
    GwyDataField *dfield;
    GwyDataField *mask;
    GdkPixbuf *colourpixbuf;
    GwySIValueFormat *vf;
    GArray *terracecoords;      /* Non-NULL if we have segmented terraces. */
    gboolean fit_ok;            /* We have fitted terraces. */
    gdouble xc, yc;
} TerraceControls;

typedef struct {
    guint nterrparam;
    guint npowers;
    guint nterraces;
    gdouble msq;
    gdouble deltares;
    gdouble *solution;
    gdouble *invdiag;
} FitResult;

typedef struct {
    GwyRGBA colour;
    gdouble height;   /* estimate from free fit */
    gdouble error;    /* difference from free fit estimate */
    gdouble residuum; /* final fit residuum */
    guint npixels;
    gint level;
} TerraceInfo;

typedef struct {
    gint poly_degree;
    gdouble edge_kernel_size;
    gdouble edge_threshold;
    gdouble edge_broadening;
    gdouble min_area_frac;
    gint fit_ok;
    gint nterraces;
    gdouble step;
    gdouble step_err;
    gdouble msq;
    gdouble discrep;
} TerraceSurveyRow;

static gboolean   module_register          (void);
static void       terrace                  (GwyContainer *data,
                                            GwyRunType run);
static void       terrace_dialogue         (TerraceArgs *args,
                                            GwyContainer *data,
                                            GwyDataField *dfield,
                                            GwyDataField *mask,
                                            gint id);
static void       create_output_fields     (TerraceControls *controls,
                                            GwyContainer *data,
                                            gint id);
static void       terrace_dialog_update    (TerraceControls *controls,
                                            TerraceArgs *args);
static void       terrace_fit              (TerraceControls *controls);
static void       terrace_segment_do       (TerraceControls *controls);
static FitResult* terrace_do               (GwyDataField *marked,
                                            GwyDataField *residuum,
                                            GwyDataField *background,
                                            GwyDataField *terraces,
                                            GArray *terracecoords,
                                            GArray *terraceinfo,
                                            const TerraceArgs *args,
                                            gdouble xc,
                                            gdouble yc,
                                            gboolean fill_bg_and_terraces,
                                            const gchar **message);
static void       update_results           (TerraceControls *controls,
                                            FitResult *fres);
static void       update_terrace_colours   (TerraceControls *controls);
static GtkWidget* parameters_tab_new       (TerraceControls *controls);
static GtkWidget* terrace_list_tab_new     (TerraceControls *controls);
static GtkWidget* output_tab_new           (TerraceControls *controls);
static GtkWidget* survey_tab_new           (TerraceControls *controls);
static void       reset_images             (TerraceControls *controls);
static void       poly_degree_changed      (TerraceControls *controls,
                                            GtkAdjustment *adj);
static void       edge_kernel_size_changed (TerraceControls *controls,
                                            GtkAdjustment *adj);
static void       edge_threshold_changed   (TerraceControls *controls,
                                            GtkAdjustment *adj);
static void       edge_broadening_changed  (TerraceControls *controls,
                                            GtkAdjustment *adj);
static void       min_area_frac_changed    (TerraceControls *controls,
                                            GtkAdjustment *adj);
static void       masking_changed          (GtkComboBox *combo,
                                            TerraceControls *controls);
static void       independent_changed      (GtkToggleButton *toggle,
                                            TerraceControls *controls);
static void       use_only_mask_changed    (GtkToggleButton *toggle,
                                            TerraceControls *controls);
static void       preview_mode_changed     (GtkComboBox *combo,
                                            TerraceControls *controls);
static void       report_style_changed     (TerraceControls *controls,
                                            GwyResultsExport *rexport);
static void       output_flags_changed     (GtkWidget *button,
                                            TerraceControls *controls);
static void       survey_poly_changed      (GtkToggleButton *toggle,
                                            TerraceControls *controls);
static void       poly_degree_min_changed  (TerraceControls *controls,
                                            GtkAdjustment *adj);
static void       poly_degree_max_changed  (TerraceControls *controls,
                                            GtkAdjustment *adj);
static void       survey_broadening_changed(GtkToggleButton *toggle,
                                            TerraceControls *controls);
static void       broadening_min_changed   (TerraceControls *controls,
                                            GtkAdjustment *adj);
static void       broadening_max_changed   (TerraceControls *controls,
                                            GtkAdjustment *adj);
static void       update_sensitivity       (TerraceControls *controls);
static void       update_diff_gradient     (TerraceControls *controls);
static void       copy_report              (TerraceControls *controls);
static void       save_report              (TerraceControls *controls);
static gchar*     format_report            (TerraceControls *controls);
static guint      count_survey_items       (TerraceArgs *args,
                                            guint *pndegrees,
                                            guint *pnbroadenings);
static void       run_survey               (TerraceControls *controls);
static void       load_args                (GwyContainer *settings,
                                            TerraceArgs *args);
static void       save_args                (GwyContainer *settings,
                                            TerraceArgs *args);
static void       sanitize_args            (TerraceArgs *args);

static const GwyEnum output_flags[] = {
    { N_("Marked terraces"),       OUTPUT_SEGMENTED,  },
    { N_("Fitted shape"),          OUTPUT_FITTED,     },
    { N_("Difference"),            OUTPUT_RESIDUUM,   },
    { N_("Terraces (ideal)"),      OUTPUT_TERRACES,   },
    { N_("Leveled surface"),       OUTPUT_LEVELLED,   },
    { N_("Polynomial background"), OUTPUT_BACKGROUND, },
};

enum {
    OUTPUT_NFLAGS = G_N_ELEMENTS(output_flags)
};

static const TerraceArgs terrace_defaults = {
    8,
    3.5, 25.0,
    GWY_MASK_IGNORE, GWY_RESULTS_REPORT_TABSEP,
    1.5, 3.0,
    FALSE, FALSE,
    OUTPUT_SEGMENTED,

    FALSE, 0, MAX_DEGREE,
    FALSE, 0, MAX_BROADEN,

    PREVIEW_DATA,
};

static GwyModuleInfo module_info = {
    GWY_MODULE_ABI_VERSION,
    &module_register,
    N_("Fits terraces with polynomial background."),
    "Yeti <yeti@gwyddion.net>",
    "1.2",
    "David Nečas (Yeti)",
    "2019",
};

GWY_MODULE_QUERY2(module_info, terracefit)

static gboolean
module_register(void)
{
    gwy_process_func_register("terracefit",
                              (GwyProcessFunc)&terrace,
                              N_("/Measure _Features/_Terraces..."),
                              GWY_STOCK_TERRACE_MEASURE,
                              TERRACE_RUN_MODES,
                              GWY_MENU_FLAG_DATA,
                              N_("Fit terraces with polynomial background"));

    return TRUE;
}

static void
terrace(GwyContainer *data, GwyRunType run)
{
    GwyDataField *dfield, *mfield;
    TerraceArgs args;
    gint id;

    g_return_if_fail(run & TERRACE_RUN_MODES);
    gwy_app_data_browser_get_current(GWY_APP_DATA_FIELD, &dfield,
                                     GWY_APP_MASK_FIELD, &mfield,
                                     GWY_APP_DATA_FIELD_ID, &id,
                                     0);
    g_return_if_fail(dfield);

    load_args(gwy_app_settings_get(), &args);
    terrace_dialogue(&args, data, dfield, mfield, id);
    save_args(gwy_app_settings_get(), &args);
}

static void
set_up_other_field(GwyContainer *data, gint id,
                   GwyContainer *mydata, gint myid,
                   GwyDataField *dfield)
{
    GwyDataField *otherfield;
    GQuark quark;

    otherfield = gwy_data_field_new_alike(dfield, TRUE);
    quark = gwy_app_get_data_key_for_id(myid);
    gwy_container_set_object(mydata, quark, otherfield);
    quark = gwy_app_get_data_range_type_key_for_id(myid);
    gwy_container_set_enum(mydata, quark, GWY_LAYER_BASIC_RANGE_FULL);
    gwy_app_sync_data_items(data, mydata, id, myid, FALSE,
                            GWY_DATA_ITEM_GRADIENT,
                            GWY_DATA_ITEM_REAL_SQUARE,
                            0);
    g_object_unref(otherfield);
}

static void
free_terrace_coordinates(GArray *terracecoords)
{
    guint g;

    if (!terracecoords)
        return;

    for (g = 0; g < terracecoords->len; g++) {
        TerraceCoords *tc = &coord_index(terracecoords, g);
        g_free(tc->xyz);
        g_free(tc->pixels);
    }
    g_array_free(terracecoords, TRUE);
}

static void
terrace_dialogue(TerraceArgs *args,
                 GwyContainer *data,
                 GwyDataField *dfield,
                 GwyDataField *mask,
                 gint id)
{
    GtkWidget *hbox, *alignment, *notebook, *widget;
    GtkDialog *dialogue;
    TerraceControls controls;
    GwyResults *results;
    GwyDataField *otherfield;
    PreviewMode preview_mode;
    gint response, width, height;

    gwy_clear(&controls, 1);
    controls.args = args;
    controls.dfield = dfield;
    controls.mask = mask;
    controls.vf = gwy_data_field_get_value_format_z(dfield,
                                                     GWY_SI_UNIT_FORMAT_MARKUP,
                                                     NULL);
    controls.vf->precision++;
    controls.diff_gradient = gwy_inventory_new_item(gwy_gradients(),
                                                    GWY_GRADIENT_DEFAULT,
                                                    FIT_GRADIENT_NAME);
    gwy_resource_use(GWY_RESOURCE(controls.diff_gradient));

    gtk_icon_size_lookup(GTK_ICON_SIZE_MENU, &width, &height);
    height |= 1;
    controls.colourpixbuf = gdk_pixbuf_new(GDK_COLORSPACE_RGB, FALSE, 8,
                                           height, height);

    controls.results = results = gwy_results_new();
    gwy_results_add_header(results, N_("Fit Results"));
    gwy_results_add_value_str(results, "file", N_("File"));
    gwy_results_add_value_str(results, "image", N_("Image"));
    gwy_results_add_value_yesno(results, "masking", N_("Mask in use"));
    gwy_results_add_separator(results);
    /* TODO: We might also want to output the segmentation & fit settings. */
    gwy_results_add_value_z(results, "step", N_("Fitted step height"));
    gwy_results_add_value_z(results, "resid", N_("Mean square difference"));
    gwy_results_add_value_z(results, "discrep", N_("Terrace discrepancy"));
    gwy_results_add_value_int(results, "nterraces", N_("Number of terraces"));
    gwy_results_set_unit(results, "z", gwy_data_field_get_si_unit_z(dfield));
    gwy_results_fill_filename(results, "file", data);
    gwy_results_fill_channel(results, "image", data, id);

    controls.dialogue = gtk_dialog_new_with_buttons(_("Fit Terraces"),
                                                    NULL, 0, NULL);
    dialogue = GTK_DIALOG(controls.dialogue);
    gtk_dialog_add_button(dialogue, _("_Reset"), RESPONSE_RESET);
    gtk_dialog_add_button(dialogue, gwy_sgettext("verb|_Fit"), RESPONSE_REFINE);
    gtk_dialog_add_button(dialogue, GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);
    gtk_dialog_add_button(dialogue, GTK_STOCK_OK, GTK_RESPONSE_OK);
    gtk_dialog_set_default_response(dialogue, GTK_RESPONSE_OK);
    gwy_help_add_to_proc_dialog(dialogue, GWY_HELP_DEFAULT);

    hbox = gtk_hbox_new(FALSE, 2);
    gtk_box_pack_start(GTK_BOX(dialogue->vbox), hbox, TRUE, TRUE, 4);

    controls.mydata = gwy_container_new();
    gwy_container_set_object_by_name(controls.mydata, "/0/data", dfield);
    otherfield = gwy_data_field_new_alike(dfield, TRUE);
    gwy_container_set_object_by_name(controls.mydata, "/0/mask", otherfield);
    gwy_app_sync_data_items(data, controls.mydata, id, PREVIEW_DATA, FALSE,
                            GWY_DATA_ITEM_RANGE_TYPE,
                            GWY_DATA_ITEM_RANGE,
                            GWY_DATA_ITEM_GRADIENT,
                            GWY_DATA_ITEM_REAL_SQUARE,
                            GWY_DATA_ITEM_MASK_COLOR,
                            0);
    g_object_unref(otherfield);

    set_up_other_field(data, id, controls.mydata, PREVIEW_SEGMENTED, dfield);
    gwy_container_set_const_string_by_name(controls.mydata, "/1/base/palette",
                                           "DFit");
    set_up_other_field(data, id, controls.mydata, PREVIEW_FITTED, dfield);
    set_up_other_field(data, id, controls.mydata, PREVIEW_RESIDUUM, dfield);
    gwy_container_set_const_string_by_name(controls.mydata, "/3/base/palette",
                                           FIT_GRADIENT_NAME);
    set_up_other_field(data, id, controls.mydata, PREVIEW_TERRACES, dfield);
    set_up_other_field(data, id, controls.mydata, PREVIEW_LEVELLED, dfield);
    set_up_other_field(data, id, controls.mydata, PREVIEW_BACKGROUND, dfield);

    alignment = gtk_alignment_new(0.0, 0.0, 0.0, 0.0);
    gtk_box_pack_start(GTK_BOX(hbox), alignment, FALSE, FALSE, 4);
    controls.view = create_preview(controls.mydata, 0, PREVIEW_SIZE, TRUE);
    gtk_container_add(GTK_CONTAINER(alignment), controls.view);

    notebook = gtk_notebook_new();
    gtk_box_pack_start(GTK_BOX(hbox), notebook, TRUE, TRUE, 0);

    widget = parameters_tab_new(&controls);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), widget,
                             gtk_label_new(_("Parameters")));

    widget = terrace_list_tab_new(&controls);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), widget,
                             gtk_label_new(_("Terrace List")));

    widget = output_tab_new(&controls);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), widget,
                             gtk_label_new(_("Output")));

    widget = survey_tab_new(&controls);
    gtk_notebook_append_page(GTK_NOTEBOOK(notebook), widget,
                             gtk_label_new(_("Survey")));

    update_sensitivity(&controls);
    terrace_segment_do(&controls);

    gtk_widget_show_all(controls.dialogue);
    do {
        response = gtk_dialog_run(dialogue);
        switch (response) {
            case GTK_RESPONSE_CANCEL:
            case GTK_RESPONSE_DELETE_EVENT:
            gtk_widget_destroy(controls.dialogue);
            case GTK_RESPONSE_NONE:
            goto finalise;
            break;

            case GTK_RESPONSE_OK:
            break;

            case RESPONSE_RESET:
            preview_mode = args->preview_mode;
            *args = terrace_defaults;
            terrace_dialog_update(&controls, args);
            args->preview_mode = preview_mode;
            break;

            case RESPONSE_REFINE:
            terrace_fit(&controls);
            update_sensitivity(&controls);
            break;

            default:
            g_assert_not_reached();
            break;
        }
    } while (response != GTK_RESPONSE_OK);

    create_output_fields(&controls, data, id);
    gtk_widget_destroy(controls.dialogue);

finalise:
    gwy_resource_release(GWY_RESOURCE(controls.diff_gradient));
    gwy_inventory_delete_item(gwy_gradients(), FIT_GRADIENT_NAME);
    gwy_si_unit_value_format_free(controls.vf);
    g_object_unref(controls.results);
    g_object_unref(controls.colourpixbuf);
    g_object_unref(controls.mydata);
    free_terrace_coordinates(controls.terracecoords);
    g_array_free(controls.terraceinfo, TRUE);
}

static void
create_output_fields(TerraceControls *controls, GwyContainer *data, gint id)
{
    const gint id_output_map[OUTPUT_NFLAGS] = { -1, 2, 3, 4, 5, 6 };
    const gboolean add_inv_mask[OUTPUT_NFLAGS] = {
        FALSE, TRUE, TRUE, TRUE, FALSE, FALSE
    };
    guint oflags = controls->args->output_flags;
    GwyContainer *mydata = controls->mydata;
    GwyDataField *mask, *invmask, *dfield;
    GQuark quark;
    gint newid;
    guint i;

    mask = gwy_container_get_object_by_name(mydata, "/0/mask");
    if (oflags & OUTPUT_SEGMENTED) {
        quark = gwy_app_get_mask_key_for_id(id);
        gwy_app_undo_qcheckpointv(data, 1, &quark);
        gwy_container_set_object(data, quark, mask);
        gwy_app_sync_data_items(mydata, data, 0, id, FALSE,
                                GWY_DATA_ITEM_MASK_COLOR,
                                0);
        gwy_app_channel_log_add_proc(data, id, id);
    }
    for (i = 1; i < OUTPUT_NFLAGS; i++) {
        if (!(oflags & (1 << i)))
            continue;

        quark = gwy_app_get_data_key_for_id(id_output_map[i]);
        dfield = gwy_container_get_object(mydata, quark);
        newid = gwy_app_data_browser_add_data_field(dfield, data, TRUE);
        gwy_app_sync_data_items(data, data, id, newid, FALSE,
                                GWY_DATA_ITEM_GRADIENT,
                                GWY_DATA_ITEM_REAL_SQUARE,
                                GWY_DATA_ITEM_MASK_COLOR,
                                0);
        if (add_inv_mask[i]) {
            invmask = gwy_data_field_duplicate(mask);
            gwy_data_field_grains_invert(invmask);
            quark = gwy_app_get_mask_key_for_id(newid);
            gwy_container_set_object(data, quark, invmask);
            g_object_unref(invmask);
        }
        gwy_app_set_data_field_title(data, newid, _(output_flags[i].name));
        gwy_app_channel_log_add_proc(data, id, newid);
    }
}

static void
terrace_dialog_update(TerraceControls *controls, TerraceArgs *args)
{
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->poly_degree),
                             args->poly_degree);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->edge_kernel_size),
                             args->edge_kernel_size);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->edge_threshold),
                             args->edge_threshold);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->min_area_frac),
                             args->min_area_frac);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->edge_broadening),
                             args->edge_broadening);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->independent),
                                 args->independent);
    if (controls->mask) {
        gwy_enum_combo_box_set_active(GTK_COMBO_BOX(controls->masking),
                                      args->masking);
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->use_only_mask),
                                    args->use_only_mask);
    }
}

static void
reset_images(TerraceControls *controls)
{
    GwyDataField *dfield;
    GQuark quark;
    guint i;

    for (i = PREVIEW_DATA; i < PREVIEW_NTYPES; i++) {
        /* These are always available. */
        if (i == PREVIEW_DATA || i == PREVIEW_SEGMENTED)
            continue;
        quark = gwy_app_get_data_key_for_id(i);
        dfield = gwy_container_get_object(controls->mydata, quark);
        gwy_data_field_clear(dfield);
        gwy_data_field_data_changed(dfield);
    }
}

static GtkWidget*
parameters_tab_new(TerraceControls *controls)
{
    TerraceArgs *args = controls->args;
    GwyResults *results = controls->results;
    GtkWidget *table, *spin, *label;
    GwyResultsExport *rexport;
    GString *str;
    guint i;
    gint row;

    table = gtk_table_new(14 + 2*(!!controls->mask), 4, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    row = 0;

    controls->edge_kernel_size = gtk_adjustment_new(args->edge_kernel_size,
                                                    1.0, 64.0, 0.1, 5.0, 0.0);
    gwy_table_attach_adjbar(table, row, _("_Step detection kernel:"), _("px"),
                            controls->edge_kernel_size, GWY_HSCALE_SQRT);
    g_signal_connect_swapped(controls->edge_kernel_size, "value-changed",
                             G_CALLBACK(edge_kernel_size_changed), controls);
    row++;

    controls->edge_threshold = gtk_adjustment_new(args->edge_threshold,
                                                  0.0, 100.0, 0.01, 0.1, 0.0);
    gwy_table_attach_adjbar(table, row, _("Step detection _threshold:"), "%",
                            controls->edge_threshold, GWY_HSCALE_SQRT);
    g_signal_connect_swapped(controls->edge_threshold, "value-changed",
                             G_CALLBACK(edge_threshold_changed), controls);
    row++;

    controls->edge_broadening = gtk_adjustment_new(args->edge_broadening,
                                                   0.0, 16.0, 1.0, 2.0, 0.0);
    spin = gwy_table_attach_adjbar(table, row, _("Step _broadening:"), _("px"),
                                   controls->edge_broadening, GWY_HSCALE_SQRT);
    gtk_spin_button_set_digits(GTK_SPIN_BUTTON(spin), 1);
    g_signal_connect_swapped(controls->edge_broadening, "value-changed",
                             G_CALLBACK(edge_broadening_changed), controls);
    row++;

    controls->min_area_frac = gtk_adjustment_new(args->min_area_frac,
                                                 0.1, 40.0, 0.01, 1.0, 0.0);
    gwy_table_attach_adjbar(table, row, _("Minimum terrace _area:"), "%",
                            controls->min_area_frac, GWY_HSCALE_SQRT);
    g_signal_connect_swapped(controls->min_area_frac, "value-changed",
                             G_CALLBACK(min_area_frac_changed), controls);
    row++;

    controls->poly_degree = gtk_adjustment_new(args->poly_degree,
                                               0.0, MAX_DEGREE, 1.0, 2.0, 0.0);
    gwy_table_attach_adjbar(table, row, _("_Polynomial degree:"), NULL,
                            controls->poly_degree,
                            GWY_HSCALE_LINEAR | GWY_HSCALE_SNAP);
    g_signal_connect_swapped(controls->poly_degree, "value-changed",
                             G_CALLBACK(poly_degree_changed), controls);
    row++;

    controls->independent
        = gtk_check_button_new_with_mnemonic(_("_Independent heights"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->independent),
                                 args->independent);
    gtk_table_attach(GTK_TABLE(table), controls->independent,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    g_signal_connect(controls->independent, "toggled",
                     G_CALLBACK(independent_changed), controls);
    row++;

    controls->preview_mode
        = gwy_enum_combo_box_newl(G_CALLBACK(preview_mode_changed), controls,
                                  args->preview_mode,
                                  _("Data"), PREVIEW_DATA,
                                  _("Marked terraces"), PREVIEW_SEGMENTED,
                                  _("Fitted shape"), PREVIEW_FITTED,
                                  _("Difference"), PREVIEW_RESIDUUM,
                                  _("Terraces (ideal)"), PREVIEW_TERRACES,
                                  _("Leveled surface"), PREVIEW_LEVELLED,
                                  _("Polynomial background"), PREVIEW_BACKGROUND,
                                  NULL);
    gwy_table_attach_adjbar(table, row++, _("_Display:"), NULL,
                            GTK_OBJECT(controls->preview_mode),
                            GWY_HSCALE_WIDGET_NO_EXPAND);

    if (controls->mask) {
        controls->masking
            = gwy_enum_combo_box_new(gwy_masking_type_get_enum(), -1,
                                     G_CALLBACK(masking_changed),
                                     controls, args->masking, TRUE);
        gwy_table_attach_adjbar(table, row++, _("_Masking:"), NULL,
                                GTK_OBJECT(controls->masking),
                                GWY_HSCALE_WIDGET_NO_EXPAND);

        controls->use_only_mask
            = gtk_check_button_new_with_mnemonic(_("Do not _segment, "
                                                   "use only mask"));
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->use_only_mask),
                                     args->use_only_mask);
        gtk_table_attach(GTK_TABLE(table), controls->use_only_mask,
                         0, 2, row, row+1, GTK_FILL, 0, 0, 0);
        g_signal_connect(controls->use_only_mask, "toggled",
                         G_CALLBACK(use_only_mask_changed), controls);
        row++;
    }

    controls->color_button = create_mask_color_button(controls->mydata,
                                                      controls->dialogue, 0);
    gwy_table_attach_adjbar(table, row++, _("_Mask color:"), NULL,
                            GTK_OBJECT(controls->color_button),
                            GWY_HSCALE_WIDGET_NO_EXPAND);

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);

    label = gwy_label_new_header(_("Result"));
    gtk_table_attach(GTK_TABLE(table), label,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    str = g_string_new(NULL);
    results = controls->results;
    for (i = 0; i < G_N_ELEMENTS(guivalues); i++) {
        g_string_assign(str, gwy_results_get_label_with_symbol(results,
                                                               guivalues[i]));
        g_string_append_c(str, ':');
        controls->guivalues[i] = gtk_label_new(NULL);
        gwy_table_attach_adjbar(table, row++, str->str, NULL,
                                GTK_OBJECT(controls->guivalues[i]),
                                GWY_HSCALE_WIDGET_NO_EXPAND);
    }
    g_string_free(str, TRUE);

    controls->message = gtk_label_new(NULL);
    gtk_misc_set_alignment(GTK_MISC(controls->message), 0.0, 0.5);
    set_widget_as_error_message(controls->message);
    gtk_table_attach(GTK_TABLE(table), controls->message,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    controls->rexport_result = gwy_results_export_new(args->report_style);
    rexport = GWY_RESULTS_EXPORT(controls->rexport_result);
    gwy_results_export_set_results(rexport, results);
    gwy_results_export_set_title(rexport, _("Save Fit Report"));
    gwy_results_export_set_actions_sensitive(rexport, FALSE);
    gtk_table_attach(GTK_TABLE(table), controls->rexport_result,
                     0, 3, row, row+1, GTK_FILL, 0, 0, 0);
    g_signal_connect_swapped(rexport, "format-changed",
                             G_CALLBACK(report_style_changed), controls);

    return table;
}

static void
render_id(G_GNUC_UNUSED GtkTreeViewColumn *column,
          GtkCellRenderer *renderer,
          GtkTreeModel *model,
          GtkTreeIter *iter,
          G_GNUC_UNUSED gpointer user_data)
{
    gchar buf[16];
    guint i;

    gtk_tree_model_get(model, iter, 0, &i, -1);
    g_snprintf(buf, sizeof(buf), "%u", i+1);
    g_object_set(renderer, "text", buf, NULL);
}

static void
render_colour(G_GNUC_UNUSED GtkTreeViewColumn *column,
              G_GNUC_UNUSED GtkCellRenderer *renderer,
              GtkTreeModel *model,
              GtkTreeIter *iter,
              gpointer user_data)
{
    TerraceControls *controls = (TerraceControls*)user_data;
    TerraceInfo *info;
    guint i, pixel;

    gtk_tree_model_get(model, iter, 0, &i, -1);
    info = &info_index(controls->terraceinfo, i);
    pixel = 0xff | gwy_rgba_to_pixbuf_pixel(&info->colour);
    gdk_pixbuf_fill(controls->colourpixbuf, pixel);
}

static void
render_height(G_GNUC_UNUSED GtkTreeViewColumn *column,
              GtkCellRenderer *renderer,
              GtkTreeModel *model,
              GtkTreeIter *iter,
              gpointer user_data)
{
    TerraceControls *controls = (TerraceControls*)user_data;
    GwySIValueFormat *vf = controls->vf;
    TerraceInfo *info;
    gchar buf[32];
    guint i;

    if (!controls->fit_ok) {
        g_object_set(renderer, "text", "", NULL);
        return;
    }

    gtk_tree_model_get(model, iter, 0, &i, -1);
    info = &info_index(controls->terraceinfo, i);
    g_snprintf(buf, sizeof(buf), "%.*f",
               vf->precision, info->height/vf->magnitude);
    g_object_set(renderer, "text", buf, NULL);
}

static void
render_level(G_GNUC_UNUSED GtkTreeViewColumn *column,
            GtkCellRenderer *renderer,
            GtkTreeModel *model,
            GtkTreeIter *iter,
            gpointer user_data)
{
    TerraceControls *controls = (TerraceControls*)user_data;
    TerraceInfo *info;
    gchar buf[16];
    guint i;

    if (!controls->fit_ok) {
        g_object_set(renderer, "text", "", NULL);
        return;
    }

    gtk_tree_model_get(model, iter, 0, &i, -1);
    info = &info_index(controls->terraceinfo, i);
    g_snprintf(buf, sizeof(buf), "%d", info->level);
    g_object_set(renderer, "text", buf, NULL);
}

static void
render_area(G_GNUC_UNUSED GtkTreeViewColumn *column,
            GtkCellRenderer *renderer,
            GtkTreeModel *model,
            GtkTreeIter *iter,
            gpointer user_data)
{
    TerraceControls *controls = (TerraceControls*)user_data;
    TerraceInfo *info;
    gchar buf[16];
    guint i;

    gtk_tree_model_get(model, iter, 0, &i, -1);
    info = &info_index(controls->terraceinfo, i);
    g_snprintf(buf, sizeof(buf), "%u", info->npixels);
    g_object_set(renderer, "text", buf, NULL);
}

static void
render_error(G_GNUC_UNUSED GtkTreeViewColumn *column,
             GtkCellRenderer *renderer,
             GtkTreeModel *model,
             GtkTreeIter *iter,
             gpointer user_data)
{
    TerraceControls *controls = (TerraceControls*)user_data;
    GwySIValueFormat *vf = controls->vf;
    TerraceInfo *info;
    gchar buf[32];
    guint i;

    if (!controls->fit_ok) {
        g_object_set(renderer, "text", "", NULL);
        return;
    }

    gtk_tree_model_get(model, iter, 0, &i, -1);
    info = &info_index(controls->terraceinfo, i);
    g_snprintf(buf, sizeof(buf), "%.*f",
               vf->precision, info->error/vf->magnitude);
    g_object_set(renderer, "text", buf, NULL);
}

static void
render_residuum(G_GNUC_UNUSED GtkTreeViewColumn *column,
                GtkCellRenderer *renderer,
                GtkTreeModel *model,
                GtkTreeIter *iter,
                gpointer user_data)
{
    TerraceControls *controls = (TerraceControls*)user_data;
    GwySIValueFormat *vf = controls->vf;
    TerraceInfo *info;
    gchar buf[32];
    guint i;

    if (!controls->fit_ok) {
        g_object_set(renderer, "text", "", NULL);
        return;
    }

    gtk_tree_model_get(model, iter, 0, &i, -1);
    info = &info_index(controls->terraceinfo, i);
    g_snprintf(buf, sizeof(buf), "%.*f",
               vf->precision, info->residuum/vf->magnitude);
    g_object_set(renderer, "text", buf, NULL);
}

static GtkTreeViewColumn*
append_text_column(GtkTreeCellDataFunc render_func, const gchar *title,
                   TerraceControls *controls, gboolean is_z)
{
    GtkTreeViewColumn *column;
    GtkCellRenderer *renderer;
    GtkWidget *label;
    gchar *s;

    column = gtk_tree_view_column_new();
    gtk_tree_view_column_set_expand(column, TRUE);
    gtk_tree_view_column_set_alignment(column, 0.5);
    renderer = gtk_cell_renderer_text_new();
    g_object_set(renderer, "xalign", 1.0, NULL);
    gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(column), renderer, TRUE);
    gtk_tree_view_column_set_cell_data_func(column, renderer,
                                            render_func, controls, NULL);

    label = gtk_label_new(NULL);
    if (is_z && *controls->vf->units)
        s = g_strdup_printf("<b>%s</b> [%s]", title, controls->vf->units);
    else
        s = g_strdup_printf("<b>%s</b>", title);
    gtk_label_set_markup(GTK_LABEL(label), s);
    gtk_tree_view_column_set_widget(column, label);
    gtk_widget_show(label);

    gtk_tree_view_append_column(GTK_TREE_VIEW(controls->terracelist), column);

    return column;
}

static GtkWidget*
terrace_list_tab_new(TerraceControls *controls)
{
    TerraceArgs *args = controls->args;
    GtkWidget *vbox, *scwin;
    GtkTreeViewColumn *column;
    GtkCellRenderer *renderer;
    GwyNullStore *store;
    GwyResultsExport *rexport;

    vbox = gtk_vbox_new(FALSE, 2);
    gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);

    scwin = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scwin),
                                   GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_box_pack_start(GTK_BOX(vbox), scwin, TRUE, TRUE, 0);

    controls->terraceinfo = g_array_new(FALSE, FALSE, sizeof(TerraceInfo));

    store = gwy_null_store_new(0);
    controls->terracelist = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));
    gtk_container_add(GTK_CONTAINER(scwin), controls->terracelist);

    column = append_text_column(render_id, "n", controls, FALSE);
    renderer = gtk_cell_renderer_pixbuf_new();
    gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(column), renderer, FALSE);
    g_object_set(renderer, "pixbuf", controls->colourpixbuf, NULL);
    gtk_tree_view_column_set_cell_data_func(column, renderer,
                                            render_colour, controls, NULL);
    append_text_column(render_height, "h", controls, TRUE);
    append_text_column(render_level, "k", controls, FALSE);
    append_text_column(render_area, "A<sub>px</sub>", controls, FALSE);
    append_text_column(render_error, "Δ", controls, TRUE);
    append_text_column(render_residuum, "r", controls, TRUE);

    controls->rexport_list = gwy_results_export_new(args->report_style);
    rexport = GWY_RESULTS_EXPORT(controls->rexport_list);
    gwy_results_export_set_style(rexport, GWY_RESULTS_EXPORT_TABULAR_DATA);
    gwy_results_export_set_title(rexport, _("Save Terrace Table"));
    gwy_results_export_set_actions_sensitive(rexport, FALSE);
    gtk_box_pack_start(GTK_BOX(vbox), controls->rexport_list, FALSE, FALSE, 0);
    g_signal_connect_swapped(rexport, "format-changed",
                             G_CALLBACK(report_style_changed), controls);
    g_signal_connect_swapped(rexport, "copy",
                             G_CALLBACK(copy_report), controls);
    g_signal_connect_swapped(rexport, "save",
                             G_CALLBACK(save_report), controls);

    return vbox;
}

static GtkWidget*
output_tab_new(TerraceControls *controls)
{
    TerraceArgs *args = controls->args;
    GtkWidget *table;
    gint row;

    table = gtk_table_new(OUTPUT_NFLAGS, 1, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    row = 0;

    controls->output_flags
        = gwy_check_boxes_create(output_flags, OUTPUT_NFLAGS,
                                 G_CALLBACK(output_flags_changed), controls,
                                 args->output_flags);
    row = gwy_check_boxes_attach_to_table(controls->output_flags,
                                          GTK_TABLE(table), 1, row);

    return table;
}

static GtkWidget*
survey_tab_new(TerraceControls *controls)
{
    TerraceArgs *args = controls->args;
    GtkWidget *table, *align, *spin;
    gint row;

    table = controls->survey_table = gtk_table_new(8, 3, FALSE);
    gtk_table_set_row_spacings(GTK_TABLE(table), 2);
    gtk_table_set_col_spacings(GTK_TABLE(table), 6);
    gtk_container_set_border_width(GTK_CONTAINER(table), 4);
    row = 0;

    controls->survey_poly
        = gtk_check_button_new_with_mnemonic(_("_Polynomial degree"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->survey_poly),
                                 args->survey_poly);
    gtk_table_attach(GTK_TABLE(table), controls->survey_poly,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    g_signal_connect(controls->survey_poly, "toggled",
                     G_CALLBACK(survey_poly_changed), controls);
    row++;

    controls->poly_degree_min = gtk_adjustment_new(args->poly_degree_min,
                                                   0.0, MAX_DEGREE, 1.0, 2.0,
                                                   0.0);
    gwy_table_attach_adjbar(table, row, _("M_inimum polynomial degree:"), NULL,
                            controls->poly_degree_min,
                            GWY_HSCALE_LINEAR | GWY_HSCALE_SNAP);
    g_signal_connect_swapped(controls->poly_degree_min, "value-changed",
                             G_CALLBACK(poly_degree_min_changed), controls);
    row++;

    controls->poly_degree_max = gtk_adjustment_new(args->poly_degree_max,
                                                   0.0, MAX_DEGREE, 1.0, 2.0,
                                                   0.0);
    gwy_table_attach_adjbar(table, row, _("_Maximum polynomial degree:"), NULL,
                            controls->poly_degree_max,
                            GWY_HSCALE_LINEAR | GWY_HSCALE_SNAP);
    g_signal_connect_swapped(controls->poly_degree_max, "value-changed",
                             G_CALLBACK(poly_degree_max_changed), controls);
    row++;

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    controls->survey_broadening
        = gtk_check_button_new_with_mnemonic(_("Step _broadening"));
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(controls->survey_broadening),
                                 args->survey_broadening);
    gtk_table_attach(GTK_TABLE(table), controls->survey_broadening,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    g_signal_connect(controls->survey_broadening, "toggled",
                     G_CALLBACK(survey_broadening_changed), controls);
    row++;

    controls->broadening_min = gtk_adjustment_new(args->broadening_min,
                                                  0.0, MAX_BROADEN, 1.0, 10.0,
                                                  0.0);
    spin = gwy_table_attach_adjbar(table, row,
                                   _("Minimum broadening:"), _("px"),
                                   controls->broadening_min, GWY_HSCALE_SQRT);
    gtk_spin_button_set_digits(GTK_SPIN_BUTTON(spin), 1);
    g_signal_connect_swapped(controls->broadening_min, "value-changed",
                             G_CALLBACK(broadening_min_changed), controls);
    row++;

    controls->broadening_max = gtk_adjustment_new(args->broadening_max,
                                                  0.0, MAX_BROADEN, 1.0, 10.0,
                                                  0.0);
    spin = gwy_table_attach_adjbar(table, row,
                                   _("Maximum broadening:"), _("px"),
                                   controls->broadening_max, GWY_HSCALE_SQRT);
    gtk_spin_button_set_digits(GTK_SPIN_BUTTON(spin), 1);
    g_signal_connect_swapped(controls->broadening_max, "value-changed",
                             G_CALLBACK(broadening_max_changed), controls);
    row++;

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    controls->survey_message = gtk_label_new(NULL);
    gtk_misc_set_alignment(GTK_MISC(controls->survey_message), 0.0, 0.5);
    gtk_table_attach(GTK_TABLE(table), controls->survey_message,
                     0, 2, row, row+1, GTK_FILL, 0, 0, 0);
    row++;

    gtk_table_set_row_spacing(GTK_TABLE(table), row-1, 8);
    controls->run_survey = gtk_button_new_from_stock(GTK_STOCK_EXECUTE);
    align = gtk_alignment_new(0.0, 0.5, 0.0, 0.0);
    gtk_container_add(GTK_CONTAINER(align), controls->run_survey);
    gtk_table_attach(GTK_TABLE(table), align,
                     0, 1, row, row+1, GTK_FILL, 0, 0, 0);
    g_signal_connect_swapped(controls->run_survey, "clicked",
                             G_CALLBACK(run_survey), controls);
    row++;

    return table;
}

static void
poly_degree_changed(TerraceControls *controls, GtkAdjustment *adj)
{
    TerraceArgs *args = controls->args;
    args->poly_degree = gwy_adjustment_get_int(adj);
}

static void
edge_kernel_size_changed(TerraceControls *controls, GtkAdjustment *adj)
{
    TerraceArgs *args = controls->args;
    args->edge_kernel_size = gtk_adjustment_get_value(adj);
    terrace_segment_do(controls);
}

static void
edge_threshold_changed(TerraceControls *controls, GtkAdjustment *adj)
{
    TerraceArgs *args = controls->args;
    args->edge_threshold = gtk_adjustment_get_value(adj);
    terrace_segment_do(controls);
}

static void
edge_broadening_changed(TerraceControls *controls, GtkAdjustment *adj)
{
    TerraceArgs *args = controls->args;
    args->edge_broadening = gtk_adjustment_get_value(adj);
    terrace_segment_do(controls);
}

static void
min_area_frac_changed(TerraceControls *controls, GtkAdjustment *adj)
{
    TerraceArgs *args = controls->args;
    args->min_area_frac = gtk_adjustment_get_value(adj);
    terrace_segment_do(controls);
}

static void
masking_changed(GtkComboBox *combo, TerraceControls *controls)
{
    TerraceArgs *args = controls->args;
    args->masking = gwy_enum_combo_box_get_active(combo);
    update_sensitivity(controls);
    terrace_segment_do(controls);
}

static void
use_only_mask_changed(GtkToggleButton *toggle, TerraceControls *controls)
{
    TerraceArgs *args = controls->args;
    args->use_only_mask = gtk_toggle_button_get_active(toggle);
    update_sensitivity(controls);
    terrace_segment_do(controls);
}

static void
independent_changed(GtkToggleButton *toggle, TerraceControls *controls)
{
    TerraceArgs *args = controls->args;
    args->independent = gtk_toggle_button_get_active(toggle);
}

static void
preview_mode_changed(GtkComboBox *combo, TerraceControls *controls)
{
    GwyPixmapLayer *blayer, *mlayer;
    TerraceArgs *args = controls->args;
    gchar key[24];

    args->preview_mode = gwy_enum_combo_box_get_active(combo);

    blayer = gwy_data_view_get_base_layer(GWY_DATA_VIEW(controls->view));
    g_snprintf(key, sizeof(key), "/%d/data", args->preview_mode);
    gwy_pixmap_layer_set_data_key(blayer, key);
    g_snprintf(key, sizeof(key), "/%d/base/palette", args->preview_mode);
    gwy_layer_basic_set_gradient_key(GWY_LAYER_BASIC(blayer), key);
    g_snprintf(key, sizeof(key), "/%d/base/range-type", args->preview_mode);
    gwy_layer_basic_set_range_type_key(GWY_LAYER_BASIC(blayer), key);
    g_snprintf(key, sizeof(key), "/%d/base", args->preview_mode);
    gwy_layer_basic_set_min_max_key(GWY_LAYER_BASIC(blayer), key);

    mlayer = gwy_data_view_get_alpha_layer(GWY_DATA_VIEW(controls->view));
    g_snprintf(key, sizeof(key), "/%d/mask", args->preview_mode);
    gwy_pixmap_layer_set_data_key(mlayer, key);
}

static void
report_style_changed(TerraceControls *controls, GwyResultsExport *rexport)
{
    controls->args->report_style = gwy_results_export_get_format(rexport);
}

static void
output_flags_changed(GtkWidget *button, TerraceControls *controls)
{
    GSList *group = gwy_check_box_get_group(button);
    TerraceArgs *args = controls->args;

    args->output_flags = gwy_check_boxes_get_selected(group);
}

static void
survey_poly_changed(GtkToggleButton *toggle, TerraceControls *controls)
{
    TerraceArgs *args = controls->args;
    args->survey_poly = gtk_toggle_button_get_active(toggle);
    update_sensitivity(controls);
}

static void
poly_degree_min_changed(TerraceControls *controls, GtkAdjustment *adj)
{
    TerraceArgs *args = controls->args;
    args->poly_degree_min = gwy_adjustment_get_int(adj);
    if (args->poly_degree_min > args->poly_degree_max) {
        gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->poly_degree_max),
                                 args->poly_degree_min);
    }
    update_sensitivity(controls);
}

static void
poly_degree_max_changed(TerraceControls *controls, GtkAdjustment *adj)
{
    TerraceArgs *args = controls->args;
    args->poly_degree_max = gwy_adjustment_get_int(adj);
    if (args->poly_degree_min > args->poly_degree_max) {
        gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->poly_degree_min),
                                 args->poly_degree_max);
    }
    update_sensitivity(controls);
}

static void
survey_broadening_changed(GtkToggleButton *toggle, TerraceControls *controls)
{
    TerraceArgs *args = controls->args;
    args->survey_broadening = gtk_toggle_button_get_active(toggle);
    update_sensitivity(controls);
}

static void
broadening_min_changed(TerraceControls *controls, GtkAdjustment *adj)
{
    TerraceArgs *args = controls->args;
    args->broadening_min = gtk_adjustment_get_value(adj);
    if (args->broadening_min > args->broadening_max + 1e-14) {
        gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->broadening_max),
                                 args->broadening_min);
    }
    update_sensitivity(controls);
}

static void
broadening_max_changed(TerraceControls *controls, GtkAdjustment *adj)
{
    TerraceArgs *args = controls->args;
    args->broadening_max = gtk_adjustment_get_value(adj);
    if (args->broadening_min > args->broadening_max + 1e-14) {
        gtk_adjustment_set_value(GTK_ADJUSTMENT(controls->broadening_min),
                                 args->broadening_max);
    }
    update_sensitivity(controls);
}

static void
update_sensitivity(TerraceControls *controls)
{
    TerraceArgs *args = controls->args;
    gboolean sens;

    sens = (controls->mask && args->masking != GWY_MASK_IGNORE);
    if (controls->mask)
        gtk_widget_set_sensitive(controls->use_only_mask, sens);

    sens = !(controls->mask && args->use_only_mask);
    gwy_table_hscale_set_sensitive(controls->edge_kernel_size, sens);
    gwy_table_hscale_set_sensitive(controls->edge_threshold, sens);
    gwy_table_hscale_set_sensitive(controls->edge_broadening, sens);

    gtk_dialog_set_response_sensitive(GTK_DIALOG(controls->dialogue),
                                      GTK_RESPONSE_OK, controls->fit_ok);

    gtk_widget_set_sensitive(controls->survey_table, !args->independent);
    if (args->independent) {
        gtk_label_set_text(GTK_LABEL(controls->survey_message),
                           _("Survey cannot be run with independent degrees."));
    }
    else {
        sens = args->survey_poly || args->survey_broadening;
        if (sens) {
            TerraceArgs myargs = *args;
            guint n = count_survey_items(&myargs, NULL, NULL);
            gchar *s = g_strdup_printf(_("Number of combinations: %u."), n);

            gtk_label_set_text(GTK_LABEL(controls->survey_message), s);
            g_free(s);
        }
        else {
            gtk_label_set_text(GTK_LABEL(controls->survey_message),
                               _("No free parameters are selected."));
        }
        gtk_widget_set_sensitive(controls->run_survey, sens);
        sens = args->survey_poly;
        gwy_table_hscale_set_sensitive(controls->poly_degree_min, sens);
        gwy_table_hscale_set_sensitive(controls->poly_degree_max, sens);
        sens = args->survey_broadening;
        gwy_table_hscale_set_sensitive(controls->broadening_min, sens);
        gwy_table_hscale_set_sensitive(controls->broadening_max, sens);
    }
}

static void
update_diff_gradient(TerraceControls *controls)
{
    GwyDataField *residuum;
    GQuark quark;
    gdouble min, max, dispmin, dispmax;

    quark = gwy_app_get_data_key_for_id(PREVIEW_RESIDUUM);
    residuum = gwy_container_get_object(controls->mydata, quark);
    gwy_data_field_get_min_max(residuum, &min, &max);
    gwy_data_field_get_autorange(residuum, &dispmin, &dispmax);
    gwy_debug("residuum min %g, max %g", min, max);
    set_gradient_for_residuum(controls->diff_gradient, min, max,
                              &dispmin, &dispmax);

    quark = gwy_app_get_data_range_type_key_for_id(PREVIEW_RESIDUUM);
    gwy_container_set_enum(controls->mydata, quark,
                           GWY_LAYER_BASIC_RANGE_FIXED);
    quark = gwy_app_get_data_range_min_key_for_id(PREVIEW_RESIDUUM);
    gwy_container_set_double(controls->mydata, quark, dispmin);
    quark = gwy_app_get_data_range_max_key_for_id(PREVIEW_RESIDUUM);
    gwy_container_set_double(controls->mydata, quark, dispmax);
}

static void
improve_edge_connectivity(GwyDataField *steps,
                          GwyDataField *tmp,
                          gdouble radius)
{
    gint xres, yres, i, r, r2lim;
    const gdouble *d;
    gdouble *t;

    gwy_data_field_clear(tmp);
    xres = gwy_data_field_get_xres(steps);
    yres = gwy_data_field_get_yres(steps);
    d = gwy_data_field_get_data_const(steps);
    t = gwy_data_field_get_data(tmp);
    r = (gint)floor(radius);
    r2lim = (gint)0.7*radius*radius;

#ifdef _OPENMP
#pragma omp parallel for if (gwy_threads_are_enabled()) default(none) \
            shared(xres,yres,r,r2lim,t,d) \
            private(i)
#endif
    for (i = r; i < yres-r; i++) {
        gint j;

        for (j = r; j < xres-r; j++) {
            gint ii, jj;

            if (d[i*xres + j] <= 0.0)
                continue;

            for (ii = -r; ii <= r; ii++) {
                for (jj = -r; jj <= r; jj++) {
                    if (ii*ii + jj*jj > 0.7*r2lim
                        && d[(i+ii)*xres + (j+jj)] >= 1.0
                        && d[(i-ii)*xres + (j-jj)] >= 1.0) {
                        gint ic = (ii > 0 ? i + ii/2 : i - (-ii)/2);
                        gint jc = (jj > 0 ? j + jj/2 : j - (-jj)/2);
                        if (d[ic*xres + jc] <= 0.0)
                            t[ic*xres + jc] += 1.0;
                    }
                }
            }
        }
    }

    gwy_data_field_max_of_fields(steps, steps, tmp);
}

static GArray*
find_terrace_coordinates(GwyDataField *dfield,
                         GwyDataField *mask,
                         const TerraceArgs *args,
                         GwyDataField *marked,
                         GwyDataField *terraceids,
                         gdouble *pxc, gdouble *pyc)
{
    gint xres = gwy_data_field_get_xres(dfield);
    gint yres = gwy_data_field_get_yres(dfield);
    gint i, j, g, minsize, ngrains, n = xres*yres, npixels;
    const gdouble *d = gwy_data_field_get_data_const(dfield);
    gint *grains, *sizes;
    GArray *terracecoords;
    gdouble threshold, xc, yc;
    gdouble *ti;

    if (mask && args->use_only_mask) {
        /* Use provided mask as requested. */
        gwy_data_field_copy(mask, marked, FALSE);
        if (args->masking == GWY_MASK_EXCLUDE)
            gwy_data_field_grains_invert(marked);
    }
    else {
        /* Mark flat areas in the field. */
        gwy_data_field_copy(dfield, marked, FALSE);
        gwy_data_field_filter_gauss_step(marked, args->edge_kernel_size);
        threshold = args->edge_threshold/100.0*gwy_data_field_get_max(marked);
        gwy_data_field_threshold(marked, threshold, 0.0, 1.0);
        /* Use terraceids as a buffer. */
        improve_edge_connectivity(marked, terraceids, 11.5);
        improve_edge_connectivity(marked, terraceids, 9.5);
        gwy_data_field_grains_invert(marked);
        gwy_data_field_grains_shrink(marked, args->edge_broadening,
                                     GWY_DISTANCE_TRANSFORM_EUCLIDEAN, FALSE);

        /* Combine with existing mask if required. */
        if (mask && args->masking != GWY_MASK_IGNORE) {
            if (args->masking == GWY_MASK_INCLUDE)
                gwy_data_field_grains_intersect(marked, mask);
            else {
                gwy_data_field_grains_invert(marked);
                gwy_data_field_grains_add(marked, mask);
                gwy_data_field_grains_invert(marked);
            }
        }
    }

    /* Keep only large areas.  This inherently limits the maximum number of
     * areas too. */
    minsize = GWY_ROUND(args->min_area_frac/100.0 * n);
    gwy_data_field_grains_remove_by_size(marked, minsize);

    /* Gather coordinates for each terrace into an array. */
    grains = g_new0(gint, n);
    ngrains = gwy_data_field_number_grains(marked, grains);
    if (!ngrains) {
        g_free(grains);
        return NULL;
    }

    sizes = gwy_data_field_get_grain_sizes(marked, ngrains, grains, NULL);
    terracecoords = g_array_sized_new(FALSE, FALSE, sizeof(TerraceCoords),
                                      ngrains);
    for (g = 1; g <= ngrains; g++) {
        TerraceCoords tc;

        tc.ncoords = 0;
        tc.xyz = g_new(GwyXYZ, sizes[g]);
        tc.pixels = g_new(guint, sizes[g]);
        g_array_append_val(terracecoords, tc);
    }

    /* Normalise coordinates to have centre of mass at 0. */
    ti = gwy_data_field_get_data(terraceids);
    gwy_clear(sizes, ngrains+1);
    xc = yc = 0.0;
    npixels = 0;
    for (i = 0; i < yres; i++) {
        gdouble y = (2.0*i + 1 - yres)/(yres - 1);
        for (j = 0; j < xres; j++) {
            guint k = i*xres + j;
            if ((g = grains[k])) {
                TerraceCoords *tc = &coord_index(terracecoords, g-1);
                GwyXYZ *xyz = tc->xyz + tc->ncoords;

                tc->pixels[tc->ncoords] = k;
                xyz->x = (2.0*j + 1 - xres)/(xres - 1);
                xyz->y = y;
                xyz->z = d[k];
                xc += xyz->x;
                yc += xyz->y;
                tc->ncoords++;
                npixels++;
            }
            ti[k] = g;
        }
    }
    xc /= npixels;
    yc /= npixels;

    for (g = 0; g < ngrains; g++) {
        TerraceCoords *tc = &coord_index(terracecoords, g);
        GwyXYZ *xyz = tc->xyz;
        guint ncoords = tc->ncoords;

        for (i = 0; i < ncoords; i++) {
            xyz[i].x -= xc;
            xyz[i].y -= yc;
        }
    }

    g_free(sizes);
    g_free(grains);

    *pxc = xc;
    *pyc = yc;

    return terracecoords;
}

static gint*
make_term_powers_except0(gint poly_degree, gint *pnterms)
{
    gint nterms = (poly_degree + 1)*(poly_degree + 2)/2 - 1;
    gint *term_powers = g_new(gint, 2*nterms);
    gint i, j, k;

    for (i = k = 0; i <= poly_degree; i++) {
        for (j = 0; j <= poly_degree - i; j++) {
            if (i || j) {
                term_powers[k++] = i;
                term_powers[k++] = j;
            }
        }
    }

    *pnterms = nterms;
    return term_powers;
}

static guint
find_maximum_power(guint npowers, const gint *term_powers)
{
    guint k, maxpower;

    maxpower = 0;
    for (k = 0; k < 2*npowers; k++)
        maxpower = MAX(maxpower, term_powers[k]);

    return maxpower;
}

/* Diagonal power-power matrix block.  Some of the entries could be
 * calculated from the per-terrace averages; the higher powers are only
 * used here though.  This is the slow part.  */
static gdouble*
calculate_power_matrix_block(GArray *terracecoords,
                             guint npowers, const gint *term_powers)
{
    guint maxpower, nterraces, kp, mp;
    gdouble *power_block;

    /* We multiply two powers together so the maximum power in the product is
     * twice the single maximum power. */
    maxpower = 2*find_maximum_power(npowers, term_powers);
    nterraces = terracecoords->len;
    power_block = g_new0(gdouble, npowers*npowers);

#ifdef _OPENMP
#pragma omp parallel if (gwy_threads_are_enabled()) default(none) \
            shared(terracecoords,maxpower,npowers,term_powers,power_block,nterraces)
#endif
    {
        gdouble *xpowers = g_new(gdouble, maxpower+1);
        gdouble *ypowers = g_new(gdouble, maxpower+1);
        gdouble *tpower_block = gwy_omp_if_threads_new0(power_block,
                                                        npowers*npowers);
        guint gfrom = gwy_omp_chunk_start(nterraces);
        guint gto = gwy_omp_chunk_end(nterraces);
        guint m, k, g, i;

        xpowers[0] = ypowers[0] = 1.0;

        for (g = gfrom; g < gto; g++) {
            TerraceCoords *tc = &coord_index(terracecoords, g);
            GwyXYZ *xyz = tc->xyz;
            guint ncoords = tc->ncoords;

            for (i = 0; i < ncoords; i++) {
                gdouble x = xyz[i].x, y = xyz[i].y;

                for (k = 1; k <= maxpower; k++) {
                    xpowers[k] = xpowers[k-1]*x;
                    ypowers[k] = ypowers[k-1]*y;
                }

                for (k = 0; k < npowers; k++) {
                    for (m = 0; m <= k; m++) {
                        gint powx = term_powers[2*k + 0] + term_powers[2*m + 0];
                        gint powy = term_powers[2*k + 1] + term_powers[2*m + 1];
                        gdouble xp = xpowers[powx];
                        gdouble yp = ypowers[powy];
                        tpower_block[k*npowers + m] += xp*yp;
                    }
                }
            }
        }
        g_free(xpowers);
        g_free(ypowers);
        gwy_omp_if_threads_sum_double(power_block, tpower_block,
                                      npowers*npowers);
    }

    /* Redundant, but keep for simplicity. */
    for (kp = 0; kp < npowers; kp++) {
        for (mp = kp+1; mp < npowers; mp++)
            power_block[kp*npowers + mp] = power_block[mp*npowers + kp];
    }

    return power_block;
}

static void
free_fit_result(FitResult *fres)
{
    g_free(fres->solution);
    g_free(fres->invdiag);
    g_free(fres);
}

static void
calculate_residuum(GArray *terracecoords, FitResult *fres,
                   GwyDataField *residuum,
                   const gint *term_powers, guint npowers, guint maxpower,
                   gdouble *xpowers, gdouble *ypowers,
                   gboolean indep)
{
    const gdouble *solution = fres->solution, *solution_block;
    gdouble *resdata;
    guint g, i, k, nterraces = terracecoords->len, npixels;

    solution_block = solution + (indep ? nterraces : 2);
    gwy_data_field_clear(residuum);
    resdata = gwy_data_field_get_data(residuum);

    fres->msq = fres->deltares = 0.0;
    npixels = 0;
    for (g = 0; g < nterraces; g++) {
        TerraceCoords *tc = &coord_index(terracecoords, g);
        GwyXYZ *xyz = tc->xyz;
        guint ncoords = tc->ncoords;
        gint ng = tc->level;
        gdouble z0 = (indep ? solution[g] : ng*solution[0] + solution[1]);
        gdouble ts = 0.0, toff = 0.0;

        for (i = 0; i < ncoords; i++) {
            gdouble x = xyz[i].x, y = xyz[i].y, z = xyz[i].z;
            gdouble s = z0;

            for (k = 1; k <= maxpower; k++) {
                xpowers[k] = xpowers[k-1]*x;
                ypowers[k] = ypowers[k-1]*y;
            }

            for (k = 0; k < npowers; k++) {
                gint powx = term_powers[2*k + 0];
                gint powy = term_powers[2*k + 1];
                gdouble xp = xpowers[powx];
                gdouble yp = ypowers[powy];
                s += xp*yp*solution_block[k];
            }
            s = z - s;
            resdata[tc->pixels[i]] = s;
            ts += s*s;
            toff += s;
        }
        tc->msq = ts/ncoords;
        tc->off = toff/ncoords;
        fres->msq += ts;
        fres->deltares += tc->off*tc->off * ncoords;
        npixels += ncoords;
    }
    fres->msq = sqrt(fres->msq/npixels);
    fres->deltares = sqrt(fres->deltares/npixels);
}

static FitResult*
fit_terraces_arbitrary(GArray *terracecoords,
                       const gint *term_powers, guint npowers,
                       const gdouble *power_block,
                       GwyDataField *residuum,
                       const gchar **message)
{
    gint g, k, nterraces, matn, matsize, npixels, maxpower;
    gdouble *mixed_block, *matrix, *invmat, *rhs, *xpowers, *ypowers;
    FitResult *fres;
    guint i, j;
    gboolean ok;

    fres = g_new0(FitResult, 1);
    nterraces = fres->nterrparam = fres->nterraces = terracecoords->len;
    fres->npowers = npowers;
    matn = nterraces + npowers;

    maxpower = find_maximum_power(npowers, term_powers);
    xpowers = g_new(gdouble, maxpower+1);
    ypowers = g_new(gdouble, maxpower+1);
    xpowers[0] = ypowers[0] = 1.0;

    /* Calculate the matrix by pieces, the put it together.  The terrace block
     * is identity matrix I so we do not need to compute it. */
    mixed_block = g_new0(gdouble, npowers*nterraces);
    rhs = fres->solution = g_new0(gdouble, nterraces + npowers);
    fres->invdiag = g_new0(gdouble, matn);

    /* Mixed off-diagonal power-terrace matrix block (we represent it as the
     * upper right block) and power block on the right hand side. */
    for (g = 0; g < nterraces; g++) {
        TerraceCoords *tc = &coord_index(terracecoords, g);
        GwyXYZ *xyz = tc->xyz;
        guint ncoords = tc->ncoords;
        gdouble *mixed_row = mixed_block + g*npowers;
        gdouble *rhs_block = rhs + nterraces;

        for (i = 0; i < ncoords; i++) {
            gdouble x = xyz[i].x, y = xyz[i].y, z = xyz[i].z;

            for (k = 1; k <= maxpower; k++) {
                xpowers[k] = xpowers[k-1]*x;
                ypowers[k] = ypowers[k-1]*y;
            }

            for (k = 0; k < npowers; k++) {
                gint powx = term_powers[2*k + 0];
                gint powy = term_powers[2*k + 1];
                gdouble xp = xpowers[powx];
                gdouble yp = ypowers[powy];
                mixed_row[k] += xp*yp;
                rhs_block[k] += xp*yp*z;
            }
        }
    }

    /* Terrace block of right hand side. */
    npixels = 0;
    for (g = 0; g < nterraces; g++) {
        TerraceCoords *tc = &coord_index(terracecoords, g);
        GwyXYZ *xyz = tc->xyz;
        guint ncoords = tc->ncoords;

        for (i = 0; i < ncoords; i++) {
            gdouble z = xyz[i].z;
            rhs[g] += z;
        }
        npixels += ncoords;
    }

    /* Construct the matrix. */
    matsize = (matn + 1)*matn/2;
    matrix = g_new(gdouble, matsize);
    gwy_debug("matrix (%u)", matn);
    for (i = 0; i < matn; i++) {
        for (j = 0; j <= i; j++) {
            gdouble t;

            if (i < nterraces && j < nterraces) {
                t = (i == j)*(coord_index(terracecoords, i).ncoords);
            }
            else if (j < nterraces)
                t = mixed_block[j*npowers + (i - nterraces)];
            else
                t = power_block[(i - nterraces)*npowers + (j - nterraces)];

            SLi(matrix, i, j) = t/npixels;
#ifdef DEBUG_MATRIX
            printf("% .2e ", t);
#endif
        }
#ifdef DEBUG_MATRIX
        printf("\n");
#endif
    }
    g_free(mixed_block);

    invmat = g_memdup(matrix, matsize*sizeof(gdouble));
    ok = gwy_math_choleski_decompose(matn, matrix);
    gwy_debug("decomposition: %s", ok ? "OK" : "FAIL");
    if (!ok) {
        *message = _("Fit failed");
        free_fit_result(fres);
        fres = NULL;
        goto finalise;
    }
    for (i = 0; i < matn; i++)
        rhs[i] /= npixels;

    gwy_math_choleski_solve(matn, matrix, rhs);

    if (residuum) {
        calculate_residuum(terracecoords, fres, residuum,
                           term_powers, npowers, maxpower, xpowers, ypowers,
                           TRUE);
    }

    ok = gwy_math_choleski_invert(matn, invmat);
    gwy_debug("inversion: %s", ok ? "OK" : "FAIL");
    if (!ok) {
        *message = _("Fit failed");
        free_fit_result(fres);
        fres = NULL;
        goto finalise;
    }
    for (i = 0; i < matn; i++)
        fres->invdiag[i] = SLi(invmat, i, i);

finalise:
    g_free(xpowers);
    g_free(ypowers);
    g_free(matrix);
    g_free(invmat);

    return fres;
}

static FitResult*
fit_terraces_same_step(GArray *terracecoords,
                       const gint *term_powers, guint npowers,
                       const gdouble *power_block,
                       GwyDataField *residuum,
                       const gchar **message)
{
    gint g, k, nterraces, matn, matsize, npixels, maxpower;
    gdouble *sheight_block, *offset_block, *matrix, *invmat, *rhs;
    gdouble *xpowers, *ypowers;
    gdouble stepstep, stepoff, offoff;
    FitResult *fres;
    guint i, j;
    gboolean ok;

    fres = g_new0(FitResult, 1);
    nterraces = fres->nterraces = terracecoords->len;
    fres->npowers = npowers;
    fres->nterrparam = 2;
    matn = 2 + npowers;

    maxpower = find_maximum_power(npowers, term_powers);
    xpowers = g_new(gdouble, maxpower+1);
    ypowers = g_new(gdouble, maxpower+1);
    xpowers[0] = ypowers[0] = 1.0;

    /* Calculate the matrix by pieces, the put it together.  */
    sheight_block = g_new0(gdouble, npowers);
    offset_block = g_new0(gdouble, npowers);
    rhs = fres->solution = g_new0(gdouble, matn);
    fres->invdiag = g_new0(gdouble, matn);

    /* Mixed two first upper right matrix rows and power block of right hand
     * side. */
    for (g = 0; g < nterraces; g++) {
        TerraceCoords *tc = &coord_index(terracecoords, g);
        GwyXYZ *xyz = tc->xyz;
        guint ncoords = tc->ncoords;
        gint ng = tc->level;
        gdouble *rhs_block = rhs + 2;

        for (i = 0; i < ncoords; i++) {
            gdouble x = xyz[i].x, y = xyz[i].y, z = xyz[i].z;

            for (k = 1; k <= maxpower; k++) {
                xpowers[k] = xpowers[k-1]*x;
                ypowers[k] = ypowers[k-1]*y;
            }

            for (k = 0; k < npowers; k++) {
                gint powx = term_powers[2*k + 0];
                gint powy = term_powers[2*k + 1];
                gdouble xp = xpowers[powx];
                gdouble yp = ypowers[powy];
                sheight_block[k] += xp*yp*ng;
                offset_block[k] += xp*yp;
                rhs_block[k] += xp*yp*z;
            }
        }
    }

    /* Remaining three independent elements in the top left corner of the
     * matrix. */
    stepstep = stepoff = 0.0;
    npixels = 0;
    for (g = 0; g < nterraces; g++) {
        TerraceCoords *tc = &coord_index(terracecoords, g);
        guint ncoords = tc->ncoords;
        gint ng = tc->level;

        /* Ensure ng does not converted to unsigned, with disasterous
         * consequences. */
        stepstep += ng*ng*(gdouble)ncoords;
        stepoff += ng*(gdouble)ncoords;
        npixels += ncoords;
    }
    offoff = npixels;

    /* Remaining first two elements of the right hand side. */
    for (g = 0; g < nterraces; g++) {
        TerraceCoords *tc = &coord_index(terracecoords, g);
        GwyXYZ *xyz = tc->xyz;
        guint ncoords = tc->ncoords;
        gint ng = tc->level;

        for (i = 0; i < ncoords; i++) {
            gdouble z = xyz[i].z;
            rhs[0] += ng*z;
            rhs[1] += z;
        }
    }

    /* Construct the matrix. */
    matsize = (matn + 1)*matn/2;
    matrix = g_new(gdouble, matsize);

    gwy_debug("matrix (%u)", matn);
    SLi(matrix, 0, 0) = stepstep/npixels;
#ifdef DEBUG_MATRIX
    printf("% .2e\n", SLi(matrix, 0, 0));
#endif
    SLi(matrix, 1, 0) = stepoff/npixels;
    SLi(matrix, 1, 1) = offoff/npixels;
#ifdef DEBUG_MATRIX
    printf("% .2e % .2e\n", SLi(matrix, 1, 0), SLi(matrix, 1, 1));
#endif

    for (i = 2; i < matn; i++) {
        for (j = 0; j <= i; j++) {
            gdouble t;

            if (j == 0)
                t = sheight_block[i-2];
            else if (j == 1)
                t = offset_block[i-2];
            else
                t = power_block[(i - 2)*npowers + (j - 2)];

            SLi(matrix, i, j) = t/npixels;
#ifdef DEBUG_MATRIX
            printf("% .2e ", t);
#endif
        }
#ifdef DEBUG_MATRIX
        printf("\n");
#endif
    }
    g_free(sheight_block);
    g_free(offset_block);

    invmat = g_memdup(matrix, matsize*sizeof(gdouble));
    ok = gwy_math_choleski_decompose(matn, matrix);
    gwy_debug("decomposition: %s", ok ? "OK" : "FAIL");
    if (!ok) {
        *message = _("Fit failed");
        free_fit_result(fres);
        fres = NULL;
        goto finalise;
    }
    for (i = 0; i < matn; i++)
        rhs[i] /= npixels;

    gwy_math_choleski_solve(matn, matrix, rhs);

    if (residuum) {
        calculate_residuum(terracecoords, fres, residuum,
                           term_powers, npowers, maxpower, xpowers, ypowers,
                           FALSE);
    }

    ok = gwy_math_choleski_invert(matn, invmat);
    gwy_debug("inversion: %s", ok ? "OK" : "FAIL");
    if (!ok) {
        *message = _("Fit failed");
        free_fit_result(fres);
        fres = NULL;
        goto finalise;
    }

    /* Compensate division of all matrix elements by npixels. */
    for (i = 0; i < matsize; i++)
        invmat[i] /= npixels;
    for (i = 0; i < matn; i++)
        fres->invdiag[i] = SLi(invmat, i, i);

finalise:
    g_free(xpowers);
    g_free(ypowers);
    g_free(invmat);
    g_free(matrix);

    return fres;
}

static gboolean
estimate_step_parameters(const gdouble *heights, guint n,
                         gdouble *stepheight, gdouble *offset,
                         const gchar **message)
{
    gdouble *steps;
    gdouble sh, smin, bestoff, p;
    gint i, g, ns, noff = 120;

    if (n < 2) {
        *message = _("No suitable terrace steps found");
        return FALSE;
    }

    steps = g_memdup(heights, n*sizeof(gdouble));
    gwy_math_sort(n, steps);
    ns = n-1;
    for (g = 0; g < ns; g++) {
        steps[g] = steps[g+1] - steps[g];
        gwy_debug("step%d: height %g nm", g, steps[g]/1e-9);
    }

    p = 85.0;
    gwy_math_percentiles(ns, steps, GWY_PERCENTILE_INTERPOLATION_LINEAR,
                         1, &p, &sh);
    gwy_debug("estimated step height %g nm", sh/1e-9);
    g_free(steps);

    *stepheight = sh;

    /* Find a good offset value. */
    smin = G_MAXDOUBLE;
    bestoff = 0.0;
    for (i = 0; i < noff; i++) {
        gdouble off = sh*i/noff;
        gdouble s = 0.0;

        for (g = 0; g < n; g++) {
            gint ng = GWY_ROUND((heights[g] - off)/sh);

            s += fabs(heights[g] - off - ng*sh);
        }
        if (s < smin) {
            smin = s;
            bestoff = off;
        }
    }
    gwy_debug("estimated base offset %g nm", bestoff/1e-9);

    *offset = bestoff;

    return TRUE;
}

static void
fill_fitted_image(GwyDataField *dfield,
                  GwyDataField *marked,
                  GwyDataField *residuum,
                  GwyDataField *fitted)
{
    gint xres = gwy_data_field_get_xres(dfield);
    gint yres = gwy_data_field_get_yres(dfield);
    const gdouble *d = gwy_data_field_get_data_const(dfield);
    const gdouble *r = gwy_data_field_get_data_const(residuum);
    const gdouble *m = gwy_data_field_get_data_const(marked);
    gdouble *f;
    gint n = xres*yres, k;
    gdouble avg;

    avg = gwy_data_field_get_avg(dfield);
    gwy_data_field_fill(fitted, avg);

    f = gwy_data_field_get_data(fitted);
    for (k = 0; k < n; k++) {
        if (m[k] > 0.0)
            f[k] = d[k] - r[k];
    }
}

static void
terrace_fit(TerraceControls *controls)
{
    GwyDataField *marked, *fitted, *residuum, *background,
                 *levelled, *terraces;
    GtkTreeModel *model;
    GwyResultsExport *rexport_list, *rexport_result;
    TerraceArgs *args = controls->args;
    FitResult *fres;
    GQuark quark;
    const gchar *message = "";

    if (!controls->terracecoords)
        return;

    gwy_app_wait_cursor_start(GTK_WINDOW(controls->dialogue));
    controls->fit_ok = FALSE;

    marked = gwy_container_get_object_by_name(controls->mydata, "/0/mask");
    quark = gwy_app_get_data_key_for_id(PREVIEW_FITTED);
    fitted = gwy_container_get_object(controls->mydata, quark);
    quark = gwy_app_get_data_key_for_id(PREVIEW_RESIDUUM);
    residuum = gwy_container_get_object(controls->mydata, quark);
    quark = gwy_app_get_data_key_for_id(PREVIEW_TERRACES);
    terraces = gwy_container_get_object(controls->mydata, quark);
    quark = gwy_app_get_data_key_for_id(PREVIEW_LEVELLED);
    levelled = gwy_container_get_object(controls->mydata, quark);
    quark = gwy_app_get_data_key_for_id(PREVIEW_BACKGROUND);
    background = gwy_container_get_object(controls->mydata, quark);

    rexport_list = GWY_RESULTS_EXPORT(controls->rexport_list);
    rexport_result = GWY_RESULTS_EXPORT(controls->rexport_result);
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(controls->terracelist));
    fres = terrace_do(marked, residuum, background, terraces,
                      controls->terracecoords, controls->terraceinfo,
                      args, controls->xc, controls->yc, TRUE, &message);
    gtk_label_set_text(GTK_LABEL(controls->message), message);
    update_results(controls, fres);
    if (!fres) {
        gwy_results_export_set_actions_sensitive(rexport_list, FALSE);
        gwy_results_export_set_actions_sensitive(rexport_result, FALSE);
        reset_images(controls);
        gwy_app_wait_cursor_finish(GTK_WINDOW(controls->dialogue));
        return;
    }

    controls->fit_ok = TRUE;
    gtk_label_set_text(GTK_LABEL(controls->message), "");
    gwy_results_export_set_actions_sensitive(rexport_list, TRUE);
    gwy_results_export_set_actions_sensitive(rexport_result, TRUE);

    /* Rerender the terrace table. */
    gwy_null_store_set_n_rows(GWY_NULL_STORE(model), 0);
    gwy_null_store_set_n_rows(GWY_NULL_STORE(model), fres->nterraces);

#ifdef DEBUG
    printf("%d %g %g %g %g %u\n",
           controls->args->poly_degree,
           fres->solution[0],
           sqrt(fres->invdiag[0])*fres->msq,
           fres->msq,
           fres->deltares,
           fres->nterraces);
#endif

    free_fit_result(fres);
    fill_fitted_image(controls->dfield, marked, residuum, fitted);
    gwy_data_field_subtract_fields(levelled, controls->dfield, background);

    update_diff_gradient(controls);
    gwy_data_field_data_changed(fitted);
    gwy_data_field_data_changed(residuum);
    gwy_data_field_data_changed(terraces);
    gwy_data_field_data_changed(levelled);
    gwy_data_field_data_changed(background);

    gwy_app_wait_cursor_finish(GTK_WINDOW(controls->dialogue));
}

static void
terrace_segment_do(TerraceControls *controls)
{
    GwyDataField *marked, *terraceids;
    GwyResultsExport *rexport_list, *rexport_result;
    GtkTreeModel *model;
    TerraceArgs *args = controls->args;
    GArray *terracecoords, *terraceinfo;
    GQuark quark;
    guint g, nterraces;

    controls->fit_ok = FALSE;
    free_terrace_coordinates(controls->terracecoords);
    controls->terracecoords = NULL;
    terraceinfo = controls->terraceinfo;

    gtk_dialog_set_response_sensitive(GTK_DIALOG(controls->dialogue),
                                      GTK_RESPONSE_OK, FALSE);

    marked = gwy_container_get_object_by_name(controls->mydata, "/0/mask");
    quark = gwy_app_get_data_key_for_id(PREVIEW_SEGMENTED);
    terraceids = gwy_container_get_object(controls->mydata, quark);

    rexport_list = GWY_RESULTS_EXPORT(controls->rexport_list);
    rexport_result = GWY_RESULTS_EXPORT(controls->rexport_result);
    gwy_results_export_set_actions_sensitive(rexport_list, FALSE);
    gwy_results_export_set_actions_sensitive(rexport_result, FALSE);
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(controls->terracelist));
    gwy_null_store_set_n_rows(GWY_NULL_STORE(model), 0);
    g_array_set_size(terraceinfo, 0);
    terracecoords = find_terrace_coordinates(controls->dfield, controls->mask,
                                             args, marked, terraceids,
                                             &controls->xc, &controls->yc);
    controls->terracecoords = terracecoords;

    if (terracecoords) {
        nterraces = terracecoords->len;
        gtk_label_set_text(GTK_LABEL(controls->message), "");
        for (g = 0; g < nterraces; g++) {
            TerraceCoords *tc = &coord_index(terracecoords, g);
            TerraceInfo info;

            gwy_clear(&info, 1);
            info.npixels = tc->ncoords;
            g_array_append_val(terraceinfo, info);
        }
        gwy_null_store_set_n_rows(GWY_NULL_STORE(model), nterraces);
    }
    else {
        gtk_label_set_text(GTK_LABEL(controls->message),
                           _("No terraces were found"));
    }
    gtk_dialog_set_response_sensitive(GTK_DIALOG(controls->dialogue),
                                      RESPONSE_REFINE,
                                      !!terracecoords);

    update_results(controls, NULL);
    update_terrace_colours(controls);
    gwy_data_field_data_changed(marked);
    gwy_data_field_data_changed(terraceids);
    reset_images(controls);
}

static void
update_results(TerraceControls *controls, FitResult *fres)
{
    GwyResults *results = controls->results;
    GwyMaskingType masking;
    guint i;

    for (i = 0; i < G_N_ELEMENTS(guivalues); i++)
        gtk_label_set_text(GTK_LABEL(controls->guivalues[i]), "");
    if (!controls->terracecoords)
        return;

    masking = controls->args->masking;
    if (!controls->mask)
        masking = GWY_MASK_IGNORE;
    gwy_results_fill_values(controls->results, "masking", masking, NULL);
    gwy_results_fill_values(results,
                            "nterraces", controls->terracecoords->len,
                            NULL);

    if (fres) {
        if (controls->args->independent)
            gwy_results_set_na(results, "step", "discrep", NULL);
        else {
            gwy_results_fill_values_with_errors(results,
                                                "step",
                                                fres->solution[0],
                                                sqrt(fres->invdiag[0])*fres->msq,
                                                NULL);
            gwy_results_fill_values(results, "discrep", fres->deltares, NULL);
        }
        gwy_results_fill_values(results, "resid", fres->msq, NULL);
    }

    for (i = 0; i < G_N_ELEMENTS(guivalues); i++) {
        if (!fres && !gwy_strequal(guivalues[i], "nterraces"))
            continue;

        gtk_label_set_markup(GTK_LABEL(controls->guivalues[i]),
                             gwy_results_get_full(results, guivalues[i]));
    }
}

static void
update_terrace_colours(TerraceControls *controls)
{
    GArray *terraceinfo = controls->terraceinfo;
    guint g, nterraces = terraceinfo->len;
    GwyGradient *gradient;

    gradient = gwy_inventory_get_item_or_default(gwy_gradients(), "DFit");
    g_return_if_fail(gradient);

    for (g = 0; g < nterraces; g++) {
        TerraceInfo *info = &info_index(terraceinfo, g);
        gwy_gradient_get_color(gradient, (g + 1.0)/nterraces, &info->colour);
    }
}

G_GNUC_UNUSED
static void
write_gwy_image(GwyDataField *dfield, const gchar *filename)
{
    GwyContainer *container = gwy_container_new();
    GByteArray *bytes;

    gwy_container_set_object_by_name(container, "/0/data", dfield);
    bytes = g_byte_array_new();
    g_byte_array_append(bytes, "GWYP", 4);
    gwy_serializable_serialize(G_OBJECT(container), bytes);
    g_file_set_contents(filename, bytes->data, bytes->len, NULL);
    g_byte_array_free(bytes, TRUE);
    g_object_unref(container);
}

static void
fill_terraces(GwyDataField *terraces, GwyDataField *marked,
              GArray *terracecoords,
              const gdouble *sheights, gboolean independent)
{
    GwyDataField *mask;
    guint i, g, nterraces, xres, yres, ng;
    gint minlevel = 0;
    gint *grains;
    gdouble *d, *zmap;

    nterraces = terracecoords->len;
    if (!independent) {
        minlevel = G_MAXINT;
        for (g = 0; g < nterraces; g++) {
            TerraceCoords *tc = &coord_index(terracecoords, g);
            minlevel = MIN(minlevel, tc->level);
        }
    }

    mask = gwy_data_field_duplicate(marked);
    xres = gwy_data_field_get_xres(mask);
    yres = gwy_data_field_get_yres(mask);
    gwy_data_field_grains_grow(mask, 25,
                               GWY_DISTANCE_TRANSFORM_EUCLIDEAN, TRUE);
    gwy_data_field_grains_grow(mask, 25,
                               GWY_DISTANCE_TRANSFORM_EUCLIDEAN, TRUE);
    gwy_data_field_grains_grow(mask, 25,
                               GWY_DISTANCE_TRANSFORM_EUCLIDEAN, TRUE);
    gwy_data_field_grains_grow(mask, 25,
                               GWY_DISTANCE_TRANSFORM_EUCLIDEAN, TRUE);

    grains = g_new0(gint, xres*yres);
    ng = gwy_data_field_number_grains(mask, grains);
    zmap = g_new(gdouble, ng+1);
    zmap[0] = 0.0;

    for (g = 0; g < nterraces; g++) {
        TerraceCoords *tc = &coord_index(terracecoords, g);

        i = grains[tc->pixels[0]];
        zmap[i] = (independent
                   ? sheights[g]
                   : (tc->level+1 - minlevel)*sheights[0]);
    }

    gwy_data_field_clear(terraces);
    d = gwy_data_field_get_data(terraces);
    for (i = 0; i < xres*yres; i++) {
        d[i] = zmap[grains[i]];
    }
    g_free(grains);

    gwy_data_field_laplace_solve(terraces, mask, 0, 1.0);
    write_gwy_image(terraces, "terraces.gwy");
    g_object_unref(mask);
}

/* XXX: The background is generally bogus far outside the fitted region.  This
 * usually means image corners because they contain too small terrace bits. It
 * is more meaningful to only calculate it for marked area. */
static void
fill_background(GwyDataField *background,
                const gint *term_powers, guint npowers,
                const gdouble *coeffs, gdouble xc, gdouble yc)
{
    gint maxpower, xres, yres, i, j, k;
    gdouble *xpowers, *ypowers, *d;

    maxpower = find_maximum_power(npowers, term_powers);
    xpowers = g_new(gdouble, maxpower+1);
    ypowers = g_new(gdouble, maxpower+1);
    xpowers[0] = ypowers[0] = 1.0;

    xres = gwy_data_field_get_xres(background);
    yres = gwy_data_field_get_yres(background);
    d = gwy_data_field_get_data(background);
    for (i = 0; i < yres; i++) {
        gdouble y = (2.0*i + 1 - yres)/(yres - 1) - yc;
        for (j = 0; j < xres; j++) {
            gdouble x = (2.0*j + 1 - xres)/(xres - 1) - xc;
            gdouble s = 0.0;

            for (k = 1; k <= maxpower; k++) {
                xpowers[k] = xpowers[k-1]*x;
                ypowers[k] = ypowers[k-1]*y;
            }

            for (k = 0; k < npowers; k++) {
                gint powx = term_powers[2*k + 0];
                gint powy = term_powers[2*k + 1];
                gdouble xp = xpowers[powx];
                gdouble yp = ypowers[powy];
                s += xp*yp*coeffs[k];
            }
            d[i*xres + j] = s;
        }
    }
    g_free(xpowers);
    g_free(ypowers);
}

static gboolean
analyse_topology(GArray *terracecoords, const TerraceArgs *args,
                 GwyDataField *terraces,
                 const gdouble *heights, gdouble sheight)
{
    guint xres, yres, nterraces, g, gg, i, j, k, kk, nreached;
    gint maxdist2, ldiff;
    GArray **boundaries;
    guint *ids, *neighcounts, *neighter;
    gboolean *reached = NULL;
    gboolean ok = FALSE;

    nterraces = terracecoords->len;
    xres = gwy_data_field_get_xres(terraces);
    yres = gwy_data_field_get_yres(terraces);

    /* Find boundary pixels of all terraces. */
    ids = g_new0(guint, xres*yres);
    for (g = 0; g < nterraces; g++) {
        TerraceCoords *tc = &coord_index(terracecoords, g);
        guint ncoords = tc->ncoords;

        for (k = 0; k < ncoords; k++)
            ids[tc->pixels[k]] = g+1;
    }

    boundaries = g_new0(GArray*, nterraces);
    for (g = 0; g < nterraces; g++) {
        TerraceCoords *tc = &coord_index(terracecoords, g);
        guint ncoords = tc->ncoords;

        boundaries[g] = g_array_new(FALSE, FALSE, sizeof(guint));
        for (k = 0; k < ncoords; k++) {
            kk = tc->pixels[k];
            i = kk/xres;
            j = kk % xres;
            if ((i > 0 && ids[kk-xres] != g+1)
                || (j > 0 && ids[kk-1] != g+1)
                || (j < xres-1 && ids[kk+1] != g+1)
                || (i < yres-1 && ids[kk+xres] != g+1)) {
                g_array_append_val(boundaries[g], i);
                g_array_append_val(boundaries[g], j);
            }
        }
        gwy_debug("terrace #%u has %u boundary pixels",
                  g, boundaries[g]->len/2);
    }

    g_free(ids);

    /* Go through all pairs of terraces and check if their have pixels which
     * are sufficiently close.  We base the criterion on kernel size and edge
     * broadening as they give natural neighbour terrace separation. */
    maxdist2 = GWY_ROUND(gwy_powi(2.0*(args->edge_kernel_size
                                       + args->edge_broadening), 2)
                         + 0.5*log(xres*yres));
    neighcounts = g_new0(guint, nterraces*nterraces);
#ifdef _OPENMP
#pragma omp parallel for if (gwy_threads_are_enabled()) default(none) \
            shared(nterraces,boundaries,maxdist2,neighcounts) \
            private(k,g,gg)
#endif
    for (k = 0; k < (nterraces-1)*nterraces/2; k++) {
        guint nb, nb2, ib, ib2, n;
        guint *b, *b2;

        g = (guint)floor(0.5*(sqrt(8*k+1) + 1) + 0.00001);
        gg = k - g*(g-1)/2;

        nb = boundaries[g]->len/2;
        nb2 = boundaries[gg]->len/2;
        b = &g_array_index(boundaries[g], guint, 0);
        n = 0;

        for (ib = 0; ib < nb; ib++) {
            gint xb1, yb1;

            yb1 = *(b++);
            xb1 = *(b++);
            b2 = &g_array_index(boundaries[gg], guint, 0);
            for (ib2 = 0; ib2 < nb2; ib2++) {
                gint dxb2, dyb2;

                dyb2 = *(b2++) - yb1;
                dxb2 = *(b2++) - xb1;

                if (dxb2*dxb2 + dyb2*dyb2 <= maxdist2)
                    n++;
            }
        }
        if (n < sqrt(fmin(nb, nb2)))
            continue;

        neighcounts[g*nterraces + gg] = neighcounts[gg*nterraces + g] = n;
    }

    for (g = 0; g < nterraces; g++)
        g_array_free(boundaries[g], TRUE);
    g_free(boundaries);

    /* Here comes the difficult part.  Make a consistent guess which terrace
     * is at what level based on relations to neighbours. */
    neighter = g_new0(guint, nterraces);
    for (g = 0; g < nterraces; g++) {
        for (gg = 0; gg < nterraces; gg++) {
            if (neighcounts[g*nterraces + gg]) {
                neighter[g]++;
                gwy_debug("%u and %u are neighbours (%u), "
                          "level diff %d (%g nm)",
                          g+1, gg+1, neighcounts[g*nterraces + gg],
                          GWY_ROUND((heights[gg] - heights[g])/sheight),
                          (heights[gg] - heights[g])/1e-9);
            }
        }
    }

    /* Find a terrace with the most neighbours. */
    g = 0;
    k = 0;
    for (gg = 0; gg < nterraces; gg++) {
        if (neighter[gg] > k) {
            k = neighter[gg];
            g = gg;
        }
    }
    if (!k) {
        /* Nothing is a neigbour of anything else.  So we cannot proceed. */
        gwy_debug("no neighbours");
        goto fail;
    }

    reached = g_new0(gboolean, nterraces);
    reached[g] = TRUE;
    nreached = 1;
    coord_index(terracecoords, g).level = 0;

    while (nreached < nterraces) {
        gboolean did_anything = FALSE;

        for (g = 0; g < nterraces; g++) {
            TerraceCoords *tc = &coord_index(terracecoords, g);
            for (gg = 0; gg < nterraces; gg++) {
                TerraceCoords *tc2 = &coord_index(terracecoords, gg);

                if (!reached[g]
                    || reached[gg]
                    || !neighcounts[g*nterraces + gg])
                    continue;

                reached[gg] = TRUE;
                ldiff = GWY_ROUND((heights[gg] - heights[g])/sheight);
                tc2->level = tc->level + ldiff;
                gwy_debug("%u level is %d, based on connection to %u (%d)",
                          gg+1, tc2->level, g+1, tc->level);
                nreached++;
                did_anything = TRUE;
            }
        }

        if (!did_anything) {
            /* The graph is not connected.  We could perhaps still proceed,
             * but for now just give up.  */
            gwy_debug("neighbour graph is not connected.");
            goto fail;
        }

        for (g = 0; g < nterraces; g++) {
            TerraceCoords *tc = &coord_index(terracecoords, g);
            for (gg = 0; gg < nterraces; gg++) {
                TerraceCoords *tc2 = &coord_index(terracecoords, gg);

                if (!reached[g]
                    || !reached[gg]
                    || !neighcounts[g*nterraces + gg])
                    continue;

                ldiff = GWY_ROUND((heights[gg] - heights[g])/sheight);
                if (tc2->level != tc->level + ldiff) {
                    gwy_debug("inconsistent level differences");
                    gwy_debug("%u level should be %d, "
                              "based on connection to %u (%d), but it is %d",
                              gg+1, tc->level + ldiff, g+1, tc->level,
                              tc2->level);
                    goto fail;
                }
            }
        }
    }
    gwy_debug("level assignment OK");
    ok = TRUE;

fail:
    g_free(reached);
    g_free(neighter);
    g_free(neighcounts);

    return ok;
}

static FitResult*
terrace_do(GwyDataField *marked, GwyDataField *residuum,
           GwyDataField *background, GwyDataField *terraces,
           GArray *terracecoords, GArray *terraceinfo,
           const TerraceArgs *args,
           gdouble xc, gdouble yc,
           gboolean fill_bg_and_terraces,
           const gchar **message)
{
    guint nterraces, npowers;
    FitResult *fres;
    gdouble sheight, offset;
    gint *term_powers;
    gdouble *power_block;
    gboolean indep = args->independent;
    gint g;

    nterraces = terracecoords->len;
    if (!nterraces) {
        *message = _("No terraces were found");
        return NULL;
    }

    term_powers = make_term_powers_except0(args->poly_degree, &npowers);
    power_block = calculate_power_matrix_block(terracecoords,
                                               npowers, term_powers);
    if (!(fres = fit_terraces_arbitrary(terracecoords,
                                        term_powers, npowers, power_block,
                                        indep ? residuum : NULL,
                                        message)))
        goto finalise;
    if (!estimate_step_parameters(fres->solution, nterraces,
                                  &sheight, &offset, message)) {
        free_fit_result(fres);
        fres = NULL;
        goto finalise;
    }

    if (!analyse_topology(terracecoords, args, terraces,
                          fres->solution, sheight)) {
        gwy_debug("assigning levels by plain rounding");
        for (g = 0; g < nterraces; g++) {
            TerraceCoords *tc = &coord_index(terracecoords, g);
            tc->level = GWY_ROUND((fres->solution[g] - offset)/sheight);
        }
    }
    for (g = 0; g < nterraces; g++) {
        TerraceCoords *tc = &coord_index(terracecoords, g);
        TerraceInfo *info = &info_index(terraceinfo, g);

        /* This does not depend on whether we run the second stage fit. */
        info->level = tc->level;
        info->height = fres->solution[g];
        /* This will be recalculated in the second stage fit.  Note that error
         * is anyway with respect to the multiple of estimated step height
         * and normally similar in both fit types. */
        info->error = fres->solution[g] - offset - tc->level*sheight;
        info->residuum = sqrt(tc->msq);
    }

    /* Normally also perform the second stage fitting with a single common
     * step height.  But if requested, avoid it, keeping the heights
     * independents. */
    if (!indep) {
        free_fit_result(fres);
        if (!(fres = fit_terraces_same_step(terracecoords,
                                            term_powers, npowers, power_block,
                                            indep ? NULL : residuum,
                                            message)))
            goto finalise;

        for (g = 0; g < nterraces; g++) {
            TerraceCoords *tc = &coord_index(terracecoords, g);
            TerraceInfo *info = &info_index(terraceinfo, g);

            info->error = tc->off;
            info->residuum = sqrt(tc->msq);
        }
    }
    if (fill_bg_and_terraces) {
        fill_background(background, term_powers, npowers,
                        fres->solution + (indep ? nterraces : 2), xc, yc);
        fill_terraces(terraces, marked, terracecoords, fres->solution, indep);
    }

finalise:
    g_free(power_block);
    g_free(term_powers);

    return fres;
}

static void
save_report(TerraceControls *controls)
{
    gchar *text;

    text = format_report(controls);
    gwy_save_auxiliary_data(_("Save Table"),
                            GTK_WINDOW(controls->dialogue), -1, text);
    g_free(text);
}

static void
copy_report(TerraceControls *controls)
{
    GtkClipboard *clipboard;
    GdkDisplay *display;
    gchar *text;

    text = format_report(controls);
    display = gtk_widget_get_display(controls->dialogue);
    clipboard = gtk_clipboard_get_for_display(display, GDK_SELECTION_CLIPBOARD);
    gtk_clipboard_set_text(clipboard, text, -1);
    g_free(text);
}

static gchar*
format_report(TerraceControls *controls)
{
    GwyResultsReportType report_style = controls->args->report_style;
    GArray *terraceinfo = controls->terraceinfo;
    GwySIUnitFormatStyle style = GWY_SI_UNIT_FORMAT_UNICODE;
    GwySIValueFormat *vfz;
    GwySIUnit *zunit;
    GString *text;
    const gchar *k_header, *Apx_header;
    gchar *retval, *h_header, *Delta_header, *r_header;
    guint g, nterraces;

    text = g_string_new(NULL);
    zunit = gwy_data_field_get_si_unit_z(controls->dfield);
    if (!(report_style & GWY_RESULTS_REPORT_MACHINE))
        vfz = gwy_si_unit_value_format_copy(controls->vf);
    else
        vfz = gwy_si_unit_get_format_for_power10(zunit, style, 0, NULL);

    h_header = g_strdup_printf("h [%s]", vfz->units);
    k_header = "k";
    Apx_header = "Apx";
    Delta_header = g_strdup_printf("Δ [%s]", vfz->units);
    r_header = g_strdup_printf("r [%s]", vfz->units);
    gwy_format_result_table_strings(text, report_style, 5,
                                    h_header, k_header, Apx_header,
                                    Delta_header, r_header);
    g_free(h_header);
    g_free(Delta_header);
    g_free(r_header);

    nterraces = terraceinfo->len;
    for (g = 0; g < nterraces; g++) {
        TerraceInfo *info = &info_index(terraceinfo, g);

        gwy_format_result_table_mixed(text, report_style, "viivv",
                                      info->height/vfz->magnitude,
                                      info->level,
                                      info->npixels,
                                      info->error/vfz->magnitude,
                                      info->residuum/vfz->magnitude);
    }

    retval = text->str;
    gwy_si_unit_value_format_free(vfz);
    g_string_free(text, FALSE);

    return retval;
}

static gdouble
interpolate_broadening(gdouble a, gdouble b, gdouble t)
{
    return pow((1.0 - t)*pow(a, pwr) + t*pow(b, pwr), 1.0/pwr);
}

/* NB: Modifies args! */
static guint
count_survey_items(TerraceArgs *args,
                   guint *pndegrees, guint *pnbroadenings)
{
    guint ndegrees, nbroadenings;

    if (!args->survey_poly)
        args->poly_degree_min = args->poly_degree_max = args->poly_degree;

    ndegrees = args->poly_degree_max - args->poly_degree_min + 1;
    if (pndegrees)
        *pndegrees = ndegrees;

    if (!args->survey_broadening)
        args->broadening_min = args->broadening_max = args->edge_broadening;
    nbroadenings = GWY_ROUND(2.0*(pow(args->broadening_max, pwr)
                                  - pow(args->broadening_min, pwr))) + 1;

    if (pnbroadenings)
        *pnbroadenings = nbroadenings;

    return nbroadenings*ndegrees;
}

static void
run_survey(TerraceControls *controls)
{
    GwyDataField *marked, *residuum, *terraces, *terraceids, *dfield, *mask;
    TerraceArgs myargs;
    FitResult *fres;
    GArray *terracecoords = NULL, *terraceinfo, *surveyout;
    GwyResultsReportType report_style;
    const gchar *message;
    GString *str;
    guint i, w, ndegrees, nbroadenings, totalwork;
    gdouble *broadening_values;
    gint *degree_values;
    gdouble xc, yc;

    myargs = *controls->args;

    report_style = myargs.report_style & ~GWY_RESULTS_REPORT_MACHINE;
    if (report_style == GWY_RESULTS_REPORT_COLON)
        report_style = GWY_RESULTS_REPORT_TABSEP;
    report_style |= GWY_RESULTS_REPORT_MACHINE;

    dfield = controls->dfield;
    mask = controls->mask;
    marked = gwy_data_field_new_alike(dfield, FALSE);
    terraceids = gwy_data_field_new_alike(dfield, FALSE);
    residuum = gwy_data_field_new_alike(dfield, FALSE);
    terraces = gwy_data_field_new_alike(dfield, FALSE);

    terraceinfo = g_array_new(FALSE, FALSE, sizeof(TerraceInfo));
    g_array_append_vals(terraceinfo,
                        controls->terraceinfo->data,
                        controls->terraceinfo->len);
    surveyout = g_array_new(FALSE, FALSE, sizeof(TerraceSurveyRow));

    totalwork = count_survey_items(&myargs, &ndegrees, &nbroadenings);
    degree_values = g_new(gint, ndegrees);
    for (i = 0; i < ndegrees; i++)
        degree_values[i] = myargs.poly_degree_min + i;

    broadening_values = g_new(gdouble, nbroadenings);
    for (i = 0; i < nbroadenings; i++) {
        gdouble t = (nbroadenings == 1 ? 0.5 : i/(nbroadenings - 1.0));
        broadening_values[i] = interpolate_broadening(myargs.broadening_min,
                                                      myargs.broadening_max,
                                                      t);
    }

    gwy_app_wait_start(GTK_WINDOW(controls->dialogue),
                       _("Fitting in progress..."));

    /* We only want to re-run segmentation when broadening changes.  This means
     * we must have broadening (or any other segmentation parameter) in the
     * outer cycle and polynomial degree as the inner cycle! */
    for (w = 0; w < totalwork; w++) {
        TerraceSurveyRow srow;

        myargs.poly_degree = degree_values[w % ndegrees];
        myargs.edge_broadening = broadening_values[w/ndegrees];
        if (w/nbroadenings != (w-1)/nbroadenings) {
            free_terrace_coordinates(terracecoords);
            terracecoords = find_terrace_coordinates(dfield, mask,
                                                     &myargs,
                                                     marked, terraceids,
                                                     &xc, &yc);
        }

        fres = terrace_do(marked, residuum, NULL, terraces,
                          terracecoords, terraceinfo,
                          &myargs, xc, yc, FALSE,
                          &message);

        gwy_clear(&srow, 1);
        srow.poly_degree = myargs.poly_degree;
        srow.edge_kernel_size = myargs.edge_kernel_size;
        srow.edge_threshold = myargs.edge_threshold;
        srow.edge_broadening = myargs.edge_broadening;
        srow.min_area_frac = myargs.min_area_frac;
        srow.fit_ok = !!fres;
        if (fres) {
            srow.nterraces = fres->nterraces;
            srow.step = fres->solution[0];
            srow.step_err = sqrt(fres->invdiag[0])*fres->msq;
            srow.msq = fres->msq;
            srow.discrep = fres->deltares;
        }
        g_array_append_val(surveyout, srow);

        free_fit_result(fres);

        if (!gwy_app_wait_set_fraction((w + 1.0)/totalwork))
            break;
    }

    gwy_app_wait_finish();

    free_terrace_coordinates(terracecoords);
    g_free(degree_values);
    g_free(broadening_values);
    g_array_free(terraceinfo, TRUE);
    g_object_unref(terraces);
    g_object_unref(residuum);
    g_object_unref(marked);
    g_object_unref(terraceids);

    if (w != totalwork) {
        g_array_free(surveyout, TRUE);
        return;
    }

    str = g_string_new(NULL);
    gwy_format_result_table_strings(str, report_style, 11,
                                    "Poly degree", "Edge kernel size",
                                    "Edge threshold", "Edge broadening",
                                    "Min area frac", "Fit OK", "Num terraces",
                                    "Step height", "Step height err",
                                    "Msq residual", "Discrepancy");
    for (i = 0; i < surveyout->len; i++) {
        TerraceSurveyRow *srow = &g_array_index(surveyout, TerraceSurveyRow, i);
        gwy_format_result_table_mixed(str, report_style, "ivvvvyivvvv",
                                      srow->poly_degree,
                                      srow->edge_kernel_size,
                                      srow->edge_threshold,
                                      srow->edge_broadening,
                                      srow->min_area_frac,
                                      srow->fit_ok,
                                      srow->nterraces,
                                      srow->step,
                                      srow->step_err,
                                      srow->msq,
                                      srow->discrep);
    }
    g_array_free(surveyout, TRUE);

    gwy_save_auxiliary_data(_("Save Terrace Fit Survey"),
                            GTK_WINDOW(controls->dialogue),
                            str->len, str->str);
    g_string_free(str, TRUE);
}

static const gchar edge_broadening_key[]   = "/module/terracefit/edge_broadening";
static const gchar edge_kernel_size_key[]  = "/module/terracefit/edge_kernel_size";
static const gchar edge_threshold_key[]    = "/module/terracefit/edge_threshold";
static const gchar independent_key[]       = "/module/terracefit/independent";
static const gchar masking_key[]           = "/module/terracefit/masking";
static const gchar min_area_frac_key[]     = "/module/terracefit/min_area_frac";
static const gchar output_flags_key[]      = "/module/terracefit/output_flags";
static const gchar poly_degree_key[]       = "/module/terracefit/poly_degree";
static const gchar poly_degree_max_key[]   = "/module/terracefit/poly_degree_max";
static const gchar poly_degree_min_key[]   = "/module/terracefit/poly_degree_min";
static const gchar broadening_max_key[]    = "/module/terracefit/broadening_max";
static const gchar broadening_min_key[]    = "/module/terracefit/broadening_min";
static const gchar report_style_key[]      = "/module/terracefit/report_style";
static const gchar use_only_mask_key[]     = "/module/terracefit/use_only_mask";
static const gchar survey_poly_key[]       = "/module/terracefit/survey_poly";
static const gchar survey_broadening_key[] = "/module/terracefit/survey_broadening";

static void
sanitize_args(TerraceArgs *args)
{
    args->poly_degree = CLAMP(args->poly_degree, 0, MAX_DEGREE);
    args->edge_kernel_size = CLAMP(args->edge_kernel_size, 1.0, 64.0);
    args->edge_threshold = CLAMP(args->edge_threshold, 0.0, 100.0);
    args->masking = gwy_enum_sanitize_value(args->masking,
                                            GWY_TYPE_MASKING_TYPE);
    args->edge_broadening = CLAMP(args->edge_broadening, 0.0, 16.0);
    args->min_area_frac = CLAMP(args->min_area_frac, 0.1, 40.0);
    args->independent = !!args->independent;
    args->use_only_mask = !!args->use_only_mask;
    args->output_flags &= OUTPUT_ALL;
    args->survey_poly = !!args->survey_poly;
    args->survey_broadening = !!args->survey_broadening;
    args->poly_degree_min = CLAMP(args->poly_degree_min, 0, MAX_DEGREE);
    args->poly_degree_max = CLAMP(args->poly_degree_max,
                                  args->poly_degree_min, MAX_DEGREE);
    args->broadening_min = CLAMP(args->broadening_min, 0, MAX_BROADEN);
    args->broadening_max = CLAMP(args->broadening_max,
                                 args->broadening_min, MAX_BROADEN);
}

static void
load_args(GwyContainer *settings, TerraceArgs *args)
{
    *args = terrace_defaults;

    gwy_container_gis_int32_by_name(settings, poly_degree_key,
                                    &args->poly_degree);
    gwy_container_gis_double_by_name(settings, edge_kernel_size_key,
                                     &args->edge_kernel_size);
    gwy_container_gis_double_by_name(settings, edge_threshold_key,
                                     &args->edge_threshold);
    gwy_container_gis_enum_by_name(settings, masking_key,
                                   &args->masking);
    gwy_container_gis_enum_by_name(settings, report_style_key,
                                   &args->report_style);
    gwy_container_gis_double_by_name(settings, min_area_frac_key,
                                     &args->min_area_frac);
    gwy_container_gis_double_by_name(settings, edge_broadening_key,
                                     &args->edge_broadening);
    gwy_container_gis_boolean_by_name(settings, independent_key,
                                      &args->independent);
    gwy_container_gis_boolean_by_name(settings, use_only_mask_key,
                                      &args->use_only_mask);
    gwy_container_gis_int32_by_name(settings, output_flags_key,
                                    (gint32*)&args->output_flags);
    gwy_container_gis_boolean_by_name(settings, survey_poly_key,
                                      &args->survey_poly);
    gwy_container_gis_int32_by_name(settings, poly_degree_min_key,
                                    &args->poly_degree_min);
    gwy_container_gis_int32_by_name(settings, poly_degree_max_key,
                                    &args->poly_degree_max);
    gwy_container_gis_boolean_by_name(settings, survey_broadening_key,
                                      &args->survey_broadening);
    gwy_container_gis_int32_by_name(settings, broadening_min_key,
                                    &args->broadening_min);
    gwy_container_gis_int32_by_name(settings, broadening_max_key,
                                    &args->broadening_max);
    sanitize_args(args);
}

static void
save_args(GwyContainer *settings, TerraceArgs *args)
{
    gwy_container_set_int32_by_name(settings, poly_degree_key,
                                    args->poly_degree);
    gwy_container_set_double_by_name(settings, edge_kernel_size_key,
                                     args->edge_kernel_size);
    gwy_container_set_double_by_name(settings, edge_threshold_key,
                                     args->edge_threshold);
    gwy_container_set_enum_by_name(settings, masking_key,
                                   args->masking);
    gwy_container_set_enum_by_name(settings, report_style_key,
                                   args->report_style);
    gwy_container_set_double_by_name(settings, min_area_frac_key,
                                     args->min_area_frac);
    gwy_container_set_double_by_name(settings, edge_broadening_key,
                                     args->edge_broadening);
    gwy_container_set_boolean_by_name(settings, independent_key,
                                      args->independent);
    gwy_container_set_boolean_by_name(settings, use_only_mask_key,
                                      args->use_only_mask);
    gwy_container_set_int32_by_name(settings, output_flags_key,
                                    args->output_flags);
    gwy_container_set_boolean_by_name(settings, survey_poly_key,
                                      args->survey_poly);
    gwy_container_set_int32_by_name(settings, poly_degree_min_key,
                                    args->poly_degree_min);
    gwy_container_set_int32_by_name(settings, poly_degree_max_key,
                                    args->poly_degree_max);
    gwy_container_set_boolean_by_name(settings, survey_broadening_key,
                                      args->survey_broadening);
    gwy_container_set_int32_by_name(settings, broadening_min_key,
                                    args->broadening_min);
    gwy_container_set_int32_by_name(settings, broadening_max_key,
                                    args->broadening_max);
}

/* vim: set cin et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
