#include "EXTERN.h"
#include "perl.h"
#include "XSUB.h"

#include <stdbool.h>
#include <string.h>

=pod
static void
data_dumper (SV *sv)
{
  dSP;

  PUSHMARK(SP);
  XPUSHs(sv);
  PUTBACK;

  SV *cvrv = eval_pv("sub {use Data::Dumper; print Dumper($_[0])}", TRUE);
  call_sv(cvrv, G_VOID);
}
=cut

static void
retry_tokens (SV *self, AV *tokens)
{
  int i;

  dSP;

  PUSHMARK(SP);
  XPUSHs(self);
  if (tokens)
    {
      int l = av_len(tokens);
      for (i = 0; i <= l; i++)
        {
          SV **token = av_fetch(tokens, i, 0);
          XPUSHs(*token);
        }
    }
  PUTBACK;
  call_method("retry_tokens", G_DISCARD);
}

static void
syntax_error (SV *self, char *thing)
{
  dSP;

  PUSHMARK(SP);
  XPUSHs(self);
  XPUSHs(sv_2mortal(newSVpv(thing, 0)));
  PUTBACK;
  call_method("syntax_error", G_DISCARD);
}

static bool commit = false;

static AV *
try_parse_setup (SV *self)
{
  SV **trying_tokens = hv_fetch((HV *)SvRV(self), "trying_tokens", strlen("trying_tokens"), 0);
  AV *old_trying_tokens = (AV *)SvRV(*trying_tokens);
  /* Keep this for later - we'll put it back into the hash */
  SvREFCNT_inc((SV *)old_trying_tokens);
  SV *new_trying_tokens = newRV_noinc((SV *)newAV());
  hv_store((HV *)SvRV(self), "trying_tokens", strlen("trying_tokens"), new_trying_tokens, 0);
  return old_trying_tokens;
}

static void
try_parse_cleanup (SV *self, char *thing, SV *parsed, AV *old_trying_tokens)
{
  if (parsed)
    {
      int i;
      SV **trying_tokens = hv_fetch((HV *)SvRV(self), "trying_tokens", strlen("trying_tokens"), 0);
      AV *new_tokens = (AV *)SvRV(*trying_tokens);
      for (i = 0; i <= av_len(new_tokens); i++)
        {
          SV **token = av_fetch(new_tokens, i, 0);
          SvREFCNT_inc(*token);
          av_push(old_trying_tokens, *token);
        }
      hv_store((HV *)SvRV(self), "trying_tokens", strlen("trying_tokens"), newRV_noinc((SV *)old_trying_tokens), 0);
    }
  else
    {
      if (commit)
        {
          syntax_error(self, thing);
          hv_store((HV *)SvRV(self), "skip_errors", strlen("skip_errors"), newSViv(1), 0);
        }

      SV **trying_tokens = hv_fetch((HV *)SvRV(self), "trying_tokens", strlen("trying_tokens"), 0);
      retry_tokens(self, (AV *)SvRV(*trying_tokens));

      hv_store((HV *)SvRV(self), "trying_tokens", strlen("trying_tokens"), newRV_noinc((SV *)old_trying_tokens), 0);
    }
}

#define try_parse(self, thing, ...) ({                                  \
      bool old_commit = commit;                                         \
      commit = false;                                                   \
      AV *old_trying_tokens = try_parse_setup(self);                    \
      SV *parsed = try_ ## thing (self, ##__VA_ARGS__);                 \
      try_parse_cleanup(self, #thing, parsed, old_trying_tokens);       \
      commit = old_commit;                                              \
      parsed; })

#define try_op_list_(self, thing, end, op, ...) ({                      \
      SV *arg1 = try_parse(self, thing);                                \
      AV *av = NULL;                                                    \
      SV *res = NULL;                                                   \
      if (arg1)                                                         \
        {                                                               \
          av = newAV();                                                 \
          SvREFCNT_inc(arg1);                                           \
          av_push(av, arg1);                                            \
          while (1)                                                     \
            {                                                           \
              if (end)                                                  \
                {                                                       \
                  if (try_parse(self, punctuator, end))                 \
                    break;                                              \
                                                                        \
                  SV *operand = try_parse(self, op, ##__VA_ARGS__);     \
                  if (!operand)                                         \
                    {                                                   \
                      SvREFCNT_dec(av);                                 \
                      av = NULL;                                        \
                      break;                                            \
                    }                                                   \
                  SvREFCNT_inc(operand);                                \
                  av_push(av, operand);                                 \
                }                                                       \
              else                                                      \
                {                                                       \
                  SV *operand = try_parse(self, op, ##__VA_ARGS__);     \
                  if (!operand)                                         \
                    break;                                              \
                  SvREFCNT_inc(operand);                                \
                  av_push(av, operand);                                 \
                }                                                       \
                                                                        \
              SV *arg = try_parse(self, thing);                         \
              if (!arg)                                                 \
                {                                                       \
                  SvREFCNT_dec(av);                                     \
                  av = NULL;                                            \
                  break;                                                \
                }                                                       \
              SvREFCNT_inc(arg);                                        \
              av_push(av, arg);                                         \
            }                                                           \
          if (av)                                                       \
            {                                                           \
              SV *ref = newRV_noinc((SV *)av);                          \
              res = ref;                                                \
            }                                                           \
        }                                                               \
      res;})

#define try_op_list(self, thing, end, op, ...) ({                       \
      bool old_commit = commit;                                         \
      AV *old_trying_tokens = try_parse_setup(self);                    \
      SV *parsed = try_op_list_(self, thing, end, op, ##__VA_ARGS__);   \
      try_parse_cleanup(self, #thing, parsed, old_trying_tokens);       \
      commit = old_commit;                                              \
      parsed; })

static SV *
new_obj4 (char *class, SV *arg1, SV *arg2, SV *arg3, SV *arg4)
{
  dSP;

  PUSHMARK(SP);
  XPUSHs(sv_2mortal(newSVpv(class, 0)));
  if (arg1)
    XPUSHs(arg1);
  if (arg2)
    XPUSHs(arg2);
  if (arg3)
    XPUSHs(arg3);
  if (arg4)
    XPUSHs(arg4);
  PUTBACK;

  int count = call_method("new", G_SCALAR);
  if (count != 1)
    croak("Bad return count from new");
  SPAGAIN;

  SV *obj = POPs;
  if (!SvOK(obj))
    return NULL;
  else
    return obj;
}

#define new_obj(class, arg1) new_obj4(class, arg1, NULL, NULL, NULL)
#define new_obj2(class, arg1, arg2) new_obj4(class, arg1, arg2, NULL, NULL)
#define new_obj3(class, arg1, arg2, arg3) new_obj4(class, arg1, arg2, arg3, NULL)

static SV *try_logical_or_expression(SV *self);
static SV *try_unary_expression(SV *self);
static SV *try_type_name(SV *self);
static SV *try_declarator(SV *self);
static SV *try_abstract_declarator(SV *self);
static SV *try_assignment_expression(SV *self);
static SV *try_designated_initialiser (SV *self);
static SV *try_cast_expression (SV *self);
static SV *try_type_specifier(SV *self);
static SV *try_statement(SV *self);
static SV *try_compound_statement(SV *self);

static SV *
try_token (SV *self)
{
  dSP;

  PUSHMARK(SP);
  XPUSHs(self);
  PUTBACK;
  int count = call_method("try_token", G_SCALAR);
  if (count != 1)
    croak("Bad return count from try_token");
  SPAGAIN;

  SV *token = POPs;
  if (!SvOK(token))
    return NULL;
  else
    return token;
}

static SV *
get_string (SV *token)
{
  dSP;

  PUSHMARK(SP);
  XPUSHs(token);
  PUTBACK;

  int count = call_method("string", G_SCALAR);
    
  if (count != 1)
    croak("Bad result count");

  SPAGAIN;
  SV *string = POPs;
  return string;
}

static bool
check_string (SV *token, char *name)
{
  if (!name)
    return true;

  SV *token_string_sv = get_string(token);

  if (!SvOK(token_string_sv))
    croak("Bad result value");

  char *token_string = SvPV_nolen(token_string_sv);
  return strcmp(name, token_string) == 0;
}

static SV *
try_identifier (SV *self)
{
  SV *token = try_token(self);

  if (!token)
    return NULL;

  if (!sv_derived_from(token, "CParse::Parser::Token::Identifier"))
    return NULL;

  return token;
}

static SV *
try_keyword (SV *self, char *name)
{
  SV *token = try_token(self);

  if (!token)
    return NULL;

  if (!sv_derived_from(token, "CParse::Parser::Token::Keyword"))
    return NULL;

  if (!check_string(token, name))
    return NULL;

  return token;
}

static SV *
try_punctuator (SV *self, char *name)
{
  SV *token = try_token(self);

  if (!token)
    return NULL;

  if (!sv_derived_from(token, "CParse::Parser::Token::Punctuator"))
    return NULL;

  if (!check_string(token, name))
    return NULL;

  return token;
}

static SV *
try_integer_constant (SV *self)
{
  SV *token = try_token(self);

  if (!token)
    return NULL;

  if (!sv_derived_from(token, "CParse::Parser::Token::Integer"))
    return NULL;

  return token;
}

static SV *
try_floating_constant (SV *self)
{
  SV *token = try_token(self);

  if (!token)
    return NULL;

  if (!sv_derived_from(token, "CParse::Parser::Token::Float"))
    return NULL;

  return token;
}

static SV *
try_character_constant (SV *self)
{
  SV *token = try_token(self);

  if (!token)
    return NULL;

  if (!sv_derived_from(token, "CParse::Parser::Token::Character"))
    return NULL;

  return token;
}

static SV *
try_one_string_literal (SV *self)
{
  SV *token = try_token(self);

  if (!token)
    return NULL;

  if (!sv_derived_from(token, "CParse::Parser::Token::String"))
    return NULL;

  return token;
}

static void
concatenate (SV *str, SV *tail)
{
  dSP;

  PUSHMARK(SP);
  XPUSHs(str);
  XPUSHs(tail);
  PUTBACK;

  call_method("concatenate", G_DISCARD);
}

static SV *
try_string_literal (SV *self)
{
  SV *str = try_parse(self, one_string_literal);
  if (!SvTRUE(str))
    return NULL;

  SV *next;
  while ((next = try_parse(self, one_string_literal)))
    {
      concatenate(str, next);
    }
  return str;
}

static void
push_arrayref(AV *dest, SV *source)
{
  int i;
  AV *list = (AV *)SvRV(source);
  for (i = 0; i <= av_len(list); i++)
    {
      SV **elem = av_fetch(list, i, 0);
      SvREFCNT_inc(*elem);
      av_push(dest, *elem);
    }
}

static SV *
process (SV *token)
{
  dSP;

  PUSHMARK(SP);
  XPUSHs(token);
  PUTBACK;

  int count = call_method("process", G_SCALAR);
    
  if (count != 1)
    croak("Bad result count");

  SPAGAIN;
  SV *obj = POPs;
  return obj;
}

static SV *
try_attribute_name (SV *self)
{
  SV *keyword = try_parse(self, keyword, NULL);
  if (keyword)
    return get_string(keyword);

  SV *identifier = try_parse(self, identifier);
  if (identifier)
    return get_string(identifier);

  return NULL;
}

static SV *
try_attribute (SV *self)
{
  SV *name = try_parse(self, attribute_name);
  if (!name)
    return NULL;

  AV *args = newAV();
  if (try_parse(self, punctuator, "("))
    {
      if (!try_parse(self, punctuator, ")"))
        {
          SV *identifier = try_parse(self, identifier);
          if (identifier)
            {
              av_push(args, process(identifier));
              SV *p = try_parse(self, punctuator, NULL);
              char *pstr = SvPV_nolen(get_string(p));
              if (strcmp(pstr, ",") == 0)
                {
                  SV *list = try_op_list(self, assignment_expression, ")", punctuator, ",");
                  if (!list)
                    {
                      SvREFCNT_dec(args);
                      return NULL;
                    }
                  push_arrayref(args, list);
                  SvREFCNT_dec(list);
                }
              else if (strcmp(pstr, ")") != 0)
                {
                  SvREFCNT_dec(args);
                  return NULL;
                }
            }
          else
            {
              SV *list = try_op_list(self, assignment_expression, ")", punctuator, ",");
              if (!list)
                {
                  SvREFCNT_dec(args);
                  return NULL;
                }
              push_arrayref(args, list);
              SvREFCNT_dec(list);
            }
        }
    }

  return new_obj2("CParse::Attribute", name, newRV_noinc((SV *)args));
}

static SV *
try_attribute_specifier (SV *self)
{
  if (try_parse(self, keyword, "__asm__"))
    {
      commit = true;
      if (!try_parse(self, punctuator, "("))
        return NULL;
      SV *name = try_parse(self, string_literal);
      if (!try_parse(self, punctuator, ")"))
        return NULL;

      SV *attr = new_obj2("CParse::Attribute", sv_2mortal(newSVpv("asm_name", 0)), name);

      return new_obj("CParse::AttributeList", newRV_noinc((SV *)av_make(1, &attr)));
    }

  if (!try_parse(self, keyword, "__attribute__"))
    return NULL;

  commit = true;

  if (!try_parse(self, punctuator, "("))
    return NULL;
  if (!try_parse(self, punctuator, "("))
    return NULL;

  SV *list = try_op_list(self, attribute, ")", punctuator, ",");
  if (!list)
    return NULL;
  if (!try_parse(self, punctuator, ")"))
    return NULL;

  /* We're going to discard this list, and build a new one out of some
   * of its members
   */

  AV *attrs = newAV();
  AV *alist = (AV *)SvRV(list);
  int i;
  for (i = 0; i <= av_len(alist); i++)
    {
      SV **arg = av_fetch(alist, i, 0);
      if (!sv_isa(*arg, "CParse::Parser::Token::Punctuator"))
        {
          SvREFCNT_inc(*arg);
          av_push(attrs, *arg);
        }
    }

  SvREFCNT_dec(list);

  return new_obj("CParse::AttributeList", newRV_noinc((SV *)attrs));
}

static SV *
try_attribute_specifier_list (SV *self)
{
  dSP;
  I32 ax;

  AV *specs = newAV();
  while (1)
    {
      SV *spec = try_parse(self, attribute_specifier);
      if (!spec)
        break;

      ENTER;
      SAVETMPS;

      PUSHMARK(SP);
      XPUSHs(spec);
      PUTBACK;
      int count = call_method("attributes", G_ARRAY);

      /* Evil duplicated from perlcall. I don't want to understand
         this. It makes ST work, and thusly lets me access the list
         returned by that method call using a loop. */
      SPAGAIN;
      SP -= count;
      ax = (SP - PL_stack_base) + 1;

      int i;
      for (i = 0; i < count; i++)
        {
          SV *s = ST(i);
          SvREFCNT_inc(s);
          av_push(specs, s);
        }

      PUTBACK;
      FREETMPS;
      LEAVE;
    }

  if (av_len(specs) == -1)
    {
      SvREFCNT_dec(specs);
      return NULL;
    }

  return new_obj("CParse::AttributeList", newRV_noinc((SV *)specs));
}

static SV *
try_storage_class_specifier (SV *self, char *name)
{
  SV *keyword = try_parse(self, keyword, name);
  if (!SvTRUE(keyword))
    return NULL;

  SV *str = get_string(keyword);
  char *pstr = SvPV_nolen(str);
  if (strcmp(pstr, "typedef") != 0
      && strcmp(pstr, "extern") != 0
      && strcmp(pstr, "static") != 0
      && strcmp(pstr, "auto") != 0
      && strcmp(pstr, "register") != 0)
    return NULL;

  SV *obj = new_obj("CParse::StorageClass", str);
  return obj;
}

static SV *
try_type_qualifier (SV *self, char *name)
{
  SV *keyword = try_parse(self, keyword, name);
  if (!SvTRUE(keyword))
    return NULL;

  SV *str = get_string(keyword);
  char *pstr = SvPV_nolen(str);
  if (strcmp(pstr, "const") != 0
      && strcmp(pstr, "restrict") != 0
      && strcmp(pstr, "volatile") != 0)
    return NULL;

  SV *obj = new_obj("CParse::TypeQualifier", str);
  return obj;
}

static SV *
try_function_specifier (SV *self, char *name)
{
  SV *keyword = try_parse(self, keyword, name);
  if (!SvTRUE(keyword))
    return NULL;

  SV *str = get_string(keyword);
  char *pstr = SvPV_nolen(str);
  if (strcmp(pstr, "inline") != 0)
    return NULL;

  SV *obj = new_obj("CParse::FunctionSpecifier", str);
  return obj;
}

static SV *
try_assignment_operator (SV *self)
{
  SV *punctuator = try_parse(self, punctuator, NULL);
  if (!SvTRUE(punctuator))
    return NULL;

  SV *str = get_string(punctuator);
  char *pstr = SvPV_nolen(str);
  if (strcmp(pstr, "=") != 0
      && strcmp(pstr, "*=") != 0
      && strcmp(pstr, "/=") != 0
      && strcmp(pstr, "%=") != 0
      && strcmp(pstr, "+=") != 0
      && strcmp(pstr, "-=") != 0
      && strcmp(pstr, "<<=") != 0
      && strcmp(pstr, ">>=") != 0
      && strcmp(pstr, "&=") != 0
      && strcmp(pstr, "^=") != 0
      && strcmp(pstr, "|=") != 0)
    return NULL;

  return get_string(punctuator);
}

static SV *
try_unary_operator (SV *self)
{
  SV *punctuator = try_parse(self, punctuator, NULL);
  if (!SvTRUE(punctuator))
    return NULL;

  SV *str = get_string(punctuator);
  char *pstr = SvPV_nolen(str);
  if (strcmp(pstr, "&") != 0
      && strcmp(pstr, "*") != 0
      && strcmp(pstr, "+") != 0
      && strcmp(pstr, "-") != 0
      && strcmp(pstr, "~") != 0
      && strcmp(pstr, "!") != 0)
    return NULL;

  return get_string(punctuator);
}

static SV *
try_equality_operator (SV *self)
{
  SV *punctuator = try_parse(self, punctuator, NULL);
  if (!SvTRUE(punctuator))
    return NULL;

  SV *str = get_string(punctuator);
  char *pstr = SvPV_nolen(str);
  if (strcmp(pstr, "==") != 0
      && strcmp(pstr, "!=") != 0)
    return NULL;

  return get_string(punctuator);
}

static SV *
try_relational_operator (SV *self)
{
  SV *punctuator = try_parse(self, punctuator, NULL);
  if (!SvTRUE(punctuator))
    return NULL;

  SV *str = get_string(punctuator);
  char *pstr = SvPV_nolen(str);
  if (strcmp(pstr, "<") != 0
      && strcmp(pstr, ">") != 0
      && strcmp(pstr, "<=") != 0
      && strcmp(pstr, ">=") != 0)
    return NULL;

  return get_string(punctuator);
}

static SV *
try_shift_operator (SV *self)
{
  SV *punctuator = try_parse(self, punctuator, NULL);
  if (!SvTRUE(punctuator))
    return NULL;

  SV *str = get_string(punctuator);
  char *pstr = SvPV_nolen(str);
  if (strcmp(pstr, "<<") != 0
      && strcmp(pstr, ">>") != 0)
    return NULL;

  return get_string(punctuator);
}

static SV *
try_additive_operator (SV *self)
{
  SV *punctuator = try_parse(self, punctuator, NULL);
  if (!SvTRUE(punctuator))
    return NULL;

  SV *str = get_string(punctuator);
  char *pstr = SvPV_nolen(str);
  if (strcmp(pstr, "+") != 0
      && strcmp(pstr, "-") != 0)
    return NULL;

  return get_string(punctuator);
}

static SV *
try_multiplicative_operator (SV *self)
{
  SV *punctuator = try_parse(self, punctuator, NULL);
  if (!SvTRUE(punctuator))
    return NULL;

  SV *str = get_string(punctuator);
  char *pstr = SvPV_nolen(str);
  if (strcmp(pstr, "*") != 0
      && strcmp(pstr, "%") != 0
      && strcmp(pstr, "/") != 0)
    return NULL;

  return get_string(punctuator);
}

static SV *
try_expression (SV *self)
{
  SV *list = try_op_list(self, assignment_expression, NULL, punctuator, ",");
  if (!SvTRUE(list))
    return NULL;

  return new_obj2("CParse::Op", list, sv_2mortal(newSVpv("CParse::Op::Expression", 0)));
}

static SV *
try_conditional_expression (SV *self)
{
  SV *cond_expr = try_parse(self, logical_or_expression);
  if (!cond_expr)
    return NULL;

  if (try_parse(self, punctuator, "?"))
    {
      commit = true;

      SV *true_expr = try_parse(self, expression);
      if (!true_expr)
        return NULL;
      if (!try_parse(self, punctuator, ":"))
        return NULL;

      SV *false_expr = try_parse(self, conditional_expression);
      if (!false_expr)
        return NULL;

      return new_obj3("CParse::Op::Conditional", cond_expr, true_expr, false_expr);
    }
  else
    {
      return cond_expr;
    }
}

static SV *
try_assignment (SV *self)
{
  SV *left_exp = try_parse(self, unary_expression);
  if (!left_exp)
    return NULL;
  SV *op = try_parse(self, assignment_operator);
  if (!op)
    return NULL;
  SV *right_exp = try_parse(self, assignment_expression);
  if (!right_exp)
    return NULL;

  return new_obj3("CParse::Op::Assign", left_exp, right_exp, op);
}

static SV *
try_assignment_expression (SV *self)
{
  SV *exp = try_parse(self, assignment);
  if (exp)
    return exp;
  exp = try_parse(self, conditional_expression);
  return exp;
}

static SV *
try_constant_expression (SV *self)
{
  return try_conditional_expression(self);
}

static SV *
try_constant (SV *self)
{
  SV *constant = try_parse(self, integer_constant);
  if (constant)
    return process(constant);
  constant = try_parse(self, floating_constant);
  if (constant)
    return process(constant);
  constant = try_parse(self, character_constant);
  if (constant)
    return process(constant);

  return NULL;
}

static SV *
try_primary_expression (SV *self)
{
  if (try_parse(self, punctuator, "("))
    {
      SV *exp = try_parse(self, expression);
      if (!exp)
        return NULL;
      if (!try_parse(self, punctuator, ")"))
        return NULL;
      return exp;
    }

  SV *string = try_parse(self, string_literal);
  if (string)
    return process(string);

  SV *identifier = try_parse(self, identifier);
  if (identifier)
    return process(identifier);

  return try_parse(self, constant);
}

static SV *
try_initialiser (SV *self)
{
  if (try_parse(self, punctuator, "{"))
    {
      AV *inits = newAV();
      bool first = true;
      while (1)
        {
          if (!try_parse(self, punctuator, "}"))
            {
              SvREFCNT_dec(inits);
              return NULL;
            }

          if (!first && !try_parse(self, punctuator, ","))
            {
              SvREFCNT_dec(inits);
              return NULL;
            }
          first = false;

          SV *init = try_parse(self, designated_initialiser);
          if (!init)
            {
              SvREFCNT_dec(inits);
              return NULL;
            }

          SvREFCNT_inc(init);
          av_push(inits, init);
        }
    }

  return try_parse(self, assignment_expression);
}

static SV *
try_designated_initialiser (SV *self)
{
  if (try_parse(self, punctuator, "["))
    {
      commit = true;
      SV *exp = try_parse(self, constant_expression);
      if (!exp)
        return NULL;
      if (!try_parse(self, punctuator, "]"))
        return NULL;
      return try_parse(self, initialiser);
    }

  if (try_parse(self, punctuator, "."))
    {
      commit = true;
      SV *id = try_parse(self, identifier);
      if (!id)
        return NULL;
      return try_parse(self, initialiser);
    }

  return try_parse(self, initialiser);
}

static SV *
try_compound_literal (SV *self)
{
  if (!try_parse(self, punctuator, "("))
    return NULL;
  SV *type = try_parse(self, type_name);
  if (!type)
    return NULL;
  if (!try_parse(self, punctuator, ")"))
    return NULL;
  if (!try_parse(self, punctuator, "{"))
    return NULL;

  AV *inits = newAV();
  bool first = true;
  while (1)
    {
      if (!try_parse(self, punctuator, "}"))
        {
          SvREFCNT_dec(inits);
          return NULL;
        }

      if (!first && !try_parse(self, punctuator, ","))
        {
          SvREFCNT_dec(inits);
          return NULL;
        }
      first = false;

      if (!try_parse(self, punctuator, "}"))
        {
          SvREFCNT_dec(inits);
          return NULL;
        }

      SV *init = try_parse(self, designated_initialiser);
      if (!init)
        {
          SvREFCNT_dec(inits);
          return NULL;
        }

      SvREFCNT_inc(init);
      av_push(inits, init);
    }

  SV *ref = newRV_noinc((SV *)inits);
  return new_obj2("CParse::Op::CompoundLiteral", type, ref);
}

static SV *
try_postfix_expression_prefix (SV *self)
{
  SV *compound = try_parse(self, compound_literal);
  if (compound)
    return compound;

  return try_parse(self, primary_expression);
}

static SV *
try_postfix_expression_array_suffix (SV *self)
{
  if (!try_parse(self, punctuator, "["))
    return NULL;

  SV *expr = try_parse(self, expression);
  if (!expr)
    return NULL;

  if (!try_parse(self, punctuator, "]"))
    return NULL;

  return new_obj("CParse::Op::ArraySubscript", expr);
}

static SV *
try_postfix_expression_call_suffix (SV *self)
{
  if (!try_parse(self, punctuator, "("))
    return NULL;

  SV *args = try_op_list(self, assignment_expression, NULL, punctuator, ",");

  if (!try_parse(self, punctuator, ")"))
    return NULL;

  if (!args)
    args = newRV_noinc((SV *)newAV());

  return new_obj("CParse::Op::Call", args);
}

static SV *
try_postfix_expression_member_suffix (SV *self)
{
  if (!try_parse(self, punctuator, "."))
    return NULL;

  SV *identifier = try_parse(self, identifier);
  if (!identifier)
    return NULL;

  SV *str = get_string(identifier);
  return new_obj("CParse::Op::Member", str);
}

static SV *
try_postfix_expression_member_indirect_suffix (SV *self)
{
  if (!try_parse(self, punctuator, "->"))
    return NULL;

  SV *identifier = try_parse(self, identifier);
  if (!identifier)
    return NULL;

  SV *str = get_string(identifier);
  return new_obj("CParse::Op::MemberIndirect", str);
}

static SV *
try_postfix_expression_post_increment_suffix (SV *self)
{
  if (!try_parse(self, punctuator, "++"))
    return NULL;

  return new_obj("CParse::Op::Postinc", NULL);
}

static SV *
try_postfix_expression_post_decrement_suffix (SV *self)
{
  if (!try_parse(self, punctuator, "--"))
    return NULL;

  return new_obj("CParse::Op::Postdec", NULL);
}

static SV *
try_postfix_expression (SV *self)
{
  SV *prefix = try_parse(self, postfix_expression_prefix);
  if (!prefix)
    return NULL;

  AV *suffixes = newAV();
  while (1)
    {
      SV *suffix;

      if ((suffix = try_parse(self, postfix_expression_array_suffix)))
        {
          SvREFCNT_inc(suffix);
          av_push(suffixes, suffix);
          continue;
        }

      if ((suffix = try_parse(self, postfix_expression_call_suffix)))
        {
          SvREFCNT_inc(suffix);
          av_push(suffixes, suffix);
          continue;
        }

      if ((suffix = try_parse(self, postfix_expression_member_suffix)))
        {
          SvREFCNT_inc(suffix);
          av_push(suffixes, suffix);
          continue;
        }

      if ((suffix = try_parse(self, postfix_expression_member_indirect_suffix)))
        {
          SvREFCNT_inc(suffix);
          av_push(suffixes, suffix);
          continue;
        }

      if ((suffix = try_parse(self, postfix_expression_post_increment_suffix)))
        {
          SvREFCNT_inc(suffix);
          av_push(suffixes, suffix);
          continue;
        }

      if ((suffix = try_parse(self, postfix_expression_post_decrement_suffix)))
        {
          SvREFCNT_inc(suffix);
          av_push(suffixes, suffix);
          continue;
        }

      break;
    }

  SV *ref = newRV_noinc((SV *)suffixes);
  return new_obj2("CParse::Op::Postfix", prefix, ref);
}

static SV *
try_unary_expression (SV *self)
{
  if (try_parse(self, keyword, "sizeof"))
    {
      commit = true;

      if (try_parse(self, punctuator, "("))
        {
          SV *type = try_parse(self, type_name);
          if (!type)
            return NULL;
          if (!try_parse(self, punctuator, ")"))
            return NULL;
          return new_obj("CParse::Op::Sizeof", type);
        }

      SV *expr = try_parse(self, unary_expression);
      if (!expr)
        return NULL;
      return new_obj("CParse::Op::SizeofExpr", expr);
    }

  if (try_parse(self, keyword, "__alignof__"))
    {
      commit = true;

      if (!try_parse(self, punctuator, "("))
        return NULL;

      SV *arg = try_parse(self, type_name);
      if (!arg)
        arg = try_parse(self, unary_expression);
      if (!arg)
        return NULL;

      if (!try_parse(self, punctuator, ")"))
        return NULL;

      return new_obj("CParse::Op::Alignof", arg);
    }

  if (try_parse(self, punctuator, "++"))
    {
      SV *expr = try_parse(self, unary_expression);
      if (!expr)
        return NULL;
      return new_obj("CParse::Op::Preinc", expr);
    }

  if (try_parse(self, punctuator, "--"))
    {
      SV *expr = try_parse(self, unary_expression);
      if (!expr)
        return NULL;
      return new_obj("CParse::Op::Predec", expr);
    }

  if (try_parse(self, keyword, "__extension__"))
    {
      commit = 1;
      return try_parse(self, cast_expression);
    }

  SV *op;
  if ((op = try_parse(self, unary_operator)))
    {
      SV *expr = try_parse(self, cast_expression);
      return new_obj2("CParse::Op::Unary", expr, op);
    }

  return try_parse(self, postfix_expression);
}

static SV *
try_explicit_cast (SV *self)
{
  if (!try_parse(self, punctuator, "("))
    return NULL;

  SV *type = try_parse(self, type_name);
  if (!type)
    return NULL;

  if (!try_parse(self, punctuator, ")"))
    return NULL;

  SV *expr = try_parse(self, cast_expression);
  if (!expr)
    return NULL;

  return new_obj2("CParse::Op::Cast", expr, type);
}

static SV *
try_cast_expression (SV *self)
{
  SV *expr = try_parse(self, explicit_cast);
  if (expr)
    return expr;
  return try_parse(self, unary_expression);
}

#define def_arith1_func(this, next, class)                              \
  static SV *                                                           \
  try_ ## this ## _expression (SV *self)                                \
  {                                                                     \
    SV *list = try_op_list(self, next ## _expression, NULL, this ## _operator); \
    if (!list)                                                          \
      return NULL;                                                      \
    return new_obj2("CParse::Op", list, sv_2mortal(newSVpv("CParse::Op::" #class, 0))); \
  }

#define def_arith2_func(this, next, class, op)                          \
  static SV *                                                           \
  try_ ## this ## _expression (SV *self)                                \
  {                                                                     \
    SV *list = try_op_list(self, next ## _expression, NULL, punctuator, op); \
    if (!list)                                                          \
      return NULL;                                                      \
    return new_obj2("CParse::Op", list, sv_2mortal(newSVpv("CParse::Op::" #class, 0))); \
  }

def_arith1_func(multiplicative, cast, Multiply)
def_arith1_func(additive, multiplicative, Add)
def_arith1_func(shift, additive, Shift)
def_arith1_func(relational, shift, Relation)
def_arith1_func(equality, relational, Equal)
def_arith2_func(and, equality, BitAnd, "&")
def_arith2_func(exclusive_or, and, BitXor, "^")
def_arith2_func(inclusive_or, exclusive_or, BitOr, "|")
def_arith2_func(logical_and, inclusive_or, BoolAnd, "&&")
def_arith2_func(logical_or, logical_and, BoolOr, "||")

static SV *
try_specifier_qualifier (SV *self)
{
  SV *sv;

  if ((sv = try_parse(self, attribute_specifier)))
    return sv;
  if ((sv = try_parse(self, type_specifier)))
    return sv;
  if ((sv = try_parse(self, type_qualifier, NULL)))
    return sv;

  return NULL;
}

static SV *
try_struct_declarator (SV *self)
{
  SV *attr1 = try_parse(self, attribute_specifier_list);
  SV *declarator = try_parse(self, declarator);
  if (!declarator)
    declarator = new_obj("CParse::Declarator", NULL);

  if (try_parse(self, punctuator, ":"))
    {
      SV *expr = try_parse(self, constant_expression);
      if (!expr)
        return NULL;
      SV *attr2 = try_parse(self, attribute_specifier_list);
      return new_obj4("CParse::StructDeclarator",
                      declarator, expr,
                      attr1 ? attr1 : &PL_sv_undef,
                      attr2 ? attr2 : &PL_sv_undef);
    }
  else
    {
      SV *attr2 = try_parse(self, attribute_specifier_list);
      return new_obj4("CParse::StructDeclarator",
                      declarator, &PL_sv_undef,
                      attr1 ? attr1 : &PL_sv_undef,
                      attr2 ? attr2 : &PL_sv_undef);
    }
}

static SV *
try_struct_declaration_declarator_list (SV *self)
{
  AV *decls = newAV();
  bool first = true;

  while (1)
    {
      if (try_parse(self, punctuator, ";"))
        break;

      if (!first)
        {
          if (!try_parse(self, punctuator, ","))
            {
              SvREFCNT_dec(decls);
              return NULL;
            }
        }
      first = false;

      SV *decl = try_parse(self, struct_declarator);
      if (!decl)
        {
          SvREFCNT_dec(decls);
          return NULL;
        }
      SvREFCNT_inc(decl);
      av_push(decls, decl);
    }

  return newRV_noinc((SV *)decls);
}

static SV *
try_struct_declaration (SV *self)
{
  SV *declarators;
  AV *specifiers = newAV();

  /* This loop accumulates specifiers until we find a single declarator list */
  while (1)
    {
      declarators = try_parse(self, struct_declaration_declarator_list);
      if (declarators)
        break;

      SV *specifier = try_parse(self, specifier_qualifier);
      if (specifier)
        {
          SvREFCNT_inc(specifier);
          av_push(specifiers, specifier);
          continue;
        }

      SvREFCNT_dec((SV *)specifiers);
      return NULL;
    }

  return new_obj2("CParse::StructDeclaration", newRV_noinc((SV *)specifiers), declarators);
}

static SV *
try_struct_specifier (SV *self)
{
  SV *attr1 = try_parse(self, attribute_specifier_list);

  SV *id = try_parse(self, identifier);

  if (try_parse(self, punctuator, "{"))
    {
      commit = true;

      AV *decls = newAV();
      while (1)
        {
          if (try_parse(self, punctuator, "}"))
            break;

          SV *decl = try_parse(self, struct_declaration);
          if (!decl)
            {
              SvREFCNT_dec(decls);
              return NULL;
            }
          SvREFCNT_inc(decl);
          av_push(decls, decl);
        }

      SV *attr2 = try_parse(self, attribute_specifier_list);

      return new_obj4("CParse::Struct",
                      id ? get_string(id) : &PL_sv_undef,
                      newRV_noinc((SV *)decls),
                      attr1 ? attr1 : &PL_sv_undef,
                      attr2 ? attr2 : &PL_sv_undef);
    }
  else
    {
      if (!id)
        return NULL;

      return new_obj("CParse::StructRef", get_string(id));
    }
}

static SV *
try_union_specifier (SV *self)
{
  SV *attr1 = try_parse(self, attribute_specifier_list);

  SV *id = try_parse(self, identifier);

  if (try_parse(self, punctuator, "{"))
    {
      commit = true;

      AV *decls = newAV();
      while (1)
        {
          if (try_parse(self, punctuator, "}"))
            break;

          SV *decl = try_parse(self, struct_declaration);
          if (!decl)
            {
              SvREFCNT_dec(decls);
              return NULL;
            }
          SvREFCNT_inc(decl);
          av_push(decls, decl);
        }

      SV *attr2 = try_parse(self, attribute_specifier_list);

      return new_obj4("CParse::Union",
                      id ? get_string(id) : &PL_sv_undef,
                      newRV_noinc((SV *)decls),
                      attr1 ? attr1 : &PL_sv_undef,
                      attr2 ? attr2 : &PL_sv_undef);
    }
  else
    {
      if (!id)
        return NULL;

      return new_obj("CParse::UnionRef", get_string(id));
    }
}

static SV *
try_enumerator (SV *self)
{
  SV *id = try_parse(self, identifier);
  if (!id)
    return NULL;

  SV *expr = &PL_sv_undef;
  if (try_parse(self, punctuator, "="))
    {
      expr = try_parse(self, constant_expression);
      if (!expr)
        return NULL;
    }

  return new_obj2("CParse::Enumerator", get_string(id), expr);
}

static SV *
try_enum_specifier (SV *self)
{
  SV *attr1 = try_parse(self, attribute_specifier_list);

  SV *id = try_parse(self, identifier);

  if (try_parse(self, punctuator, "{"))
    {
      commit = true;

      AV *decls = newAV();
      bool first = true;
      while (1)
        {
          if (try_parse(self, punctuator, "}"))
            break;

          if (!first)
            {
              if (!try_parse(self, punctuator, ","))
                {
                  SvREFCNT_dec(decls);
                  return NULL;
                }
            }
          first = false;

          if (try_parse(self, punctuator, "}"))
            break;

          SV *decl = try_parse(self, enumerator);
          if (!decl)
            {
              SvREFCNT_dec(decls);
              return NULL;
            }
          SvREFCNT_inc(decl);
          av_push(decls, decl);
        }

      SV *attr2 = try_parse(self, attribute_specifier_list);

      return new_obj4("CParse::Enum",
                      id ? get_string(id) : &PL_sv_undef,
                      newRV_noinc((SV *)decls),
                      attr1 ? attr1 : &PL_sv_undef,
                      attr2 ? attr2 : &PL_sv_undef);
    }
  else
    {
      if (!id)
        return NULL;

      return new_obj("CParse::EnumRef", get_string(id));
    }
}

static SV *
try_type_specifier (SV *self)
{
  SV *id = try_parse(self, identifier);
  if (id)
    return process(id);

  SV *keyword = try_parse(self, keyword, NULL);
  if (!keyword)
    return NULL;

  char *str = SvPV_nolen(get_string(keyword));
  static const char *basic_types[] = {"void",
                                      "char", "short", "int", "long",
                                      "float", "double",
                                      "signed", "unsigned",
                                      "_Bool", "_Complex", "_Imaginary",
                                      NULL};
  const char **t;
  for (t = basic_types; *t; t++)
    {
      if (strcmp(str, *t) == 0)
        return new_obj("CParse::TypeSpecifier", get_string(keyword));
    }

  if (strcmp(str, "__extension__") == 0)
    return new_obj("CParse::Extension", NULL);

  if (strcmp(str, "struct") == 0)
    return try_parse(self, struct_specifier);

  if (strcmp(str, "union") == 0)
    return try_parse(self, union_specifier);

  if (strcmp(str, "enum") == 0)
    return try_parse(self, enum_specifier);

  return NULL;
}

static SV *
try_type_name (SV *self)
{
  AV *specs = newAV();
  SV *decl;

  while (1)
    {
      decl = try_parse(self, abstract_declarator);
      if (decl)
        break;

      SV *spec = try_parse(self, specifier_qualifier);
      if (spec)
        {
          SvREFCNT_inc(spec);
          av_push(specs, spec);
          continue;
        }

      break;
    }

  if (av_len(specs) == -1)
    {
      SvREFCNT_dec(specs);
      return NULL;
    }

  if (!decl)
    decl = new_obj("CParse::Declarator", NULL);

  return new_obj2("CParse::TypeName", newRV_noinc((SV *)specs), decl);
}

static SV *
try_declaration_specifier (SV *self)
{
  SV *spec;

  if ((spec = try_parse(self, storage_class_specifier, NULL)))
    return spec;
  if ((spec = try_parse(self, type_qualifier, NULL)))
    return spec;
  if ((spec = try_parse(self, function_specifier, NULL)))
    return spec;
  if ((spec = try_parse(self, attribute_specifier)))
    return spec;

  return try_parse(self, type_specifier);
}

static SV *
try_pointer (SV *self)
{
  if (!try_parse(self, punctuator, "*"))
    return NULL;

  AV *quals = newAV();
  while (1)
    {
      SV *attr = try_parse(self, attribute_specifier);
      if (attr)
        {
          SvREFCNT_inc(attr);
          av_push(quals, attr);
          continue;
        }

      SV *qual = try_parse(self, type_qualifier, NULL);
      if (!qual)
        break;

      SvREFCNT_inc(qual);
      av_push(quals, qual);
    }

  SV *pointer = try_parse(self, pointer);

  return new_obj2("CParse::Pointer", newRV_noinc((SV *)quals), pointer ? pointer : &PL_sv_undef);
}

static SV *
try_direct_declarator_prefix (SV *self)
{
  if (try_parse(self, punctuator, "("))
    {
      SV *prefix = try_parse(self, declarator);
      if (!try_parse(self, punctuator, ")"))
        return NULL;
      return prefix;
    }
  else
    {
      SV *prefix = try_parse(self, identifier);
      if (prefix)
        return process(prefix);
      else
        return NULL;
    }
  croak("not reached");
}

static SV *
try_direct_declarator_array_suffix (SV *self)
{
  if (!try_parse(self, punctuator, "["))
    return NULL;

  commit = true;

  if (try_parse(self, punctuator, "*"))
    croak("Unhandled foo[*] construct");

  bool restrict = false;
  SV *expr = &PL_sv_undef;
  if (try_parse(self, type_qualifier, "restrict"))
    {
      restrict = true;
    }
  else
    {
      expr = try_parse(self, assignment_expression);
    }

  if (!try_parse(self, punctuator, "]"))
    return NULL;

  return new_obj2("CParse::Declarator::Array", expr ? expr : &PL_sv_undef, restrict ? sv_2mortal(newSViv(1)) : &PL_sv_undef);
}

static SV *
try_parameter_declaration (SV *self)
{
  AV *specs = newAV();

  while (1)
    {
      SV *spec = try_parse(self, declaration_specifier);
      if (!spec)
        break;
      SvREFCNT_inc(spec);
      av_push(specs, spec);
    }

  SV *decl = try_parse(self, declarator);
  if (!decl)
    decl = try_parse(self, abstract_declarator);
  if (!decl)
    decl = new_obj("CParse::Declarator", NULL);

  return new_obj2("CParse::ParameterDeclaration", newRV_noinc((SV *)specs), decl);
}

static SV *
try_direct_declarator_function_suffix (SV *self)
{
  if (!try_parse(self, punctuator, "("))
    return NULL;

  if (try_parse(self, punctuator, ")"))
    return new_obj2("CParse::Declarator::Function", newRV_noinc((SV *)newAV()), sv_2mortal(newSViv(1)));

  AV *parms = newAV();
  bool variadic = false;
  bool first = true;
  while (1)
    {
      if (try_parse(self, punctuator, ")"))
        break;

      if (!first)
        {
          if (!try_parse(self, punctuator, ","))
            {
              SvREFCNT_dec(parms);
              return NULL;
            }
        }
      first = false;

      if (try_parse(self, punctuator, "..."))
        {
          if (!try_parse(self, punctuator, ")"))
            {
              SvREFCNT_dec(parms);
              return NULL;
            }
          variadic = true;
          break;
        }

      SV *parm = try_parse(self, parameter_declaration);
      if (!parm)
        {
          SvREFCNT_dec(parms);
          return NULL;
        }
      SvREFCNT_inc(parm);
      av_push(parms, parm);
    }

  return new_obj2("CParse::Declarator::Function", newRV_noinc((SV *)parms), sv_2mortal(newSViv(variadic ? 1 : 0)));
}

static SV *
try_declarator (SV *self)
{
  SV *attr1 = try_parse(self, attribute_specifier_list);
  SV *pointer = try_parse(self, pointer);
  SV *prefix = try_parse(self, direct_declarator_prefix);
  if (!prefix)
    return NULL;

  bool had_function_suffix = false;
  AV *suffixes = newAV();
  while (1)
    {
      SV *suffix = try_parse(self, direct_declarator_array_suffix);
      if (suffix)
        {
          SvREFCNT_inc(suffix);
          av_push(suffixes, suffix);
          continue;
        }

      if (!had_function_suffix)
        {
          suffix = try_parse(self, direct_declarator_function_suffix);
          if (suffix)
            {
              had_function_suffix = true;
              SvREFCNT_inc(suffix);
              av_push(suffixes, suffix);
              continue;
            }
        }

      break;
    }

  SV *attr2 = try_parse(self, attribute_specifier_list);

  SV *direct = new_obj2("CParse::Declarator::Direct", prefix, newRV_noinc((SV *)suffixes));
  return new_obj4("CParse::Declarator", direct,
                  pointer ? pointer : &PL_sv_undef, 
                  attr1 ? attr1 : &PL_sv_undef,
                  attr2 ? attr2 : &PL_sv_undef);
}

static SV *
try_abstract_declarator_prefix (SV *self)
{
  if (!try_parse(self, punctuator, "("))
    return NULL;
  SV *decl = try_parse(self, abstract_declarator);
  if (!try_parse(self, punctuator, ")"))
    return NULL;
  return decl;
}

static SV *
try_direct_abstract_declarator (SV *self)
{
  SV *prefix = try_parse(self, abstract_declarator_prefix);
  if (!prefix)
    prefix = &PL_sv_undef;

  AV *suffixes = newAV();
  while (1)
    {
      SV *suffix = try_parse(self, direct_declarator_array_suffix);
      if (suffix)
        {
          SvREFCNT_inc(suffix);
          av_push(suffixes, suffix);
          continue;
        }

      suffix = try_parse(self, direct_declarator_function_suffix);
      if (suffix)
        {
          SvREFCNT_inc(suffix);
          av_push(suffixes, suffix);
          continue;
        }

      break;
    }

  return new_obj2("CParse::Declarator::Direct", prefix, newRV_noinc((SV *)suffixes));
}

static SV *
try_abstract_declarator (SV *self)
{
  SV *attr1 = try_parse(self, attribute_specifier_list);
  SV *pointer = try_parse(self, pointer);
  SV *decl = try_parse(self, direct_abstract_declarator);
  SV *attr2 = try_parse(self, attribute_specifier_list);

  if (!pointer && (!decl || !SvOK(decl)))
    return NULL;

  return new_obj4("CParse::Declarator",
                  decl ? decl : &PL_sv_undef,
                  pointer ? pointer : &PL_sv_undef, 
                  attr1 ? attr1 : &PL_sv_undef,
                  attr2 ? attr2 : &PL_sv_undef);
}

static SV *
try_init_declarator (SV *self)
{
  SV *decl = try_parse(self, declarator);
  SV *init = NULL;

  if (try_parse(self, punctuator, "="))
    {
      init = try_parse(self, initialiser);
      if (!init)
        return NULL;
    }

  return decl;
}

static SV *
try_declaration_declarator_list (SV *self)
{
  AV *decls = newAV();
  bool first = true;
  while (1)
    {
      if (try_parse(self, punctuator, ";"))
        break;

      if (!first)
        {
          if (!try_parse(self, punctuator, ","))
            {
              SvREFCNT_dec(decls);
              return NULL;
            }
        }
      first = false;

      SV *decl = try_parse(self, init_declarator);
      if (!decl)
        {
          SvREFCNT_dec(decls);
          return NULL;
        }

      SvREFCNT_inc(decl);
      av_push(decls, decl);
    }

  return newRV_noinc((SV *)decls);
}

static SV *
try_declaration (SV *self)
{
  SV *decls;
  AV *decl_specs = newAV();

  while (1)
    {
      decls = try_parse(self, declaration_declarator_list);
      if (decls)
        break;

      SV *spec = try_parse(self, declaration_specifier);
      if (spec)
        {
          SvREFCNT_inc(spec);
          av_push(decl_specs, spec);
          continue;
        }

      SvREFCNT_dec(decl_specs);
      return NULL;
    }

  return new_obj2("CParse::Declaration", newRV_noinc((SV *)decl_specs), decls);
}

static SV *
try_function_declarator (SV *self)
{
  SV *pointer = try_parse(self, pointer);
  SV *prefix = try_parse(self, direct_declarator_prefix);
  if (!prefix)
    return NULL;

  SV *suffix = try_parse(self, direct_declarator_function_suffix);
  if (!suffix)
    return NULL;

  SV *direct = new_obj2("CParse::Declarator::Direct", prefix, newRV_noinc((SV *)av_make(1, &suffix)));
  return new_obj2("CParse::Declarator", direct,
                  pointer ? pointer : &PL_sv_undef);
}

static SV *
try_function (SV *self)
{
  SV *decl;
  AV *decl_specs = newAV();

  while (1)
    {
      decl = try_parse(self, function_declarator);
      if (decl)
        break;

      SV *spec = try_parse(self, declaration_specifier);
      if (spec)
        {
          SvREFCNT_inc(spec);
          av_push(decl_specs, spec);
          continue;
        }

      SvREFCNT_dec(decl_specs);
      return NULL;
    }

  AV *decls = newAV();
  while (1)
    {
      SV *arg_decl = try_parse(self, declaration);
      if (!arg_decl)
        break;
      SvREFCNT_inc(arg_decl);
      av_push(decls, arg_decl);
    }

  SV *body = try_parse(self, compound_statement);
  if (!body)
    {
      SvREFCNT_dec(decl_specs);
      SvREFCNT_dec(decls);
      return NULL;
    }

  return new_obj3("CParse::Function", newRV_noinc((SV *)decl_specs), decl, newRV_noinc((SV *)decls));
}

static SV *
try_asm_clobbers (SV *self)
{
  return try_op_list(self, string_literal, NULL, punctuator, ",");
}

static SV *
try_asm_operand (SV *self)
{
  if (try_parse(self, punctuator, "["))
    {
      commit = true;
      if (!try_parse(self, identifier))
        return NULL;
      if (!try_parse(self, punctuator, "]"))
        return NULL;
    }
  SV *op = try_parse(self, string_literal);
  if (!op)
    return NULL;
  commit = true;
  if (!try_parse(self, punctuator, "("))
    return NULL;
  if (!try_parse(self, expression))
    return NULL;
  if (!try_parse(self, punctuator, ")"))
    return NULL;
  return op;
}

static SV *
try_asm_operands (SV *self)
{
  return try_op_list(self, asm_operand, NULL, punctuator, ",");
}

static SV *
try_asm_statement (SV *self)
{
  if (!try_parse(self, keyword, "__asm__"))
    return NULL;
  commit = true;
  try_parse(self, keyword, "volatile");
  if (!try_parse(self, punctuator, "("))
    return NULL;

  SV *expr = try_parse(self, expression);
  if (!expr)
    return NULL;
  if (try_parse(self, punctuator, ":"))
    {
      if (!try_parse(self, asm_operands))
        return NULL;
      
      if (try_parse(self, punctuator, ":"))
        {
          if (!try_parse(self, asm_operands))
            return NULL;

          if (try_parse(self, punctuator, ":"))
            {
              if (!try_parse(self, asm_clobbers))
                return NULL;
            }
        }
    }

  if (!try_parse(self, punctuator, ")"))
    return NULL;
  if (!try_parse(self, punctuator, ";"))
    return NULL;

  return expr;
}

static SV *
try_jump_statement (SV *self)
{
  if (try_parse(self, keyword, "goto"))
    {
      commit = true;
      SV *label = try_parse(self, identifier);
      if (!label)
        return NULL;
      if (!try_parse(self, punctuator, ";"))
        return NULL;
      return label;
    }
  if (try_parse(self, keyword, "continue"))
    {
      commit = true;
      if (!try_parse(self, punctuator, ";"))
        return NULL;
      return newRV_noinc((SV *)newAV());
    }
  if (try_parse(self, keyword, "break"))
    {
      commit = true;
      if (!try_parse(self, punctuator, ";"))
        return NULL;
      return newRV_noinc((SV *)newAV());
    }
  if (try_parse(self, keyword, "return"))
    {
      commit = true;
      try_parse(self, expression);
      if (!try_parse(self, punctuator, ";"))
        return NULL;
      return newRV_noinc((SV *)newAV());
    }

  return NULL;
}

static SV *
try_iteration_statement (SV *self)
{
  if (try_parse(self, keyword, "while"))
    {
      commit = true;

      if (!try_parse(self, punctuator, "("))
        return NULL;
      if (!try_parse(self, expression))
        return NULL;
      if (!try_parse(self, punctuator, ")"))
        return NULL;
      if (!try_parse(self, statement))
        return NULL;

      return newRV_noinc((SV *)newAV());
    }

  if (try_parse(self, keyword, "do"))
    {
      commit = true;

      if (!try_parse(self, statement))
        return NULL;
      if (!try_parse(self, keyword, "while"))
        return NULL;
      if (!try_parse(self, punctuator, "("))
        return NULL;
      if (!try_parse(self, expression))
        return NULL;
      if (!try_parse(self, punctuator, ")"))
        return NULL;
      if (!try_parse(self, punctuator, ";"))
        return NULL;

      return newRV_noinc((SV *)newAV());
    }

  if (try_parse(self, keyword, "for"))
    {
      commit = true;

      if (!try_parse(self, punctuator, "("))
        return NULL;
      if (!try_parse(self, declaration))
        {
          try_parse(self, expression);
          if (!try_parse(self, punctuator, ";"))
            return NULL;
        }
      try_parse(self, expression);
      if (!try_parse(self, punctuator, ";"))
        return NULL;
      try_parse(self, expression);
      if (!try_parse(self, punctuator, ")"))
        return NULL;
      if (!try_parse(self, statement))
        return NULL;

      return newRV_noinc((SV *)newAV());
    }

  return NULL;
}

static SV *
try_selection_statement (SV *self)
{
  if (try_parse(self, keyword, "if"))
    {
      commit = true;

      if (!try_parse(self, punctuator, "("))
        return NULL;
      if (!try_parse(self, expression))
        return NULL;
      if (!try_parse(self, punctuator, ")"))
        return NULL;
      if (!try_parse(self, statement))
        return NULL;
      if (try_parse(self, keyword, "else"))
        {
          if (!try_parse(self, statement))
            return NULL;
        }

      return newRV_noinc((SV *)newAV());
    }

  if (try_parse(self, keyword, "switch"))
    {
      commit = true;

      if (!try_parse(self, punctuator, "("))
        return NULL;
      if (!try_parse(self, expression))
        return NULL;
      if (!try_parse(self, punctuator, ")"))
        return NULL;
      if (!try_parse(self, statement))
        return NULL;

      return newRV_noinc((SV *)newAV());
    }

  return NULL;
}

static SV *
try_expression_statement (SV *self)
{
  SV *expr = try_parse(self, expression);
  if (!try_parse(self, punctuator, ";"))
    return NULL;
  return expr;
}

static SV *
try_labelled_statement (SV *self)
{
  if (try_parse(self, keyword, "case"))
    {
      commit = true;

      if (!try_parse(self, constant_expression))
        return NULL;
      if (!try_parse(self, punctuator, ":"))
        return NULL;
      if (!try_parse(self, statement))
        return NULL;

      return newRV_noinc((SV *)newAV());
    }

  if (try_parse(self, keyword, "default"))
    {
      commit = true;

      if (!try_parse(self, punctuator, ":"))
        return NULL;
      if (!try_parse(self, statement))
        return NULL;

      return newRV_noinc((SV *)newAV());
    }

  if (!try_parse(self, identifier))
    return NULL;
  if (!try_parse(self, punctuator, ":"))
    return NULL;
  try_parse(self, attribute_specifier_list);
  if (!try_parse(self, statement))
    return NULL;

  return newRV_noinc((SV *)newAV());
}

static SV *
try_null_statement (SV *self)
{
  if (!try_parse(self, punctuator, ";"))
    return NULL;
  return newRV_noinc((SV *)newAV());
}

static SV *
try_statement (SV *self)
{
  SV *stmt;
  if ((stmt = try_parse(self, null_statement)))
    return stmt;
  if ((stmt = try_parse(self, compound_statement)))
    return stmt;
  if ((stmt = try_parse(self, labelled_statement)))
    return stmt;
  if ((stmt = try_parse(self, selection_statement)))
    return stmt;
  if ((stmt = try_parse(self, iteration_statement)))
    return stmt;
  if ((stmt = try_parse(self, jump_statement)))
    return stmt;
  if ((stmt = try_parse(self, asm_statement)))
    return stmt;
  if ((stmt = try_parse(self, expression_statement)))
    return stmt;
  return NULL;
}

static SV *
try_compound_statement (SV *self)
{
  if (!try_parse(self, punctuator, "{"))
    return NULL;
  commit = true;

  while (1)
    {
      if (try_parse(self, punctuator, "}"))
        break;

      if (try_parse(self, statement))
        continue;
      if (try_parse(self, declaration))
        continue;

      return NULL;
    }

  return newRV_noinc((SV *)newAV());
}

MODULE = CParse::Parser::PerlXS		PACKAGE = CParse::Parser::PerlXS		
PROTOTYPES: ENABLE

SV *
try_parse(self, thing)
    SV *self;
    char *thing;
  CODE:
    SV *token = NULL;
    if (strcmp(thing, "declaration") == 0)
      token = try_parse(self, declaration);
    else if (strcmp(thing, "function") == 0)
      token = try_parse(self, function);
    else
      croak("Unhandled try_parse argument: '%s'", thing);
    SvREFCNT_inc(token);
    RETVAL = token;
  OUTPUT:
    RETVAL
