#!/usr/bin/env python

"""
General handler support for incoming calendar objects.

Copyright (C) 2014, 2015, 2016, 2017 Paul Boddie <paul@boddie.org.uk>

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 3 of the License, or (at your option) any later
version.

This program is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
details.

You should have received a copy of the GNU General Public License along with
this program.  If not, see <http://www.gnu.org/licenses/>.
"""

from email.mime.text import MIMEText
from imiptools.client import ClientForObject
from imiptools.config import settings
from imiptools.data import check_delegation, get_address, get_uri, \
                           get_sender_identities, uri_dict, uri_item
from socket import gethostname

MANAGER_PATH = settings["MANAGER_PATH"]
MANAGER_URL = settings["MANAGER_URL"]
MANAGER_URL_SCHEME = settings["MANAGER_URL_SCHEME"]

# References to the Web interface.

def get_manager_url():
    url_base = MANAGER_URL or \
               "%s%s/" % (MANAGER_URL_SCHEME or "https://", gethostname())
    return "%s/%s" % (url_base.rstrip("/"), MANAGER_PATH.lstrip("/"))

def get_object_url(uid, recurrenceid=None):
    return "%s/%s%s" % (
        get_manager_url().rstrip("/"), uid,
        recurrenceid and "/%s" % recurrenceid or ""
        )

class Handler(ClientForObject):

    "General handler support."

    def __init__(self, senders=None, recipient=None, messenger=None, store=None,
                 publisher=None, journal=None, preferences_dir=None):

        """
        Initialise the handler with any specifically indicated 'senders' and
        'recipient' of a calendar object. The object is initially undefined.

        The optional 'messenger' provides a means of interacting with the mail
        system.

        The optional 'store', 'publisher' and 'journal' can be specified to
        override the default store and publisher objects.
        """

        ClientForObject.__init__(self, None, recipient and get_uri(recipient),
            messenger, store, publisher, journal, preferences_dir)

        self.senders = senders and set(map(get_address, senders))
        self.recipient = recipient and get_address(recipient)

        self.results = []
        self.outgoing_methods = set()

    def wrap(self, text, link=True):

        "Wrap any valid message for passing to the recipient."

        _ = self.get_translator()

        texts = []
        texts.append(text)

        # Add a link to the manager application if available and requested.

        if link and self.have_manager():
            texts.append(_("If your mail program cannot handle this "
                           "message, you may view the details here:\n\n%s\n") %
                         get_object_url(self.uid, self.recurrenceid))

        # Create the text part, tagging it with a header that allows this part
        # to be merged with other calendar information.

        text_part = MIMEText("\n\n".join(texts))
        text_part["X-IMIP-Agent"] = "info"
        return self.add_result(None, None, text_part)

    # Result registration.

    def add_result(self, method, outgoing_recipients, part):

        """
        Record a result having the given 'method', 'outgoing_recipients' and
        message 'part'.
        """

        if outgoing_recipients:
            self.outgoing_methods.add(method)
        self.results.append((outgoing_recipients, part))

    def add_results(self, methods, outgoing_recipients, parts):

        """
        Record results having the given 'methods', 'outgoing_recipients' and
        message 'parts'.
        """

        if outgoing_recipients:
            self.outgoing_methods.update(methods)
        for part in parts:
            self.results.append((outgoing_recipients, part))

    def get_results(self):
        return self.results

    def get_outgoing_methods(self):
        return self.outgoing_methods

    # Logic, filtering and access to calendar structures and other data.

    def filter_by_senders(self, mapping):

        """
        Return a list of items from 'mapping' filtered using sender information.
        """

        if self.senders:

            # Get a mapping from senders to identities.

            identities = get_sender_identities(mapping)

            # Find the senders that are valid.

            senders = map(get_address, identities)
            valid = self.senders.intersection(senders)

            # Return the true identities.

            return reduce(lambda a, b: a + b, [identities[get_uri(address)] for address in valid], [])
        else:
            return mapping

    def filter_by_recipient(self, mapping):

        """
        Return a list of items from 'mapping' filtered using recipient
        information.
        """

        if self.recipient:
            addresses = set(map(get_address, mapping))
            return map(get_uri, addresses.intersection([self.recipient]))
        else:
            return mapping

    def is_delegation(self):

        """
        Return whether delegation is occurring by returning any delegator.
        """

        attendee_map = uri_dict(self.obj.get_value_map("ATTENDEE"))
        attendee_attr = attendee_map.get(self.user)
        return check_delegation(attendee_map, self.user, attendee_attr)

    def require_organiser(self, from_organiser=True):

        """
        Return the normalised organiser for the current object, filtered for the
        sender or recipient of interest. Return None if no identities are
        eligible.

        If the sender is not the organiser but is delegating to the recipient,
        the actual organiser is returned.
        """

        organiser, organiser_attr = organiser_item = uri_item(self.obj.get_item("ORGANIZER"))

        if not organiser:
            return None

        # Check the delegate status of the recipient.

        delegated = from_organiser and self.is_delegation()

        # Only provide details for an organiser who sent/receives the message or
        # is presiding over a delegation.

        organiser_filter_fn = from_organiser and self.filter_by_senders or self.filter_by_recipient

        if not delegated and not organiser_filter_fn(dict([organiser_item])):
            return None

        # Test against any previously-received organiser details.

        if not self.is_recognised_organiser(organiser):
            replacement = self.get_organiser_replacement()

            # Allow any organiser as a replacement where indicated.

            if replacement == "any":
                pass

            # Allow any recognised attendee as a replacement where indicated.

            elif replacement != "attendee" or not self.is_recognised_attendee(organiser):
                return None

        return organiser_item

    def require_attendees(self, from_organiser=True):

        """
        Return the attendees for the current object, filtered for the sender or
        recipient of interest. Return None if no identities are eligible.

        The attendee identities are normalized.
        """

        attendee_map = uri_dict(self.obj.get_value_map("ATTENDEE"))

        # Only provide details for attendees who sent/receive the message.

        attendee_filter_fn = from_organiser and self.filter_by_recipient or self.filter_by_senders

        attendees = {}
        for attendee in attendee_filter_fn(attendee_map):
            if attendee:
                attendees[attendee] = attendee_map[attendee]

        return attendees

    def require_organiser_and_attendees(self, from_organiser=True):

        """
        Return the organiser and attendees for the current object, filtered for
        the recipient of interest. Return None if no identities are eligible.

        Organiser and attendee identities are normalized.
        """

        organiser_item = self.require_organiser(from_organiser)
        attendees = self.require_attendees(from_organiser)

        if not attendees or not organiser_item:
            return None

        return organiser_item, attendees

# vim: tabstop=4 expandtab shiftwidth=4
