/**
   \file lcmaps_if.h 
   \brief interface functions for lcmaps.

   This file contains the macros that come with the LCMAPS
   API. Strictly speaking, they are not part of the API functions
   in the LCMAPS libraries, but help to get LCMAPS loaded and ready to run.

   The macros can be used in two modes:

   -# when linking with LCMAPS at build-time
   -* when using dlopen() at run-time. In this mode, 
      set #define LCMAPS_USE_DLOPEN


   The LCMAPS API for client applications makes use of run-time linking
   with dlopen(), in order to support all kinds of deployment scenarios.
   
   Since all of the actual linking is figured out at run-time, it becomes
   possible to use the resulting client in combination with many older (as
   well as newer) versions of LCMAPS.

   It is also possible to use the API with straightforward linking almost
   without any change in the code (but the compile and link steps will be
   different).

   By defining \b LCMAPS_USE_DLOPEN you select the dlopen() style linking
   of LCMAPS. By leaving it undefined, direct linking is used and the flag
   \verbatim -llcmaps_return_account_from_pem \endverbatim
   must be given at link time.

   \ingroup LcmapsInterface
*/


#include "_lcmaps.h"

/**
   \brief Macro to retrieve the error message in case of a failure
   In some failure cases of the other macros, an error message can be retrieved
   with this macro. It is of type char * and may be overwritten by subsequent
   macro calls. 
   \param l the LCMAPS_HANDLE
   \return string containing the error message
*/
#define LCMAPS_ERRMSG(l) _LCMAPS_ERRMSG(l)

/**
   \brief Macro to load specific extra functions

   Newer versions of LCMAPS offer some features not found in older versions.
   Applications that want to exploit these functions should declare their
   intentions by calling LCMAPS_REQUIRE_FUNC for each function separately. 
   \param l the LCMAPS_HANDLE
   \param f the function to include
   \retval 1 success
   \retval 0 failure
*/
#define LCMAPS_REQUIRE_FUNC(l,f) _LCMAPS_REQUIRE_FUNC(l,f)

/** 

   \defgroup VersionMacros Version macros

  The version of the LCMAPS library can be found by the functions
  lcmaps_get_major_version, lcmaps_get_minor_version and
  lcmaps_get_patch_version, but older versions of LCMAPS did not
  feature them. The following macros provide a safe way to retrieve
  these numbers, and they will return 0 in case the functions are not
  there.

  @{
*/
/**
  \def LCMAPS_MAJOR_VERSION(handle)
  \brief the major version of the LCMAPS library
  \param handle the LCMAPS handle
*/
/**
  \def LCMAPS_MINOR_VERSION(handle)
  \brief the minor version of the LCMAPS library
  \param handle the LCMAPS handle
*/
/**
  \def LCMAPS_PATCH_VERSION(handle)
  \brief the patch version of the LMAPS library
  \param handle the LCMAPS handle
 */
#define LCMAPS_MAJOR_VERSION(handle) _LCMAPS_MAJOR_VERSION(handle)
#define LCMAPS_MINOR_VERSION(handle) _LCMAPS_MINOR_VERSION(handle)
#define LCMAPS_PATCH_VERSION(handle) _LCMAPS_PATCH_VERSION(handle)

/**@}*/

/**
  The LCMAPS_HANDLE \a l is set to load the LCMAPS interface library
  from the system default locations. Applications may have reasons
  to load from a different location, so this macro can be used to
  set the name \a p of the shared object. This is going to be 
  passed straight to dlopen(). Be mindful that as \a p is not copied,
  it remains within scope throughout the use of the handle.
  For convenience we define empty macros in the case we do not use dlopen().

  \param l the LCMAPS_HANDLE
  \param p path (absolute or relative) to the shared object to be opened
 */
#define LCMAPS_SET_LIBFILE_PATH(l,p) _LCMAPS_SET_LIBFILE_PATH(l,p)

/**
  The LCMAPS_HANDLE \a l will load a helper library as a workaround
  for a bug in older versions of LCMAPS. This macro can be used to
  override the name of this helper. The same caveat applies as with 
  LCMAPS_SET_LIBFILE_PATH.
  For convenience we define empty macros in the case we do not use dlopen().
  \param l the LCMAPS_HANDLE
  \param p path (absolute or relative) to the helper library to use.

 */
#define LCMAPS_SET_HELPER_PATH(l,p) _LCMAPS_SET_HELPER_PATH(l,p)

/**
   \brief Macro to call a function

   Calls to the LCMAPS interface functions should be done through this
   macro. The function parameter list should follow the macro call, like so:

   LCMAPS_CALL(handle,lcmaps_init_t)(logfile)

   The list of functions that can be used is defined by each of the individual
   interface files.

   \param l the LCMAPS_HANDLE
   \param f the LCMAPS function to call.

 */

#define LCMAPS_CALL(l,f) _LCMAPS_CALL(l,f)

