% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bit64-package.R, R/integer64.R
\docType{package}
\name{bit64-package}
\alias{bit64-package}
\alias{bit64}
\alias{is.integer.integer64}
\alias{is.vector.integer64}
\alias{integer64}
\alias{is.integer64}
\alias{length<-.integer64}
\alias{print.integer64}
\alias{str.integer64}
\title{A S3 class for vectors of 64bit integers}
\usage{
integer64(length = 0L)

is.integer64(x)

\method{length}{integer64}(x) <- value

\method{print}{integer64}(x, quote = FALSE, ...)

\method{str}{integer64}(
  object,
  vec.len = strO$vec.len,
  give.head = TRUE,
  give.length = give.head,
  ...
)
}
\arguments{
\item{length}{length of vector using \code{\link[=integer]{integer()}}}

\item{x}{an integer64 vector}

\item{value}{an integer64 vector of values to be assigned}

\item{quote}{logical, indicating whether or not strings should be printed with surrounding quotes.}

\item{...}{further arguments to the \code{\link[=NextMethod]{NextMethod()}}}

\item{object}{an integer64 vector}

\item{vec.len, give.head, give.length}{see \code{\link[utils:str]{utils::str()}}}
}
\value{
\code{integer64} returns a vector of 'integer64', i.e.,
a vector of \code{\link[=double]{double()}} decorated with class 'integer64'.
}
\description{
Package 'bit64' provides fast serializable S3 atomic 64bit (signed) integers
that can be used in vectors, matrices, arrays and data.frames. Methods are
available for coercion from and to logicals, integers, doubles, characters
and factors as well as many elementwise and summary functions.
\subsection{Version 0.8}{

With 'integer64' vectors you can store very large integers at the expense
of 64 bits, which is by factor 7 better than 'int64' from package 'int64'.
Due to the smaller memory footprint, the atomic vector architecture and
using only S3 instead of S4 classes, most operations are one to three orders
of magnitude faster: Example speedups are 4x for serialization, 250x for
adding, 900x for coercion and 2000x for object creation. Also 'integer64'
avoids an ongoing (potentially infinite) penalty for garbage collection
observed during existence of 'int64' objects (see code in example section).
}

\subsection{Version 0.9}{

Package 'bit64' - which extends R with fast 64-bit integers - now has fast
(single-threaded) implementations the most important univariate algorithmic
operations (those based on hashing and sorting). We now have methods for
'match', '\%in\%', 'duplicated', 'unique', 'table', 'sort', 'order', 'rank',
'quantile', 'median' and 'summary'. Regarding data management we also have
novel generics 'unipos' (positions of the unique values), 'tiepos' (
positions of ties), 'keypos' (positions of foreign keys in a sorted
dimension table) and derived methods 'as.factor' and 'as.ordered'. This 64-
bit functionality is implemented carefully to be not slower than the
respective 32-bit operations in Base R and also to avoid outlying waiting
times observed with 'order', 'rank' and 'table' (speedup factors 20/16/200
respective). This increases the dataset size with wich we can work truly
interactive. The speed is achieved by simple heuristic optimizers in high-
level functions choosing the best from multiple low-level algorithms and
further taking advantage of a novel caching if activated. In an example R
session using a couple of these operations the 64-bit integers performed 22x
faster than base 32-bit integers, hash-caching improved this to 24x,
sortorder-caching was most efficient with 38x (caching hashing and sorting
is not worth it with 32x at duplicated RAM consumption).
}
}
\note{
\code{integer64} are useful for handling database keys and exact counting in +-2^63.
Do not use them as replacement for 32bit integers, integer64 are not supported for
subscripting by R-core and they have different semantics when combined with double.
Do understand that \code{integer64} can only be useful over \code{double} if we do not coerce
it to \code{double}.

While

integer + double -> double + double -> double

or

1L + 0.5 -> 1.5

for additive operations we coerce to \code{integer64}

integer64 + double ->  integer64 + integer64 -> integer64

hence

as.integer64(1) + 0.5 -> 1LL + 0LL -> 1LL

see section "Arithmetic precision and coercion" above
}
\section{Design considerations}{
64 bit integers are related to big data: we need them to overcome address space
limitations. Therefore performance of the 64 bit integer type is critical. In the
S language -- designed in 1975 -- atomic objects were defined to be vectors for a
couple of good reasons: simplicity, option for implicit parallelization, good
cache locality. In recent years many analytical databases have learnt that lesson:
column based data bases provide superior performance for many applications, the
result are products such as MonetDB, Sybase IQ, Vertica, Exasol, Ingres Vectorwise.
If we introduce 64 bit integers not natively in Base R but as an external package,
we should at least strive to make them as 'basic' as possible. Therefore the design
choice of bit64 not only differs from package int64, it is obvious: Like the other
atomic types in Base R, we model data type 'integer64' as a contiguous \code{\link{atomic}}
vector in memory, and we use the more basic \link{S3} class system, not \link{S4}. Like
package int64 we want our 'integer64' to be \code{\link{serialize}}able, therefore we also
use an existing data type as the basis. Again the choice is obvious: R has only one
64 bit data type: doubles. By using \code{\link{double}}s, \code{integer64} \code{\link{inherits}} some
functionality such as \code{\link[=is.atomic]{is.atomic()}}, \code{\link[=length]{length()}}, \code{\link{length<-}}, \code{\link[=names]{names()}},
\code{\link{names<-}}, \code{\link[=dim]{dim()}}, \code{\link{dim<-}}, \code{\link[=dimnames]{dimnames()}}, \code{\link{dimnames<-}}.

Our R level functions strictly follow the functional programming paradigm:
no modification of arguments or other side-effects. Before version 0.93  we
internally deviated from the strict paradigm in order to boost performance. Our C
functions do not create new return values, instead we pass-in the memory to be
returned as an argument. This gives us the freedom to apply the C-function to new
or old vectors, which helps to avoid unnecessary memory allocation, unnecessary
copying and unnecessary garbage collection. Prior to 0.93 \emph{within} our R functions
we also deviated from conventional R programming by not using \code{\link{attr<-}} and
\code{\link{attributes<-}} because they always did new memory allocation and copying in older
R versions. If we wanted to set attributes of return values that we have freshly
created, we instead used functions \code{\link[bit:getsetattr]{bit::setattr()}} and \code{\link[bit:getsetattr]{bit::setattributes()}}.
From version 0.93 \code{bit::setattr()} is only used for manipulating \code{\link{cache}} objects,
in \code{\link[=ramsort.integer64]{ramsort.integer64()}}, \code{\link[=sort.integer64]{sort.integer64()}}, and \code{\link[=as.data.frame.integer64]{as.data.frame.integer64()}}.
}

\section{Arithmetic precision and coercion}{
The fact that we introduce 64 bit long long integers -- without introducing 128-bit
long doubles -- creates some subtle challenges: Unlike 32 bit \code{\link{integer}}s, the
\code{integer64} are no longer a proper subset of \code{\link{double}}. If a binary arithmetic
operation does involve a \code{double} and a \code{integer}, it is a no-brainer to return
\code{double} without loss of information. If an \code{integer64} meets a \code{double}, it is not
trivial what type to return. Switching to \code{integer64} limits our ability to
represent very large numbers, switching to \code{double} limits our ability to
distinguish \code{x} from \code{x+1}. Since the latter is the purpose of introducing 64 bit
integers, we usually return \code{integer64} from functions involving \code{integer64}, for
example in \code{\link[=c.integer64]{c()}}, \code{\link[=cbind.integer64]{cbind()}}, and
\code{\link[=rbind.integer64]{rbind()}}

Different from Base R, our operators \code{\link[=+.integer64]{+}}, \code{\link[=-.integer64]{-}},
\code{\link[=\%/\%.integer64]{\%/\%}}, and \code{\link[=\%\%.integer64]{\%\%}} coerce their arguments to
\code{integer64} and always return \code{integer64}.

The multiplication operator \code{\link[=*.integer64]{*}} coerces its first argument to
\code{integer64} but allows its second argument to be also \code{double}: the second
argument is internaly coerced to 'long double' and the result of the
multiplication is returned as \code{integer64}.

The division \code{\link[=/.integer64]{/}} and power \code{\link[=^.integer64]{^}} operators also
coerce their first argument to \code{integer64} and coerce internally their second
argument to 'long double', they return as \code{double}, like
\code{\link[=sqrt.integer64]{sqrt()}}, \code{\link[=log.integer64]{log()}},
\code{\link[=log2.integer64]{log2()}}, and \code{\link[=log10.integer64]{log10()}} do.\tabular{ccccccccc}{
   \strong{argument1} \tab \strong{op} \tab \strong{argument2} \tab \strong{->} \tab \strong{coerced1} \tab \strong{op} \tab \strong{coerced2} \tab \strong{->} \tab \strong{result} \cr
   integer64 \tab + \tab double \tab -> \tab integer64 \tab + \tab integer64 \tab -> \tab integer64 \cr
   double \tab + \tab integer64 \tab -> \tab integer64 \tab + \tab integer64 \tab -> \tab integer64 \cr
   integer64 \tab - \tab double \tab -> \tab integer64 \tab - \tab integer64 \tab -> \tab integer64 \cr
   double \tab - \tab integer64 \tab -> \tab integer64 \tab - \tab integer64 \tab -> \tab integer64 \cr
   integer64 \tab \%/\% \tab double \tab -> \tab integer64 \tab \%/\% \tab integer64 \tab -> \tab integer64 \cr
   double \tab \%/\% \tab integer64 \tab -> \tab integer64 \tab \%/\% \tab integer64 \tab -> \tab integer64 \cr
   integer64 \tab \%\% \tab double \tab -> \tab integer64 \tab \%\% \tab integer64 \tab -> \tab integer64 \cr
   double \tab \%\% \tab integer64 \tab -> \tab integer64 \tab \%\% \tab integer64 \tab -> \tab integer64 \cr
   integer64 \tab * \tab double \tab -> \tab integer64 \tab * \tab long double \tab -> \tab integer64 \cr
   double \tab * \tab integer64 \tab -> \tab integer64 \tab * \tab integer64 \tab -> \tab integer64 \cr
   integer64 \tab / \tab double \tab -> \tab integer64 \tab / \tab long double \tab -> \tab double \cr
   double \tab / \tab integer64 \tab -> \tab integer64 \tab / \tab long double \tab -> \tab double \cr
   integer64 \tab ^ \tab double \tab -> \tab integer64 \tab / \tab long double \tab -> \tab double \cr
   double \tab ^ \tab integer64 \tab -> \tab integer64 \tab / \tab long double \tab -> \tab double \cr
}
}

\section{Creating and testing S3 class 'integer64'}{
Our creator function \code{integer64} takes an argument \code{length}, creates an atomic
double vector of this length, attaches an S3 class attribute 'integer64' to it,
and that's it. We simply rely on S3 method dispatch and interpret those 64-bit
elements as 'long long int'.

\code{\link[=is.double]{is.double()}} currently returns TRUE for \code{integer64} and might return \code{FALSE} in
a later release. Consider \code{is.double()} to have undefined behavior and do query
\code{\link[=is.integer64]{is.integer64()}} \emph{before} querying \code{is.double()}.

The methods \code{\link[=is.integer64]{is.integer64()}} and \code{\link[=is.vector]{is.vector()}} both return \code{TRUE} for \code{integer64}.
Note that we did not patch \code{\link[=storage.mode]{storage.mode()}} and \code{\link[=typeof]{typeof()}}, which both continue
returning 'double'. Like for 32 bit \code{\link{integer}}, \code{\link[=mode]{mode()}} returns 'numeric' and
\code{\link[=as.double]{as.double()}} tries coercing to \code{\link{double}}. It is possible that 'integer64' becomes
a \code{vmode} in package ff.

Further methods for creating \code{integer64} are \code{\link[=range.integer64]{range()}} which
returns the range of the data type if calles without arguments,
\code{\link[=rep.integer64]{rep()}}, \code{\link[=seq.integer64]{seq()}}.

For all available methods on \code{integer64} vectors see the index below and the examples.
}

\section{Index of implemented methods}{
\tabular{rrl}{
   \strong{creating, testing, printing} \tab \strong{see also} \tab \strong{description} \cr
   \code{NA_integer64_} \tab \code{\link{NA_integer_}} \tab NA constant \cr
   \code{integer64} \tab \code{\link{integer}} \tab create zero atomic vector \cr
   \code{\link[=runif64]{runif64()}} \tab \code{\link[=runif]{runif()}} \tab create random vector \cr
   \code{\link[=rep.integer64]{rep.integer64()}} \tab \code{\link[=rep]{rep()}} \tab  \cr
   \code{\link[=seq.integer64]{seq.integer64()}} \tab \code{\link[=seq]{seq()}} \tab  \cr
   \code{\link[=is.integer64]{is.integer64()}} \tab \code{\link[=is]{is()}} \tab  \cr
    \tab \code{\link[=is.integer]{is.integer()}} \tab inherited from Base R \cr
   \code{\link[=is.vector.integer64]{is.vector.integer64()}} \tab \code{\link[=is.vector]{is.vector()}} \tab  \cr
   \code{\link[=identical.integer64]{identical.integer64()}} \tab \code{\link[=identical]{identical()}} \tab  \cr
   \code{\link{length<-.integer64}} \tab \code{\link{length<-}} \tab  \cr
    \tab \code{\link[=length]{length()}} \tab inherited from Base R \cr
   \code{\link{names<-}} \tab inherited from Base R \tab  \cr
    \tab \code{\link[=names]{names()}} \tab inherited from Base R \cr
    \tab \code{\link{dim<-}} \tab inherited from Base R \cr
    \tab \code{\link[=dim]{dim()}} \tab inherited from Base R \cr
    \tab \code{\link{dimnames<-}} \tab inherited from Base R \cr
    \tab \code{\link[=dimnames]{dimnames()}} \tab inherited from Base R \cr
    \tab \code{\link[=str]{str()}} \tab inherited from Base R, does not print values correctly \cr
   \code{\link[=print.integer64]{print.integer64()}} \tab \code{\link[=print]{print()}} \tab  \cr
   \code{\link[=str.integer64]{str.integer64()}} \tab \code{\link[=str]{str()}} \tab  \cr
}
\tabular{rrl}{
   \strong{coercing to integer64} \tab \strong{see also} \tab \strong{description} \cr
   \code{\link[=as.integer64]{as.integer64()}} \tab  \tab generic \cr
   \code{\link[=as.integer64.bitstring]{as.integer64.bitstring()}} \tab \code{\link[=as.bitstring]{as.bitstring()}} \tab  \cr
   \code{\link[=as.integer64.character]{as.integer64.character()}} \tab \code{\link[=character]{character()}} \tab  \cr
   \code{\link[=as.integer64.double]{as.integer64.double()}} \tab \code{\link[=double]{double()}} \tab  \cr
   \code{\link[=as.integer64.integer]{as.integer64.integer()}} \tab \code{\link[=integer]{integer()}} \tab  \cr
   \code{\link[=as.integer64.integer64]{as.integer64.integer64()}} \tab \code{integer64} \tab  \cr
   \code{\link[=as.integer64.logical]{as.integer64.logical()}} \tab \code{\link[=logical]{logical()}} \tab  \cr
   \code{\link[=as.integer64.NULL]{as.integer64.NULL()}} \tab \code{\link[=NULL]{NULL()}} \tab  \cr
}
\tabular{rrl}{
   \strong{coercing from integer64} \tab \strong{see also} \tab \strong{description} \cr
   \code{\link[=as.list.integer64]{as.list.integer64()}} \tab \code{\link[=as.list]{as.list()}} \tab generic \cr
   \code{\link[=as.bitstring]{as.bitstring()}} \tab \code{\link[=as.bitstring]{as.bitstring()}} \tab generic \cr
   \code{\link[=as.bitstring.integer64]{as.bitstring.integer64()}} \tab  \tab  \cr
   \code{\link[=as.character.integer64]{as.character.integer64()}} \tab \code{\link[=as.character]{as.character()}} \tab  \cr
   \code{\link[=as.double.integer64]{as.double.integer64()}} \tab \code{\link[=as.double]{as.double()}} \tab  \cr
   \code{\link[=as.integer.integer64]{as.integer.integer64()}} \tab \code{\link[=as.integer]{as.integer()}} \tab  \cr
   \code{\link[=as.logical.integer64]{as.logical.integer64()}} \tab \code{\link[=as.logical]{as.logical()}} \tab  \cr
}
\tabular{rrl}{
   \strong{data structures} \tab \strong{see also} \tab \strong{description} \cr
   \code{\link[=c.integer64]{c.integer64()}} \tab \code{\link[=c]{c()}} \tab vector concatenate \cr
   \code{\link[=cbind.integer64]{cbind.integer64()}} \tab \code{\link[=cbind]{cbind()}} \tab column bind \cr
   \code{\link[=rbind.integer64]{rbind.integer64()}} \tab \code{\link[=rbind]{rbind()}} \tab row bind \cr
   \code{\link[=as.data.frame.integer64]{as.data.frame.integer64()}} \tab \code{\link[=as.data.frame]{as.data.frame()}} \tab coerce atomic object to data.frame \cr
    \tab \code{\link[=data.frame]{data.frame()}} \tab inherited from Base R since we have coercion \cr
}
\tabular{rrl}{
   \strong{subscripting} \tab \strong{see also} \tab \strong{description} \cr
   \code{\link[=extract.replace.integer64]{[.integer64}} \tab \code{\link[base:Extract]{[}} \tab vector and array extract \cr
   \code{\link[=extract.replace.integer64]{[<-.integer64}} \tab \code{\link[base:Extract]{[<-}} \tab vector and array assign \cr
   \code{\link[=extract.replace.integer64]{[[.integer64}} \tab \code{\link[base:Extract]{[[}} \tab scalar extract \cr
   \code{\link[=extract.replace.integer64]{[[<-.integer64}} \tab \code{\link[base:Extract]{[[<-}} \tab scalar assign \cr
}
\tabular{rrl}{
   \strong{binary operators} \tab \strong{see also} \tab \strong{description} \cr
   \code{\link{+.integer64}} \tab \code{\link{+}} \tab returns integer64 \cr
   \code{\link{-.integer64}} \tab \code{\link{-}} \tab returns integer64 \cr
   \code{\link{*.integer64}} \tab \code{\link{*}} \tab returns integer64 \cr
   \code{\link{^.integer64}} \tab \code{\link{^}} \tab returns double \cr
   \code{\link{/.integer64}} \tab \code{\link{/}} \tab returns double \cr
   \code{\link{\%/\%.integer64}} \tab \code{\link{\%/\%}} \tab returns integer64 \cr
   \code{\link{\%\%.integer64}} \tab \code{\link{\%\%}} \tab returns integer64 \cr
}
\tabular{rrl}{
   \strong{comparison operators} \tab \strong{see also} \tab \strong{description} \cr
   \code{\link{==.integer64}} \tab \code{\link{==}} \tab  \cr
   \code{\link{!=.integer64}} \tab \code{\link{!=}} \tab  \cr
   \code{\link{<.integer64}} \tab \code{\link{<}} \tab  \cr
   \code{\link{<=.integer64}} \tab \code{\link{<=}} \tab  \cr
   \code{\link{>.integer64}} \tab \code{\link{>}} \tab  \cr
   \code{\link{>=.integer64}} \tab \code{\link{>=}} \tab  \cr
}


\tabular{rrl}{
\strong{logical operators} \tab \strong{see also} \tab \strong{description} \cr
\code{\link{!.integer64}} \tab \code{\link{!}} \tab  \cr
\code{\link{&.integer64}} \tab \code{\link{&}} \tab  \cr
\code{\link[=xor.integer64]{|.integer64}} \tab \code{\link[base:Logic]{|}} \tab  \cr
\code{\link{xor.integer64}} \tab \code{\link[=xor]{xor()}} \tab  \cr
}\tabular{rrl}{
   \strong{math functions} \tab \strong{see also} \tab \strong{description} \cr
   \code{\link[=is.na.integer64]{is.na.integer64()}} \tab \code{\link[=is.na]{is.na()}} \tab returns logical \cr
   \code{\link[=format.integer64]{format.integer64()}} \tab \code{\link[=format]{format()}} \tab returns character \cr
   \code{\link[=abs.integer64]{abs.integer64()}} \tab \code{\link[=abs]{abs()}} \tab returns integer64 \cr
   \code{\link[=sign.integer64]{sign.integer64()}} \tab \code{\link[=sign]{sign()}} \tab returns integer64 \cr
   \code{\link[=log.integer64]{log.integer64()}} \tab \code{\link[=log]{log()}} \tab returns double \cr
   \code{\link[=log10.integer64]{log10.integer64()}} \tab \code{\link[=log10]{log10()}} \tab returns double \cr
   \code{\link[=log2.integer64]{log2.integer64()}} \tab \code{\link[=log2]{log2()}} \tab returns double \cr
   \code{\link[=sqrt.integer64]{sqrt.integer64()}} \tab \code{\link[=sqrt]{sqrt()}} \tab returns double \cr
   \code{\link[=ceiling.integer64]{ceiling.integer64()}} \tab \code{\link[=ceiling]{ceiling()}} \tab dummy returning its argument \cr
   \code{\link[=floor.integer64]{floor.integer64()}} \tab \code{\link[=floor]{floor()}} \tab dummy returning its argument \cr
   \code{\link[=trunc.integer64]{trunc.integer64()}} \tab \code{\link[=trunc]{trunc()}} \tab dummy returning its argument \cr
   \code{\link[=round.integer64]{round.integer64()}} \tab \code{\link[=round]{round()}} \tab dummy returning its argument \cr
   \code{\link[=signif.integer64]{signif.integer64()}} \tab \code{\link[=signif]{signif()}} \tab dummy returning its argument \cr
}
\tabular{rrl}{
   \strong{cumulative functions} \tab \strong{see also} \tab \strong{description} \cr
   \code{\link[=cummin.integer64]{cummin.integer64()}} \tab \code{\link[=cummin]{cummin()}} \tab  \cr
   \code{\link[=cummax.integer64]{cummax.integer64()}} \tab \code{\link[=cummax]{cummax()}} \tab  \cr
   \code{\link[=cumsum.integer64]{cumsum.integer64()}} \tab \code{\link[=cumsum]{cumsum()}} \tab  \cr
   \code{\link[=cumprod.integer64]{cumprod.integer64()}} \tab \code{\link[=cumprod]{cumprod()}} \tab  \cr
   \code{\link[=diff.integer64]{diff.integer64()}} \tab \code{\link[=diff]{diff()}} \tab  \cr
}
\tabular{rrl}{
   \strong{summary functions} \tab \strong{see also} \tab \strong{description} \cr
   \code{\link[=range.integer64]{range.integer64()}} \tab \code{\link[=range]{range()}} \tab  \cr
   \code{\link[=min.integer64]{min.integer64()}} \tab \code{\link[=min]{min()}} \tab  \cr
   \code{\link[=max.integer64]{max.integer64()}} \tab \code{\link[=max]{max()}} \tab  \cr
   \code{\link[=sum.integer64]{sum.integer64()}} \tab \code{\link[=sum]{sum()}} \tab  \cr
   \code{\link[=mean.integer64]{mean.integer64()}} \tab \code{\link[=mean]{mean()}} \tab  \cr
   \code{\link[=prod.integer64]{prod.integer64()}} \tab \code{\link[=prod]{prod()}} \tab  \cr
   \code{\link[=all.integer64]{all.integer64()}} \tab \code{\link[=all]{all()}} \tab  \cr
   \code{\link[=any.integer64]{any.integer64()}} \tab \code{\link[=any]{any()}} \tab  \cr
}
\tabular{rrl}{
   \strong{algorithmically complex functions} \tab \strong{see also} \tab \strong{description (caching)} \cr
   \code{\link[=match.integer64]{match.integer64()}} \tab \code{\link[=match]{match()}} \tab position of x in table (h//o/so) \cr
   \code{\link{\%in\%.integer64}} \tab \code{\link{\%in\%}} \tab is x in table? (h//o/so) \cr
   \code{\link[=duplicated.integer64]{duplicated.integer64()}} \tab \code{\link[=duplicated]{duplicated()}} \tab is current element duplicate of previous one? (h//o/so) \cr
   \code{\link[=unique.integer64]{unique.integer64()}} \tab \code{\link[=unique]{unique()}} \tab (shorter) vector of unique values only (h/s/o/so) \cr
   \code{\link[=unipos.integer64]{unipos.integer64()}} \tab \code{\link[=unipos]{unipos()}} \tab positions corresponding to unique values (h/s/o/so) \cr
   \code{\link[=tiepos.integer64]{tiepos.integer64()}} \tab \code{\link[=tiepos]{tiepos()}} \tab positions of values that are tied (//o/so) \cr
   \code{\link[=keypos.integer64]{keypos.integer64()}} \tab \code{\link[=keypos]{keypos()}} \tab position of current value in sorted list of unique values (//o/so) \cr
   \code{\link[=table.integer64]{table.integer64()}} \tab \code{\link[=table]{table()}} \tab unique values and their frequencies (h/s/o/so) \cr
   \code{\link[=sort.integer64]{sort.integer64()}} \tab \code{\link[=sort]{sort()}} \tab sorted vector (/s/o/so) \cr
   \code{\link[=order.integer64]{order.integer64()}} \tab \code{\link[=order]{order()}} \tab positions of elements that would create sorted vector (//o/so) \cr
   \code{\link[=rank.integer64]{rank.integer64()}} \tab \code{\link[=rank]{rank()}} \tab (average) ranks of non-NAs, NAs kept in place (/s/o/so) \cr
   \code{\link[=quantile.integer64]{quantile.integer64()}} \tab \code{\link[=quantile]{quantile()}} \tab (existing) values at specified percentiles (/s/o/so) \cr
   \code{\link[=median.integer64]{median.integer64()}} \tab \code{\link[=median]{median()}} \tab (existing) value at percentile 0.5 (/s/o/so) \cr
   \code{\link[=summary.integer64]{summary.integer64()}} \tab \code{\link[=summary]{summary()}} \tab (/s/o/so) \cr
   \code{\link[=all.equal.integer64]{all.equal.integer64()}} \tab \code{\link[=all.equal]{all.equal()}} \tab test if two objects are (nearly) equal (/s/o/so) \cr
}
\tabular{rrl}{
   \strong{helper functions} \tab \strong{see also} \tab \strong{description} \cr
   \code{\link[=minusclass]{minusclass()}} \tab \code{\link[=minusclass]{minusclass()}} \tab removing class attritbute \cr
   \code{\link[=plusclass]{plusclass()}} \tab \code{\link[=plusclass]{plusclass()}} \tab inserting class attribute \cr
   \code{\link[=binattr]{binattr()}} \tab \code{\link[=binattr]{binattr()}} \tab define binary op behaviour \cr
}
\tabular{rrl}{
   \strong{tested I/O functions} \tab \strong{see also} \tab \strong{description} \cr
    \tab \code{\link[=read.table]{read.table()}} \tab inherited from Base R \cr
    \tab \code{\link[=write.table]{write.table()}} \tab inherited from Base R \cr
    \tab \code{\link[=serialize]{serialize()}} \tab inherited from Base R \cr
    \tab \code{\link[=unserialize]{unserialize()}} \tab inherited from Base R \cr
    \tab \code{\link[=save]{save()}} \tab inherited from Base R \cr
    \tab \code{\link[=load]{load()}} \tab inherited from Base R \cr
    \tab \code{\link[=dput]{dput()}} \tab inherited from Base R \cr
    \tab \code{\link[=dget]{dget()}} \tab inherited from Base R \cr
}
}

\section{Limitations inherited from implementing 64 bit integers via an external package}{
\itemize{
\item \strong{vector size} of atomic vectors is still limited to
\code{\link[=.Machine]{.Machine$integer.max}}. However, external memory extending packages
such as ff or bigmemory can extend their address space now with \code{integer64}.
Having 64 bit integers also help with those not so obvious address issues that
arise once we exchange data with SQL databases and datawarehouses, which use big
integers as surrogate keys, e.g. on indexed primary key columns. This puts R into
a relatively strong position compared to certain commercial statistical softwares,
which sell database connectivity but neither have the range of 64 bit integers,
nor have integers at all, nor have a single numeric data type in their
macro-glue-language.
\item \strong{literals} such as \verb{123LL} would require changes to Base R, up to then we need
to write (and call) \code{as.integer64(123L)} or \code{as.integer64(123)} or
\code{as.integer64('123')}. Only the latter allows to specify numbers beyond Base R's
numeric data types and therefore is the recommended way to use -- using only one
way may facilitate migrating code to literals at a later stage.
}
}

\section{Limitations inherited from Base R, Core team, can you change this?}{
\itemize{
\item \strong{\code{\link[=identical]{identical()}}} with default parameters does not distinguish all bit-patterns of
doubles. For testing purposes we provide a wrapper \code{\link[=identical.integer64]{identical.integer64()}} that
will distinguish all bit-patterns. It would be desireable to have a single call
of \code{identical()} handle both, \code{\link{double}} and \code{integer64}.
\item the \strong{colon} operator \link{:} officially does not dispatch S3 methods, however, we
have made it generic:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{from <- lim.integer64()[1]
to <- from+99
from:to
}\if{html}{\out{</div>}}

As a limitation remains: it will only dispatch at its first argument \code{from} but
not at its second \code{to}.
\item \strong{\code{\link[=is.double]{is.double()}}} does not dispatch S3 methods, However, we have made it generic
and it will return \code{FALSE} on \code{integer64}.
\item \strong{\code{\link[=c]{c()}}} only dispatches \code{\link[=c.integer64]{c.integer64()}} if the first argument is \code{integer64}
and it does not recursively dispatch the proper method when called with argument
\code{recursive=TRUE}. Therefore \code{c(list(integer64, integer64))} does not work and
for now you can only call \code{c.integer64(list(x, x))}.
\item \strong{generic binary operators} fail to dispatch \emph{any} user-defined S3 method
if the two arguments have two different S3 classes. For example we have two
classes \code{\link[bit:bit]{bit::bit}} and \code{\link[bit:bitwhich]{bit::bitwhich}} sparsely representing boolean vectors
and we have methods \code{\link[bit:xor]{&.bit}} and
\code{\link[bit:xor]{&.bitwhich}}. For an expression involving both as in
\code{bit & bitwhich}, none of the two methods is dispatched. Instead a standard
method is dispatched, which neither handles \code{bit} nor \code{bitwhich}. Although
it lacks symmetry, the better choice would be to dispatch simply the method
of the class of the first argument in case of class conflict. This choice would
allow authors of extension packages providing coherent behaviour at least within
their contributed classes. But as long as none of the package author's methods is
dispatched, they cannot handle the conflicting classes at all.
\item \strong{\code{\link[=unlist]{unlist()}}} is not generic and if it were, we would face similar problems as
with \code{\link[=c]{c()}}
\item \strong{\code{\link[=vector]{vector()}}} with argument \code{mode='integer64'} cannot work without adjustment
of Base R
\item \strong{\code{\link[=as.vector]{as.vector()}}} with argument \code{mode='integer64'} cannot work without adjustment
of Base R
\item \strong{\code{\link[=is.vector]{is.vector()}}} does not dispatch its method \code{\link[=is.vector.integer64]{is.vector.integer64()}}
\item \strong{\code{\link[=mode<-]{mode<-()}}} drops the class 'integer64' which is returned from
\code{as.integer64()}. Also it does not remove an existing class 'integer64' when
assigning mode 'integer'.
\item \strong{\code{\link[=storage.mode<-]{storage.mode<-()}}} does not support external data types such as \code{integer64}
\item \strong{\code{\link[=matrix]{matrix()}}} does drop the 'integer64' class attribute.
\item \strong{\code{\link[=array]{array()}}}  does drop the 'integer64' class attribute.
\itemize{
\item In current R versions (1.15.1) this can be circumvented by activating the
function \code{as.vector.integer64()}. However, the CRAN maintainer has requested
to remove \code{as.vector.integer64()}, even at the price of breaking previously
working functionality of the package.
}
\item \strong{\code{\link[=str]{str()}}} does not print the values of \code{integer64} correctly
}
}

\section{Further limitations}{
\itemize{
\item \strong{subscripting} non-existing elements and subscripting with \code{NA}s is currently
not supported. Such subscripting currently returns \code{9218868437227407266} instead
of \code{NA} (the \code{NA} value of the underlying double code). Following the full R
behaviour here would either destroy performance or require extensive C-coding.
}
}

\examples{
message("Using integer64 in vector")
x <- integer64(8)    # create 64 bit vector
x
is.atomic(x)         # TRUE
is.integer64(x)      # TRUE
is.numeric(x)        # TRUE
is.integer(x)        # FALSE - debatable
is.double(x)         # FALSE - might change
x[] <- 1:2           # assigned value is recycled as usual
x[1:6]               # subscripting as usual
length(x) <- 13      # changing length as usual
x
rep(x, 2)            # replicate as usual
seq(as.integer64(1), 10)     # seq.integer64 is dispatched on first given argument
seq(to=as.integer64(10), 1)  # seq.integer64 is dispatched on first given argument
seq.integer64(along.with=x)  # or call seq.integer64 directly
# c.integer64 is dispatched only if *first* argument is integer64 ...
x <- c(x,runif(length(x), max=100))
# ... and coerces everything to integer64 - including double
x
names(x) <- letters  # use names as usual
x

message("Using integer64 in array - note that 'matrix' currently does not work")
message("as.vector.integer64 removed as requested by the CRAN maintainer")
message("as consequence 'array' also does not work anymore")
message("we still can create a matrix or array by assigning 'dim'")
y <- rep(as.integer64(NA), 12)
dim(y) <- c(3,4)
dimnames(y) <- list(letters[1:3], LETTERS[1:4])
y["a",] <- 1:2       # assigning as usual
y
y[1:2,-4]            # subscripting as usual
# cbind.integer64 dispatched on any argument and coerces everything to integer64
cbind(E=1:3, F=runif(3, 0, 100), G=c("-1","0","1"), y)

message("Using integer64 in data.frame")
str(as.data.frame(x))
str(as.data.frame(y))
str(data.frame(y))
str(data.frame(I(y)))
d <- data.frame(x=x, y=runif(length(x), 0, 100))
d
d$x

message("Using integer64 with csv files")
fi64 <- tempfile()
write.csv(d, file=fi64, row.names=FALSE)
e <- read.csv(fi64, colClasses=c("integer64", NA))
unlink(fi64)
str(e)
identical.integer64(d$x,e$x)

message("Serializing and unserializing integer64")
dput(d, fi64)
e <- dget(fi64)
identical.integer64(d$x,e$x)
e <- d[,]
save(e, file=fi64)
rm(e)
load(file=fi64)
identical.integer64(d,e)

  \dontrun{
message("== Differences between integer64 and int64 ==")
require(bit64)
require(int64)

message("-- integer64 is atomic --")
is.atomic(integer64())
#is.atomic(int64())
str(integer64(3))
#str(int64(3))

message("-- The following performance numbers are measured under RWin64  --")
message("-- under RWin32 the advantage of integer64 over int64 is smaller --")

message("-- integer64 needs 7x/5x less RAM than int64 under 64/32 bit OS
(and twice the RAM of integer as it should be) --")
#as.vector(object.size(int64(1e6))/object.size(integer64(1e6)))
as.vector(object.size(integer64(1e6))/object.size(integer(1e6)))

message("-- integer64 creates 2000x/1300x faster than int64 under 64/32 bit OS
(and 3x the time of integer) --")
t32 <- system.time(integer(1e8))
t64 <- system.time(integer64(1e8))
#T64 <- system.time(int64(1e7))*10  # using 1e8 as above stalls our R on an i7 8 GB RAM Thinkpad
#T64/t64
t64/t32

i32 <- sample(1e6)
d64 <- as.double(i32)

message("-- the following timings are rather conservative since timings
 of integer64 include garbage collection -- due to looped calls")
message("-- integer64 coerces 900x/100x faster than int64
 under 64/32 bit OS (and 2x the time of coercing to integer) --")
t32 <- system.time(for(i in 1:1000)as.integer(d64))
t64 <- system.time(for(i in 1:1000)as.integer64(d64))
#T64 <- system.time(as.int64(d64))*1000
#T64/t64
t64/t32
td64 <- system.time(for(i in 1:1000)as.double(i32))
t64 <- system.time(for(i in 1:1000)as.integer64(i32))
#T64 <- system.time(for(i in 1:10)as.int64(i32))*100
#T64/t64
t64/td64

message("-- integer64 serializes 4x/0.8x faster than int64
 under 64/32 bit OS (and less than 2x/6x the time of integer or double) --")
t32 <- system.time(for(i in 1:10)serialize(i32, NULL))
td64 <- system.time(for(i in 1:10)serialize(d64, NULL))
i64 <- as.integer64(i32);
t64 <- system.time(for(i in 1:10)serialize(i64, NULL))
rm(i64); gc()
#I64 <- as.int64(i32);
#T64 <- system.time(for(i in 1:10)serialize(I64, NULL))
#rm(I64); gc()
#T64/t64
t64/t32
t64/td64


message("-- integer64 adds 250x/60x faster than int64
 under 64/32 bit OS (and less than 6x the time of integer or double) --")
td64 <- system.time(for(i in 1:100)d64+d64)
t32 <- system.time(for(i in 1:100)i32+i32)
i64 <- as.integer64(i32);
t64 <- system.time(for(i in 1:100)i64+i64)
rm(i64); gc()
#I64 <- as.int64(i32);
#T64 <- system.time(for(i in 1:10)I64+I64)*10
#rm(I64); gc()
#T64/t64
t64/t32
t64/td64

message("-- integer64 sums 3x/0.2x faster than int64
(and at about 5x/60X the time of integer and double) --")
td64 <- system.time(for(i in 1:100)sum(d64))
t32 <- system.time(for(i in 1:100)sum(i32))
i64 <- as.integer64(i32);
t64 <- system.time(for(i in 1:100)sum(i64))
rm(i64); gc()
#I64 <- as.int64(i32);
#T64 <- system.time(for(i in 1:100)sum(I64))
#rm(I64); gc()
#T64/t64
t64/t32
t64/td64

message("-- integer64 diffs 5x/0.85x faster than integer and double
(int64 version 1.0 does not support diff) --")
td64 <- system.time(for(i in 1:10)diff(d64, lag=2L, differences=2L))
t32 <- system.time(for(i in 1:10)diff(i32, lag=2L, differences=2L))
i64 <- as.integer64(i32);
t64 <- system.time(for(i in 1:10)diff(i64, lag=2L, differences=2L))
rm(i64); gc()
t64/t32
t64/td64


message("-- integer64 subscripts 1000x/340x faster than int64
(and at the same speed / 10x slower as integer) --")
ts32 <- system.time(for(i in 1:1000)sample(1e6, 1e3))
t32<- system.time(for(i in 1:1000)i32[sample(1e6, 1e3)])
i64 <- as.integer64(i32);
t64 <- system.time(for(i in 1:1000)i64[sample(1e6, 1e3)])
rm(i64); gc()
#I64 <- as.int64(i32);
#T64 <- system.time(for(i in 1:100)I64[sample(1e6, 1e3)])*10
#rm(I64); gc()
#(T64-ts32)/(t64-ts32)
(t64-ts32)/(t32-ts32)

message("-- integer64 assigns 200x/90x faster than int64
(and 50x/160x slower than integer) --")
ts32 <- system.time(for(i in 1:100)sample(1e6, 1e3))
t32 <- system.time(for(i in 1:100)i32[sample(1e6, 1e3)] <- 1:1e3)
i64 <- as.integer64(i32);
i64 <- system.time(for(i in 1:100)i64[sample(1e6, 1e3)] <- 1:1e3)
rm(i64); gc()
#I64 <- as.int64(i32);
#I64 <- system.time(for(i in 1:10)I64[sample(1e6, 1e3)] <- 1:1e3)*10
#rm(I64); gc()
#(T64-ts32)/(t64-ts32)
(t64-ts32)/(t32-ts32)


tdfi32 <- system.time(dfi32 <- data.frame(a=i32, b=i32, c=i32))
tdfsi32 <- system.time(dfi32[1e6:1,])
fi32 <- tempfile()
tdfwi32 <- system.time(write.csv(dfi32, file=fi32, row.names=FALSE))
tdfri32 <- system.time(read.csv(fi32, colClasses=rep("integer", 3)))
unlink(fi32)
rm(dfi32); gc()

i64 <- as.integer64(i32);
tdfi64 <- system.time(dfi64 <- data.frame(a=i64, b=i64, c=i64))
tdfsi64 <- system.time(dfi64[1e6:1,])
fi64 <- tempfile()
tdfwi64 <- system.time(write.csv(dfi64, file=fi64, row.names=FALSE))
tdfri64 <- system.time(read.csv(fi64, colClasses=rep("integer64", 3)))
unlink(fi64)
rm(i64, dfi64); gc()

#I64 <- as.int64(i32);
#tdfI64 <- system.time(dfI64<-data.frame(a=I64, b=I64, c=I64))
#tdfsI64 <- system.time(dfI64[1e6:1,])
#fI64 <- tempfile()
#tdfwI64 <- system.time(write.csv(dfI64, file=fI64, row.names=FALSE))
#tdfrI64 <- system.time(read.csv(fI64, colClasses=rep("int64", 3)))
#unlink(fI64)
#rm(I64, dfI64); gc()

message("-- integer64 coerces 40x/6x faster to data.frame than int64
(and factor 1/9 slower than integer) --")
#tdfI64/tdfi64
tdfi64/tdfi32
message("-- integer64 subscripts from data.frame 20x/2.5x faster than int64
 (and 3x/13x slower than integer) --")
#tdfsI64/tdfsi64
tdfsi64/tdfsi32
message("-- integer64 csv writes about 2x/0.5x faster than int64
(and about 1.5x/5x slower than integer) --")
#tdfwI64/tdfwi64
tdfwi64/tdfwi32
message("-- integer64 csv reads about 3x/1.5 faster than int64
(and about 2x slower than integer) --")
#tdfrI64/tdfri64
tdfri64/tdfri32

rm(i32, d64); gc()


message("-- investigating the impact on garbage collection: --")
message("-- the fragmented structure of int64 messes up R's RAM --")
message("-- and slows down R's gargbage collection just by existing --")

td32 <- double(21)
td32[1] <- system.time(d64 <- double(1e7))[3]
for (i in 2:11)td32[i] <- system.time(gc(), gcFirst=FALSE)[3]
rm(d64)
for (i in 12:21)td32[i] <- system.time(gc(), gcFirst=FALSE)[3]

t64 <- double(21)
t64[1] <- system.time(i64 <- integer64(1e7))[3]
for (i in 2:11)t64[i] <- system.time(gc(), gcFirst=FALSE)[3]
rm(i64)
for (i in 12:21)t64[i] <- system.time(gc(), gcFirst=FALSE)[3]

#T64 <- double(21)
#T64[1] <- system.time(I64 <- int64(1e7))[3]
#for (i in 2:11)T64[i] <- system.time(gc(), gcFirst=FALSE)[3]
#rm(I64)
#for (i in 12:21)T64[i] <- system.time(gc(), gcFirst=FALSE)[3]

#matplot(1:21, cbind(td32, t64, T64), pch=c("d","i","I"), log="y")
matplot(1:21, cbind(td32, t64), pch=c("d","i"), log="y")
  }
}
\seealso{
\code{\link[=integer]{integer()}} in base R
}
\author{
\strong{Maintainer}: Michael Chirico \email{michaelchirico4@gmail.com}

Authors:
\itemize{
  \item Jens Oehlschlägel
}

Other contributors:
\itemize{
  \item Leonardo Silvestri [contributor]
  \item Ofek Shilon [contributor]
}

}
\keyword{classes}
\keyword{internal}
\keyword{manip}
\keyword{package}
