/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

import {
  BackgroundPlatformAPI,
  CrossBrowserPermissionsApi,
  ForegroundPlatformAPI,
  Settings,
  defaultSettings
} from "./api.js";
import chromePlatform, {
  containsClipboardPermissions as chromeClipContains,
  removeClipboardPermissions as chromeClipRemove,
  requestClipboardPermissions as chromeClipRequest,
} from "./chrome.js";

const api: BackgroundPlatformAPI & ForegroundPlatformAPI = {
  ...chromePlatform,
  isFirefox,
  getSettingsFromStorage,
  getPermissionsApi,
  notifyWhenAppIsReady,
  redirectTabToWalletPage,
  useServiceWorkerAsBackgroundProcess,
};

export default api;

function isFirefox(): boolean {
  return true;
}

function getPermissionsApi(): CrossBrowserPermissionsApi {
  return {
    containsClipboardPermissions: chromeClipContains,
    removeClipboardPermissions: chromeClipRemove,
    requestClipboardPermissions: chromeClipRequest,
  };
}

async function getSettingsFromStorage(): Promise<Settings> {
  //@ts-ignore
  const data = await browser.storage.local.get("wallet-settings");
  if (!data) return defaultSettings;
  const settings = data["wallet-settings"];
  if (!settings) return defaultSettings;
  try {
    const parsed = JSON.parse(settings);
    return parsed;
  } catch (e) {
    return defaultSettings;
  }
}

/**
 *
 * @param callback function to be called
 */
function notifyWhenAppIsReady(): Promise<void> {
  return new Promise((resolve) => {
    if (chrome.runtime && chrome.runtime.getManifest().manifest_version === 3) {
      resolve();
    } else {
      window.addEventListener("load", () => {
        resolve();
      });
    }
  });
}

function redirectTabToWalletPage(tabId: number, page: string): void {
  const url = chrome.runtime.getURL(`/static/wallet.html#${page}`);
  chrome.tabs.update(tabId, { url, loadReplace: true } as any);
}

function useServiceWorkerAsBackgroundProcess(): false {
  return false;
}
