/*
 This file is part of GNU Taler
 (C) 2022 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Main entry point for extension pages.
 *
 * @author sebasjm
 */

import {
  Amounts,
  ScopeInfo,
  TalerUri,
  TalerUriAction,
  TransactionIdStr,
  TranslatedString,
  parsePaytoUri,
  parseScopeInfoShort,
  parseTalerUri,
  stringifyScopeInfoShort,
  stringifyTalerUri
} from "@gnu-taler/taler-util";
import {
  TranslationProvider,
  decodeCrockFromURI,
  encodeCrockForURI,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { createHashHistory } from "history";
import { ComponentChildren, Fragment, VNode, h } from "preact";
import { Route, Router, route } from "preact-router";
import { useEffect } from "preact/hooks";
import {
  Pages,
  WalletNavBar,
  WalletNavBarOptions,
  getPathnameForTalerURI,
} from "../NavigationBar.js";
import { AlertView, CurrentAlerts } from "../components/CurrentAlerts.js";
import { EnabledBySettings } from "../components/EnabledBySettings.js";
import { LogoHeader } from "../components/LogoHeader.js";
import PendingTransactions from "../components/PendingTransactions.js";
import { WalletActivity } from "../components/WalletActivity.js";
import {
  LinkPrimary,
  RedBanner,
  SubTitle,
  WalletAction,
  WalletBox,
} from "../components/styled/index.js";
import { AlertProvider } from "../context/alert.js";
import { IoCProviderForRuntime } from "../context/iocContext.js";
import { DepositPage as DepositPageCTA } from "../cta/Deposit/index.js";
import { DevExperimentPage } from "../cta/DevExperiment/index.js";
import { InvoiceCreatePage } from "../cta/InvoiceCreate/index.js";
import { InvoicePayPage } from "../cta/InvoicePay/index.js";
import { PaymentPage } from "../cta/Payment/index.js";
import { PaymentTemplatePage } from "../cta/PaymentTemplate/index.js";
import { RecoveryPage } from "../cta/Recovery/index.js";
import { RefundPage } from "../cta/Refund/index.js";
import { TransferCreatePage } from "../cta/TransferCreate/index.js";
import { TransferPickupPage } from "../cta/TransferPickup/index.js";
import {
  WithdrawPageFromParams,
  WithdrawPageFromURI,
} from "../cta/Withdraw/index.js";
import { useIsOnline } from "../hooks/useIsOnline.js";
import { strings } from "../i18n/strings.js";
import CloseIcon from "../svg/close_24px.inline.svg";
import { AddBackupProviderPage } from "./AddBackupProvider/index.js";
import { AddExchange } from "./AddExchange/index.js";
import { ConfirmAddExchangeView } from "./AddExchange/views.js";
import { AddContact } from "./AddContact/index.js";
import { ConfirmAddContactView } from "./AddContact/views.js";
import { BackupPage } from "./BackupPage.js";
import { DepositPage } from "./DepositPage/index.js";
import { DestinationSelectionPage } from "./DestinationSelection/index.js";
import { DeveloperPage } from "./DeveloperPage.js";
import { HistoryPage } from "./History.js";
import { ManageAccountPage } from "./ManageAccount/index.js";
import { NotificationsPage } from "./Notifications/index.js";
import { ProviderDetailPage } from "./ProviderDetailPage.js";
import { QrReaderPage } from "./QrReader.js";
import { SettingsPage } from "./Settings.js";
import { ContactsPage } from "./Contacts.js";
import { SupportedBanksForAccount } from "./SupportedBanksForAccount.js";
import { TransactionPage } from "./Transaction.js";
import { WelcomePage } from "./Welcome.js";
import { MailboxPage } from "./Mailbox.js";

export function Application(): VNode {
  const { i18n } = useTranslationContext();
  const hash_history = createHashHistory();

  async function redirectToTxInfo(tid: string): Promise<void> {
    redirectTo(Pages.balanceTransaction({ tid }));
  }
  function redirectToURL(str: string): void {
    window.location.href = new URL(str).href;
  }

  return (
    <TranslationProvider source={strings}>
      <IoCProviderForRuntime>
        <Router history={hash_history}>
          <Route
            path={Pages.welcome}
            component={() => (
              <WalletTemplate goToURL={redirectToURL}>
                <WelcomePage />
              </WalletTemplate>
            )}
          />

          <Route
            path={Pages.qr}
            component={() => (
              <WalletTemplate
                goToTransaction={redirectToTxInfo}
                goToURL={redirectToURL}
              >
                <QrReaderPage
                  onDetected={(talerActionUrl: TalerUri) => {
                    redirectTo(
                      Pages.defaultCta({
                        uri: encodeCrockForURI(
                          stringifyTalerUri(talerActionUrl),
                        ),
                      }),
                    );
                  }}
                />
              </WalletTemplate>
            )}
          />

          <Route
            path={Pages.settings}
            component={() => (
              <WalletTemplate
                goToTransaction={redirectToTxInfo}
                goToURL={redirectToURL}
              >
                <SettingsPage />
              </WalletTemplate>
            )}
          />
          <Route
          path={Pages.contacts.pattern}
            component={({ tid }: { tid?: string }) => (
              <WalletTemplate
                goToTransaction={redirectToTxInfo}
                goToURL={redirectToURL}
              >
                <ContactsPage
                  onMessageSent={() => {
                      return redirectTo(Pages.contacts({}));
                    }
                  }
                  tid={tid}
                  />
              </WalletTemplate>
            )}
          />
          <Route
          path={Pages.mailbox}
            component={() => (
              <WalletTemplate
                goToTransaction={redirectToTxInfo}
                goToURL={redirectToURL}
              >
                <MailboxPage
                  onTalerUri={(talerActionUrl: string) => {
                    redirectTo(
                      Pages.defaultCta({
                        uri: encodeCrockForURI(
                          talerActionUrl
                        ),
                      }),
                    );
                  }}
                />
              </WalletTemplate>
            )}
          />
          <Route
            path={Pages.notifications}
            component={() => (
              <WalletTemplate goToURL={redirectToURL}>
                <NotificationsPage />
              </WalletTemplate>
            )}
          />
          {/**
           * SETTINGS
           */}
          <Route
          path={Pages.settingsExchangeAdd.pattern}
            component={() => (
              <WalletTemplate goToURL={redirectToURL}>
              <AddExchange onBack={() => redirectTo(Pages.balance)} />
              </WalletTemplate>
            )}
          />
          <Route
            path={Pages.contactsAdd}
            component={() => (
              <WalletTemplate goToURL={redirectToURL}>
                <AddContact onBack={() => redirectTo(Pages.contacts({}))} />
              </WalletTemplate>
            )}
          />


          <Route
            path={Pages.balanceHistory.pattern}
            component={({ scope }: { scope?: string }) => (
              <WalletTemplate
                path="balance"
                goToTransaction={redirectToTxInfo}
                goToURL={redirectToURL}
              >
                <HistoryPage
                  scope={
                    !scope
                      ? undefined
                      : parseScopeInfoShort(decodeCrockFromURI(scope))
                  }
                  goToWalletDeposit={(scope: ScopeInfo) =>
                    redirectTo(
                      Pages.sendCash({
                        scope: encodeCrockForURI(
                          stringifyScopeInfoShort(scope),
                        ),
                      }),
                    )
                  }
                  goToWalletManualWithdraw={(scope?: ScopeInfo) =>
                    redirectTo(
                      Pages.receiveCash({
                        scope: !scope
                          ? undefined
                          : encodeCrockForURI(stringifyScopeInfoShort(scope)),
                      }),
                    )
                  }
                />
              </WalletTemplate>
            )}
          />
          <Route
            path={Pages.searchHistory.pattern}
            component={({ scope }: { scope?: string }) => (
              <WalletTemplate
                path="balance"
                goToTransaction={redirectToTxInfo}
                goToURL={redirectToURL}
              >
                <HistoryPage
                  scope={
                    !scope
                      ? undefined
                      : parseScopeInfoShort(decodeCrockFromURI(scope))
                  }
                  search
                  goToWalletDeposit={(scope: ScopeInfo) =>
                    redirectTo(
                      Pages.sendCash({
                        scope: encodeCrockForURI(
                          stringifyScopeInfoShort(scope),
                        ),
                      }),
                    )
                  }
                  goToWalletManualWithdraw={(scope?: ScopeInfo) =>
                    redirectTo(
                      Pages.receiveCash({
                        scope: !scope
                          ? undefined
                          : encodeCrockForURI(stringifyScopeInfoShort(scope)),
                      }),
                    )
                  }
                />
              </WalletTemplate>
            )}
          />
          <Route
            path={Pages.sendCash.pattern}
            component={({ scope }: { scope?: string }) => {
              if (!scope) return <Redirect to={Pages.balanceHistory({})} />;
              const s = parseScopeInfoShort(decodeCrockFromURI(scope));
              if (!s) return <Redirect to={Pages.balanceHistory({})} />;

              return (
                <WalletTemplate path="balance" goToURL={redirectToURL}>
                  <DestinationSelectionPage
                    type="send"
                    scope={s}
                    goToWalletKnownBankDeposit={(s, p) =>
                      redirectTo(
                        Pages.ctaDeposit({
                          scope: encodeCrockForURI(stringifyScopeInfoShort(s)),
                          account: encodeCrockForURI(p),
                        }),
                      )
                    }
                    goToWalletNewBankDeposit={(s) =>
                      redirectTo(
                        Pages.bankManange({
                          scope: encodeCrockForURI(stringifyScopeInfoShort(s)),
                        }),
                      )
                    }
                    goToWalletWalletSend={(s) =>
                      redirectTo(
                        Pages.ctaTransferCreate({
                          scope: encodeCrockForURI(stringifyScopeInfoShort(s)),
                        }),
                      )
                    }
                  />
                </WalletTemplate>
              );
            }}
          />
          <Route
            path={Pages.bankManange.pattern}
            component={({ scope }: { scope?: string }) => {
              const s = !scope
                ? undefined
                : parseScopeInfoShort(decodeCrockFromURI(scope));
              if (!s) return <div>missing scope</div>;

              return (
                <WalletTemplate path="balance" goToURL={redirectToURL}>
                  <ManageAccountPage
                    scope={s}
                    onAccountSelected={(account) =>
                      redirectTo(
                        Pages.ctaDeposit({
                          scope: encodeCrockForURI(stringifyScopeInfoShort(s)),
                          account: encodeCrockForURI(account),
                        }),
                      )
                    }
                    onCancel={() => {
                      redirectTo(
                        Pages.balanceHistory({
                          scope: !scope
                            ? undefined
                            : encodeCrockForURI(stringifyScopeInfoShort(s)),
                        }),
                      );
                    }}
                  />
                </WalletTemplate>
              );
            }}
          />
          <Route
            path={Pages.receiveCashForPurchase.pattern}
            component={({ id: _purchaseId }: { id?: string }) => {
              return (
                <WalletTemplate path="balance" goToURL={redirectToURL}>
                  not yet implemented
                </WalletTemplate>
              );
            }}
          />
          <Route
            path={Pages.receiveCashForInvoice.pattern}
            component={({ id: _invoiceId }: { id?: string }) => {
              return (
                <WalletTemplate path="balance" goToURL={redirectToURL}>
                  not yet implemented
                </WalletTemplate>
              );
            }}
          />
          <Route
            path={Pages.receiveCash.pattern}
            component={({ scope }: { scope?: string }) => {
              const s = !scope
                ? undefined
                : parseScopeInfoShort(decodeCrockFromURI(scope));

              return (
                <WalletTemplate path="balance" goToURL={redirectToURL}>
                  <DestinationSelectionPage
                    type="get"
                    scope={s}
                    goToWalletManualWithdraw={(s) =>
                      redirectTo(
                        Pages.ctaWithdrawManualForScope({
                          scope: encodeCrockForURI(stringifyScopeInfoShort(s)),
                        }),
                      )
                    }
                    goToWalletWalletInvoice={(s) =>
                      redirectTo(
                        Pages.ctaInvoiceCreate({
                          scope: encodeCrockForURI(stringifyScopeInfoShort(s)),
                        }),
                      )
                    }
                  />
                </WalletTemplate>
              );
            }}
          />

          <Route
            path={Pages.balanceTransaction.pattern}
            component={({ tid }: { tid: string }) => (
              <WalletTemplate path="balance" goToURL={redirectToURL}>
                <TransactionPage
                  tid={tid}
                  goToWalletHistory={(scope?: ScopeInfo) =>
                    redirectTo(
                      Pages.balanceHistory({
                        scope: !scope
                          ? undefined
                          : encodeCrockForURI(stringifyScopeInfoShort(scope)),
                      }),
                    )
                  }
                  onForwardToContact={(tid: TransactionIdStr) =>
                    redirectTo(Pages.contacts({tid}))
                  }
                />
              </WalletTemplate>
            )}
          />

          <Route
            path={Pages.balanceDeposit.pattern}
            component={({ scope }: { scope: string }) => {
              const s = parseScopeInfoShort(decodeCrockFromURI(scope));
              if (!s) {
                return <div>missing scope</div>;
              }
              return (
                <WalletTemplate path="balance" goToURL={redirectToURL}>
                  <DepositPage
                    scope={s}
                    onCancel={(scope: ScopeInfo) => {
                      redirectTo(
                        Pages.balanceHistory({
                          scope: encodeCrockForURI(
                            stringifyScopeInfoShort(scope),
                          ),
                        }),
                      );
                    }}
                    onSuccess={(scope: ScopeInfo) => {
                      redirectTo(
                        Pages.balanceHistory({
                          scope: encodeCrockForURI(
                            stringifyScopeInfoShort(scope),
                          ),
                        }),
                      );
                    }}
                  />
                </WalletTemplate>
              );
            }}
          />

          <Route
            path={Pages.backup}
            component={() => (
              <WalletTemplate
                path="backup"
                goToTransaction={redirectToTxInfo}
                goToURL={redirectToURL}
              >
                <BackupPage
                  onAddProvider={() => redirectTo(Pages.backupProviderAdd)}
                />
              </WalletTemplate>
            )}
          />
          <Route
            path={Pages.backupProviderDetail.pattern}
            component={({ pid }: { pid: string }) => (
              <WalletTemplate goToURL={redirectToURL}>
                <ProviderDetailPage
                  pid={pid}
                  onPayProvider={(uri: string) =>
                    redirectTo(`${Pages.ctaPay}?talerPayUri=${uri}`)
                  }
                  onWithdraw={(_amount: string) =>
                    // FIXME: use receiveCashForPurchase
                    redirectTo(Pages.receiveCash({ scope: "FIXME missing" }))
                  }
                  onBack={() => redirectTo(Pages.backup)}
                />
              </WalletTemplate>
            )}
          />
          <Route
            path={Pages.backupProviderAdd}
            component={() => (
              <WalletTemplate goToURL={redirectToURL}>
                <AddBackupProviderPage
                  onPaymentRequired={(uri: string) =>
                    redirectTo(`${Pages.ctaPay}?talerPayUri=${uri}`)
                  }
                  onComplete={(pid: string) =>
                    redirectTo(Pages.backupProviderDetail({ pid }))
                  }
                  onBack={() => redirectTo(Pages.backup)}
                />
              </WalletTemplate>
            )}
          />

          {/**
           * DEV
           */}
          <Route
            path={Pages.dev}
            component={() => (
              <WalletTemplate
                path="dev"
                goToTransaction={redirectToTxInfo}
                goToURL={redirectToURL}
              >
                <DeveloperPage />
              </WalletTemplate>
            )}
          />

          {/**
           * CALL TO ACTION
           */}
          <Route
            path={Pages.defaultCta.pattern}
            component={({ uri }: { uri: string }) => {
              const path = getPathnameForTalerURI(decodeCrockFromURI(uri));
              if (!path) {
                return (
                  <CallToActionTemplate title={i18n.str`Taler URI handler`}>
                    <AlertView
                      alert={{
                        type: "warning",
                        message: i18n.str`Could not found a handler for the Taler URI`,
                        description: i18n.str`The uri read in the path parameter is not valid: "${uri}"`,
                      }}
                    />
                  </CallToActionTemplate>
                );
              }
              return <Redirect to={path} />;
            }}
          />
          {/* // FIXME: mem leak problems */}
          <Route
            path={Pages.defaultCtaSimple.pattern}
            component={({ uri }: { uri: string }) => {
              const path = getPathnameForTalerURI(decodeURIComponent(uri));
              if (!path) {
                return (
                  <CallToActionTemplate title={i18n.str`Taler URI handler`}>
                    <AlertView
                      alert={{
                        type: "warning",
                        message: i18n.str`Could not found a handler for the Taler URI`,
                        description: i18n.str`The uri read in the path parameter is not valid: "${uri}"`,
                      }}
                    />
                  </CallToActionTemplate>
                );
              }
              return <Redirect to={path} />;
            }}
          />

          <Route
            path={Pages.ctaPay}
            component={({ talerUri }: { talerUri: string }) => (
              <CallToActionTemplate title={i18n.str`Digital cash payment`}>
                <PaymentPage
                  talerPayUri={decodeCrockFromURI(talerUri)}
                  goToWalletManualWithdraw={(_amount?: string) =>
                    // FIXME: use receiveCashForPruchase
                    redirectTo(Pages.receiveCash({}))
                  }
                  cancel={() => redirectTo(Pages.balance)}
                  onSuccess={(tid: string) =>
                    redirectTo(Pages.balanceTransaction({ tid }))
                  }
                />
              </CallToActionTemplate>
            )}
          />
          <Route
            path={Pages.ctaPayTemplate}
            component={({ talerUri }: { talerUri: string }) => (
              <CallToActionTemplate title={i18n.str`Digital cash payment`}>
                <PaymentTemplatePage
                  talerTemplateUri={decodeCrockFromURI(talerUri)}
                  goToWalletManualWithdraw={(_amount?: string) =>
                    // FIXME: use receiveCashForPruchase
                    redirectTo(Pages.receiveCash({}))
                  }
                  cancel={() => redirectTo(Pages.balance)}
                  onSuccess={(tid: string) =>
                    redirectTo(Pages.balanceTransaction({ tid }))
                  }
                />
              </CallToActionTemplate>
            )}
          />
          <Route
            path={Pages.ctaRefund}
            component={({ talerUri }: { talerUri: string }) => (
              <CallToActionTemplate title={i18n.str`Digital cash refund`}>
                <RefundPage
                  talerRefundUri={decodeCrockFromURI(talerUri)}
                  cancel={() => redirectTo(Pages.balance)}
                  onSuccess={(tid: string) =>
                    redirectTo(Pages.balanceTransaction({ tid }))
                  }
                />
              </CallToActionTemplate>
            )}
          />
          <Route
            path={Pages.ctaWithdraw}
            component={({ talerUri }: { talerUri: string }) => (
              <CallToActionTemplate title={i18n.str`Digital cash withdrawal`}>
                <WithdrawPageFromURI
                  talerWithdrawUri={
                    !talerUri ? undefined : decodeCrockFromURI(talerUri)
                  }
                  cancel={() => redirectTo(Pages.balance)}
                  onSuccess={(tid: string) =>
                    redirectTo(Pages.balanceTransaction({ tid }))
                  }
                />
              </CallToActionTemplate>
            )}
          />
          <Route
            path={Pages.ctaWithdrawManual.pattern}
            component={({
              // scope,
              amount,
              talerUri,
            }: {
              // scope: string;
              amount: string;
              talerUri: string;
            }) => {
              return (
                <CallToActionTemplate title={i18n.str`Digital cash withdrawal`}>
                  <WithdrawPageFromParams
                    scope={undefined}
                    talerExchangeWithdrawUri={
                      !talerUri ? undefined : decodeCrockFromURI(talerUri)
                    }
                    amount={Amounts.parse(amount)}
                    cancel={() => redirectTo(Pages.balance)}
                    onSuccess={(tid: string) =>
                      redirectTo(Pages.balanceTransaction({ tid }))
                    }
                  />
                </CallToActionTemplate>
              );
            }}
          />
          <Route
            path={Pages.ctaWithdrawManualForScope.pattern}
            component={({
              scope,
              amount,
            }: {
              scope: string;
              amount: string;
            }) => {
              if (!scope) return <Redirect to={Pages.balanceHistory({})} />;
              const s = parseScopeInfoShort(decodeCrockFromURI(scope));
              if (!s) return <Redirect to={Pages.balanceHistory({})} />;

              return (
                <CallToActionTemplate title={i18n.str`Digital cash withdrawal`}>
                  <WithdrawPageFromParams
                    talerExchangeWithdrawUri={undefined}
                    scope={s}
                    amount={Amounts.parse(amount)}
                    cancel={() => redirectTo(Pages.balance)}
                    onSuccess={(tid: string) =>
                      redirectTo(Pages.balanceTransaction({ tid }))
                    }
                  />
                </CallToActionTemplate>
              );
            }}
          />
          <Route
            path={Pages.ctaDeposit.pattern}
            component={({
              scope,
              account,
            }: {
              scope: string;
              account: string;
            }) => {
              const s = parseScopeInfoShort(decodeCrockFromURI(scope));
              if (!s) {
                return <div>missing scope</div>;
              }
              const p = parsePaytoUri(decodeCrockFromURI(account));
              if (!p) {
                return <div>missing account</div>;
              }

              return (
                <CallToActionTemplate title={i18n.str`Digital cash deposit`}>
                  <DepositPageCTA
                    scope={s}
                    account={p}
                    cancel={() => redirectTo(Pages.balance)}
                    onSuccess={(tid: string) =>
                      redirectTo(Pages.balanceTransaction({ tid }))
                    }
                  />
                </CallToActionTemplate>
              );
            }}
          />
          <Route
            path={Pages.ctaInvoiceCreate.pattern}
            component={({ scope }: { scope: string }) => (
              <CallToActionTemplate title={i18n.str`Digital cash invoice`}>
                <InvoiceCreatePage
                  scope={parseScopeInfoShort(decodeCrockFromURI(scope))!}
                  onClose={() => redirectTo(Pages.balance)}
                  onSuccess={(tid: string) =>
                    redirectTo(Pages.balanceTransaction({ tid }))
                  }
                />
              </CallToActionTemplate>
            )}
          />
          <Route
            path={Pages.ctaTransferCreate.pattern}
            component={({ scope }: { scope: string }) => (
              <CallToActionTemplate title={i18n.str`Digital cash transfer`}>
                <TransferCreatePage
                  scope={parseScopeInfoShort(decodeCrockFromURI(scope))!}
                  onClose={() => redirectTo(Pages.balance)}
                  onSuccess={(tid: string) =>
                    redirectTo(Pages.balanceTransaction({ tid }))
                  }
                />
              </CallToActionTemplate>
            )}
          />
          <Route
            path={Pages.ctaInvoicePay}
            component={({ talerUri }: { talerUri: string }) => {
              const uri = decodeCrockFromURI(talerUri);
              if (!uri) {
                return <div>missing taler uri</div>;
              }

              return (
                <CallToActionTemplate title={i18n.str`Digital cash invoice`}>
                  <InvoicePayPage
                    talerPayPullUri={uri}
                    goToWalletManualWithdraw={(_amount?: string) =>
                      // FIXME: use receiveCashForInvoice
                      redirectTo(Pages.receiveCash({}))
                    }
                    onClose={() => redirectTo(Pages.balance)}
                    onSuccess={(tid: string) =>
                      redirectTo(Pages.balanceTransaction({ tid }))
                    }
                  />
                </CallToActionTemplate>
              );
            }}
          />
          <Route
            path={Pages.ctaTransferPickup}
            component={({ talerUri }: { talerUri: string }) => {
              const uri = decodeCrockFromURI(talerUri);
              if (!uri) {
                return <div>missing taler uri</div>;
              }

              return (
                <CallToActionTemplate title={i18n.str`Digital cash transfer`}>
                  <TransferPickupPage
                    talerPayPushUri={uri}
                    onClose={() => redirectTo(Pages.balance)}
                    onSuccess={(tid: string) =>
                      redirectTo(Pages.balanceTransaction({ tid }))
                    }
                  />
                </CallToActionTemplate>
              );
            }}
          />
          <Route
            path={Pages.ctaRecovery}
            component={({ talerRecoveryUri }: { talerRecoveryUri: string }) => (
              <CallToActionTemplate title={i18n.str`Digital cash recovery`}>
                <RecoveryPage
                  talerRecoveryUri={
                    !talerRecoveryUri
                      ? undefined
                      : decodeCrockFromURI(talerRecoveryUri)
                  }
                  onCancel={() => redirectTo(Pages.balance)}
                  onSuccess={() => redirectTo(Pages.backup)}
                />
              </CallToActionTemplate>
            )}
          />
          <Route
            path={Pages.ctaExperiment}
            component={({ talerUri }: { talerUri: string }) => (
              <CallToActionTemplate title={i18n.str`Development experiment`}>
                <DevExperimentPage
                  talerExperimentUri={
                    !talerUri ? undefined : decodeCrockFromURI(talerUri)
                  }
                  onCancel={() => redirectTo(Pages.balanceHistory({}))}
                  onSuccess={() => redirectTo(Pages.balanceHistory({}))}
                />
              </CallToActionTemplate>
            )}
          />
          <Route
            path={Pages.ctaAddExchange}
            component={({ talerUri }: { talerUri: string }) => {
              const tUri = parseTalerUri(
                decodeCrockFromURI(talerUri).toLowerCase(),
              );
              const baseUrl =
                tUri?.type === TalerUriAction.AddExchange
                  ? tUri.exchangeBaseUrl
                  : undefined;
              if (!baseUrl) {
                redirectTo(Pages.balanceHistory({}));
                return <div>invalid url {talerUri}</div>;
              }
              return (
                <CallToActionTemplate title={i18n.str`Add exchange`}>
                  <ConfirmAddExchangeView
                    url={baseUrl}
                    status="confirm"
                    error={undefined}
                    onCancel={() => redirectTo(Pages.balanceHistory({}))}
                    onConfirm={() => redirectTo(Pages.balanceHistory({}))}
                  />
                </CallToActionTemplate>
              );
            }}
          />
          <Route
            path={Pages.ctaAddContact}
            component={({ talerUri }: { talerUri: string }) => {
              const tUri = parseTalerUri(
                decodeCrockFromURI(talerUri).toLowerCase(),
              );
              const contact =
                tUri?.type === TalerUriAction.AddContact
                  ? {
                    alias: tUri.alias,
                    aliasType: tUri.aliasType,
                    mailboxBaseUri: tUri.mailboxBaseUri,
                    mailboxAddress: tUri.mailboxIdentity,
                    source: tUri.sourceBaseUrl,
                  }
                  : undefined;
              if (!contact) {
                redirectTo(Pages.balanceHistory({}));
                return <div>invalid url {talerUri}</div>;
              }
              return (
                <CallToActionTemplate title={i18n.str`Add contact`}>
                  <ConfirmAddContactView
                    contact={contact}
                    status="confirm"
                    error={undefined}
                    onCancel={() => redirectTo(Pages.balanceHistory({}))}
                    onConfirm={() => redirectTo(Pages.contacts({}))}
                  />
                </CallToActionTemplate>
              );
            }}
          />
          <Route
            path={Pages.paytoBanks.pattern}
            component={({ payto }: { payto: string }) => {
              const pUri = parsePaytoUri(
                decodeCrockFromURI(payto).toLowerCase(),
              );
              if (!pUri) {
                redirectTo(Pages.balanceHistory({}));
                return <div>invalid uri {pUri}</div>;
              }
              return (
                <WalletTemplate goToURL={redirectToURL}>
                  <SupportedBanksForAccount account={pUri} />
                </WalletTemplate>
              );
            }}
          />
          <Route
            path={Pages.paytoQrs.pattern}
            component={({ payto }: { payto: string }) => {
              const pUri = parsePaytoUri(
                decodeCrockFromURI(payto).toLowerCase(),
              );
              if (!pUri) {
                redirectTo(Pages.balanceHistory({}));
                return <div>invalid uri {pUri}</div>;
              }
              return (
                <WalletTemplate goToURL={redirectToURL}>
                  {/* <AllQrsForAccount account={pUri} /> */}
                  <pre>{JSON.stringify({ title: "QRS", pUri })}</pre>
                </WalletTemplate>
              );
            }}
          />
          {/**
           * NOT FOUND
           * all redirects should be at the end
           */}
          <Route
            path={Pages.balance}
            component={() => <Redirect to={Pages.balanceHistory({})} />}
          />

          <Route
            default
            component={() => <Redirect to={Pages.balanceHistory({})} />}
          />
        </Router>
        <EnabledBySettings name="showWalletActivity">
          <WalletActivity />
        </EnabledBySettings>
      </IoCProviderForRuntime>
    </TranslationProvider>
  );
}

async function redirectTo(location: string): Promise<void> {
  route(location);
}

function Redirect({ to }: { to: string }): null {
  useEffect(() => {
    route(to, true);
  });
  return null;
}

// function matchesRoute(url: string, route: string): boolean {
//   type MatcherFunc = (
//     url: string,
//     route: string,
//     opts: any,
//   ) => Record<string, string> | false;

//   const internalPreactMatcher: MatcherFunc = (Router as any).exec;
//   const result = internalPreactMatcher(url, route, {});
//   return !result ? false : true;
// }

function CallToActionTemplate({
  title,
  children,
}: {
  title: TranslatedString;
  children: ComponentChildren;
}): VNode {
  const { i18n } = useTranslationContext();
  return (
    <WalletAction>
      <LogoHeader />
      <section style={{ display: "flex", justifyContent: "right", margin: 0 }}>
        <LinkPrimary href={`#${Pages.balance}`}>
          <div
            style={{
              height: 24,
              width: 24,
              marginLeft: 4,
              marginRight: 4,
              border: "1px solid black",
              borderRadius: 12,
            }}
            dangerouslySetInnerHTML={{ __html: CloseIcon }}
          />
        </LinkPrimary>
      </section>
      <SubTitle>{title}</SubTitle>
      <AlertProvider>
        <CurrentAlerts />
        {children}
      </AlertProvider>
      <section style={{ display: "flex", justifyContent: "right" }}>
        <LinkPrimary href={`#${Pages.balance}`}>
          <i18n.Translate>Return to wallet</i18n.Translate>
        </LinkPrimary>
      </section>
    </WalletAction>
  );
}

function WalletTemplate({
  path,
  children,
  goToTransaction,
  goToURL,
}: {
  path?: WalletNavBarOptions;
  children: ComponentChildren;
  goToTransaction?: (id: string) => Promise<void>;
  goToURL: (url: string) => void;
}): VNode {
  const online = useIsOnline();
  const { i18n } = useTranslationContext();
  return (
    <Fragment>
      {!online && (
        <div style={{ display: "flex", justifyContent: "center" }}>
          <RedBanner>{i18n.str`Network is offline`}</RedBanner>
        </div>
      )}
      <LogoHeader />
      <WalletNavBar path={path} />
      <PendingTransactions
        goToTransaction={goToTransaction}
        goToURL={goToURL}
      />
      <WalletBox>
        <AlertProvider>
          <CurrentAlerts />
          {children}
        </AlertProvider>
      </WalletBox>
    </Fragment>
  );
}
