#include "gamestatemanager.h"
#include "gamestate.h"

void
GameStateManager::init ()
{

  maxFPS = 50;

  if (SDL_Init (SDL_INIT_VIDEO | SDL_INIT_AUDIO) < 0)
    {
      printf ("Unable to initialize SDL: %s\n", SDL_GetError ());
      exit (1);
    }

  initVideo ();
  initAudio ();

  loadFonts ();
  loadSettings ();
  if (m_fullscreen)
    initVideo ();
  loadSkin ();

  m_running = true;

  m_lastTime = SDL_GetTicks ();

  srand (SDL_GetTicks ());
  for (int i = 0; i < 5; i++)
    srand (rand () + rand ());
}

void
GameStateManager::initVideo ()
{
  if (m_fullscreen)
    {
      screen = SDL_SetVideoMode (640, 480, 0, SDL_DOUBLEBUF | SDL_FULLSCREEN);
      if (screen == NULL)
	{
	  printf ("Unable to set 640x480 video: %s\n", SDL_GetError ());
	  m_fullscreen = false;
	  initVideo ();
	}
    }
  else
    {
      screen = SDL_SetVideoMode (640, 480, 0, SDL_DOUBLEBUF);
      if (screen == NULL)
	{
	  printf ("Unable to set video mode: %s\n", SDL_GetError ());
	  SDL_Quit ();
	  return;
	}
    }
}

void
GameStateManager::initAudio ()
{
  if (Mix_OpenAudio (MIX_DEFAULT_FREQUENCY, MIX_DEFAULT_FORMAT, 2, 1024) ==
      -1)
    printf ("Failed to open audio\n");
  Mix_AllocateChannels (40);

  m_soundManager = new SoundManager ();
}

void
GameStateManager::makeFont (string location, int charWidth, int charHeight,
			    int r, int g, int b)
{
  SDL_Surface *imgFontMask = NULL;
  SDL_Surface *temp = IMG_Load (location.c_str ());
  if (temp != NULL)
    imgFontMask = SDL_DisplayFormat (temp);
  SDL_FreeSurface (temp);
  SDL_SetColorKey (imgFontMask, SDL_SRCCOLORKEY | SDL_RLEACCEL, 0x0000FF00);
  SDL_Surface *imgFont =
    SDL_CreateRGBSurface (SDL_HWSURFACE, 256, 174, 32, 0, 0, 0, 0);
  SDL_FillRect (imgFont, NULL, SDL_MapRGB (imgFont->format, r, g, b));
  SDL_BlitSurface (imgFontMask, NULL, imgFont, NULL);
  SDL_SetColorKey (imgFont, SDL_SRCCOLORKEY | SDL_RLEACCEL, 0x00FF00FF);
  fonts.push_back (new Font (imgFont, charWidth, charHeight));
}

void
GameStateManager::loadFonts ()
{
  makeFont (IMAGEPATH + "/fontMask.png", 16, 29, 0, 0, 0);
  FONT_INACTIVE = fonts.size () - 1;
  makeFont (IMAGEPATH + "/fontMask.png", 16, 29, 224, 224, 96);
  FONT_ACTIVE = fonts.size () - 1;
  makeFont (IMAGEPATH + "/fontMask.png", 16, 29, 255, 255, 255);
  FONT_WHITE = fonts.size () - 1;
  makeFont (IMAGEPATH + "/fontMask_small.png", 8, 14, 255, 255, 255);
  FONT_SMALL_WHITE = fonts.size () - 1;
}

void
GameStateManager::loadSounds ()
{
  m_soundManager->addSound ("blockbreak",
			    LEVELPATH + "/" + m_strLevelsetFilename +
			    "/blockbreak.wav");
  m_soundManager->addSound ("ballhitwall",
			    LEVELPATH + "/" + m_strLevelsetFilename +
			    "/ballhitwall.wav");
  m_soundManager->addSound ("laser",
			    LEVELPATH + "/" + m_strLevelsetFilename +
			    "/laser.wav");
  m_soundManager->addSound ("bonustick",
			    LEVELPATH + "/" + m_strLevelsetFilename +
			    "/bonustick.wav");
}

void
GameStateManager::playSound (string name)
{
  if (m_playSounds)
    m_soundManager->playSound (name);
}

void
GameStateManager::loopSound (string name)
{
  if (m_playSounds)
    m_soundManager->loopSound (name);
}

void
GameStateManager::stopSound ()
{
  if (m_playSounds)
    Mix_HaltChannel (-1);
}

void
GameStateManager::toggleFullscreen ()
{
  m_fullscreen = !m_fullscreen;
  initVideo ();
}

void
GameStateManager::setSkin (string skinName, string skinFilename)
{
  m_strSkinName = skinName;
  m_strSkinFilename = skinFilename;
}

void
GameStateManager::setLevelset (string levelsetName, string levelsetFilename)
{
  m_strLevelsetName = levelsetName;
  m_strLevelsetFilename = levelsetFilename;
}

void
GameStateManager::setStartLevel (int startLevel)
{
  m_startLevel = startLevel;
}

void
GameStateManager::setEnteringHighScore (bool enteringHighScore)
{
  m_enteringHighScore = enteringHighScore;
}

void
GameStateManager::setScoreIndex (int scoreIndex)
{
  m_scoreIndex = scoreIndex;
}

void
GameStateManager::setScore (int score)
{
  m_score = score;
}

void
GameStateManager::loadSkin ()
{
  SDL_FreeSurface (imgSkin);
  string location = IMAGEPATH + "/" + m_strSkinFilename;
  SDL_Surface *temp = IMG_Load (location.c_str ());
  if (temp != NULL)
    imgSkin = SDL_DisplayFormat (temp);
  SDL_FreeSurface (temp);
  if (imgSkin != NULL)
    SDL_SetColorKey (imgSkin, SDL_SRCCOLORKEY | SDL_RLEACCEL, 0x00ff00ff);
}

void
GameStateManager::loadSettings ()
{
  ifstream file;
#ifndef SINGLEDIR
  string location = string (getenv ("HOME")) + "/.ballandpaddle/config";
#endif
#ifdef SINGLEDIR
  string location = "config";
#endif
  file.open (location.c_str (), ifstream::in);
  if (!file.fail ())
    {
      file >> m_fullscreen;
      file >> m_strSkinName;
      file >> m_strSkinFilename;
      file >> m_playSounds;
      file.close ();
    }
  else
    {
#ifndef SINGLEDIR
      location = string (getenv ("HOME")) + "/.ballandpaddle";
      mkdir (location.c_str (), 0777);
#endif
      m_strSkinName = "Blue";
      m_strSkinFilename = "skin.png";
      m_playSounds = true;
    }
}

void
GameStateManager::saveSettings ()
{
  ofstream file;
#ifndef SINGLEDIR
  string location = string (getenv ("HOME")) + "/.ballandpaddle/config";
#endif
#ifdef SINGLEDIR
  string location = "config";
#endif
  file.open (location.c_str ());
  if (!file.fail ())
    {
      file << m_fullscreen << endl;
      file << m_strSkinName << endl;
      file << m_strSkinFilename << endl;
      file << m_playSounds << endl;
      file.close ();
    }
  else
    {
      cout << "couldn't open file: " << location << endl;
    }
}

void
GameStateManager::drawString (int font, string text, int x, int y)
{
  fonts[font]->drawString (text, screen, x, y);
}

void
GameStateManager::drawStringCentered (int font, string text, int x, int y)
{
  fonts[font]->drawStringCentered (text, screen, x, y);
}

void
GameStateManager::drawString (int font, string text, int x, int y,
			      SDL_Surface * dest)
{
  fonts[font]->drawString (text, dest, x, y);
}

void
GameStateManager::drawStringCentered (int font, string text, int x, int y,
				      SDL_Surface * dest)
{
  fonts[font]->drawStringCentered (text, dest, x, y);
}

void
GameStateManager::cleanup ()
{
  Mix_HaltChannel (-1);
  m_soundManager->freeSounds ();
  Mix_CloseAudio ();
  SDL_Quit ();
}

void
GameStateManager::pushState (GameState * state)
{
  if (!states.empty ())
    {
      states.back ()->pause ();
    }
  states.push_back (state);
  states.back ()->init ();
}

void
GameStateManager::changeState (GameState * state)
{
  if (!states.empty ())
    {
      states.back ()->cleanup ();
      states.pop_back ();
    }

  states.push_back (state);
  states.back ()->init ();
}

void
GameStateManager::popState ()
{
  states.back ()->cleanup ();
  states.pop_back ();
  if (!states.empty ())
    {
      states.back ()->resume ();
    }
}

void
GameStateManager::handleEvents ()
{
  states.back ()->handleEvents (this);
}

void
GameStateManager::update ()
{
  if (SDL_GetTicks () - m_lastTime < 1000 / maxFPS)
    {
      SDL_Delay (m_lastTime + 1000 / maxFPS - SDL_GetTicks ());
    }
  long now = SDL_GetTicks ();
  m_time = now - m_lastTime;
  m_lastTime = now;
  states.back ()->update (this);
}

void
GameStateManager::draw ()
{
  if (!states.back ()->getTransitioning ())
    {
      states.back ()->draw (this);
      SDL_Flip (screen);
    }
}

void
GameStateManager::quit ()
{
  m_running = false;
}
