#include "Maxon49HX.h"
#include "WT49HXTuner.h"

#include <qwt_knob.h>
#include <qlabel.h>
#include <qbuttongroup.h>
#include <qpushbutton.h>
#include <qlayout.h>
#include <qtimer.h>

#include <VrAR5000Source.h>
#include <VrComplexFIRfilter.h>
#include <VrSquelch.h>
#include <VrQuadratureDemod.h>
#include <VrRealFIRfilter.h>
#include <VrAudioSink.h>

const float initVolume = 5.0;
const float initSquelch = 5.0;

const float freqOffset = 0.0;

const int inputRate = 33000000;
const float inputFreq = 10720000;

const int CFIRdecimate = 825;
const int chanTaps = 800;
const float chanGain = 2.0;

const float squelchMult = 4000;

const float demodGain = 2200;

const int RFIRdecimate = 5;
const int ifTaps = 20;
const float ifGain = 1.0;

const int quadRate = inputRate / CFIRdecimate;
const int audioRate = quadRate / RFIRdecimate;

Maxon49HX::Maxon49HX(VrAR5000Source<char> *source,
		     QWidget *parent, const char *name) :
  QWidget(parent, name),
  source(source),
  on(false)
{
  chanFilter = new VrComplexFIRfilter<char>(CFIRdecimate, chanTaps, 
					    inputFreq + freqOffset,
					    chanGain);
  squelch = new VrSquelch<complex>(0, .01);
  demod = new VrQuadratureDemod<float>(0.0);
  ifFilter = new VrRealFIRfilter<float,short>(RFIRdecimate, audioRate/2,
					      ifTaps, ifGain);
  audio = new VrAudioSink<short>();


  CONNECT(audio, ifFilter, audioRate, 16);
  CONNECT(ifFilter, demod, quadRate, 32);
  CONNECT(demod, squelch, quadRate, 64);
  CONNECT(squelch, chanFilter, quadRate, 64);
  CONNECT(chanFilter, source, inputRate, 8);

  audio->setup();

  QBoxLayout *layout = new QHBoxLayout(this);

  vol = new QwtKnob(this);
  vol->setFixedSize(80, 80);
  vol->setRange(0, 10, 0.1);
  vol->setScaleMaxMajor(0);

  layout->addWidget(vol);
  layout->addSpacing(10);

  sq = new QwtKnob(this);
  sq->setFixedSize(80,80);
  sq->setRange(0, 10, 0.1);
  sq->setScaleMaxMajor(0);
  sq->setValue(0.0);

  QLabel *sqLabel = new QLabel("Sq", sq);
  sqLabel->setAlignment(AlignCenter);
  sqLabel->adjustSize();
  sqLabel->move((sq->width() - sqLabel->width()) / 2,
		 sq->height() - sqLabel->height());

  layout->addWidget(sq);
  layout->addSpacing(10);


  QLabel *volLabel = new QLabel("Vol", vol);
  volLabel->adjustSize();
  volLabel->setAlignment(AlignCenter);
  volLabel->move((vol->width() - volLabel->width()) / 2,
		 vol->height() - volLabel->height());

  buttons = new QButtonGroup("Channel", this);
  buttons->setAlignment(AlignHCenter);
  buttons->setExclusive(true);

  QBoxLayout *blayout = new QHBoxLayout(buttons, 20);

  QPushButton *b = new QPushButton("A", buttons);
  b->setMinimumSize(b->sizeHint());
  b->setToggleButton(true);
  blayout->addWidget(b);

  b = new QPushButton("B", buttons);
  b->setMinimumSize(b->sizeHint());
  b->setToggleButton(true);
  blayout->addWidget(b);

  b = new QPushButton("C", buttons);  
  b->setMinimumSize(b->sizeHint());
  b->setToggleButton(true);
  blayout->addWidget(b);
  
  b = new QPushButton("D", buttons);
  b->setMinimumSize(b->sizeHint());
  b->setToggleButton(true);
  blayout->addWidget(b);

  b = new QPushButton("E", buttons);
  b->setMinimumSize(b->sizeHint());
  b->setToggleButton(true);
  blayout->addWidget(b);

  blayout->activate();

  layout->addWidget(buttons);
  layout->activate();

  tuner = new WT49HXTuner(this);
  QTimer *timer = new QTimer(this);

  connect(vol, SIGNAL(valueChanged(double)), SLOT(setVolume(double)));
  connect(sq, SIGNAL(valueChanged(double)), SLOT(setSquelch(double)));
  connect(buttons, SIGNAL(clicked(int)), SLOT(buttonClicked(int)));
  connect(tuner, SIGNAL(channelChanged(int)), SLOT(setChannel(int)));
  connect(timer, SIGNAL(timeout()), SLOT(run()));

  setVolume(initVolume);
  setSquelch(initSquelch);

  timer->start(10);
}

Maxon49HX::~Maxon49HX()
{
  delete audio;
  delete ifFilter;
  delete demod;
  delete squelch;
  delete chanFilter;  
}

QSize
Maxon49HX::sizeHint() const
{
  return minimumSize();
}

void
Maxon49HX::buttonClicked(int id)
{
  if (buttons->find(id)->isOn())
    {
      setChannel(id);
      on = true;
    }
  else
    on = false;
}

void
Maxon49HX::setChannel(int chan)
{
  tuner->setChannel(chan);
  
  source->setRxFrequency(tuner->frequency() + freqOffset);
  static_cast<QPushButton *>(buttons->find(tuner->channel()))->setOn(true);
}

void
Maxon49HX::setVolume(double volume)
{
  vol->moveTo(volume);
  demod->setGain(vol->value() * demodGain);
}

void
Maxon49HX::setSquelch(double thresh)
{
  sq->moveTo(thresh);
  squelch->setThresh(sq->value() * squelchMult);
}

void
Maxon49HX::run()
{
  if (on)
    audio->start(0.1);
}


