/*
 * GNUsound - a sound editor for GNOME.
 * Copyright (C) 2002-2004  Pascal Haakmat <a.haakmat@chello.nl>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifndef CONFIG_H
#define CONFIG_H

#define _GNU_SOURCE 1

#include <inttypes.h>
#include <sys/types.h>
#include <defines.h>
#include "emergency.h"

#ifndef LIBDIR
#define LIBDIR                  "/usr/lib"
#endif

#define MODULE_SEARCH_PATH      ("modules:~/." PACKAGE "/modules:" LIBDIR "/modules")
#define MAX_MODULES                       50

/* Whether marker changes can be in the history (this is broken right
   now so don't enable it) */

#define MARKER_CHANGES_IN_HISTORY          0

/* Minimum scaling factor. */

#define HRES_MIN                      0.0625 /* inverse power of 2 */
#define HRES_MIN_SHIFT_BITS                4 /* bits to shift x to obtain ((1/x) * HRES_MIN)! */

/* Maximum zoom step when zooming in or out. */

#define ZOOM_STEP_MAX                   2048

/** Maximum scaling factor. Large scaling factors cause very slow
    redraw times. You can increase PEAK_HRES to allow larger
    values here, but then you get worse performance at smaller scaling
    factors. */

#define HRES_MAX                       65536 /* power of 2 */

#define GRAPH_BITS_VRES                  256 /* pixels, has to fit in a graph_bits_unit_t */

/** The range of a peak cache element. */
#define PEAK_VRES                        256

/** The number of samples described by a peak cache element. */
#define PEAK_HRES                     128.0f /* frames, power of 2 */
/** The number of samples described by a RMS cache element. */
#define RMS_HRES                      128.0f

/* Number of frames to feed to processors. Too large = UI
   unresponsiveness, too small = block fragmentation = slower
   everything. */

#define EFFECT_BUF_SIZE                32768 /* frames */

/* The maximum number of frames per block. */

#define MAX_BLOCK_SIZE  (PEAK_HRES * 4096) /* frames */
#define LOAD_BUF_SIZE       (MAX_BLOCK_SIZE / 2)
#define SAVE_BUF_SIZE      (MAX_BLOCK_SIZE / 16)

#define DEBUG_FLAG                         1

#if SIZEOF_INT > 4
#define CONVSPEC_INT "ld"
#else
#define CONVSPEC_INT "d"
#endif

#if SIZEOF_SIZE_T > 4
#define CONVSPEC_SIZE_T "lu"
#else
#define CONVSPEC_SIZE_T "u"
#endif

#include <stdlib.h>
#define MAX_TRACKS             32 /* same as number of bits in track_map_t */
typedef u_int32_t track_map_t;    /* limited to 32 bits for various reasons */
typedef void *frame_bits_t;
typedef int8_t graph_bits_unit_t;
typedef int8_t peak_unit_t;
typedef int8_t rms_unit_t;

#ifdef DISABLE_FASTMINMAX
# define USE_MMX_MINMAX 0
#else /* DISABLE_FASTMINMAX */
# ifdef HAVE_ARCH_X86 
#  define USE_MMX_MINMAX          1 /* 0 = never, 1 = auto-detect, 2 = always MMX2, 3 = always 3DNow! */
#  define USE_FLOAT_TO_INT_METHOD 1 /* 0 = lrintf, 1 = x86 ASM, 2 = ISO C */
# else  /* HAVE_ARCH_X86 */
#  define USE_MMX_MINMAX          0 /* 0 = never, 1 = auto-detect, 2 = always MMX2. 3 = always 3DNow! */
#  define USE_FLOAT_TO_INT_METHOD 0 /* 0 = lrintf, 1 = x86 ASM, 2 = ISO C */
# endif /* HAVE_ARCH_X86 */
#endif

/* 
 * Various methods for floating point conversion, generally much
 * faster than the code generated by GCC from C casts. Inspired by
 * work by Erik de Castro Lopo.
 */

#if USE_FLOAT_TO_INT_METHOD == 0
#include <math.h>
#define FLOAT_TO_INT(in)  \
  ({ int __out = lrintf((in)); __out; })
#elif USE_FLOAT_TO_INT_METHOD == 1
#define FLOAT_TO_INT(in)  \
  ({ int __out; __asm__ __volatile__ ("fistpl %0" : "=m" (__out) : "t" (in) : "st"); __out; })
#else
#define FLOAT_TO_INT(in)  \
  (in)
#endif

/* Many systems come with an old sys/soundcard.h which is missing a
   number of defines so they are defined here. */

#include <sys/soundcard.h>
#ifndef AFMT_S32_LE
#define AFMT_S32_LE              0x00001000
#endif
#ifndef AFMT_S32_BE
#define AFMT_S32_BE              0x00002000
#endif
#ifndef AFMT_S32_NE
#define AFMT_S32_NE              (AFMT_S16_NE == AFMT_S16_BE ? AFMT_S32_BE : AFMT_S32_LE)
#endif

#include <pthread.h>
#include <stdio.h>  /* for debug output */
extern FILE *stderr, *stdout;
#if DEBUG_FLAG == 1
# if __GNUC__ > 2 
#  define WHATEVER_GCC_USES_FOR_FUNCTION_NAME __func__
# else
#  define WHATEVER_GCC_USES_FOR_FUNCTION_NAME __FUNCTION__
# endif
# define DEBUG(fmt, args...) \
  do { if(!is_emergency) fprintf(stdout, "%s:%d: " fmt , WHATEVER_GCC_USES_FOR_FUNCTION_NAME, __LINE__ , ## args); } while(0)
  /*fprintf(stdout, "debug: "__FILE__ ":%s:%d: " fmt , \
    WHATEVER_GCC_USES_FOR_FUNCTION_NAME, __LINE__ , ## args)*/
#else
# define DEBUG(fmt, args...)
#endif

#define FAIL(fmt, args...) \
  do { if(!is_emergency) fprintf(stderr, "FAIL : "__FILE__ ":%s:%d: " fmt , WHATEVER_GCC_USES_FOR_FUNCTION_NAME, __LINE__ , ## args); } while(0)

#define INFO(fmt, args...) \
  do { if(!is_emergency) fprintf(stdout, fmt, ## args); } while(0)

/*
 * Configure fast_memcpy code from MPlayer. Disable if you get
 * mysterious crashes or other problems, tested only on Athlon and
 * PIII.
 * 
 * Benchmarks for PIII/600MHz:
 *
 *                  iter.  bytes/iter.         time    speed
 *                 ------+------------+-------------+--------------
 * Standard C:      1000 |    1048576 |  6.736989s. |  148.434 MB/s
 * MMX:             1000 |    1048576 |  6.256012s. |  159.846 MB/s
 * MMX2:            1000 |    1048576 |  3.573701s. |  279.822 MB/s
 * Runtime detect:  1000 |    1048576 |  3.642306s. |  274.551 MB/s 
 *
 */

#define USE_FASTMEMCPY
#ifdef DISABLE_FASTMEMCPY
#undef USE_FASTMEMCPY
#endif

#define RUNTIME_CPUDETECT
#ifdef DISABLE_CPUDETECT
#undef RUNTIME_CPUDETECT
#endif

#define HAVE_MALLOC_H
#define HAVE_MMX
#define HAVE_MMX2
#define HAVE_3DNOW
#define HAVE_3DNOWEX
/* These two don't seem to do much: */
#define HAVE_SSE 
#define HAVE_SSE2
/* MPlayer code needs <string.h>. */
#include <string.h>
#undef ARCH_X86
#ifdef HAVE_ARCH_X86
# define ARCH_X86
#endif
#include "lib/fastmemcpy.h"

#endif /* ! CONFIG_H */

