/*
 * Copyright (C) 1999, 2000, 2001  Lorenzo Bettini <bettini@gnu.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include <stdlib.h>
#include <iostream.h>
#include <string.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "colors.h"
#include "cmdline.h"
#include "genfun.h"
#include "fileutil.h"
#include "messages.h"
#include "generators.h"
#include "envmapper.h"
#include "chartranslator.h"

#include "startapp.h"
#include "copyright.h"
#include "reportbugs.h"

// for globals
#include "linenumdigit.h"
#include "globalostream.h"
#include "cmdlineargs.h"

// global output stream
ostream* sout;

int tabSpaces = 0;     // space to substitue to tabs
char *tags_file = 0; // possible tags file
unsigned int line_num_digit = 0; // num of digits to represent line number

gengetopt_args_info args_info ;     // command line structure

StartApp::StartApp() :
  entire_doc (0), verbose (0), cssUrl (0),
  use_css (0), is_cgi (0)
{
}

extern int parseTags() ;

int
StartApp::start(int argc, char * argv[])
{
  char *docTitle;  
  char *docHeader; // the buffer with the header  
  char *docFooter; // the buffer with the footer
  char *header_fileName = 0;
  char *footer_fileName = 0;
  char **temp_argv;
  unsigned i;
  int v; 

  // map environment to parameters if used as CGI
  temp_argv = map_environment(&argc, argv);
  is_cgi = temp_argv != argv;
  argv = temp_argv;

  if((v = cmdline_parser(argc, argv, &args_info)) != 0) 
    // calls cmdline parser. The user gived bag args if it doesn't return -1 
    return 1; 

  if (args_info.version_given)
    {
      print_version ();
      print_copyright ();
      exit (0);
    }

  if (args_info.help_given)
    {
      cout << "GNU ";
      cmdline_parser_print_help ();
      print_reportbugs ();
      exit (0);
    }

  if (strcmp (args_info.out_format_arg, "html") != 0)
    {
      cerr << PACKAGE << ": output format " << args_info.out_format_arg
           << " not recognized" << endl;
      exit (1);
    }

  /* initialization of global symbols */
  inputFileName = outputFileName = 0 ;
  sout = 0 ;
  docTitle = 0 ;
  docHeader = 0 ;
  docFooter = 0 ;
  
  docTitle = args_info.title_arg ;
  header_fileName = args_info.header_arg ;
  footer_fileName = args_info.footer_arg ;
  verbose = args_info.verbose_given ;
  tags_file = args_info.tags_file_arg;

  if ( args_info.tab_given > 0 )
    tabSpaces = args_info.tab_arg ;

  if (header_fileName)
    docHeader = read_file (header_fileName);

  if (footer_fileName)
    docFooter = read_file (footer_fileName);

  cssUrl = args_info.css_arg ;
  use_css = ( cssUrl != 0 ) ;

  entire_doc = ( args_info.doc_given || (docTitle != 0) || use_css ||
                 docHeader || docFooter ) ;
  
  inputFileName = args_info.input_arg ;
  if ( inputFileName && ! is_cgi ) {
    outputFileName = args_info.output_arg ;
    if ( ! outputFileName ) {
      outputFileName = createOutputFileName (inputFileName,
                                             args_info.output_dir_arg);
    }
  }
  
  if ( verbose )
    setMessager( new DefaultMessages ) ;

  printMessage( PACKAGE ) ;
  printMessage( VERSION ) ;
  
  parseTags() ;

  CharTranslator *translator = createCharTranslator ();

  if( use_css ) {
    createGeneratorsForCSS(translator, args_info.line_number_given);
  }
  else {
    createGenerators(translator, args_info.line_number_given);
  }
  
  if ( is_cgi )
    print_cgi_header() ;

  // let's start the translation :-)
  
  // first the --input file
  if ( ! args_info.inputs_num )
    processFile( args_info.src_lang_arg, inputFileName, outputFileName, 
                 docTitle, docHeader, docFooter, entire_doc, cssUrl ) ;

  // let's process other files, if there are any
  if ( args_info.inputs_num && !is_cgi ) {
    for ( i = 0 ; i < (args_info.inputs_num) ; ++i ) {
      processFile( args_info.src_lang_arg, args_info.inputs[i], 
		   createOutputFileName (args_info.inputs[i],
                                         args_info.output_dir_arg),
		   docTitle, docHeader, docFooter, entire_doc, cssUrl ) ; 
      cerr << "Processed " << args_info.inputs[i] << endl ;
    }
  }
  
  return (0 );
}

void
StartApp::print_copyright()
{
  int i;

  for (i = 1; i <= copyright_text_length; ++i)
    cout << copyright_text[i] << endl;;
}

void
StartApp::print_reportbugs()
{
  int i;

  for (i = 1; i <= reportbugs_text_length; ++i)
    cout << reportbugs_text[i] << endl;
}

void
StartApp::print_version()
{
  cout << "GNU " << PACKAGE << " " << VERSION << endl;
}
