"""Provides widgets related to submodules"""
from __future__ import absolute_import

from qtpy import QtWidgets
from qtpy.QtCore import Qt
from qtpy.QtCore import Signal

from .. import cmds
from .. import core
from .. import qtutils
from .. import icons
from ..i18n import N_
from ..widgets import defs
from ..widgets import standard


class SubmodulesWidget(QtWidgets.QFrame):

    def __init__(self, context, parent):
        QtWidgets.QFrame.__init__(self, parent)
        self.setToolTip(N_('Submodules'))

        self.tree = SubmodulesTreeWidget(context, parent=self)
        self.setFocusProxy(self.tree)

        self.main_layout = qtutils.vbox(defs.no_margin, defs.spacing, self.tree)
        self.setLayout(self.main_layout)

        # Titlebar buttons
        self.refresh_button = qtutils.create_action_button(
                tooltip=N_('Refresh'), icon=icons.sync())

        self.open_parent_button = qtutils.create_action_button(
                tooltip=N_('Open Parent'), icon=icons.repo())

        self.button_layout = qtutils.hbox(defs.no_margin, defs.spacing,
                                          self.open_parent_button,
                                          self.refresh_button)
        self.corner_widget = QtWidgets.QWidget(self)
        self.corner_widget.setLayout(self.button_layout)
        titlebar = parent.titleBarWidget()
        titlebar.add_corner_widget(self.corner_widget)

        # Connections
        qtutils.connect_button(self.refresh_button,
                               context.model.update_submodules_list)
        qtutils.connect_button(self.open_parent_button,
                               cmds.run(cmds.OpenParentRepo, context))


class SubmodulesTreeWidget(standard.TreeWidget):
    updated = Signal()

    def __init__(self, context, parent=None):
        standard.TreeWidget.__init__(self, parent=parent)

        self.context = context
        self.main_model = model = context.model

        self.setSelectionMode(QtWidgets.QAbstractItemView.SingleSelection)
        self.setHeaderHidden(True)
        # UI
        self._active = False
        self.list_helper = BuildItem()
        self.itemDoubleClicked.connect(self.tree_double_clicked)
        # Connections
        self.updated.connect(self.refresh, type=Qt.QueuedConnection)
        model.add_observer(model.message_submodules_changed,
                           self.updated.emit)

    def refresh(self):
        if not self._active:
            return

        items = [self.list_helper.get(entry) for entry in
                 self.main_model.submodules_list]
        self.clear()
        self.addTopLevelItems(items)

    def showEvent(self, event):
        """Defer updating widgets until the widget is visible"""
        if not self._active:
            self._active = True
            self.refresh()
        return super(SubmodulesTreeWidget, self).showEvent(event)

    def tree_double_clicked(self, item, _column):
        path = core.abspath(item.path)
        cmds.do(cmds.OpenRepo, self.context, path)


class BuildItem(object):

    def __init__(self):
        self.state_folder_map = {}
        self.state_folder_map[''] = icons.folder()
        self.state_folder_map['+'] = icons.staged()
        self.state_folder_map['-'] = icons.modified()
        self.state_folder_map['U'] = icons.merge()

    def get(self, entry):
        """entry: same as returned from list_submodule"""
        name = entry[2]
        path = entry[2]
        # TODO better tip
        tip = path + '\n' + entry[1]
        if entry[3]:
            tip += '\n({0})'.format(entry[3])
        icon = self.state_folder_map[entry[0]]
        return SubmodulesTreeWidgetItem(name, path, tip, icon)


class SubmodulesTreeWidgetItem(QtWidgets.QTreeWidgetItem):

    def __init__(self, name, path, tip, icon):
        QtWidgets.QTreeWidgetItem.__init__(self)
        self.path = path

        self.setIcon(0, icon)
        self.setText(0, name)
        self.setToolTip(0, tip)
