#!/usr/bin/python
# -*- coding: utf-8 -*-
# Copyright: (c) 2022, Martin Rødvand (@rodvand) <martin@rodvand.net>
# GNU General Public License v3.0+ (see COPYING or https://www.gnu.org/licenses/gpl-3.0.txt)

from __future__ import absolute_import, division, print_function

__metaclass__ = type

DOCUMENTATION = r"""
---
module: netbox_export_template
short_description: Creates, updates or deletes export templates within NetBox
description:
  - Creates, updates or removes export templates from NetBox
notes:  
  - This should be ran with connection C(local) and hosts C(localhost)
  - Use the C(!unsafe) data type if you want jinja2 code in template_code
author:
  - Martin Rødvand (@rodvand)
requirements:
  - pynetbox
version_added: "3.6.0"
extends_documentation_fragment:
  - netbox.netbox.common
options:
  data:
    type: dict
    description:
      - Defines the custom field
    suboptions:
      content_type:
        description:
          - The content type to apply this export template to
        required: false
        type: raw        
      name: 
        description: 
          - The name of the export template
        required: true
        type: str
      description:
        description:
          - Description of the export template
        required: false
        type: str
      template_code:
        description:
          - Template code of the export template
        required: true
        type: raw                
      mime_type:
        description:
          - MIME type of the export template
        required: false
        type: str
      file_extension:
        description:
          - The file extension of the export template
        required: false
        type: str            
      as_attachment:
        description:
          - Download file as attachment 
        required: false
        type: bool                                          
    required: true
"""

EXAMPLES = r"""
- name: "Test NetBox custom_link module"
  connection: local
  hosts: localhost  
  tasks:
    - name: Create a custom link on device
      netbox_custom_link:
        netbox_url: http://netbox.local
        netbox_token: thisIsMyToken
        data:
          content_type: "dcim.device"            
          name: Custom Link
          link_text: "Open Web Management"
          link_url: !unsafe https://{{ obj.name }}.domain.local                        

    - name: Delete the custom link
      netbox_custom_field:
        netbox_url: http://netbox.local
        netbox_token: thisIsMyToken
        data:
          content_type: "dcim.device"            
          name: Custom Link
          link_text: "Open Web Management"
          link_url: !unsafe https://{{ obj.name }}.domain.local
        state: absent
"""

RETURN = r"""
custom_link:
  description: Serialized object as created/existent/updated/deleted within NetBox
  returned: always
  type: dict
msg:
  description: Message indicating failure or info about what has been achieved
  returned: always
  type: str
"""

from ansible_collections.netbox.netbox.plugins.module_utils.netbox_utils import (
    NetboxAnsibleModule,
    NETBOX_ARG_SPEC,
)
from ansible_collections.netbox.netbox.plugins.module_utils.netbox_extras import (
    NetboxExtrasModule,
    NB_EXPORT_TEMPLATES,
)
from copy import deepcopy


def main():
    """
    Main entry point for module execution
    """
    argument_spec = deepcopy(NETBOX_ARG_SPEC)
    argument_spec.update(
        dict(
            data=dict(
                type="dict",
                required=True,
                options=dict(
                    content_type=dict(required=False, type="raw"),
                    name=dict(required=True, type="str"),
                    description=dict(required=False, type="str"),
                    template_code=dict(required=True, type="raw"),
                    mime_type=dict(required=False, type="str"),
                    file_extension=dict(required=False, type="str"),
                    as_attachment=dict(required=False, type="bool"),
                ),
            )
        )
    )

    required_if = [
        ("state", "present", ["content_type", "name", "template_code"]),
        ("state", "absent", ["name"]),
    ]

    module = NetboxAnsibleModule(
        argument_spec=argument_spec, supports_check_mode=True, required_if=required_if
    )

    netbox_export_template = NetboxExtrasModule(module, NB_EXPORT_TEMPLATES)
    netbox_export_template.run()


if __name__ == "__main__":
    main()
