"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.registerOpenSearchRoutes = registerOpenSearchRoutes;

var _lodash = require("lodash");

var _adHelpers = require("./utils/adHelpers");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

function registerOpenSearchRoutes(apiRouter, opensearchService) {
  apiRouter.get('/_indices', opensearchService.getIndices);
  apiRouter.get('/_aliases', opensearchService.getAliases);
  apiRouter.get('/_mappings', opensearchService.getMapping);
  apiRouter.post('/_search', opensearchService.executeSearch);
  apiRouter.put('/create_index', opensearchService.createIndex);
  apiRouter.post('/bulk', opensearchService.bulk);
  apiRouter.post('/delete_index', opensearchService.deleteIndex);
}

class OpenSearchService {
  constructor(client) {
    _defineProperty(this, "client", void 0);

    _defineProperty(this, "executeSearch", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          index,
          query,
          size = 0,
          sort = undefined,
          collapse = undefined,
          aggs = undefined,
          rawQuery = undefined
        } = request.body;
        const requestBody = rawQuery ? rawQuery : {
          query: query,
          ...(sort !== undefined && {
            sort: sort
          }),
          ...(collapse !== undefined && {
            collapse: collapse
          }),
          ...(aggs !== undefined && {
            aggs: aggs
          })
        };
        const params = {
          index,
          size,
          body: requestBody
        };
        const results = await this.client.asScoped(request).callAsCurrentUser('search', params);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: results
          }
        });
      } catch (err) {
        console.error('Anomaly detector - Unable to execute search', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "getIndices", async (context, request, opensearchDashboardsResponse) => {
      const {
        index
      } = request.query;

      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        // In case no matching indices is found it throws an error.
        if (err.statusCode === 404 && (0, _lodash.get)(err, 'body.error.type', '') === 'index_not_found_exception') {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                indices: []
              }
            }
          });
        }

        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "getAliases", async (context, request, opensearchDashboardsResponse) => {
      const {
        alias
      } = request.query;

      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('cat.aliases', {
          alias,
          format: 'json',
          h: 'alias,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              aliases: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get aliases', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "createIndex", async (context, request, opensearchDashboardsResponse) => {
      //@ts-ignore
      const index = request.body.index; //@ts-ignore

      const body = request.body.body;

      try {
        await this.client.asScoped(request).callAsCurrentUser('indices.create', {
          index: index,
          body: body
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to create index', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }

      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "bulk", async (context, request, opensearchDashboardsResponse) => {
      const body = request.body;

      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('bulk', {
          body: body
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to perform bulk action', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "deleteIndex", async (context, request, opensearchDashboardsResponse) => {
      const index = request.query;

      try {
        await this.client.asScoped(request).callAsCurrentUser('indices.delete', {
          index: index
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to perform delete index action', err); // Ignore the error if it's an index_not_found_exception

        if (!(0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: false,
              error: (0, _adHelpers.getErrorMessage)(err)
            }
          });
        }
      }

      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "getMapping", async (context, request, opensearchDashboardsResponse) => {
      const {
        index
      } = request.query;

      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('indices.getMapping', {
          index
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              mappings: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get mappings', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    this.client = client;
  }

}

exports.default = OpenSearchService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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