// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import { KeyringNode, needs, EncryptedDataKey, KeyringTraceFlag, immutableClass, readOnlyProperty, unwrapDataKey, } from '@aws-crypto/material-management-node';
import { constants, publicEncrypt, privateDecrypt, randomBytes, } from 'crypto';
import { _onEncrypt, _onDecrypt, } from '@aws-crypto/raw-keyring';
import { oaepHashSupported } from './oaep_hash_supported';
const supportedOaepHash = [
    'sha1',
    'sha256',
    'sha384',
    'sha512',
    undefined,
];
export class RawRsaKeyringNode extends KeyringNode {
    constructor(input) {
        super();
        const { rsaKey, keyName, keyNamespace, padding = constants.RSA_PKCS1_OAEP_PADDING, oaepHash, } = input;
        const { publicKey, privateKey } = rsaKey;
        /* Precondition: RsaKeyringNode needs either a public or a private key to operate. */
        needs(publicKey || privateKey, 'No Key provided.');
        /* Precondition: RsaKeyringNode needs identifying information for encrypt and decrypt. */
        needs(keyName && keyNamespace, 'Identifying information must be defined.');
        /* Precondition: The AWS ESDK only supports specific hash values for OAEP padding. */
        needs(padding === constants.RSA_PKCS1_OAEP_PADDING
            ? oaepHashSupported
                ? supportedOaepHash.includes(oaepHash)
                : !oaepHash || oaepHash === 'sha1'
            : !oaepHash, 'Unsupported oaepHash');
        const _wrapKey = async (material) => {
            /* Precondition: Public key must be defined to support encrypt. */
            if (!publicKey)
                throw new Error('No public key defined in constructor.  Encrypt disabled.');
            const { buffer, byteOffset, byteLength } = unwrapDataKey(material.getUnencryptedDataKey());
            const encryptedDataKey = publicEncrypt({ key: publicKey, padding, oaepHash }, Buffer.from(buffer, byteOffset, byteLength));
            const providerInfo = this.keyName;
            const providerId = this.keyNamespace;
            const flag = KeyringTraceFlag.WRAPPING_KEY_ENCRYPTED_DATA_KEY;
            const edk = new EncryptedDataKey({
                encryptedDataKey,
                providerInfo,
                providerId,
            });
            return material.addEncryptedDataKey(edk, flag);
        };
        const _unwrapKey = async (material, edk) => {
            /* Precondition: Private key must be defined to support decrypt. */
            if (!privateKey)
                throw new Error('No private key defined in constructor.  Decrypt disabled.');
            const trace = {
                keyName: this.keyName,
                keyNamespace: this.keyNamespace,
                flags: KeyringTraceFlag.WRAPPING_KEY_DECRYPTED_DATA_KEY,
            };
            const { buffer, byteOffset, byteLength } = edk.encryptedDataKey;
            const encryptedDataKey = Buffer.from(buffer, byteOffset, byteLength);
            const unencryptedDataKey = privateDecrypt({ key: privateKey, padding, oaepHash }, encryptedDataKey);
            return material.setUnencryptedDataKey(unencryptedDataKey, trace);
        };
        readOnlyProperty(this, 'keyName', keyName);
        readOnlyProperty(this, 'keyNamespace', keyNamespace);
        readOnlyProperty(this, '_wrapKey', _wrapKey);
        readOnlyProperty(this, '_unwrapKey', _unwrapKey);
    }
    _filter({ providerId, providerInfo }) {
        const { keyNamespace, keyName } = this;
        return providerId === keyNamespace && providerInfo === keyName;
    }
    _onEncrypt = _onEncrypt(randomBytesAsync);
    _onDecrypt = _onDecrypt();
}
immutableClass(RawRsaKeyringNode);
async function randomBytesAsync(size) {
    return new Promise((resolve, reject) => {
        randomBytes(size, (err, buffer) => {
            if (err)
                return reject(err);
            resolve(buffer);
        });
    });
}
//# sourceMappingURL=data:application/json;base64,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