"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.registerOpenSearchRoutes = registerOpenSearchRoutes;

var _lodash = require("lodash");

var _adHelpers = require("./utils/adHelpers");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; } /*
                                                                                                                                                                                                                   * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                   *
                                                                                                                                                                                                                   * The OpenSearch Contributors require contributions made to
                                                                                                                                                                                                                   * this file be licensed under the Apache-2.0 license or a
                                                                                                                                                                                                                   * compatible open source license.
                                                                                                                                                                                                                   *
                                                                                                                                                                                                                   * Modifications Copyright OpenSearch Contributors. See
                                                                                                                                                                                                                   * GitHub history for details.
                                                                                                                                                                                                                   */

function registerOpenSearchRoutes(apiRouter, opensearchService) {
  apiRouter.get('/_indices', opensearchService.getIndices);
  apiRouter.get('/_aliases', opensearchService.getAliases);
  apiRouter.get('/_mappings', opensearchService.getMapping);
  apiRouter.post('/_search', opensearchService.executeSearch);
  apiRouter.put('/create_index', opensearchService.createIndex);
  apiRouter.post('/bulk', opensearchService.bulk);
  apiRouter.post('/delete_index', opensearchService.deleteIndex);
}

class OpenSearchService {
  constructor(client) {
    _defineProperty(this, "client", void 0);

    _defineProperty(this, "executeSearch", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          index,
          query,
          size = 0,
          sort = undefined,
          collapse = undefined,
          aggs = undefined,
          rawQuery = undefined
        } = request.body;
        const requestBody = rawQuery ? rawQuery : {
          query: query,
          ...(sort !== undefined && {
            sort: sort
          }),
          ...(collapse !== undefined && {
            collapse: collapse
          }),
          ...(aggs !== undefined && {
            aggs: aggs
          })
        };
        const params = {
          index,
          size,
          body: requestBody
        };
        const results = await this.client.asScoped(request).callAsCurrentUser('search', params);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: results
          }
        });
      } catch (err) {
        console.error('Anomaly detector - Unable to execute search', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "getIndices", async (context, request, opensearchDashboardsResponse) => {
      const {
        index
      } = request.query;

      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        // In case no matching indices is found it throws an error.
        if (err.statusCode === 404 && (0, _lodash.get)(err, 'body.error.type', '') === 'index_not_found_exception') {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                indices: []
              }
            }
          });
        }

        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "getAliases", async (context, request, opensearchDashboardsResponse) => {
      const {
        alias
      } = request.query;

      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('cat.aliases', {
          alias,
          format: 'json',
          h: 'alias,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              aliases: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get aliases', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "createIndex", async (context, request, opensearchDashboardsResponse) => {
      //@ts-ignore
      const index = request.body.index; //@ts-ignore

      const body = request.body.body;

      try {
        await this.client.asScoped(request).callAsCurrentUser('indices.create', {
          index: index,
          body: body
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to create index', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }

      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "bulk", async (context, request, opensearchDashboardsResponse) => {
      const body = request.body;

      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('bulk', {
          body: body
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to perform bulk action', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "deleteIndex", async (context, request, opensearchDashboardsResponse) => {
      const index = request.query;

      try {
        await this.client.asScoped(request).callAsCurrentUser('indices.delete', {
          index: index
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to perform delete index action', err); // Ignore the error if it's an index_not_found_exception

        if (!(0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: false,
              error: (0, _adHelpers.getErrorMessage)(err)
            }
          });
        }
      }

      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "getMapping", async (context, request, opensearchDashboardsResponse) => {
      const {
        index
      } = request.query;

      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('indices.getMapping', {
          index
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              mappings: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get mappings', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    this.client = client;
  }

}

exports.default = OpenSearchService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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