"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.getDataStreams = getDataStreams;
exports.getIndexToDataStreamMapping = getIndexToDataStreamMapping;

var _constants = require("../utils/constants");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; } /*
                                                                                                                                                                                                                   * Copyright OpenSearch Contributors
                                                                                                                                                                                                                   * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                   */

class DataStreamService {
  constructor(osDriver) {
    _defineProperty(this, "osDriver", void 0);

    _defineProperty(this, "getDataStreams", async (context, request, response) => {
      try {
        const {
          search
        } = request.query;
        const client = this.osDriver.asScoped(request);
        const [dataStreams, apiAccessible, errMsg] = await getDataStreams(client, search);
        if (!apiAccessible) return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: errMsg
          }
        });
        return response.custom({
          statusCode: 200,
          body: {
            ok: true,
            response: {
              dataStreams: dataStreams,
              totalDataStreams: dataStreams.length
            }
          }
        });
      } catch (err) {
        console.error("Index Management - DataStreamService - getDataStreams:", err);
        return response.custom({
          statusCode: 200,
          body: {
            ok: false,
            error: err.message
          }
        });
      }
    });

    this.osDriver = osDriver;
  }

}

exports.default = DataStreamService;

async function getDataStreams({
  callAsCurrentUser: callWithRequest
}, search) {
  const searchPattern = search ? `*${search}*` : "*";
  let accessible = true;
  let errMsg = "";
  const dataStreamsResponse = await callWithRequest("transport.request", {
    path: `/_data_stream/${searchPattern}`,
    method: "GET"
  }).catch(e => {
    if (e.statusCode === 403 && e.message.startsWith(_constants.SECURITY_EXCEPTION_PREFIX)) {
      accessible = false;
      errMsg = e.message;
      return {
        data_streams: []
      };
    }

    throw e;
  });
  return [dataStreamsResponse["data_streams"], accessible, errMsg];
}

async function getIndexToDataStreamMapping({
  callAsCurrentUser: callWithRequest
}) {
  const [dataStreams] = await getDataStreams({
    callAsCurrentUser: callWithRequest
  });
  const mapping = {};
  dataStreams.forEach(dataStream => {
    dataStream.indices.forEach(index => {
      mapping[index.index_name] = dataStream.name;
    });
  });
  return mapping;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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