"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.loadSampleData = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _readline = _interopRequireDefault(require("readline"));
var _fs = _interopRequireDefault(require("fs"));
var _zlib = require("zlib");
var _lodash = require("lodash");
var _helpers = require("../../utils/helpers");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
/*
 * SPDX-License-Identifier: Apache-2.0
 *
 * The OpenSearch Contributors require contributions made to
 * this file be licensed under the Apache-2.0 license or a
 * compatible open source license.
 *
 * Modifications Copyright OpenSearch Contributors. See
 * GitHub history for details.
 */

//@ts-ignore

const BULK_INSERT_SIZE = 500;
const loadSampleData = (filePath, indexName, client, request, context, dataSourceEnabled) => {
  const {
    dataSourceId = ''
  } = request.params;
  return new Promise((resolve, reject) => {
    let count = 0;
    let docs = [];
    let isPaused = false;
    let offset = 0;
    const startTime = (0, _moment.default)(new Date().getTime()).subtract({
      days: 7
    });

    // Create the read stream for the file. Set a smaller buffer size here to prevent it from
    // getting too large, to prevent inserting too many docs at once into the index.
    const readStream = _fs.default.createReadStream(filePath, {
      highWaterMark: 1024 * 4
    });
    const lineStream = _readline.default.createInterface({
      input: readStream.pipe((0, _zlib.createUnzip)())
    });

    // This is only ran when the end of lineStream closes normally. It is used to
    // bulk insert the final batch of lines that are < BULK_INSERT_SIZE
    const onClose = async () => {
      if (docs.length > 0) {
        try {
          await bulkInsert(docs);
        } catch (err) {
          reject(err);
          return;
        }
      }
      resolve(count);
    };
    lineStream.on('close', onClose);
    lineStream.on('pause', async () => {
      isPaused = true;
    });
    lineStream.on('resume', async () => {
      isPaused = false;
    });
    lineStream.on('line', async doc => {
      // for the initial doc, get the timestamp to properly set an offset
      if (count === 0) {
        const initialTime = (0, _moment.default)(JSON.parse(doc).timestamp);
        offset = startTime.diff(initialTime);
      }
      count++;
      docs.push(doc);

      // If not currently paused: pause the stream to prevent concurrent bulk inserts
      // on the cluster which could cause performance issues.
      // Also, empty the current docs[] before performing the bulk insert to prevent
      // buffered docs from being dropped.
      if (docs.length >= BULK_INSERT_SIZE && !isPaused) {
        lineStream.pause();

        // save the docs to insert, and empty out current docs list
        const docsToInsert = docs.slice();
        docs = [];
        try {
          await bulkInsert(docsToInsert);
          lineStream.resume();
        } catch (err) {
          lineStream.removeListener('close', onClose);
          lineStream.close();
          reject(err);
        }
      }
    });
    const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, dataSourceEnabled, request, dataSourceId, client);
    const bulkInsert = async docs => {
      try {
        const bulkBody = prepareBody(docs, offset);
        const resp = await callWithRequest('bulk', {
          body: bulkBody
        });
        if (resp.errors) {
          const errorItems = resp.items;
          const firstErrorReason = (0, _lodash.isEmpty)(errorItems) ? 'Error while bulk inserting. Please try again.' : (0, _helpers.prettifyErrorMessage)(errorItems[0].index.error.reason);
          return Promise.reject(new Error(firstErrorReason));
        }
      } catch (err) {
        console.log('Error while bulk inserting. ', err);
        return Promise.reject(new Error('Error while bulk inserting. Please try again.'));
      }
    };
    const prepareBody = (docs, offset) => {
      const bulkBody = [];
      let docIdCount = count - docs.length;
      docs.forEach(doc => {
        bulkBody.push(getDocDetails(docIdCount));
        bulkBody.push(updateTimestamp(doc, offset));
        docIdCount++;
      });
      return bulkBody;
    };
    const updateTimestamp = (doc, offset) => {
      let docAsJSON = JSON.parse(doc);
      const updatedTimestamp = docAsJSON.timestamp + offset;
      docAsJSON.timestamp = updatedTimestamp;
      return docAsJSON;
    };
    const getDocDetails = docIdCount => {
      return `{ "index": { "_index": "${indexName}", "_id": ${docIdCount} } }`;
    };
  });
};
exports.loadSampleData = loadSampleData;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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