// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import { EncryptedDataKey, unwrapDataKey, KeyringTraceFlag, needs, } from '@aws-crypto/material-management';
import { createCipheriv, createDecipheriv, createHash, randomBytes, } from 'crypto';
import { kdfCounterMode } from '@aws-crypto/kdf-ctr-mode-node';
import { CACHE_ENTRY_ID_DIGEST_ALGORITHM, CIPHERTEXT_STRUCTURE, DECRYPT_FLAGS, DERIVED_BRANCH_KEY_LENGTH, ENCRYPT_FLAGS, KDF_DIGEST_ALGORITHM_SHA_256, KEY_DERIVATION_LABEL, PROVIDER_ID_HIERARCHY, PROVIDER_ID_HIERARCHY_AS_BYTES, } from './constants';
import { serializeFactory, uuidv4Factory } from '@aws-crypto/serialize';
export const stringToUtf8Bytes = (input) => Buffer.from(input, 'utf-8');
export const utf8BytesToString = (input) => input.toString('utf-8');
const stringToHexBytes = (input) => new Uint8Array(Buffer.from(input, 'hex'));
const hexBytesToString = (input) => Buffer.from(input).toString('hex');
export const { uuidv4ToCompressedBytes, decompressBytesToUuidv4 } = uuidv4Factory(stringToHexBytes, hexBytesToString);
export const { serializeEncryptionContext } = serializeFactory(stringToUtf8Bytes);
export function getBranchKeyId({ branchKeyId, branchKeyIdSupplier }, { encryptionContext }) {
    // use the branch key id attribute if it was set, otherwise use the branch key
    // id supplier. The constructor ensures that either the branch key id or
    // supplier is supplied to the keyring
    return (branchKeyId ||
        branchKeyIdSupplier.getBranchKeyId(encryptionContext));
}
const RESOURCE_ID = new Uint8Array([0x02]);
const NULL_BYTE = new Uint8Array([0x00]);
const DECRYPTION_SCOPE = new Uint8Array([0x02]);
const ENCRYPTION_SCOPE = new Uint8Array([0x01]);
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#appendix-a-cache-entry-identifier-formulas
//# When accessing the underlying cryptographic materials cache,
//# the hierarchical keyring MUST use the formulas specified in this appendix
//# in order to compute the [cache entry identifier](../cryptographic-materials-cache.md#cache-identifier).
export function getCacheEntryId(logicalKeyStoreName, partitionId, branchKeyId, versionAsBytes) {
    // get branch key id as a byte array
    const branchKeyIdAsBytes = stringToUtf8Bytes(branchKeyId);
    let entryInfo;
    //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#resource-suffix
    //# The aforementioned 4 definitions ([Resource Identifier](#resource-identifier),
    //# [Scope Identifier](#scope-identifier), [Partition ID](#partition-id-1), and
    //# [Resource Suffix](#resource-suffix)) MUST be appended together with the null byte, 0x00,
    //# and the SHA384 of the result should be taken as the final cache identifier.
    if (versionAsBytes) {
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#decryption-materials
        //# When the hierarchical keyring receives an OnDecrypt request,
        //# it MUST calculate the cache entry identifier as the
        //# SHA-384 hash of the following byte strings, in the order listed:
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#decryption-materials
        //# All the above fields must be separated by a single NULL_BYTE `0x00`.
        //#
        //# | Field                  | Length (bytes) | Interpreted as      |
        //# | ---------------------- | -------------- | ------------------- |
        //# | Resource ID            | 1              | bytes               |
        //# | Null Byte              | 1              | `0x00`              |
        //# | Scope ID               | 1              | bytes               |
        //# | Null Byte              | 1              | `0x00`              |
        //# | Partition ID           | Variable       | bytes               |
        //# | Null Byte              | 1              | `0x00`              |
        //# | Logical Key Store Name | Variable       | UTF-8 Encoded Bytes |
        //# | Null Byte              | 1              | `0x00`              |
        //# | Branch Key ID          | Variable       | UTF-8 Encoded Bytes |
        //# | Null Byte              | 1              | `0x00`              |
        //# | branch-key-version     | 36             | UTF-8 Encoded Bytes |
        entryInfo = Buffer.concat([
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#decryption-materials
            //# - MUST be the Resource ID for the Hierarchical Keyring (0x02)
            RESOURCE_ID,
            NULL_BYTE,
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#decryption-materials
            //# - MUST be the Scope ID for Decrypt (0x02)
            DECRYPTION_SCOPE,
            NULL_BYTE,
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#decryption-materials
            //# - MUST be the Partition ID for the Hierarchical Keyring
            partitionId,
            NULL_BYTE,
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#decryption-materials
            //# - MUST be the UTF8 encoded Logical Key Store Name of the keystore for the Hierarchical Keyring
            logicalKeyStoreName,
            NULL_BYTE,
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#decryption-materials
            //# - MUST be the UTF8 encoded branch-key-id
            branchKeyIdAsBytes,
            NULL_BYTE,
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#decryption-materials
            //# - MUST be the UTF8 encoded branch-key-version
            versionAsBytes,
        ]);
    }
    else {
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#encryption-materials
        //# When the hierarchical keyring receives an OnEncrypt request,
        //# the cache entry identifier MUST be calculated as the
        //# SHA-384 hash of the following byte strings, in the order listed:
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#encryption-materials
        //# All the above fields must be separated by a single NULL_BYTE `0x00`.
        //#
        //# | Field                  | Length (bytes) | Interpreted as      |
        //# | ---------------------- | -------------- | ------------------- |
        //# | Resource ID            | 1              | bytes               |
        //# | Null Byte              | 1              | `0x00`              |
        //# | Scope ID               | 1              | bytes               |
        //# | Null Byte              | 1              | `0x00`              |
        //# | Partition ID           | Variable       | bytes               |
        //# | Null Byte              | 1              | `0x00`              |
        //# | Logical Key Store Name | Variable       | UTF-8 Encoded Bytes |
        //# | Null Byte              | 1              | `0x00`              |
        //# | Branch Key ID          | Variable       | UTF-8 Encoded Bytes |
        entryInfo = Buffer.concat([
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#encryption-materials
            //# - MUST be the Resource ID for the Hierarchical Keyring (0x02)
            RESOURCE_ID,
            NULL_BYTE,
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#encryption-materials
            //# - MUST be the Scope ID for Encrypt (0x01)
            ENCRYPTION_SCOPE,
            NULL_BYTE,
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#encryption-materials
            //# - MUST be the Partition ID for the Hierarchical Keyring
            partitionId,
            NULL_BYTE,
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#encryption-materials
            //# - MUST be the UTF8 encoded Logical Key Store Name of the keystore for the Hierarchical Keyring
            logicalKeyStoreName,
            NULL_BYTE,
            //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#encryption-materials
            //# - MUST be the UTF8 encoded branch-key-id
            branchKeyIdAsBytes,
        ]);
    }
    // hash the branch key id buffer with sha512
    return createHash(CACHE_ENTRY_ID_DIGEST_ALGORITHM)
        .update(entryInfo)
        .digest()
        .toString();
}
export async function getBranchKeyMaterials(hKeyring, cmc, branchKeyId, cacheEntryId, branchKeyVersion) {
    const { keyStore, cacheLimitTtl } = hKeyring;
    //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
    //# The hierarchical keyring MUST attempt to find [branch key materials](../structures.md#branch-key-materials)
    //# from the underlying [cryptographic materials cache](../local-cryptographic-materials-cache.md).
    const cacheEntry = cmc.getBranchKeyMaterial(cacheEntryId);
    let branchKeyMaterials;
    // if the cache entry is false, branch key materials were not found
    if (!cacheEntry || hKeyring.cacheEntryHasExceededLimits(cacheEntry)) {
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
        //# If this is NOT true, then we MUST treat the cache entry as expired.
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
        //# If this is NOT true, then we MUST treat the cache entry as expired.
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
        //# If a cache entry is not found or the cache entry is expired, the hierarchical keyring MUST attempt to obtain the branch key materials
        //# by querying the backing branch keystore specified in the [retrieve OnEncrypt branch key materials](#query-branch-keystore-onencrypt) section.
        //# If the keyring is not able to retrieve [branch key materials](../structures.md#branch-key-materials)
        //# through the underlying cryptographic materials cache or
        //# it no longer has access to them through the backing keystore, OnEncrypt MUST fail.
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#query-branch-keystore-onencrypt
        //# Otherwise, OnEncrypt MUST fail.
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#getitem-branch-keystore-ondecrypt
        //# Otherwise, OnDecrypt MUST fail.
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#query-branch-keystore-onencrypt
        //# OnEncrypt MUST call the Keystore's [GetActiveBranchKey](../branch-key-store.md#getactivebranchkey) operation with the following inputs:
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#getitem-branch-keystore-ondecrypt
        //# OnDecrypt MUST call the Keystore's [GetBranchKeyVersion](../branch-key-store.md#getbranchkeyversion) operation with the following inputs:
        branchKeyMaterials = branchKeyVersion
            ? await keyStore.getBranchKeyVersion(branchKeyId, branchKeyVersion)
            : // The complice needs a line
                //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#query-branch-keystore-onencrypt
                //# OnEncrypt MUST call the Keystore's [GetActiveBranchKey](../branch-key-store.md#getactivebranchkey) operation with the following inputs:
                //# - the `branchKeyId` used in this operation
                await keyStore.getActiveBranchKey(branchKeyId);
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#query-branch-keystore-onencrypt
        //# If the Keystore's GetActiveBranchKey operation succeeds
        //# the keyring MUST put the returned branch key materials in the cache using the
        //# formula defined in [Appendix A](#appendix-a-cache-entry-identifier-formulas).
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#getitem-branch-keystore-ondecrypt
        //# If the Keystore's GetBranchKeyVersion operation succeeds
        //# the keyring MUST put the returned branch key materials in the cache using the
        //# formula defined in [Appendix A](#appendix-a-cache-entry-identifier-formulas).
        cmc.putBranchKeyMaterial(cacheEntryId, branchKeyMaterials, cacheLimitTtl);
    }
    else {
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
        //# If a cache entry is found and the entry's TTL has not expired, the hierarchical keyring MUST use those branch key materials for key unwrapping.
        //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
        //# If a cache entry is found and the entry's TTL has not expired, the hierarchical keyring MUST use those branch key materials for key wrapping.
        branchKeyMaterials = cacheEntry.response;
    }
    return branchKeyMaterials;
}
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
//# If the input [encryption materials](../structures.md#encryption-materials) do not contain a plaintext data key,
//# OnEncrypt MUST generate a random plaintext data key, according to the key length defined in the [algorithm suite](../algorithm-suites.md#encryption-key-length).
//# The process used to generate this random plaintext data key MUST use a secure source of randomness.
export function getPlaintextDataKey(material) {
    // get the pdk from the encryption material whether it is already set or we
    // must randomly generate it
    return new Uint8Array(material.hasUnencryptedDataKey
        ? unwrapDataKey(material.getUnencryptedDataKey())
        : randomBytes(material.suite.keyLengthBytes));
}
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#branch-key-wrapping
//# To derive and encrypt a data key the keyring will follow the same key derivation and encryption as [AWS KMS](https://rwc.iacr.org/2018/Slides/Gueron.pdf).
//# The hierarchical keyring MUST:
//# 1. Generate a 16 byte random `salt` using a secure source of randomness
//# 1. Generate a 12 byte random `IV` using a secure source of randomness
//# 1. Use a [KDF in Counter Mode with a Pseudo Random Function with HMAC SHA 256](https://nvlpubs.nist.gov/nistpubs/SpecialPublications/NIST.SP.800-108r1.pdf) to derive a 32 byte `derivedBranchKey` data key with the following inputs:
//#    - Use the `salt` as the salt.
//#    - Use the branch key as the `key`.
//#    - Use the UTF8 Encoded value "aws-kms-hierarchy" as the label.
//# 1. Encrypt a plaintext data key with the `derivedBranchKey` using `AES-GCM-256` with the following inputs:
//#    - MUST use the `derivedBranchKey` as the AES-GCM cipher key.
//#    - MUST use the plain text data key that will be wrapped by the `derivedBranchKey` as the AES-GCM message.
//#    - MUST use the derived `IV` as the AES-GCM IV.
//#    - MUST use an authentication tag byte of length 16.
//#    - MUST use the serialized [AAD](#branch-key-wrapping-and-unwrapping-aad) as the AES-GCM AAD.
//# If OnEncrypt fails to do any of the above, OnEncrypt MUST fail.
export function wrapPlaintextDataKey(pdk, branchKeyMaterials, { encryptionContext }) {
    // get what we need from branch key material to wrap the pdk
    const branchKey = branchKeyMaterials.branchKey();
    const { branchKeyIdentifier, branchKeyVersion: branchKeyVersionAsBytes } = branchKeyMaterials;
    // compress the branch key version utf8 bytes
    const branchKeyVersionAsBytesCompressed = Buffer.from(uuidv4ToCompressedBytes(utf8BytesToString(branchKeyVersionAsBytes)));
    const branchKeyIdAsBytes = stringToUtf8Bytes(branchKeyIdentifier);
    // generate salt and IV
    const salt = randomBytes(CIPHERTEXT_STRUCTURE.saltLength);
    const iv = randomBytes(CIPHERTEXT_STRUCTURE.ivLength);
    // derive a key from the branch key
    const derivedBranchKey = kdfCounterMode({
        digestAlgorithm: KDF_DIGEST_ALGORITHM_SHA_256,
        ikm: branchKey,
        nonce: salt,
        purpose: KEY_DERIVATION_LABEL,
        expectedLength: DERIVED_BRANCH_KEY_LENGTH,
    });
    // set up additional auth data
    const wrappedAad = wrapAad(branchKeyIdAsBytes, branchKeyVersionAsBytesCompressed, encryptionContext);
    // encrypt the pdk into an edk
    const cipher = createCipheriv('aes-256-gcm', derivedBranchKey, iv).setAAD(wrappedAad);
    const edkCiphertext = Buffer.concat([cipher.update(pdk), cipher.final()]);
    const authTag = cipher.getAuthTag();
    // wrap the edk into a ciphertext
    const ciphertext = new Uint8Array(Buffer.concat([
        salt,
        iv,
        branchKeyVersionAsBytesCompressed,
        edkCiphertext,
        authTag,
    ]));
    return ciphertext;
}
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#branch-key-wrapping-and-unwrapping-aad
//# To Encrypt and Decrypt the `wrappedDerivedBranchKey` the keyring MUST include the following values as part of the AAD for
//# the AES Encrypt/Decrypt calls.
//# To construct the AAD, the keyring MUST concatenate the following values
//# 1. "aws-kms-hierarchy" as UTF8 Bytes
//# 1. Value of `branch-key-id` as UTF8 Bytes
//# 1. [version](../structures.md#branch-key-version) as Bytes
//# 1. [encryption context](structures.md#encryption-context-1) from the input
//#    [encryption materials](../structures.md#encryption-materials) according to the [encryption context serialization specification](../structures.md#serialization).
//# | Field               | Length (bytes) | Interpreted as                                       |
//# | ------------------- | -------------- | ---------------------------------------------------- |
//# | "aws-kms-hierarchy" | 17             | UTF-8 Encoded                                        |
//# | branch-key-id       | Variable       | UTF-8 Encoded                                        |
//# | version             | 16             | Bytes                                                |
//# | encryption context  | Variable       | [Encryption Context](../structures.md#serialization) |
//# If the keyring cannot serialize the encryption context, the operation MUST fail.
export function wrapAad(branchKeyIdAsBytes, version, encryptionContext) {
    /* Precondition: Branch key version must be 16 bytes */
    needs(version.length === 16, 'Branch key version must be 16 bytes');
    /* The AAD section is uInt16BE(length) + AAD
     * see: https://docs.aws.amazon.com/encryption-sdk/latest/developer-guide/message-format.html#header-aad
     * However, we  _only_ need the ADD.
     * So, I just slice off the length.
     */
    const aad = Buffer.from(serializeEncryptionContext(encryptionContext).slice(2));
    return Buffer.concat([
        PROVIDER_ID_HIERARCHY_AS_BYTES,
        branchKeyIdAsBytes,
        version,
        aad,
    ]);
}
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#onencrypt
//# Otherwise, OnEncrypt MUST append a new [encrypted data key](../structures.md#encrypted-data-key)
//# to the encrypted data key list in the [encryption materials](../structures.md#encryption-materials), constructed as follows:
//# - [ciphertext](../structures.md#ciphertext): MUST be serialized as the [hierarchical keyring ciphertext](#ciphertext)
//# - [key provider id](../structures.md#key-provider-id): MUST be UTF8 Encoded "aws-kms-hierarchy"
//# - [key provider info](../structures.md#key-provider-information): MUST be the UTF8 Encoded AWS DDB response `branch-key-id`
export function modifyEncryptionMaterial(encryptionMaterial, pdk, edk, wrappingKeyName) {
    // if the pdk was already set in the encryption material, we should not reset
    if (!encryptionMaterial.hasUnencryptedDataKey) {
        encryptionMaterial.setUnencryptedDataKey(pdk, {
            keyNamespace: PROVIDER_ID_HIERARCHY,
            keyName: wrappingKeyName,
            flags: KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY,
        });
    }
    // add the edk (that we created during onEncrypt) to the encryption material
    return encryptionMaterial.addEncryptedDataKey(new EncryptedDataKey({
        providerId: PROVIDER_ID_HIERARCHY,
        providerInfo: wrappingKeyName,
        encryptedDataKey: edk,
    }), ENCRYPT_FLAGS);
}
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ondecrypt
//# The set of encrypted data keys MUST first be filtered to match this keyring’s configuration. For the encrypted data key to match:
//# - Its provider ID MUST match the UTF8 Encoded value of “aws-kms-hierarchy”.
//# - Deserialize the key provider info, if deserialization fails the next EDK in the set MUST be attempted.
//#   - The deserialized key provider info MUST be UTF8 Decoded and MUST match this keyring's configured `Branch Key Identifier`.
export function filterEdk(branchKeyId, { providerId, providerInfo }) {
    // check if the edk matches the keyring's configuration according to provider
    // id and info (the edk object should have been wrapped by the branch key
    // configured in this keyring or decryption material's encryption context)
    //= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#getitem-branch-keystore-ondecrypt
    //# - Deserialize the UTF8-Decoded `branch-key-id` from the [key provider info](../structures.md#key-provider-information) of the [encrypted data key](../structures.md#encrypted-data-key)
    //# and verify this is equal to the configured or supplied `branch-key-id`.
    return providerId === PROVIDER_ID_HIERARCHY
        ? branchKeyId === providerInfo
        : false;
}
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#ciphertext
//# The following table describes the fields that form the ciphertext for this keyring.
//# The bytes are appended in the order shown.
//# The Encryption Key is variable.
//# It will be whatever length is represented by the algorithm suite.
//# Because all the other values are constant,
//# this variability in the encryption key does not impact the format.
//# | Field              | Length (bytes) | Interpreted as |
//# | ------------------ | -------------- | -------------- |
//# | Salt               | 16             | bytes          |
//# | IV                 | 12             | bytes          |
//# | Version            | 16             | bytes          |
//# | Encrypted Key      | Variable       | bytes          |
//# | Authentication Tag | 16             | bytes          |
export function destructureCiphertext(ciphertext, { keyLengthBytes }) {
    // what we expect the length of the edk object's ciphertext to be. This
    // depends on the byte key length specified by the algorithm suite
    const expectedCiphertextLength = CIPHERTEXT_STRUCTURE.saltLength +
        CIPHERTEXT_STRUCTURE.ivLength +
        CIPHERTEXT_STRUCTURE.branchKeyVersionCompressedLength +
        keyLengthBytes +
        CIPHERTEXT_STRUCTURE.authTagLength;
    /* Precondition: The edk ciphertext must have the correct length */
    needs(ciphertext.length === expectedCiphertextLength, `The encrypted data key ciphertext must be ${expectedCiphertextLength} bytes long`);
    let start = 0;
    let end = 0;
    // extract the salt from the edk ciphertext
    start = end;
    end += CIPHERTEXT_STRUCTURE.saltLength;
    const salt = Buffer.from(ciphertext.subarray(start, end));
    // extract the IV from the edk ciphertext
    start = end;
    end += CIPHERTEXT_STRUCTURE.ivLength;
    const iv = Buffer.from(ciphertext.subarray(start, end));
    // extract the compressed branch key version from the edk ciphertext
    start = end;
    end += CIPHERTEXT_STRUCTURE.branchKeyVersionCompressedLength;
    const branchKeyVersionAsBytesCompressed = Buffer.from(ciphertext.subarray(start, end));
    // extract the encrypted data key from the edk ciphertext
    start = end;
    end += keyLengthBytes;
    const encryptedDataKey = Buffer.from(ciphertext.subarray(start, end));
    // extract the auth tag from the edk ciphertext
    start = end;
    end += CIPHERTEXT_STRUCTURE.authTagLength;
    const authTag = Buffer.from(ciphertext.subarray(start, end));
    return {
        salt,
        iv,
        branchKeyVersionAsBytesCompressed,
        encryptedDataKey,
        authTag,
    };
}
//= aws-encryption-sdk-specification/framework/aws-kms/aws-kms-hierarchical-keyring.md#branch-key-unwrapping
//# To decrypt an encrypted data key with a branch key, the hierarchical keyring MUST:
//# 1. Deserialize the 16 byte random `salt` from the [edk ciphertext](../structures.md#ciphertext).
//# 1. Deserialize the 12 byte random `IV` from the [edk ciphertext](../structures.md#ciphertext).
//# 1. Deserialize the 16 byte `version` from the [edk ciphertext](../structures.md#ciphertext).
//# 1. Deserialize the `encrypted key` from the [edk ciphertext](../structures.md#ciphertext).
//# 1. Deserialize the `authentication tag` from the [edk ciphertext](../structures.md#ciphertext).
//# 1. Use a [KDF in Counter Mode with a Pseudo Random Function with HMAC SHA 256](https://nvlpubs.nist.gov/nistpubs/SpecialPublications/NIST.SP.800-108r1.pdf) to derive
//#    the 32 byte `derivedBranchKey` data key with the following inputs:
//#    - Use the `salt` as the salt.
//#    - Use the branch key as the `key`.
//# 1. Decrypt the encrypted data key with the `derivedBranchKey` using `AES-GCM-256` with the following inputs:
//#    - It MUST use the `encrypted key` obtained from deserialization as the AES-GCM input ciphertext.
//#    - It MUST use the `authentication tag` obtained from deserialization as the AES-GCM input authentication tag.
//#    - It MUST use the `derivedBranchKey` as the AES-GCM cipher key.
//#    - It MUST use the `IV` obtained from deserialization as the AES-GCM input IV.
//#    - It MUST use the serialized [encryption context](#branch-key-wrapping-and-unwrapping-aad) as the AES-GCM AAD.
//# If OnDecrypt fails to do any of the above, OnDecrypt MUST fail.
export function unwrapEncryptedDataKey(ciphertext, branchKeyMaterials, { encryptionContext, suite }) {
    // get what we need from the branch key materials to unwrap the edk
    const branchKey = branchKeyMaterials.branchKey();
    const { branchKeyIdentifier } = branchKeyMaterials;
    const branchKeyIdAsBytes = stringToUtf8Bytes(branchKeyIdentifier);
    // get the salt, iv, edk, and auth tag from the edk ciphertext
    const { salt, iv, encryptedDataKey, authTag, branchKeyVersionAsBytesCompressed, } = destructureCiphertext(ciphertext, suite);
    // derive a key from the branch key
    const derivedBranchKey = kdfCounterMode({
        digestAlgorithm: KDF_DIGEST_ALGORITHM_SHA_256,
        ikm: branchKey,
        nonce: salt,
        purpose: KEY_DERIVATION_LABEL,
        expectedLength: DERIVED_BRANCH_KEY_LENGTH,
    });
    // set up additional auth data
    const wrappedAad = wrapAad(branchKeyIdAsBytes, branchKeyVersionAsBytesCompressed, encryptionContext);
    // decipher the edk to get the udk/pdk
    const decipher = createDecipheriv('aes-256-gcm', derivedBranchKey, iv)
        .setAAD(wrappedAad)
        .setAuthTag(authTag);
    const udk = Buffer.concat([
        decipher.update(encryptedDataKey),
        decipher.final(),
    ]);
    return new Uint8Array(udk);
}
export function modifyDencryptionMaterial(decryptionMaterial, udk, wrappingKeyName) {
    // modify the decryption material by setting the plaintext data key
    return decryptionMaterial.setUnencryptedDataKey(udk, {
        keyNamespace: PROVIDER_ID_HIERARCHY,
        keyName: wrappingKeyName,
        flags: DECRYPT_FLAGS,
    });
}
//# sourceMappingURL=data:application/json;base64,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