/*
 * Decompiled with CFR 0.152.
 */
package ghidra.util;

import ghidra.util.Msg;
import ghidra.util.exception.AssertException;
import java.awt.Font;
import java.io.File;
import java.io.PrintStream;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.net.URLDecoder;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.Objects;
import java.util.StringTokenizer;
import java.util.TimeZone;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import javax.swing.SwingUtilities;
import utilities.util.reflection.ReflectionUtilities;

public class SystemUtilities {
    private static final String SWING_RUN_ERROR_MSG = "Unexpected exception running a task in the Swing Thread:  ";
    private static final String DATE_TIME_FORMAT = "MMM d yyyy HH:mm:ss";
    private static String userName;
    public static final String FONT_SIZE_OVERRIDE_PROPERTY_NAME = "font.size.override";
    private static final Integer FONT_SIZE_OVERRIDE_VALUE;
    public static final String TESTING_PROPERTY = "SystemUtilities.isTesting";
    public static final String TESTING_BATCH_PROPERTY = "ghidra.test.property.batch.mode";
    public static final String HEADLESS_PROPERTY = "SystemUtilities.isHeadless";
    public static final String SINGLE_JAR_MODE_PROPERTY = "SystemUtilities.isSingleJarMode";
    private static final boolean IS_IN_DEVELOPMENT_MODE;
    private static final boolean IS_IN_TESTING_BATCH_MODE;
    private static volatile Boolean isInTestingMode;

    private static boolean checkForDevelopmentMode() {
        Class<SystemUtilities> myClass = SystemUtilities.class;
        ClassLoader loader = myClass.getClassLoader();
        if (loader == null) {
            return true;
        }
        String name = myClass.getName().replace('.', '/') + ".class";
        URL url = loader.getResource(name);
        return !"jar".equals(url.getProtocol());
    }

    public static String getUserName() {
        if (userName == null) {
            String uname = System.getProperty("user.name");
            if (uname.indexOf(" ") >= 0) {
                userName = "";
                StringTokenizer tokens = new StringTokenizer(uname, " ", false);
                while (tokens.hasMoreTokens()) {
                    userName = userName + tokens.nextToken();
                }
            } else {
                userName = uname;
            }
        }
        return userName;
    }

    public static boolean getBooleanProperty(String name, boolean defaultValue) {
        String value = System.getProperty(name);
        if (value == null) {
            return defaultValue;
        }
        return Boolean.parseBoolean(value);
    }

    public static Integer getFontSizeOverrideValue() {
        return FONT_SIZE_OVERRIDE_VALUE;
    }

    public static Font adjustForFontSizeOverride(Font font) {
        if (FONT_SIZE_OVERRIDE_VALUE == null) {
            return font;
        }
        return font.deriveFont((float)FONT_SIZE_OVERRIDE_VALUE.intValue());
    }

    public static String getDateTime() {
        return SystemUtilities.getDateTime(DATE_TIME_FORMAT);
    }

    public static String currentTimeStamp() {
        SimpleDateFormat formatter = new SimpleDateFormat("dd.MMM.yyyy_HH.mm.ss");
        return formatter.format(new Date());
    }

    public static String getDateTime(String dateTimeFormat) {
        SimpleDateFormat formatter = new SimpleDateFormat(dateTimeFormat, Locale.getDefault());
        formatter.setTimeZone(TimeZone.getDefault());
        return formatter.format(new Date());
    }

    public static boolean isInTestingMode() {
        if (isInTestingMode == null) {
            isInTestingMode = Boolean.TRUE.toString().equalsIgnoreCase(System.getProperty(TESTING_PROPERTY));
        }
        return isInTestingMode;
    }

    public static boolean isInTestingBatchMode() {
        return IS_IN_TESTING_BATCH_MODE;
    }

    public static boolean isInHeadlessMode() {
        String headlessProperty = System.getProperty(HEADLESS_PROPERTY, Boolean.TRUE.toString());
        return Boolean.parseBoolean(headlessProperty);
    }

    public static <T> T runSwingNow(Supplier<T> s) {
        AtomicReference ref = new AtomicReference();
        SystemUtilities.runSwingNow(() -> ref.set(s.get()));
        return (T)ref.get();
    }

    public static void runSwingNow(Runnable r) {
        SystemUtilities.runSwing(r, true, SWING_RUN_ERROR_MSG);
    }

    public static void runSwingLater(Runnable r) {
        SystemUtilities.runSwing(r, false, SWING_RUN_ERROR_MSG);
    }

    public static void runIfSwingOrPostSwingLater(Runnable r) {
        if (SystemUtilities.isInHeadlessMode()) {
            r.run();
            return;
        }
        if (SwingUtilities.isEventDispatchThread()) {
            r.run();
        } else {
            SwingUtilities.invokeLater(r);
        }
    }

    private static void runSwing(Runnable r, boolean wait, String errorMessage) {
        if (SystemUtilities.isInHeadlessMode()) {
            r.run();
            return;
        }
        if (wait) {
            if (SwingUtilities.isEventDispatchThread()) {
                r.run();
                return;
            }
            try {
                SwingUtilities.invokeAndWait(r);
            }
            catch (InterruptedException interruptedException) {
            }
            catch (InvocationTargetException e) {
                Msg.error(SystemUtilities.class, errorMessage + "\nException Message: " + e.getMessage(), e);
            }
        } else {
            SwingUtilities.invokeLater(r);
        }
    }

    public static boolean isInDevelopmentMode() {
        return IS_IN_DEVELOPMENT_MODE;
    }

    public static boolean isInReleaseMode() {
        return !SystemUtilities.isInDevelopmentMode() && !SystemUtilities.isInTestingMode() && !SystemUtilities.isInTestingBatchMode();
    }

    public static boolean isEqual(Object o1, Object o2) {
        return Objects.equals(o1, o2);
    }

    @SafeVarargs
    public static <T> boolean isOneOf(T o, T ... others) {
        for (T t : others) {
            if (!o.equals(t)) continue;
            return true;
        }
        return false;
    }

    public static <T extends Comparable<T>> int compareTo(T c1, T c2) {
        if (c1 == null) {
            return c2 == null ? 0 : 1;
        }
        if (c2 == null) {
            return -1;
        }
        return c1.compareTo(c2);
    }

    public static boolean isArrayEqual(Object[] array1, Object[] array2) {
        if (array1 == null) {
            return array2 == null;
        }
        if (array2 == null) {
            return false;
        }
        if (array1.length != array2.length) {
            return false;
        }
        for (int i = 0; i < array1.length; ++i) {
            if (SystemUtilities.isEqual(array1[i], array2[i])) continue;
            return false;
        }
        return true;
    }

    public static void assertTrue(boolean booleanValue, String string) {
        boolean isProductionMode;
        boolean bl = isProductionMode = !SystemUtilities.isInTestingMode() && !SystemUtilities.isInDevelopmentMode();
        if (isProductionMode) {
            return;
        }
        if (!booleanValue) {
            AssertException e = new AssertException(string);
            Msg.error(SystemUtilities.class, "Assertion failed: " + string, e);
        }
    }

    public static void assertThisIsTheSwingThread(String errorMessage) {
        boolean isProductionMode;
        boolean bl = isProductionMode = !SystemUtilities.isInTestingMode() && !SystemUtilities.isInDevelopmentMode();
        if (isProductionMode) {
            return;
        }
        if (!SwingUtilities.isEventDispatchThread()) {
            Throwable t = ReflectionUtilities.filterJavaThrowable(new AssertException(errorMessage));
            Msg.error(SystemUtilities.class, errorMessage, t);
        }
    }

    public static File getSourceLocationForClass(Class<?> classObject) {
        String name = classObject.getName().replace('.', '/') + ".class";
        URL url = classObject.getClassLoader().getResource(name);
        String urlFile = url.getFile();
        try {
            urlFile = URLDecoder.decode(urlFile, "UTF-8");
        }
        catch (UnsupportedEncodingException unsupportedEncodingException) {
            // empty catch block
        }
        if ("file".equals(url.getProtocol())) {
            int packageLevel = SystemUtilities.getPackageLevel(classObject);
            File file = new File(urlFile);
            for (int i = 0; i < packageLevel + 1; ++i) {
                file = file.getParentFile();
            }
            return file;
        }
        if ("jar".equals(url.getProtocol())) {
            String jarPath = urlFile;
            if (!jarPath.startsWith("file:")) {
                return null;
            }
            jarPath = jarPath.substring(5, jarPath.indexOf(33));
            return new File(jarPath);
        }
        return null;
    }

    private static int getPackageLevel(Class<?> classObject) {
        int dotCount = 0;
        Package package1 = classObject.getPackage();
        if (package1 == null) {
            return 0;
        }
        String packageName = package1.getName();
        for (int i = 0; i < packageName.length(); ++i) {
            if (packageName.charAt(i) != '.') continue;
            ++dotCount;
        }
        return dotCount + 1;
    }

    public static boolean isEventDispatchThread() {
        if (SystemUtilities.isInHeadlessMode()) {
            return true;
        }
        return SwingUtilities.isEventDispatchThread();
    }

    public static void allowSwingToProcessEvents() {
        Runnable r = () -> {};
        SystemUtilities.runSwingNow(r);
        SystemUtilities.runSwingNow(r);
        SystemUtilities.runSwingNow(r);
    }

    public static boolean printString(String string, PrintStream printStream) {
        printStream.println(string);
        return false;
    }

    public static int getDefaultThreadPoolSize() {
        Integer cpuCoreLimit;
        Integer cpuOverride = SystemUtilities.getCPUOverride();
        if (cpuOverride != null) {
            return cpuOverride;
        }
        int numProcessors = Math.max(1, Runtime.getRuntime().availableProcessors() + 1);
        if (numProcessors > 10) {
            numProcessors = 10;
        }
        if ((cpuCoreLimit = SystemUtilities.getCPUCoreLimit()) == null) {
            return numProcessors;
        }
        int parseInt = Math.max(1, cpuCoreLimit + 1);
        return Math.min(parseInt, numProcessors);
    }

    private static Integer getCPUOverride() {
        String cpuOverrideString = System.getProperty("cpu.core.override");
        if (cpuOverrideString == null || cpuOverrideString.trim().isEmpty()) {
            return null;
        }
        try {
            return Integer.parseInt(cpuOverrideString);
        }
        catch (NumberFormatException e) {
            Msg.debug(SystemUtilities.class, "Unable to parse cpu.core.override value: " + cpuOverrideString, e);
            return null;
        }
    }

    private static Integer getCPUCoreLimit() {
        String cpuLimitString = System.getProperty("cpu.core.limit");
        if (cpuLimitString == null || cpuLimitString.trim().isEmpty()) {
            return null;
        }
        try {
            return Integer.parseInt(cpuLimitString);
        }
        catch (Exception e) {
            Msg.debug(SystemUtilities.class, "Unable to parse cpu.core.limit value: " + cpuLimitString, e);
            return null;
        }
    }

    static {
        FONT_SIZE_OVERRIDE_VALUE = Integer.getInteger(FONT_SIZE_OVERRIDE_PROPERTY_NAME);
        IS_IN_DEVELOPMENT_MODE = SystemUtilities.checkForDevelopmentMode();
        IS_IN_TESTING_BATCH_MODE = Boolean.TRUE.toString().equalsIgnoreCase(System.getProperty(TESTING_BATCH_PROPERTY));
    }
}

