/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
import semver from 'semver';
import { i18n } from '@kbn/i18n';
import { METRIC_TYPE } from '@kbn/analytics';
import { migratePanelsTo730 } from '../../../common';
/**
 * Attempts to migrate the state stored in the URL into the latest version of it.
 *
 * Once we hit a major version, we can remove support for older style URLs and get rid of this logic.
 */

export function migrateAppState(appState, kibanaVersion, usageCollection) {
  if (!appState.panels) {
    throw new Error(i18n.translate('dashboard.panel.invalidData', {
      defaultMessage: 'Invalid data in url'
    }));
  }

  var panelNeedsMigration = appState.panels.some(function (panel) {
    if (panel.version === undefined) return true;
    var version = panel.version;

    if (usageCollection) {
      // This will help us figure out when to remove support for older style URLs.
      usageCollection.reportUiStats('DashboardPanelVersionInUrl', METRIC_TYPE.LOADED, "".concat(version));
    }

    return semver.satisfies(version, '<7.3');
  });

  if (panelNeedsMigration) {
    appState.panels = migratePanelsTo730(appState.panels, kibanaVersion, appState.useMargins, appState.uiState);
    delete appState.uiState;
  }

  return appState;
}