/*
 * Copyright (c) 2001-2010 by Apple Inc.. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * This file contains Original Code and/or Modifications of Original Code
 * as defined in and that are subject to the Apple Public Source License
 * Version 2.0 (the 'License'). You may not use this file except in
 * compliance with the License. Please obtain a copy of the License at
 * http://www.opensource.apple.com/apsl/ and read it before using this
 * file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT.
 * Please see the License for the specific language governing rights and
 * limitations under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */

/*
     File:       AvailabilityMacros.h
 
     More Info:  See the SDK Compatibility Guide

     Contains:   Autoconfiguration of AVAILABLE_ macros for Mac OS X

                 This header enables a developer to specify build time
                 constraints on what Mac OS X versions the resulting
                 application will be run.  There are two bounds a developer
                 can specify:
                 
                      MAC_OS_X_VERSION_MIN_REQUIRED
                      MAC_OS_X_VERSION_MAX_ALLOWED
                      
                The lower bound controls which calls to OS functions will 
                be weak-importing (allowed to be unresolved at launch time).
                The upper bound controls which OS functionality, if used,
                will result in a compiler error because that functionality is
                not available on any OS in the specifed range.
                
                For example, suppose an application is compiled with:
                
                      MAC_OS_X_VERSION_MIN_REQUIRED = MAC_OS_X_VERSION_10_2
                      MAC_OS_X_VERSION_MAX_ALLOWED  = MAC_OS_X_VERSION_10_3
                     
                and an OS header contains:
                
                     extern void funcA(void) AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER;
                     extern void funcB(void) AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_2;
                     extern void funcC(void) AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_3;
                     extern void funcD(void) AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER;
                     extern void funcE(void) AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER;
                     extern void funcF(void) AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER;
                     extern void funcG(void) AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER;
                     
                     typedef long TypeA DEPRECATED_IN_MAC_OS_X_VERSION_10_0_AND_LATER;
                     typedef long TypeB DEPRECATED_IN_MAC_OS_X_VERSION_10_1_AND_LATER;
                     typedef long TypeC DEPRECATED_IN_MAC_OS_X_VERSION_10_2_AND_LATER;
                     typedef long TypeD DEPRECATED_IN_MAC_OS_X_VERSION_10_3_AND_LATER;
                     typedef long TypeE DEPRECATED_IN_MAC_OS_X_VERSION_10_4_AND_LATER;

                Any application code which uses these declarations will get the following:
                
                                compile         link          run 
                                -------         ------        -------
                     funcA:     normal          normal        normal
                     funcB:     warning         normal        normal
                     funcC:     normal          normal        normal
                     funcD:     normal          normal        normal
                     funcE:     normal          normal        normal
                     funcF:     normal          weak          on 10.3 normal, on 10.2 (&funcF == NULL)
                     funcG:     error           error         n/a
                     typeA:     warning
                     typeB:     warning
                     typeC:     warning
                     typeD:     normal
                     typeE:     normal
                  
  
*/
#ifndef __AVAILABILITYMACROS__
#define __AVAILABILITYMACROS__

/*
 * Set up standard Mac OS X versions
 */
#define MAC_OS_X_VERSION_10_0         1000
#define MAC_OS_X_VERSION_10_1         1010
#define MAC_OS_X_VERSION_10_2         1020
#define MAC_OS_X_VERSION_10_3         1030
#define MAC_OS_X_VERSION_10_4         1040
#define MAC_OS_X_VERSION_10_5         1050
#define MAC_OS_X_VERSION_10_6         1060
#define MAC_OS_X_VERSION_10_7         1070
#define MAC_OS_X_VERSION_10_8         1080
#define MAC_OS_X_VERSION_10_9         1090
#define MAC_OS_X_VERSION_10_10      101000
#define MAC_OS_X_VERSION_10_10_2    101002
#define MAC_OS_X_VERSION_10_10_3    101003
#define MAC_OS_X_VERSION_10_11      101100
#define MAC_OS_X_VERSION_10_11_2    101102
#define MAC_OS_X_VERSION_10_11_3    101103
#define MAC_OS_X_VERSION_10_11_4    101104
#define MAC_OS_X_VERSION_10_12      101200
#define MAC_OS_X_VERSION_10_12_1    101201
#define MAC_OS_X_VERSION_10_12_2    101202
#define MAC_OS_X_VERSION_10_12_4    101204
#define MAC_OS_X_VERSION_10_13      101300
#define MAC_OS_X_VERSION_10_13_1    101301
#define MAC_OS_X_VERSION_10_13_2    101302
#define MAC_OS_X_VERSION_10_13_4    101304
#define MAC_OS_X_VERSION_10_14      101400
#define MAC_OS_X_VERSION_10_14_1    101401
#define MAC_OS_X_VERSION_10_14_4    101404
#define MAC_OS_X_VERSION_10_15      101500
#define MAC_OS_VERSION_11_0         110000
#define MAC_OS_VERSION_11_1         110100

/* 
 * If min OS not specified, assume 10.4 for intel
 * Note: compiler driver may set _ENVIRONMENT_MAC_OS_X_VERSION_MIN_REQUIRED_ based on MACOSX_DEPLOYMENT_TARGET environment variable
 */
#ifndef MAC_OS_X_VERSION_MIN_REQUIRED
    #ifdef __ENVIRONMENT_MAC_OS_X_VERSION_MIN_REQUIRED__
        #if (__i386__ || __x86_64__) && (__ENVIRONMENT_MAC_OS_X_VERSION_MIN_REQUIRED__ < MAC_OS_X_VERSION_10_4)
            #warning Building for Intel with Mac OS X Deployment Target < 10.4 is invalid.
        #endif
        #define MAC_OS_X_VERSION_MIN_REQUIRED __ENVIRONMENT_MAC_OS_X_VERSION_MIN_REQUIRED__
    #else
        #if __i386__ || __x86_64__
            #define MAC_OS_X_VERSION_MIN_REQUIRED MAC_OS_X_VERSION_10_4
        #elif __arm__ || __arm64__
            #define MAC_OS_X_VERSION_MIN_REQUIRED MAC_OS_X_VERSION_10_5
        #else
            #define MAC_OS_X_VERSION_MIN_REQUIRED MAC_OS_X_VERSION_10_1
        #endif
     #endif
#endif

/*
 * if max OS not specified, assume larger of (10.15, min)
 */
#ifndef MAC_OS_X_VERSION_MAX_ALLOWED
    #if MAC_OS_X_VERSION_MIN_REQUIRED > MAC_OS_VERSION_11_1
        #define MAC_OS_X_VERSION_MAX_ALLOWED MAC_OS_X_VERSION_MIN_REQUIRED
    #else
        #define MAC_OS_X_VERSION_MAX_ALLOWED MAC_OS_VERSION_11_1
    #endif
#endif

/*
 * Error on bad values
 */
#if MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_MIN_REQUIRED
    #error MAC_OS_X_VERSION_MAX_ALLOWED must be >= MAC_OS_X_VERSION_MIN_REQUIRED
#endif
#if MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_0
    #error MAC_OS_X_VERSION_MIN_REQUIRED must be >= MAC_OS_X_VERSION_10_0
#endif

/*
 * only certain compilers support __attribute__((weak_import))
 */
#if defined(__GNUC__) && ((__GNUC__ >= 4) || ((__GNUC__ == 3) && (__GNUC_MINOR__ >= 1))) && (MAC_OS_X_VERSION_MIN_REQUIRED >= 1020)
    #define WEAK_IMPORT_ATTRIBUTE __attribute__((weak_import))
#elif defined(__MWERKS__) && (__MWERKS__ >= 0x3205) && (MAC_OS_X_VERSION_MIN_REQUIRED >= 1020) && !defined(__INTEL__)
    #define WEAK_IMPORT_ATTRIBUTE __attribute__((weak_import))
#else
    #define WEAK_IMPORT_ATTRIBUTE
#endif

/*
 * only certain compilers support __attribute__((deprecated))
 */
#if defined(__has_feature) && defined(__has_attribute)
    #if __has_attribute(deprecated)
        #define DEPRECATED_ATTRIBUTE        __attribute__((deprecated))
        #if __has_feature(attribute_deprecated_with_message)
            #define DEPRECATED_MSG_ATTRIBUTE(s) __attribute__((deprecated(s)))
        #else
            #define DEPRECATED_MSG_ATTRIBUTE(s) __attribute__((deprecated))
        #endif
    #else
        #define DEPRECATED_ATTRIBUTE
        #define DEPRECATED_MSG_ATTRIBUTE(s)
    #endif
#elif defined(__GNUC__) && ((__GNUC__ >= 4) || ((__GNUC__ == 3) && (__GNUC_MINOR__ >= 1)))
    #define DEPRECATED_ATTRIBUTE        __attribute__((deprecated))
    #if (__GNUC__ >= 5) || ((__GNUC__ == 4) && (__GNUC_MINOR__ >= 5))
        #define DEPRECATED_MSG_ATTRIBUTE(s) __attribute__((deprecated(s)))
    #else
        #define DEPRECATED_MSG_ATTRIBUTE(s) __attribute__((deprecated))
    #endif
#else
    #define DEPRECATED_ATTRIBUTE
    #define DEPRECATED_MSG_ATTRIBUTE(s)
#endif

/*
 * only certain compilers support __attribute__((unavailable))
 */
#if defined(__GNUC__) && ((__GNUC__ >= 4) || ((__GNUC__ == 3) && (__GNUC_MINOR__ >= 1)))
    #define UNAVAILABLE_ATTRIBUTE __attribute__((unavailable))
#else
    #define UNAVAILABLE_ATTRIBUTE
#endif


/*
 * AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER
 * 
 * Used on functions introduced in Mac OS X 10.0 
 */
#define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED
 * 
 * Used on functions introduced in Mac OS X 10.0, 
 * and deprecated in Mac OS X 10.0
 */
#define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED    DEPRECATED_ATTRIBUTE

/*
 * DEPRECATED_IN_MAC_OS_X_VERSION_10_0_AND_LATER
 * 
 * Used on types deprecated in Mac OS X 10.0 
 */
#define DEPRECATED_IN_MAC_OS_X_VERSION_10_0_AND_LATER     DEPRECATED_ATTRIBUTE

#ifndef __AVAILABILITY_MACROS_USES_AVAILABILITY
	#ifdef __has_attribute
		#if __has_attribute(availability)
			#include <Availability.h>
			#define __AVAILABILITY_MACROS_USES_AVAILABILITY 1
		#endif
	#endif
#endif

#if TARGET_OS_OSX
#define __IPHONE_COMPAT_VERSION  __IPHONE_NA
#elif TARGET_OS_MACCATALYST
#define __IPHONE_COMPAT_VERSION  __IPHONE_NA
#else
#define __IPHONE_COMPAT_VERSION  __IPHONE_4_0
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER
 * 
 * Used on declarations introduced in Mac OS X 10.1 
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER     __OSX_AVAILABLE_STARTING(__MAC_10_1, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER     UNAVAILABLE_ATTRIBUTE
#elif MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER     WEAK_IMPORT_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED
 *
 * Used on declarations introduced in Mac OS X 10.1,
 * and deprecated in Mac OS X 10.1
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED     __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_1, __MAC_10_1, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED    AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_1
 *
 * Used on declarations introduced in Mac OS X 10.0,
 * but later deprecated in Mac OS X 10.1
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_1    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_1, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_1    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_1    AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER
 * 
 * Used on declarations introduced in Mac OS X 10.2 
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER     __OSX_AVAILABLE_STARTING(__MAC_10_2, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER     UNAVAILABLE_ATTRIBUTE
#elif MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER     WEAK_IMPORT_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED
 *
 * Used on declarations introduced in Mac OS X 10.2,
 * and deprecated in Mac OS X 10.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED     __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_2, __MAC_10_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED    AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_2
 *
 * Used on declarations introduced in Mac OS X 10.0,
 * but later deprecated in Mac OS X 10.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_2    AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_2
 *
 * Used on declarations introduced in Mac OS X 10.1,
 * but later deprecated in Mac OS X 10.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_1, __MAC_10_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_2    AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER
 * 
 * Used on declarations introduced in Mac OS X 10.3 
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER     __OSX_AVAILABLE_STARTING(__MAC_10_3, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER     UNAVAILABLE_ATTRIBUTE
#elif MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER     WEAK_IMPORT_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED
 *
 * Used on declarations introduced in Mac OS X 10.3,
 * and deprecated in Mac OS X 10.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED     __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_3, __MAC_10_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED    AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_3
 *
 * Used on declarations introduced in Mac OS X 10.0,
 * but later deprecated in Mac OS X 10.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_3    AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_3
 *
 * Used on declarations introduced in Mac OS X 10.1,
 * but later deprecated in Mac OS X 10.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_1, __MAC_10_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_3    AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_3
 *
 * Used on declarations introduced in Mac OS X 10.2,
 * but later deprecated in Mac OS X 10.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_2, __MAC_10_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_3    AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER
 * 
 * Used on declarations introduced in Mac OS X 10.4 
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER     __OSX_AVAILABLE_STARTING(__MAC_10_4, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER     UNAVAILABLE_ATTRIBUTE
#elif MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER     WEAK_IMPORT_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED
 *
 * Used on declarations introduced in Mac OS X 10.4,
 * and deprecated in Mac OS X 10.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED     __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_4, __MAC_10_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED    AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_4
 *
 * Used on declarations introduced in Mac OS X 10.0,
 * but later deprecated in Mac OS X 10.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_4    AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_4
 *
 * Used on declarations introduced in Mac OS X 10.1,
 * but later deprecated in Mac OS X 10.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_1, __MAC_10_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_4    AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_4
 *
 * Used on declarations introduced in Mac OS X 10.2,
 * but later deprecated in Mac OS X 10.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_2, __MAC_10_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_4    AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_4
 *
 * Used on declarations introduced in Mac OS X 10.3,
 * but later deprecated in Mac OS X 10.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_3, __MAC_10_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_4    AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER
 * 
 * Used on declarations introduced in Mac OS X 10.5 
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER     __OSX_AVAILABLE_STARTING(__MAC_10_5, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_5
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER     UNAVAILABLE_ATTRIBUTE
#elif MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_5
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER     WEAK_IMPORT_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED
 *
 * Used on declarations introduced in Mac OS X 10.5,
 * and deprecated in Mac OS X 10.5
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED     __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_5, __MAC_10_5, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_5
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED    AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_5
 *
 * Used on declarations introduced in Mac OS X 10.0,
 * but later deprecated in Mac OS X 10.5
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_5    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_5, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_5
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_5    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_5    AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_5
 *
 * Used on declarations introduced in Mac OS X 10.1,
 * but later deprecated in Mac OS X 10.5
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_5    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_1, __MAC_10_5, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_5
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_5    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_5    AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_5
 *
 * Used on declarations introduced in Mac OS X 10.2,
 * but later deprecated in Mac OS X 10.5
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_5    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_2, __MAC_10_5, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_5
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_5    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_5    AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_5
 *
 * Used on declarations introduced in Mac OS X 10.3,
 * but later deprecated in Mac OS X 10.5
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_5    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_3, __MAC_10_5, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_5
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_5    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_5    AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_5
 *
 * Used on declarations introduced in Mac OS X 10.4,
 * but later deprecated in Mac OS X 10.5
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_5    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_4, __MAC_10_5, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_5
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_5    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_5    AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER
 * 
 * Used on declarations introduced in Mac OS X 10.6 
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER     __OSX_AVAILABLE_STARTING(__MAC_10_6, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_6
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER     UNAVAILABLE_ATTRIBUTE
#elif MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_6
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER     WEAK_IMPORT_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED
 *
 * Used on declarations introduced in Mac OS X 10.6,
 * and deprecated in Mac OS X 10.6
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED     __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_6, __MAC_10_6, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_6
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED    AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6
 *
 * Used on declarations introduced in Mac OS X 10.0,
 * but later deprecated in Mac OS X 10.6
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_6, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_6
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6    AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6
 *
 * Used on declarations introduced in Mac OS X 10.1,
 * but later deprecated in Mac OS X 10.6
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_1, __MAC_10_6, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_6
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6    AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6
 *
 * Used on declarations introduced in Mac OS X 10.2,
 * but later deprecated in Mac OS X 10.6
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_2, __MAC_10_6, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_6
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6    AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6
 *
 * Used on declarations introduced in Mac OS X 10.3,
 * but later deprecated in Mac OS X 10.6
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_3, __MAC_10_6, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_6
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6    AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6
 *
 * Used on declarations introduced in Mac OS X 10.4,
 * but later deprecated in Mac OS X 10.6
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_4, __MAC_10_6, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_6
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6    AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6
 *
 * Used on declarations introduced in Mac OS X 10.5,
 * but later deprecated in Mac OS X 10.6
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_5, __MAC_10_6, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_6
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_6    AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER
 * 
 * Used on declarations introduced in Mac OS X 10.7 
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER     __OSX_AVAILABLE_STARTING(__MAC_10_7, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_7
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER     UNAVAILABLE_ATTRIBUTE
#elif MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_7
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER     WEAK_IMPORT_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED
 *
 * Used on declarations introduced in Mac OS X 10.7,
 * and deprecated in Mac OS X 10.7
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED     __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_7, __MAC_10_7, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_7
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED    AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7
 *
 * Used on declarations introduced in Mac OS X 10.0,
 * but later deprecated in Mac OS X 10.7
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_7, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_7
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7    AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7
 *
 * Used on declarations introduced in Mac OS X 10.1,
 * but later deprecated in Mac OS X 10.7
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_1, __MAC_10_7, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_7
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7    AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7
 *
 * Used on declarations introduced in Mac OS X 10.2,
 * but later deprecated in Mac OS X 10.7
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_2, __MAC_10_7, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_7
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7    AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7
 *
 * Used on declarations introduced in Mac OS X 10.3,
 * but later deprecated in Mac OS X 10.7
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_3, __MAC_10_7, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_7
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7    AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7
 *
 * Used on declarations introduced in Mac OS X 10.4,
 * but later deprecated in Mac OS X 10.7
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_4, __MAC_10_7, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_7
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7    AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7
 *
 * Used on declarations introduced in Mac OS X 10.5,
 * but later deprecated in Mac OS X 10.7
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_5, __MAC_10_7, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_7
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7    AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7
 *
 * Used on declarations introduced in Mac OS X 10.6,
 * but later deprecated in Mac OS X 10.7
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_6, __MAC_10_7, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_7
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_7    AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_13
 *
 * Used on declarations introduced in Mac OS X 10.6,
 * but later deprecated in Mac OS X 10.13
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
#define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_13    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_6, __MAC_10_13, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_7
#define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_13    DEPRECATED_ATTRIBUTE
#else
#define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_13    AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER
 * 
 * Used on declarations introduced in Mac OS X 10.8 
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER     __OSX_AVAILABLE_STARTING(__MAC_10_8, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_8
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER     UNAVAILABLE_ATTRIBUTE
#elif MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_8
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER     WEAK_IMPORT_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED
 *
 * Used on declarations introduced in Mac OS X 10.8,
 * and deprecated in Mac OS X 10.8
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED     __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_8, __MAC_10_8, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_8
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED    AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8
 *
 * Used on declarations introduced in Mac OS X 10.0,
 * but later deprecated in Mac OS X 10.8
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_8, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_8
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8
 *
 * Used on declarations introduced in Mac OS X 10.1,
 * but later deprecated in Mac OS X 10.8
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_1, __MAC_10_8, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_8
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8
 *
 * Used on declarations introduced in Mac OS X 10.2,
 * but later deprecated in Mac OS X 10.8
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_2, __MAC_10_8, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_8
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8
 *
 * Used on declarations introduced in Mac OS X 10.3,
 * but later deprecated in Mac OS X 10.8
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_3, __MAC_10_8, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_8
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8
 *
 * Used on declarations introduced in Mac OS X 10.4,
 * but later deprecated in Mac OS X 10.8
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_4, __MAC_10_8, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_8
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8
 *
 * Used on declarations introduced in Mac OS X 10.5,
 * but later deprecated in Mac OS X 10.8
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_5, __MAC_10_8, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_8
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8
 *
 * Used on declarations introduced in Mac OS X 10.6,
 * but later deprecated in Mac OS X 10.8
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_6, __MAC_10_8, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_8
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8
 *
 * Used on declarations introduced in Mac OS X 10.7,
 * but later deprecated in Mac OS X 10.8
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_7, __MAC_10_8, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_8
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_8    AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER
 * 
 * Used on declarations introduced in Mac OS X 10.9 
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER     __OSX_AVAILABLE_STARTING(__MAC_10_9, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_9
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER     UNAVAILABLE_ATTRIBUTE
#elif MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_9
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER     WEAK_IMPORT_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED
 *
 * Used on declarations introduced in Mac OS X 10.9,
 * and deprecated in Mac OS X 10.9
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED     __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_9, __MAC_10_9, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_9
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED    AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9
 *
 * Used on declarations introduced in Mac OS X 10.0,
 * but later deprecated in Mac OS X 10.9
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_9, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_9
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9
 *
 * Used on declarations introduced in Mac OS X 10.1,
 * but later deprecated in Mac OS X 10.9
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_1, __MAC_10_9, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_9
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9
 *
 * Used on declarations introduced in Mac OS X 10.2,
 * but later deprecated in Mac OS X 10.9
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_2, __MAC_10_9, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_9
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9
 *
 * Used on declarations introduced in Mac OS X 10.3,
 * but later deprecated in Mac OS X 10.9
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_3, __MAC_10_9, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_9
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9
 *
 * Used on declarations introduced in Mac OS X 10.4,
 * but later deprecated in Mac OS X 10.9
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_4, __MAC_10_9, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_9
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9
 *
 * Used on declarations introduced in Mac OS X 10.5,
 * but later deprecated in Mac OS X 10.9
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_5, __MAC_10_9, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_9
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9
 *
 * Used on declarations introduced in Mac OS X 10.6,
 * but later deprecated in Mac OS X 10.9
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_6, __MAC_10_9, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_9
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9
 *
 * Used on declarations introduced in Mac OS X 10.7,
 * but later deprecated in Mac OS X 10.9
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_7, __MAC_10_9, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_9
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9
 *
 * Used on declarations introduced in Mac OS X 10.8,
 * but later deprecated in Mac OS X 10.9
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_8, __MAC_10_9, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_9
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_9    AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER
 * 
 * Used on declarations introduced in Mac OS X 10.10 
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER     __OSX_AVAILABLE_STARTING(__MAC_10_10, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_10
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER     UNAVAILABLE_ATTRIBUTE
#elif MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_10
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER     WEAK_IMPORT_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED
 *
 * Used on declarations introduced in Mac OS X 10.10,
 * and deprecated in Mac OS X 10.10
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED     __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10, __MAC_10_10, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED    AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10
 *
 * Used on declarations introduced in Mac OS X 10.0,
 * but later deprecated in Mac OS X 10.10
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_10, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10
 *
 * Used on declarations introduced in Mac OS X 10.1,
 * but later deprecated in Mac OS X 10.10
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_1, __MAC_10_10, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10
 *
 * Used on declarations introduced in Mac OS X 10.2,
 * but later deprecated in Mac OS X 10.10
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_2, __MAC_10_10, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10
 *
 * Used on declarations introduced in Mac OS X 10.3,
 * but later deprecated in Mac OS X 10.10
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_3, __MAC_10_10, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10
 *
 * Used on declarations introduced in Mac OS X 10.4,
 * but later deprecated in Mac OS X 10.10
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_4, __MAC_10_10, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10
 *
 * Used on declarations introduced in Mac OS X 10.5,
 * but later deprecated in Mac OS X 10.10
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_5, __MAC_10_10, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10
 *
 * Used on declarations introduced in Mac OS X 10.6,
 * but later deprecated in Mac OS X 10.10
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_6, __MAC_10_10, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10
 *
 * Used on declarations introduced in Mac OS X 10.7,
 * but later deprecated in Mac OS X 10.10
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_7, __MAC_10_10, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10
 *
 * Used on declarations introduced in Mac OS X 10.8,
 * but later deprecated in Mac OS X 10.10
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_8, __MAC_10_10, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10
 *
 * Used on declarations introduced in Mac OS X 10.9,
 * but later deprecated in Mac OS X 10.10
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_9, __MAC_10_10, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10    AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER
 * 
 * Used on declarations introduced in Mac OS X 10.10.2 
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER     __OSX_AVAILABLE_STARTING(__MAC_10_10_2, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_10_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER     UNAVAILABLE_ATTRIBUTE
#elif MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_10_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER     WEAK_IMPORT_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED
 *
 * Used on declarations introduced in Mac OS X 10.10.2,
 * and deprecated in Mac OS X 10.10.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED     __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10_2, __MAC_10_10_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED    AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2
 *
 * Used on declarations introduced in Mac OS X 10.0,
 * but later deprecated in Mac OS X 10.10.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_10_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2
 *
 * Used on declarations introduced in Mac OS X 10.1,
 * but later deprecated in Mac OS X 10.10.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_1, __MAC_10_10_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2
 *
 * Used on declarations introduced in Mac OS X 10.2,
 * but later deprecated in Mac OS X 10.10.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_2, __MAC_10_10_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2
 *
 * Used on declarations introduced in Mac OS X 10.3,
 * but later deprecated in Mac OS X 10.10.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_3, __MAC_10_10_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2
 *
 * Used on declarations introduced in Mac OS X 10.4,
 * but later deprecated in Mac OS X 10.10.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_4, __MAC_10_10_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2
 *
 * Used on declarations introduced in Mac OS X 10.5,
 * but later deprecated in Mac OS X 10.10.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_5, __MAC_10_10_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2
 *
 * Used on declarations introduced in Mac OS X 10.6,
 * but later deprecated in Mac OS X 10.10.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_6, __MAC_10_10_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2
 *
 * Used on declarations introduced in Mac OS X 10.7,
 * but later deprecated in Mac OS X 10.10.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_7, __MAC_10_10_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2
 *
 * Used on declarations introduced in Mac OS X 10.8,
 * but later deprecated in Mac OS X 10.10.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_8, __MAC_10_10_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2
 *
 * Used on declarations introduced in Mac OS X 10.9,
 * but later deprecated in Mac OS X 10.10.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_9, __MAC_10_10_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2
 *
 * Used on declarations introduced in Mac OS X 10.10,
 * but later deprecated in Mac OS X 10.10.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10, __MAC_10_10_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_2    AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER
 * 
 * Used on declarations introduced in Mac OS X 10.10.3 
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER     __OSX_AVAILABLE_STARTING(__MAC_10_10_3, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_10_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER     UNAVAILABLE_ATTRIBUTE
#elif MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_10_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER     WEAK_IMPORT_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED
 *
 * Used on declarations introduced in Mac OS X 10.10.3,
 * and deprecated in Mac OS X 10.10.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED     __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10_3, __MAC_10_10_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED    AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3
 *
 * Used on declarations introduced in Mac OS X 10.0,
 * but later deprecated in Mac OS X 10.10.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_10_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3
 *
 * Used on declarations introduced in Mac OS X 10.1,
 * but later deprecated in Mac OS X 10.10.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_1, __MAC_10_10_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3
 *
 * Used on declarations introduced in Mac OS X 10.2,
 * but later deprecated in Mac OS X 10.10.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_2, __MAC_10_10_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3
 *
 * Used on declarations introduced in Mac OS X 10.3,
 * but later deprecated in Mac OS X 10.10.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_3, __MAC_10_10_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3
 *
 * Used on declarations introduced in Mac OS X 10.4,
 * but later deprecated in Mac OS X 10.10.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_4, __MAC_10_10_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3
 *
 * Used on declarations introduced in Mac OS X 10.5,
 * but later deprecated in Mac OS X 10.10.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_5, __MAC_10_10_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3
 *
 * Used on declarations introduced in Mac OS X 10.6,
 * but later deprecated in Mac OS X 10.10.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_6, __MAC_10_10_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3
 *
 * Used on declarations introduced in Mac OS X 10.7,
 * but later deprecated in Mac OS X 10.10.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_7, __MAC_10_10_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3
 *
 * Used on declarations introduced in Mac OS X 10.8,
 * but later deprecated in Mac OS X 10.10.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_8, __MAC_10_10_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3
 *
 * Used on declarations introduced in Mac OS X 10.9,
 * but later deprecated in Mac OS X 10.10.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_9, __MAC_10_10_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3
 *
 * Used on declarations introduced in Mac OS X 10.10,
 * but later deprecated in Mac OS X 10.10.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10, __MAC_10_10_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3
 *
 * Used on declarations introduced in Mac OS X 10.10.2,
 * but later deprecated in Mac OS X 10.10.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10_2, __MAC_10_10_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_10_3    AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER
 * 
 * Used on declarations introduced in Mac OS X 10.11 
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER     __OSX_AVAILABLE_STARTING(__MAC_10_11, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_11
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER     UNAVAILABLE_ATTRIBUTE
#elif MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_11
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER     WEAK_IMPORT_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED
 *
 * Used on declarations introduced in Mac OS X 10.11,
 * and deprecated in Mac OS X 10.11
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED     __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11, __MAC_10_11, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED    AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11
 *
 * Used on declarations introduced in Mac OS X 10.0,
 * but later deprecated in Mac OS X 10.11
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_11, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11
 *
 * Used on declarations introduced in Mac OS X 10.1,
 * but later deprecated in Mac OS X 10.11
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_1, __MAC_10_11, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11
 *
 * Used on declarations introduced in Mac OS X 10.2,
 * but later deprecated in Mac OS X 10.11
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_2, __MAC_10_11, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11
 *
 * Used on declarations introduced in Mac OS X 10.3,
 * but later deprecated in Mac OS X 10.11
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_3, __MAC_10_11, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11
 *
 * Used on declarations introduced in Mac OS X 10.4,
 * but later deprecated in Mac OS X 10.11
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_4, __MAC_10_11, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11
 *
 * Used on declarations introduced in Mac OS X 10.5,
 * but later deprecated in Mac OS X 10.11
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_5, __MAC_10_11, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11
 *
 * Used on declarations introduced in Mac OS X 10.6,
 * but later deprecated in Mac OS X 10.11
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_6, __MAC_10_11, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11
 *
 * Used on declarations introduced in Mac OS X 10.7,
 * but later deprecated in Mac OS X 10.11
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_7, __MAC_10_11, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11
 *
 * Used on declarations introduced in Mac OS X 10.8,
 * but later deprecated in Mac OS X 10.11
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_8, __MAC_10_11, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11
 *
 * Used on declarations introduced in Mac OS X 10.9,
 * but later deprecated in Mac OS X 10.11
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_9, __MAC_10_11, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11
 *
 * Used on declarations introduced in Mac OS X 10.10,
 * but later deprecated in Mac OS X 10.11
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10, __MAC_10_11, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11
 *
 * Used on declarations introduced in Mac OS X 10.10.2,
 * but later deprecated in Mac OS X 10.11
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10_2, __MAC_10_11, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11
 *
 * Used on declarations introduced in Mac OS X 10.10.3,
 * but later deprecated in Mac OS X 10.11
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10_3, __MAC_10_11, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11    AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER
 * 
 * Used on declarations introduced in Mac OS X 10.11.2 
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER     __OSX_AVAILABLE_STARTING(__MAC_10_11_2, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_11_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER     UNAVAILABLE_ATTRIBUTE
#elif MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_11_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER     WEAK_IMPORT_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED
 *
 * Used on declarations introduced in Mac OS X 10.11.2,
 * and deprecated in Mac OS X 10.11.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED     __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11_2, __MAC_10_11_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED    AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2
 *
 * Used on declarations introduced in Mac OS X 10.0,
 * but later deprecated in Mac OS X 10.11.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_11_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2
 *
 * Used on declarations introduced in Mac OS X 10.1,
 * but later deprecated in Mac OS X 10.11.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_1, __MAC_10_11_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2
 *
 * Used on declarations introduced in Mac OS X 10.2,
 * but later deprecated in Mac OS X 10.11.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_2, __MAC_10_11_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2
 *
 * Used on declarations introduced in Mac OS X 10.3,
 * but later deprecated in Mac OS X 10.11.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_3, __MAC_10_11_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2
 *
 * Used on declarations introduced in Mac OS X 10.4,
 * but later deprecated in Mac OS X 10.11.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_4, __MAC_10_11_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2
 *
 * Used on declarations introduced in Mac OS X 10.5,
 * but later deprecated in Mac OS X 10.11.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_5, __MAC_10_11_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2
 *
 * Used on declarations introduced in Mac OS X 10.6,
 * but later deprecated in Mac OS X 10.11.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_6, __MAC_10_11_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2
 *
 * Used on declarations introduced in Mac OS X 10.7,
 * but later deprecated in Mac OS X 10.11.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_7, __MAC_10_11_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2
 *
 * Used on declarations introduced in Mac OS X 10.8,
 * but later deprecated in Mac OS X 10.11.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_8, __MAC_10_11_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2
 *
 * Used on declarations introduced in Mac OS X 10.9,
 * but later deprecated in Mac OS X 10.11.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_9, __MAC_10_11_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2
 *
 * Used on declarations introduced in Mac OS X 10.10,
 * but later deprecated in Mac OS X 10.11.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10, __MAC_10_11_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2
 *
 * Used on declarations introduced in Mac OS X 10.10.2,
 * but later deprecated in Mac OS X 10.11.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10_2, __MAC_10_11_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2
 *
 * Used on declarations introduced in Mac OS X 10.10.3,
 * but later deprecated in Mac OS X 10.11.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10_3, __MAC_10_11_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2
 *
 * Used on declarations introduced in Mac OS X 10.11,
 * but later deprecated in Mac OS X 10.11.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11, __MAC_10_11_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_2    AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER
 * 
 * Used on declarations introduced in Mac OS X 10.11.3 
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER     __OSX_AVAILABLE_STARTING(__MAC_10_11_3, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_11_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER     UNAVAILABLE_ATTRIBUTE
#elif MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_11_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER     WEAK_IMPORT_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED
 *
 * Used on declarations introduced in Mac OS X 10.11.3,
 * and deprecated in Mac OS X 10.11.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED     __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11_3, __MAC_10_11_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED    AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3
 *
 * Used on declarations introduced in Mac OS X 10.0,
 * but later deprecated in Mac OS X 10.11.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_11_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3
 *
 * Used on declarations introduced in Mac OS X 10.1,
 * but later deprecated in Mac OS X 10.11.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_1, __MAC_10_11_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3
 *
 * Used on declarations introduced in Mac OS X 10.2,
 * but later deprecated in Mac OS X 10.11.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_2, __MAC_10_11_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3
 *
 * Used on declarations introduced in Mac OS X 10.3,
 * but later deprecated in Mac OS X 10.11.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_3, __MAC_10_11_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3
 *
 * Used on declarations introduced in Mac OS X 10.4,
 * but later deprecated in Mac OS X 10.11.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_4, __MAC_10_11_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3
 *
 * Used on declarations introduced in Mac OS X 10.5,
 * but later deprecated in Mac OS X 10.11.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_5, __MAC_10_11_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3
 *
 * Used on declarations introduced in Mac OS X 10.6,
 * but later deprecated in Mac OS X 10.11.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_6, __MAC_10_11_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3
 *
 * Used on declarations introduced in Mac OS X 10.7,
 * but later deprecated in Mac OS X 10.11.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_7, __MAC_10_11_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3
 *
 * Used on declarations introduced in Mac OS X 10.8,
 * but later deprecated in Mac OS X 10.11.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_8, __MAC_10_11_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3
 *
 * Used on declarations introduced in Mac OS X 10.9,
 * but later deprecated in Mac OS X 10.11.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_9, __MAC_10_11_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3
 *
 * Used on declarations introduced in Mac OS X 10.10,
 * but later deprecated in Mac OS X 10.11.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10, __MAC_10_11_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3
 *
 * Used on declarations introduced in Mac OS X 10.10.2,
 * but later deprecated in Mac OS X 10.11.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10_2, __MAC_10_11_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3
 *
 * Used on declarations introduced in Mac OS X 10.10.3,
 * but later deprecated in Mac OS X 10.11.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10_3, __MAC_10_11_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3
 *
 * Used on declarations introduced in Mac OS X 10.11,
 * but later deprecated in Mac OS X 10.11.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11, __MAC_10_11_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3
 *
 * Used on declarations introduced in Mac OS X 10.11.2,
 * but later deprecated in Mac OS X 10.11.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11_2, __MAC_10_11_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_3
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_3    AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER
 * 
 * Used on declarations introduced in Mac OS X 10.11.4 
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER     __OSX_AVAILABLE_STARTING(__MAC_10_11_4, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_11_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER     UNAVAILABLE_ATTRIBUTE
#elif MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_11_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER     WEAK_IMPORT_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER_BUT_DEPRECATED
 *
 * Used on declarations introduced in Mac OS X 10.11.4,
 * and deprecated in Mac OS X 10.11.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER_BUT_DEPRECATED     __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11_4, __MAC_10_11_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER_BUT_DEPRECATED    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER_BUT_DEPRECATED    AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4
 *
 * Used on declarations introduced in Mac OS X 10.0,
 * but later deprecated in Mac OS X 10.11.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_11_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4
 *
 * Used on declarations introduced in Mac OS X 10.1,
 * but later deprecated in Mac OS X 10.11.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_1, __MAC_10_11_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4
 *
 * Used on declarations introduced in Mac OS X 10.2,
 * but later deprecated in Mac OS X 10.11.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_2, __MAC_10_11_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4
 *
 * Used on declarations introduced in Mac OS X 10.3,
 * but later deprecated in Mac OS X 10.11.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_3, __MAC_10_11_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4
 *
 * Used on declarations introduced in Mac OS X 10.4,
 * but later deprecated in Mac OS X 10.11.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_4, __MAC_10_11_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4
 *
 * Used on declarations introduced in Mac OS X 10.5,
 * but later deprecated in Mac OS X 10.11.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_5, __MAC_10_11_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4
 *
 * Used on declarations introduced in Mac OS X 10.6,
 * but later deprecated in Mac OS X 10.11.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_6, __MAC_10_11_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4
 *
 * Used on declarations introduced in Mac OS X 10.7,
 * but later deprecated in Mac OS X 10.11.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_7, __MAC_10_11_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4
 *
 * Used on declarations introduced in Mac OS X 10.8,
 * but later deprecated in Mac OS X 10.11.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_8, __MAC_10_11_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4
 *
 * Used on declarations introduced in Mac OS X 10.9,
 * but later deprecated in Mac OS X 10.11.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_9, __MAC_10_11_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4
 *
 * Used on declarations introduced in Mac OS X 10.10,
 * but later deprecated in Mac OS X 10.11.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10, __MAC_10_11_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4
 *
 * Used on declarations introduced in Mac OS X 10.10.2,
 * but later deprecated in Mac OS X 10.11.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10_2, __MAC_10_11_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4
 *
 * Used on declarations introduced in Mac OS X 10.10.3,
 * but later deprecated in Mac OS X 10.11.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10_3, __MAC_10_11_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4
 *
 * Used on declarations introduced in Mac OS X 10.11,
 * but later deprecated in Mac OS X 10.11.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11, __MAC_10_11_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4
 *
 * Used on declarations introduced in Mac OS X 10.11.2,
 * but later deprecated in Mac OS X 10.11.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11_2, __MAC_10_11_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4
 *
 * Used on declarations introduced in Mac OS X 10.11.3,
 * but later deprecated in Mac OS X 10.11.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11_3, __MAC_10_11_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_11_4    AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER
 * 
 * Used on declarations introduced in Mac OS X 10.12 
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER     __OSX_AVAILABLE_STARTING(__MAC_10_12, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_12
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER     UNAVAILABLE_ATTRIBUTE
#elif MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_12
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER     WEAK_IMPORT_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER_BUT_DEPRECATED
 *
 * Used on declarations introduced in Mac OS X 10.12,
 * and deprecated in Mac OS X 10.12
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER_BUT_DEPRECATED     __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_12, __MAC_10_12, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER_BUT_DEPRECATED    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER_BUT_DEPRECATED    AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12
 *
 * Used on declarations introduced in Mac OS X 10.0,
 * but later deprecated in Mac OS X 10.12
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_12, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12
 *
 * Used on declarations introduced in Mac OS X 10.1,
 * but later deprecated in Mac OS X 10.12
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_1, __MAC_10_12, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12
 *
 * Used on declarations introduced in Mac OS X 10.2,
 * but later deprecated in Mac OS X 10.12
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_2, __MAC_10_12, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12
 *
 * Used on declarations introduced in Mac OS X 10.3,
 * but later deprecated in Mac OS X 10.12
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_3, __MAC_10_12, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12
 *
 * Used on declarations introduced in Mac OS X 10.4,
 * but later deprecated in Mac OS X 10.12
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_4, __MAC_10_12, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12
 *
 * Used on declarations introduced in Mac OS X 10.5,
 * but later deprecated in Mac OS X 10.12
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_5, __MAC_10_12, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12
 *
 * Used on declarations introduced in Mac OS X 10.6,
 * but later deprecated in Mac OS X 10.12
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_6, __MAC_10_12, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12
 *
 * Used on declarations introduced in Mac OS X 10.7,
 * but later deprecated in Mac OS X 10.12
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_7, __MAC_10_12, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12
 *
 * Used on declarations introduced in Mac OS X 10.8,
 * but later deprecated in Mac OS X 10.12
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_8, __MAC_10_12, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12
 *
 * Used on declarations introduced in Mac OS X 10.9,
 * but later deprecated in Mac OS X 10.12
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_9, __MAC_10_12, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12
 *
 * Used on declarations introduced in Mac OS X 10.10,
 * but later deprecated in Mac OS X 10.12
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10, __MAC_10_12, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12
 *
 * Used on declarations introduced in Mac OS X 10.10.2,
 * but later deprecated in Mac OS X 10.12
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10_2, __MAC_10_12, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12
 *
 * Used on declarations introduced in Mac OS X 10.10.3,
 * but later deprecated in Mac OS X 10.12
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10_3, __MAC_10_12, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12
 *
 * Used on declarations introduced in Mac OS X 10.11,
 * but later deprecated in Mac OS X 10.12
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11, __MAC_10_12, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12
 *
 * Used on declarations introduced in Mac OS X 10.11.2,
 * but later deprecated in Mac OS X 10.12
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11_2, __MAC_10_12, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12
 *
 * Used on declarations introduced in Mac OS X 10.11.3,
 * but later deprecated in Mac OS X 10.12
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11_3, __MAC_10_12, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12
 *
 * Used on declarations introduced in Mac OS X 10.11.4,
 * but later deprecated in Mac OS X 10.12
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11_4, __MAC_10_12, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12    AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_12_1_AND_LATER
 * 
 * Used on declarations introduced in Mac OS X 10.12.1 
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_1_AND_LATER     __OSX_AVAILABLE_STARTING(__MAC_10_12_1, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_12_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_1_AND_LATER     UNAVAILABLE_ATTRIBUTE
#elif MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_12_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_1_AND_LATER     WEAK_IMPORT_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_12_1_AND_LATER_BUT_DEPRECATED
 *
 * Used on declarations introduced in Mac OS X 10.12.1,
 * and deprecated in Mac OS X 10.12.1
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_1_AND_LATER_BUT_DEPRECATED     __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_12_1, __MAC_10_12_1, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_1_AND_LATER_BUT_DEPRECATED    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_1_AND_LATER_BUT_DEPRECATED    AVAILABLE_MAC_OS_X_VERSION_10_12_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1
 *
 * Used on declarations introduced in Mac OS X 10.0,
 * but later deprecated in Mac OS X 10.12.1
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_12_1, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1
 *
 * Used on declarations introduced in Mac OS X 10.1,
 * but later deprecated in Mac OS X 10.12.1
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_1, __MAC_10_12_1, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1
 *
 * Used on declarations introduced in Mac OS X 10.2,
 * but later deprecated in Mac OS X 10.12.1
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_2, __MAC_10_12_1, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1
 *
 * Used on declarations introduced in Mac OS X 10.3,
 * but later deprecated in Mac OS X 10.12.1
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_3, __MAC_10_12_1, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1
 *
 * Used on declarations introduced in Mac OS X 10.4,
 * but later deprecated in Mac OS X 10.12.1
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_4, __MAC_10_12_1, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1
 *
 * Used on declarations introduced in Mac OS X 10.5,
 * but later deprecated in Mac OS X 10.12.1
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_5, __MAC_10_12_1, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1
 *
 * Used on declarations introduced in Mac OS X 10.6,
 * but later deprecated in Mac OS X 10.12.1
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_6, __MAC_10_12_1, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1
 *
 * Used on declarations introduced in Mac OS X 10.7,
 * but later deprecated in Mac OS X 10.12.1
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_7, __MAC_10_12_1, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1
 *
 * Used on declarations introduced in Mac OS X 10.8,
 * but later deprecated in Mac OS X 10.12.1
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_8, __MAC_10_12_1, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1
 *
 * Used on declarations introduced in Mac OS X 10.9,
 * but later deprecated in Mac OS X 10.12.1
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_9, __MAC_10_12_1, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1
 *
 * Used on declarations introduced in Mac OS X 10.10,
 * but later deprecated in Mac OS X 10.12.1
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10, __MAC_10_12_1, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1
 *
 * Used on declarations introduced in Mac OS X 10.10.2,
 * but later deprecated in Mac OS X 10.12.1
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10_2, __MAC_10_12_1, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1
 *
 * Used on declarations introduced in Mac OS X 10.10.3,
 * but later deprecated in Mac OS X 10.12.1
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10_3, __MAC_10_12_1, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1
 *
 * Used on declarations introduced in Mac OS X 10.11,
 * but later deprecated in Mac OS X 10.12.1
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11, __MAC_10_12_1, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1
 *
 * Used on declarations introduced in Mac OS X 10.11.2,
 * but later deprecated in Mac OS X 10.12.1
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11_2, __MAC_10_12_1, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1
 *
 * Used on declarations introduced in Mac OS X 10.11.3,
 * but later deprecated in Mac OS X 10.12.1
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11_3, __MAC_10_12_1, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1
 *
 * Used on declarations introduced in Mac OS X 10.11.4,
 * but later deprecated in Mac OS X 10.12.1
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11_4, __MAC_10_12_1, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1
 *
 * Used on declarations introduced in Mac OS X 10.12,
 * but later deprecated in Mac OS X 10.12.1
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_12, __MAC_10_12_1, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_1
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_1    AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_12_2_AND_LATER
 * 
 * Used on declarations introduced in Mac OS X 10.12.2 
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_2_AND_LATER     __OSX_AVAILABLE_STARTING(__MAC_10_12_2, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_12_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_2_AND_LATER     UNAVAILABLE_ATTRIBUTE
#elif MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_12_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_2_AND_LATER     WEAK_IMPORT_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_12_2_AND_LATER_BUT_DEPRECATED
 *
 * Used on declarations introduced in Mac OS X 10.12.2,
 * and deprecated in Mac OS X 10.12.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_2_AND_LATER_BUT_DEPRECATED     __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_12_2, __MAC_10_12_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_2_AND_LATER_BUT_DEPRECATED    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_2_AND_LATER_BUT_DEPRECATED    AVAILABLE_MAC_OS_X_VERSION_10_12_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2
 *
 * Used on declarations introduced in Mac OS X 10.0,
 * but later deprecated in Mac OS X 10.12.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_12_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2
 *
 * Used on declarations introduced in Mac OS X 10.1,
 * but later deprecated in Mac OS X 10.12.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_1, __MAC_10_12_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2
 *
 * Used on declarations introduced in Mac OS X 10.2,
 * but later deprecated in Mac OS X 10.12.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_2, __MAC_10_12_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2
 *
 * Used on declarations introduced in Mac OS X 10.3,
 * but later deprecated in Mac OS X 10.12.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_3, __MAC_10_12_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2
 *
 * Used on declarations introduced in Mac OS X 10.4,
 * but later deprecated in Mac OS X 10.12.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_4, __MAC_10_12_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2
 *
 * Used on declarations introduced in Mac OS X 10.5,
 * but later deprecated in Mac OS X 10.12.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_5, __MAC_10_12_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2
 *
 * Used on declarations introduced in Mac OS X 10.6,
 * but later deprecated in Mac OS X 10.12.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_6, __MAC_10_12_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2
 *
 * Used on declarations introduced in Mac OS X 10.7,
 * but later deprecated in Mac OS X 10.12.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_7, __MAC_10_12_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2
 *
 * Used on declarations introduced in Mac OS X 10.8,
 * but later deprecated in Mac OS X 10.12.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_8, __MAC_10_12_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2
 *
 * Used on declarations introduced in Mac OS X 10.9,
 * but later deprecated in Mac OS X 10.12.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_9, __MAC_10_12_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2
 *
 * Used on declarations introduced in Mac OS X 10.10,
 * but later deprecated in Mac OS X 10.12.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10, __MAC_10_12_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2
 *
 * Used on declarations introduced in Mac OS X 10.10.2,
 * but later deprecated in Mac OS X 10.12.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10_2, __MAC_10_12_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2
 *
 * Used on declarations introduced in Mac OS X 10.10.3,
 * but later deprecated in Mac OS X 10.12.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10_3, __MAC_10_12_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2
 *
 * Used on declarations introduced in Mac OS X 10.11,
 * but later deprecated in Mac OS X 10.12.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11, __MAC_10_12_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2
 *
 * Used on declarations introduced in Mac OS X 10.11.2,
 * but later deprecated in Mac OS X 10.12.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11_2, __MAC_10_12_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2
 *
 * Used on declarations introduced in Mac OS X 10.11.3,
 * but later deprecated in Mac OS X 10.12.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11_3, __MAC_10_12_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2
 *
 * Used on declarations introduced in Mac OS X 10.11.4,
 * but later deprecated in Mac OS X 10.12.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11_4, __MAC_10_12_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2
 *
 * Used on declarations introduced in Mac OS X 10.12,
 * but later deprecated in Mac OS X 10.12.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_12, __MAC_10_12_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_12_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2
 *
 * Used on declarations introduced in Mac OS X 10.12.1,
 * but later deprecated in Mac OS X 10.12.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_12_1, __MAC_10_12_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_2
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_2    AVAILABLE_MAC_OS_X_VERSION_10_12_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_12_4_AND_LATER
 * 
 * Used on declarations introduced in Mac OS X 10.12.4 
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_4_AND_LATER     __OSX_AVAILABLE_STARTING(__MAC_10_12_4, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_12_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_4_AND_LATER     UNAVAILABLE_ATTRIBUTE
#elif MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_12_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_4_AND_LATER     WEAK_IMPORT_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_12_4_AND_LATER_BUT_DEPRECATED
 *
 * Used on declarations introduced in Mac OS X 10.12.4,
 * and deprecated in Mac OS X 10.12.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_4_AND_LATER_BUT_DEPRECATED     __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_12_4, __MAC_10_12_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_4_AND_LATER_BUT_DEPRECATED    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_4_AND_LATER_BUT_DEPRECATED    AVAILABLE_MAC_OS_X_VERSION_10_12_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4
 *
 * Used on declarations introduced in Mac OS X 10.0,
 * but later deprecated in Mac OS X 10.12.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_12_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    AVAILABLE_MAC_OS_X_VERSION_10_0_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4
 *
 * Used on declarations introduced in Mac OS X 10.1,
 * but later deprecated in Mac OS X 10.12.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_1, __MAC_10_12_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    AVAILABLE_MAC_OS_X_VERSION_10_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4
 *
 * Used on declarations introduced in Mac OS X 10.2,
 * but later deprecated in Mac OS X 10.12.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_2, __MAC_10_12_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    AVAILABLE_MAC_OS_X_VERSION_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4
 *
 * Used on declarations introduced in Mac OS X 10.3,
 * but later deprecated in Mac OS X 10.12.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_3, __MAC_10_12_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    AVAILABLE_MAC_OS_X_VERSION_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4
 *
 * Used on declarations introduced in Mac OS X 10.4,
 * but later deprecated in Mac OS X 10.12.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_4, __MAC_10_12_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    AVAILABLE_MAC_OS_X_VERSION_10_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4
 *
 * Used on declarations introduced in Mac OS X 10.5,
 * but later deprecated in Mac OS X 10.12.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_5, __MAC_10_12_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    AVAILABLE_MAC_OS_X_VERSION_10_5_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4
 *
 * Used on declarations introduced in Mac OS X 10.6,
 * but later deprecated in Mac OS X 10.12.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_6, __MAC_10_12_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    AVAILABLE_MAC_OS_X_VERSION_10_6_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4
 *
 * Used on declarations introduced in Mac OS X 10.7,
 * but later deprecated in Mac OS X 10.12.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_7, __MAC_10_12_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    AVAILABLE_MAC_OS_X_VERSION_10_7_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4
 *
 * Used on declarations introduced in Mac OS X 10.8,
 * but later deprecated in Mac OS X 10.12.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_8, __MAC_10_12_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    AVAILABLE_MAC_OS_X_VERSION_10_8_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4
 *
 * Used on declarations introduced in Mac OS X 10.9,
 * but later deprecated in Mac OS X 10.12.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_9, __MAC_10_12_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    AVAILABLE_MAC_OS_X_VERSION_10_9_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4
 *
 * Used on declarations introduced in Mac OS X 10.10,
 * but later deprecated in Mac OS X 10.12.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10, __MAC_10_12_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    AVAILABLE_MAC_OS_X_VERSION_10_10_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4
 *
 * Used on declarations introduced in Mac OS X 10.10.2,
 * but later deprecated in Mac OS X 10.12.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10_2, __MAC_10_12_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    AVAILABLE_MAC_OS_X_VERSION_10_10_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4
 *
 * Used on declarations introduced in Mac OS X 10.10.3,
 * but later deprecated in Mac OS X 10.12.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_10_3, __MAC_10_12_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    AVAILABLE_MAC_OS_X_VERSION_10_10_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4
 *
 * Used on declarations introduced in Mac OS X 10.11,
 * but later deprecated in Mac OS X 10.12.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11, __MAC_10_12_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    AVAILABLE_MAC_OS_X_VERSION_10_11_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4
 *
 * Used on declarations introduced in Mac OS X 10.11.2,
 * but later deprecated in Mac OS X 10.12.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11_2, __MAC_10_12_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    AVAILABLE_MAC_OS_X_VERSION_10_11_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4
 *
 * Used on declarations introduced in Mac OS X 10.11.3,
 * but later deprecated in Mac OS X 10.12.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11_3, __MAC_10_12_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    AVAILABLE_MAC_OS_X_VERSION_10_11_3_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4
 *
 * Used on declarations introduced in Mac OS X 10.11.4,
 * but later deprecated in Mac OS X 10.12.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_11_4, __MAC_10_12_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    AVAILABLE_MAC_OS_X_VERSION_10_11_4_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4
 *
 * Used on declarations introduced in Mac OS X 10.12,
 * but later deprecated in Mac OS X 10.12.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_12, __MAC_10_12_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    AVAILABLE_MAC_OS_X_VERSION_10_12_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_12_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4
 *
 * Used on declarations introduced in Mac OS X 10.12.1,
 * but later deprecated in Mac OS X 10.12.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_12_1, __MAC_10_12_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_1_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    AVAILABLE_MAC_OS_X_VERSION_10_12_1_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_12_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4
 *
 * Used on declarations introduced in Mac OS X 10.12.2,
 * but later deprecated in Mac OS X 10.12.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_12_2, __MAC_10_12_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12_4
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    DEPRECATED_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_12_2_AND_LATER_BUT_DEPRECATED_IN_MAC_OS_X_VERSION_10_12_4    AVAILABLE_MAC_OS_X_VERSION_10_12_2_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_13_AND_LATER
 * 
 * Used on declarations introduced in Mac OS X 10.13 
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_13_AND_LATER     __OSX_AVAILABLE_STARTING(__MAC_10_13, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_13
    #define AVAILABLE_MAC_OS_X_VERSION_10_13_AND_LATER     UNAVAILABLE_ATTRIBUTE
#elif MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_13
    #define AVAILABLE_MAC_OS_X_VERSION_10_13_AND_LATER     WEAK_IMPORT_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_13_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_14_AND_LATER
 * 
 * Used on declarations introduced in Mac OS X 10.14 
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_14_AND_LATER     __OSX_AVAILABLE_STARTING(__MAC_10_14, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_14
    #define AVAILABLE_MAC_OS_X_VERSION_10_14_AND_LATER     UNAVAILABLE_ATTRIBUTE
#elif MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_14
    #define AVAILABLE_MAC_OS_X_VERSION_10_14_AND_LATER     WEAK_IMPORT_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_14_AND_LATER
#endif

/*
 * AVAILABLE_MAC_OS_X_VERSION_10_15_AND_LATER
 * 
 * Used on declarations introduced in Mac OS X 10.15 
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define AVAILABLE_MAC_OS_X_VERSION_10_15_AND_LATER     __OSX_AVAILABLE_STARTING(__MAC_10_15, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MAX_ALLOWED < MAC_OS_X_VERSION_10_15
    #define AVAILABLE_MAC_OS_X_VERSION_10_15_AND_LATER     UNAVAILABLE_ATTRIBUTE
#elif MAC_OS_X_VERSION_MIN_REQUIRED < MAC_OS_X_VERSION_10_15
    #define AVAILABLE_MAC_OS_X_VERSION_10_15_AND_LATER     WEAK_IMPORT_ATTRIBUTE
#else
    #define AVAILABLE_MAC_OS_X_VERSION_10_15_AND_LATER
#endif

/*
 * DEPRECATED_IN_MAC_OS_X_VERSION_10_1_AND_LATER
 *
 * Used on types deprecated in Mac OS X 10.1
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_1_AND_LATER    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_1, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_1
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_1_AND_LATER    DEPRECATED_ATTRIBUTE
#else
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_1_AND_LATER
#endif

/*
 * DEPRECATED_IN_MAC_OS_X_VERSION_10_2_AND_LATER
 *
 * Used on types deprecated in Mac OS X 10.2
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_2_AND_LATER    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_2, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_2
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_2_AND_LATER    DEPRECATED_ATTRIBUTE
#else
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_2_AND_LATER
#endif

/*
 * DEPRECATED_IN_MAC_OS_X_VERSION_10_3_AND_LATER
 *
 * Used on types deprecated in Mac OS X 10.3
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_3_AND_LATER    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_3, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_3
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_3_AND_LATER    DEPRECATED_ATTRIBUTE
#else
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_3_AND_LATER
#endif

/*
 * DEPRECATED_IN_MAC_OS_X_VERSION_10_4_AND_LATER
 *
 * Used on types deprecated in Mac OS X 10.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_4_AND_LATER    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_4
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_4_AND_LATER    DEPRECATED_ATTRIBUTE
#else
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_4_AND_LATER
#endif


/*
 * DEPRECATED_IN_MAC_OS_X_VERSION_10_5_AND_LATER
 *
 * Used on types deprecated in Mac OS X 10.5
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_5_AND_LATER    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_5, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_5
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_5_AND_LATER    DEPRECATED_ATTRIBUTE
#else
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_5_AND_LATER
#endif

/*
 * DEPRECATED_IN_MAC_OS_X_VERSION_10_6_AND_LATER
 *
 * Used on types deprecated in Mac OS X 10.6
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_6_AND_LATER    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_6, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_6
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_6_AND_LATER    DEPRECATED_ATTRIBUTE
#else
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_6_AND_LATER
#endif

/*
 * DEPRECATED_IN_MAC_OS_X_VERSION_10_7_AND_LATER
 *
 * Used on types deprecated in Mac OS X 10.7
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_7_AND_LATER    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_7, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_7
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_7_AND_LATER    DEPRECATED_ATTRIBUTE
#else
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_7_AND_LATER
#endif

/*
 * DEPRECATED_IN_MAC_OS_X_VERSION_10_8_AND_LATER
 *
 * Used on types deprecated in Mac OS X 10.8
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_8_AND_LATER    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_8, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_8
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_8_AND_LATER    DEPRECATED_ATTRIBUTE
#else
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_8_AND_LATER
#endif

/*
 * DEPRECATED_IN_MAC_OS_X_VERSION_10_9_AND_LATER
 *
 * Used on types deprecated in Mac OS X 10.9
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_9_AND_LATER    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_9, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_9
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_9_AND_LATER    DEPRECATED_ATTRIBUTE
#else
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_9_AND_LATER
#endif

/*
 * DEPRECATED_IN_MAC_OS_X_VERSION_10_10_AND_LATER
 *
 * Used on types deprecated in Mac OS X 10.10
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_10_AND_LATER    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_10, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_10
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_10_AND_LATER    DEPRECATED_ATTRIBUTE
#else
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_10_AND_LATER
#endif

/*
 * DEPRECATED_IN_MAC_OS_X_VERSION_10_11_AND_LATER
 *
 * Used on types deprecated in Mac OS X 10.11
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_11_AND_LATER    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_11, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_11
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_11_AND_LATER    DEPRECATED_ATTRIBUTE
#else
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_11_AND_LATER
#endif

/*
 * DEPRECATED_IN_MAC_OS_X_VERSION_10_12_AND_LATER
 *
 * Used on types deprecated in Mac OS X 10.12
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_12_AND_LATER    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_12, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_12_AND_LATER    DEPRECATED_ATTRIBUTE
#else
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_12_AND_LATER
#endif

/*
 * DEPRECATED_IN_MAC_OS_X_VERSION_10_13_AND_LATER
 *
 * Used on types deprecated in Mac OS X 10.13
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_13_AND_LATER    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_13, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_12
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_13_AND_LATER    DEPRECATED_ATTRIBUTE
#else
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_13_AND_LATER
#endif

/*
 * DEPRECATED_IN_MAC_OS_X_VERSION_10_14_4_AND_LATER
 *
 * Used on types deprecated in Mac OS X 10.14.4
 */
#if __AVAILABILITY_MACROS_USES_AVAILABILITY
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_14_4_AND_LATER    __OSX_AVAILABLE_BUT_DEPRECATED(__MAC_10_0, __MAC_10_14_4, __IPHONE_COMPAT_VERSION, __IPHONE_COMPAT_VERSION)
#elif MAC_OS_X_VERSION_MIN_REQUIRED >= MAC_OS_X_VERSION_10_14_4
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_14_4_AND_LATER    DEPRECATED_ATTRIBUTE
#else
    #define DEPRECATED_IN_MAC_OS_X_VERSION_10_14_4_AND_LATER
#endif

#endif  /* __AVAILABILITYMACROS__ */