<?php

declare(strict_types=1);

/*
 * Copyright (c) 2023 François Kooman <fkooman@tuxed.net>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

namespace fkooman\Radius;

class AccessResponse
{
    // @see https://www.iana.org/assignments/radius-types/radius-types.xhtml
    private const TEXT_ATTRIBUTES = [1, 11, 18, 19, 20, 22, 30, 31, 32, 34, 35, 39, 63, 74, 77, 78, 87, 88];

    private const CODE_MAPPING = [
        1 => 'Access-Request',
        2 => 'Access-Accept',
        3 => 'Access-Reject',
        4 => 'Accounting-Request',
        5 => 'Accounting-Response',
        11 => 'Access-Challenge',
    ];

    private const ATTRIBUTE_MAPPING = [
        // @see https://www.rfc-editor.org/rfc/rfc2865.html
        1 => 'User-Name',
        2 => 'User-Password',
        3 => 'CHAP-Password',
        4 => 'NAS-IP-Address',
        5 => 'NAS-Port',
        6 => 'Service-Type',
        7 => 'Framed-Protocol',
        8 => 'Framed-IP-Address',
        9 => 'Framed-IP-Netmask',
        10 => 'Framed-Routing',
        11 => 'Filter-Id',
        12 => 'Framed-MTU',
        13 => 'Framed-Compression',
        14 => 'Login-IP-Host',
        15 => 'Login-Service',
        16 => 'Login-TCP-Port',
        18 => 'Reply-Message',
        19 => 'Callback-Number',
        20 => 'Callback-Id',
        22 => 'Framed-Route',
        23 => 'Framed-IPX-Network',
        24 => 'State',
        25 => 'Class',
        26 => 'Vendor-Specific',
        27 => 'Session-Timeout',
        28 => 'Idle-Timeout',
        29 => 'Termination-Action',
        30 => 'Called-Station-Id',
        31 => 'Calling-Station-Id',
        32 => 'NAS-Identifier',
        33 => 'Proxy-State',
        34 => 'Login-LAT-Service',
        35 => 'Login-LAT-Node',
        36 => 'Login-LAT-Group',
        37 => 'Framed-AppleTalk-Link',
        38 => 'Framed-AppleTalk-Network',
        39 => 'Framed-AppleTalk-Zone',
        60 => 'CHAP-Challenge',
        61 => 'NAS-Port-Type',
        62 => 'Port-Limit',
        63 => 'Login-LAT-Port',
        // @see https://www.rfc-editor.org/rfc/rfc2869.html
        52 => 'Acct-Input-Gigawords',
        53 => 'Acct-Output-Gigawords',
        55 => 'Event-Timestamp',
        70 => 'ARAP-Password',
        71 => 'ARAP-Features',
        72 => 'ARAP-Zone-Access',
        73 => 'ARAP-Security',
        74 => 'ARAP-Security-Data',
        75 => 'Password-Retry',
        76 => 'Prompt',
        77 => 'Connect-Info',
        78 => 'Configuration-Token',
        79 => 'EAP-Message',
        80 => 'Message-Authenticator',
        84 => 'ARAP-Challenge-Response',
        85 => 'Acct-Interim-Interval',
        87 => 'NAS-Port-Id',
        88 => 'Framed-Pool',
    ];

    private int $responseCode;

    /** @var array<array{int,int,string}> */
    private array $attributeList;

    /**
     * @param array<array{int,int,string}> $attributeList
     */
    public function __construct(int $responseCode, array $attributeList)
    {
        $this->responseCode = $responseCode;
        $this->attributeList = $attributeList;
    }

    public function __toString(): string
    {
        $outStr = sprintf('Code: %s (%d)', $this->responseCode(), $this->responseCodeValue()) . PHP_EOL;
        if (0 !== count($this->attributeList)) {
            $outStr .= 'Attributes:' . PHP_EOL;
            foreach ($this->attributeList as $attributeInfo) {
                [$t,,$v] = $attributeInfo;
                // if the attribute value is not text, hex encode it
                if (!in_array($t, self::TEXT_ATTRIBUTES, true)) {
                    $v = bin2hex($v);
                }
                if (array_key_exists($t, self::ATTRIBUTE_MAPPING)) {
                    $outStr .= sprintf("\t%s (%d) => %s\n", self::ATTRIBUTE_MAPPING[$t], $t, $v);

                    continue;
                }

                $outStr .= sprintf("\t? (%d) => %s\n", $t, $v);
            }
        }

        return $outStr;
    }

    public function responseCodeValue(): int
    {
        return $this->responseCode;
    }

    public function responseCode(): string
    {
        if (array_key_exists($this->responseCode, self::CODE_MAPPING)) {
            return self::CODE_MAPPING[$this->responseCode];
        }

        return '?' . $this->responseCode . '?';
    }

    /**
     * @return array<array{int,string}>
     */
    public function attributeList(): array
    {
        $attributeList = [];
        foreach ($this->attributeList as $attributeInfo) {
            [$t,,$v] = $attributeInfo;
            $attributeList[] = [$t, $v];
        }

        return $attributeList;
    }

    /**
     * Get the first attribute from the list with this type.
     * XXX should we return all of them?!
     */
    public function attributeByType(int $attributeType): ?string
    {
        foreach ($this->attributeList as $attributeInfo) {
            [$t,, $v] = $attributeInfo;
            if ($attributeType === $t) {
                return $v;
            }
        }

        return null;
    }

    /**
     * Get the first attribute from the list with this name.
     * XXX should we return all of them?!
     */
    public function attributeByName(string $attributeName): ?string
    {
        // find out whether we know this name
        $keyValue = array_search($attributeName, self::ATTRIBUTE_MAPPING, true);
        if (!is_int($keyValue)) {
            // we do not know this name
            return null;
        }

        // figure out whether we have the attribute
        foreach ($this->attributeList as $attributeInfo) {
            [$t,, $v] = $attributeInfo;
            if ($keyValue === $t) {
                return $v;
            }
        }

        return null;
    }
}
