/*
 * Decompiled with CFR 0.152.
 */
package org.traccar.handler;

import jakarta.inject.Inject;
import java.util.Date;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.traccar.config.Config;
import org.traccar.config.Keys;
import org.traccar.database.StatisticsManager;
import org.traccar.handler.BasePositionHandler;
import org.traccar.helper.UnitsConverter;
import org.traccar.helper.model.AttributeUtil;
import org.traccar.model.Calendar;
import org.traccar.model.Device;
import org.traccar.model.Position;
import org.traccar.session.cache.CacheManager;
import org.traccar.storage.Storage;
import org.traccar.storage.StorageException;
import org.traccar.storage.query.Columns;
import org.traccar.storage.query.Condition;
import org.traccar.storage.query.Order;
import org.traccar.storage.query.Request;

public class FilterHandler
extends BasePositionHandler {
    private static final Logger LOGGER = LoggerFactory.getLogger(FilterHandler.class);
    private final boolean filterInvalid;
    private final boolean filterZero;
    private final boolean filterDuplicate;
    private final boolean filterOutdated;
    private final long filterFuture;
    private final long filterPast;
    private final boolean filterApproximate;
    private final int filterAccuracy;
    private final boolean filterStatic;
    private final int filterDistance;
    private final int filterMaxSpeed;
    private final long filterMinPeriod;
    private final int filterDailyLimit;
    private final long filterDailyLimitInterval;
    private final boolean filterRelative;
    private final long skipLimit;
    private final boolean skipAttributes;
    private final CacheManager cacheManager;
    private final Storage storage;
    private final StatisticsManager statisticsManager;

    @Inject
    public FilterHandler(Config config, CacheManager cacheManager, Storage storage, StatisticsManager statisticsManager) {
        this.filterInvalid = config.getBoolean(Keys.FILTER_INVALID);
        this.filterZero = config.getBoolean(Keys.FILTER_ZERO);
        this.filterDuplicate = config.getBoolean(Keys.FILTER_DUPLICATE);
        this.filterOutdated = config.getBoolean(Keys.FILTER_OUTDATED);
        this.filterFuture = config.getLong(Keys.FILTER_FUTURE) * 1000L;
        this.filterPast = config.getLong(Keys.FILTER_PAST) * 1000L;
        this.filterAccuracy = config.getInteger(Keys.FILTER_ACCURACY);
        this.filterApproximate = config.getBoolean(Keys.FILTER_APPROXIMATE);
        this.filterStatic = config.getBoolean(Keys.FILTER_STATIC);
        this.filterDistance = config.getInteger(Keys.FILTER_DISTANCE);
        this.filterMaxSpeed = config.getInteger(Keys.FILTER_MAX_SPEED);
        this.filterMinPeriod = (long)config.getInteger(Keys.FILTER_MIN_PERIOD) * 1000L;
        this.filterDailyLimit = config.getInteger(Keys.FILTER_DAILY_LIMIT);
        this.filterDailyLimitInterval = (long)config.getInteger(Keys.FILTER_DAILY_LIMIT_INTERVAL) * 1000L;
        this.filterRelative = config.getBoolean(Keys.FILTER_RELATIVE);
        this.skipLimit = config.getLong(Keys.FILTER_SKIP_LIMIT) * 1000L;
        this.skipAttributes = config.getBoolean(Keys.FILTER_SKIP_ATTRIBUTES_ENABLE);
        this.cacheManager = cacheManager;
        this.storage = storage;
        this.statisticsManager = statisticsManager;
    }

    private Position getPrecedingPosition(long deviceId, Date date) throws StorageException {
        return this.storage.getObject(Position.class, new Request(new Columns.All(), new Condition.And(new Condition.Equals("deviceId", deviceId), new Condition.Compare("fixTime", "<=", "time", date)), new Order("fixTime", true, 1)));
    }

    private boolean filterInvalid(Position position) {
        return this.filterInvalid && (!position.getValid() || position.getLatitude() > 90.0 || position.getLongitude() > 180.0 || position.getLatitude() < -90.0 || position.getLongitude() < -180.0);
    }

    private boolean filterZero(Position position) {
        return this.filterZero && position.getLatitude() == 0.0 && position.getLongitude() == 0.0;
    }

    private boolean filterDuplicate(Position position, Position last) {
        if (this.filterDuplicate && last != null && position.getFixTime().equals(last.getFixTime())) {
            for (String key : position.getAttributes().keySet()) {
                if (last.hasAttribute(key)) continue;
                return false;
            }
            return true;
        }
        return false;
    }

    private boolean filterOutdated(Position position) {
        return this.filterOutdated && position.getOutdated();
    }

    private boolean filterFuture(Position position) {
        return this.filterFuture != 0L && position.getFixTime().getTime() > System.currentTimeMillis() + this.filterFuture;
    }

    private boolean filterPast(Position position) {
        return this.filterPast != 0L && position.getFixTime().getTime() < System.currentTimeMillis() - this.filterPast;
    }

    private boolean filterAccuracy(Position position) {
        return this.filterAccuracy != 0 && position.getAccuracy() > (double)this.filterAccuracy;
    }

    private boolean filterApproximate(Position position) {
        return this.filterApproximate && position.getBoolean("approximate");
    }

    private boolean filterStatic(Position position) {
        return this.filterStatic && position.getSpeed() == 0.0;
    }

    private boolean filterDistance(Position position, Position last) {
        if (this.filterDistance != 0 && last != null) {
            return position.getDouble("distance") < (double)this.filterDistance;
        }
        return false;
    }

    private boolean filterMaxSpeed(Position position, Position last) {
        if (this.filterMaxSpeed != 0 && last != null) {
            double distance = position.getDouble("distance");
            double time = position.getFixTime().getTime() - last.getFixTime().getTime();
            return time > 0.0 && UnitsConverter.knotsFromMps(distance / (time / 1000.0)) > (double)this.filterMaxSpeed;
        }
        return false;
    }

    private boolean filterMinPeriod(Position position, Position last) {
        if (this.filterMinPeriod != 0L && last != null) {
            long time = position.getFixTime().getTime() - last.getFixTime().getTime();
            return time > 0L && time < this.filterMinPeriod;
        }
        return false;
    }

    private boolean filterDailyLimit(Position position, Position last) {
        if (this.filterDailyLimit != 0 && this.statisticsManager.messageStoredCount(position.getDeviceId()) >= this.filterDailyLimit) {
            long lastTime = last != null ? last.getFixTime().getTime() : 0L;
            long interval = position.getFixTime().getTime() - lastTime;
            return this.filterDailyLimitInterval <= 0L || interval < this.filterDailyLimitInterval;
        }
        return false;
    }

    private boolean skipLimit(Position position, Position last) {
        if (this.skipLimit != 0L && last != null) {
            return position.getServerTime().getTime() - last.getServerTime().getTime() > this.skipLimit;
        }
        return false;
    }

    private boolean skipAttributes(Position position) {
        if (this.skipAttributes) {
            String string = AttributeUtil.lookup(this.cacheManager, Keys.FILTER_SKIP_ATTRIBUTES, position.getDeviceId());
            for (String attribute : string.split("[ ,]")) {
                if (!position.hasAttribute(attribute)) continue;
                return true;
            }
        }
        return false;
    }

    protected boolean filter(Position position) {
        Calendar calendar;
        Device device;
        StringBuilder filterType = new StringBuilder();
        if (this.filterInvalid(position)) {
            filterType.append("Invalid ");
        }
        if (this.filterZero(position)) {
            filterType.append("Zero ");
        }
        if (this.filterOutdated(position)) {
            filterType.append("Outdated ");
        }
        if (this.filterFuture(position)) {
            filterType.append("Future ");
        }
        if (this.filterPast(position)) {
            filterType.append("Past ");
        }
        if (this.filterAccuracy(position)) {
            filterType.append("Accuracy ");
        }
        if (this.filterApproximate(position)) {
            filterType.append("Approximate ");
        }
        long deviceId = position.getDeviceId();
        if (this.filterDuplicate || this.filterStatic || this.filterDistance > 0 || this.filterMaxSpeed > 0 || this.filterMinPeriod > 0L || this.filterDailyLimit > 0) {
            Position preceding;
            if (this.filterRelative) {
                try {
                    Date newFixTime = position.getFixTime();
                    preceding = this.getPrecedingPosition(deviceId, newFixTime);
                }
                catch (StorageException e) {
                    LOGGER.warn("Error retrieving preceding position; fall backing to last received position.", (Throwable)e);
                    preceding = this.cacheManager.getPosition(deviceId);
                }
            } else {
                preceding = this.cacheManager.getPosition(deviceId);
            }
            if (this.filterDuplicate(position, preceding) && !this.skipLimit(position, preceding) && !this.skipAttributes(position)) {
                filterType.append("Duplicate ");
            }
            if (this.filterStatic(position) && !this.skipLimit(position, preceding) && !this.skipAttributes(position)) {
                filterType.append("Static ");
            }
            if (this.filterDistance(position, preceding) && !this.skipLimit(position, preceding) && !this.skipAttributes(position)) {
                filterType.append("Distance ");
            }
            if (this.filterMaxSpeed(position, preceding)) {
                filterType.append("MaxSpeed ");
            }
            if (this.filterMinPeriod(position, preceding)) {
                filterType.append("MinPeriod ");
            }
            if (this.filterDailyLimit(position, preceding)) {
                filterType.append("DailyLimit ");
            }
        }
        if ((device = this.cacheManager.getObject(Device.class, deviceId)).getCalendarId() > 0L && !(calendar = this.cacheManager.getObject(Calendar.class, device.getCalendarId())).checkMoment(position.getFixTime())) {
            filterType.append("Calendar ");
        }
        if (filterType.length() > 0) {
            LOGGER.info("Position filtered by {}filters from device: {}", (Object)filterType, (Object)device.getUniqueId());
            return true;
        }
        return false;
    }

    @Override
    public void handlePosition(Position position, BasePositionHandler.Callback callback) {
        callback.processed(this.filter(position));
    }
}

