/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.bmc.http.internal;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalTime;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.DateTimeParseException;
import java.time.temporal.ChronoField;
import java.time.temporal.TemporalQueries;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.TimeZone;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HttpDateUtils {
    private static final Logger LOG = LoggerFactory.getLogger(HttpDateUtils.class);
    private static final TimeZone TIMEZONE_GMT = TimeZone.getTimeZone("GMT");
    private static final Date AMBIGUOUS_YEAR_TWO_DIGIT_YEAR_START;
    private static final ThreadLocal<List<DateFormat>> RFC2616_DATE_FORMATS;
    private static DateTimeFormatter ISO_INSTANT_INCLUDE_MILLIS;

    private static List<DateFormat> createRfc2616DateFormats() {
        ArrayList<DateFormat> dateFormats = new ArrayList<DateFormat>();
        dateFormats.add(new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss z", Locale.US));
        SimpleDateFormat rfc1036Format = new SimpleDateFormat("EEE, dd-MMM-yy HH:mm:ss zzz", Locale.US);
        rfc1036Format.set2DigitYearStart(AMBIGUOUS_YEAR_TWO_DIGIT_YEAR_START);
        dateFormats.add(rfc1036Format);
        dateFormats.add(new SimpleDateFormat("EEE MMM d HH:mm:ss yyyy", Locale.US));
        HttpDateUtils.init(dateFormats);
        return dateFormats;
    }

    private static void init(List<DateFormat> formats) {
        for (DateFormat format : formats) {
            format.setTimeZone(TIMEZONE_GMT);
        }
    }

    static Date parse(String headerName, String value) {
        String lowerCasedHeaderName = headerName.toLowerCase(Locale.ROOT);
        if ("date".equals(lowerCasedHeaderName) || "last-modified".equals(lowerCasedHeaderName)) {
            Date date = HttpDateUtils.tryParse(value, RFC2616_DATE_FORMATS.get());
            if (date == null) {
                LOG.warn("Could not create Date instance from rfc2616 format: " + value);
            }
            return date;
        }
        Date date = HttpDateUtils.tryParseRfc3339(value);
        if (date == null) {
            LOG.warn("Could not create Date instance from rfc3339 format: " + value);
        }
        return date;
    }

    public static String format(Date date) {
        return DateTimeFormatter.ISO_INSTANT.format(date.toInstant());
    }

    public static String formatAlwaysIncludeMillis(Date date) {
        return ISO_INSTANT_INCLUDE_MILLIS.format(date.toInstant());
    }

    private static Date tryParse(String date, List<DateFormat> formats) {
        for (DateFormat format : formats) {
            try {
                Date result = format.parse(date);
                format.setTimeZone(TIMEZONE_GMT);
                return result;
            }
            catch (ParseException parseException) {
            }
        }
        return null;
    }

    private static Date tryParseRfc3339(String date) {
        DateTimeFormatter formatter = new DateTimeFormatterBuilder().append(DateTimeFormatter.ISO_LOCAL_DATE).optionalStart().appendLiteral('T').append(DateTimeFormatter.ISO_LOCAL_TIME).optionalEnd().optionalStart().appendOffsetId().toFormatter();
        try {
            return formatter.parse((CharSequence)date, temporal -> {
                if (temporal.isSupported(ChronoField.INSTANT_SECONDS)) {
                    return Date.from(Instant.from(temporal));
                }
                LocalDate localDate = LocalDate.from(temporal);
                LocalTime localTime = Optional.ofNullable(temporal.query(TemporalQueries.localTime())).orElse(LocalTime.MIDNIGHT);
                ZoneOffset offset = Optional.ofNullable(temporal.query(TemporalQueries.offset())).orElse(ZoneOffset.UTC);
                ZonedDateTime dateTime = ZonedDateTime.of(localDate, localTime, offset);
                return Date.from(dateTime.toInstant());
            });
        }
        catch (DateTimeParseException dateTimeParseException) {
            return null;
        }
    }

    static {
        RFC2616_DATE_FORMATS = new ThreadLocal<List<DateFormat>>(){

            @Override
            protected synchronized List<DateFormat> initialValue() {
                return HttpDateUtils.createRfc2616DateFormats();
            }
        };
        Calendar calendar = Calendar.getInstance();
        calendar.setTimeZone(TIMEZONE_GMT);
        calendar.set(2000, 0, 1, 0, 0, 0);
        calendar.set(14, 0);
        AMBIGUOUS_YEAR_TWO_DIGIT_YEAR_START = calendar.getTime();
        ISO_INSTANT_INCLUDE_MILLIS = new DateTimeFormatterBuilder().parseCaseInsensitive().appendInstant(3).toFormatter();
    }
}

