/*************************************************************************
 *
 *  $RCSfile: OfficeWriter.java,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: jsc $ $Date: 2003/02/21 08:05:22 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

import java.awt.*;
import java.awt.event.*;
import javax.swing.event.*;
import java.util.Vector;

import com.sun.star.uno.XInterface;
import com.sun.star.uno.UnoRuntime;
import com.sun.star.container.XIndexAccess;
import com.sun.star.text.XText;
import com.sun.star.text.XTextDocument;
import com.sun.star.text.XTextRange;
import com.sun.star.text.XTextCursor;
import com.sun.star.frame.XFrame;
import com.sun.star.frame.XModel;
import com.sun.star.view.XSelectionSupplier;
import com.sun.star.view.XSelectionChangeListener;

/** Implementation of the Office Writer Component as a Java Bean.
 *
 * The OfficeWriter provides methods for getting and setting the
 * contents of an embedded Office Writer document and also for adding a
 * listener for changes in the current selection in the document.
 */

public class OfficeWriter extends Office
{
	public static final String NEW_DOCUMENT = "private:factory/swriter";

	private transient XSelectionChangeListener mXSelectionChangeListener = null;
	private transient EventListenerList mSelectionChangeListenerList = null;

	/** Method to get the complete text of the current OfficeWriter document.
	 *
	 * @return the complete text of the current OfficeWriter document
	 */
	public String getString()
	{
		XText xText = getXText();
		String sText = "";

		if (xText != null) {
			XTextCursor xTextCursor = xText.createTextCursor();
			xTextCursor.gotoStart(false);
			xTextCursor.gotoEnd(true);
			sText = xTextCursor.getString();
		}
		return sText;        
	}

	/** Method to replace the complete text of the OfficeWriter document.
	 *
	 * @param s the new text
	 */
	public void setString(String s)
	{
		XText xText = getXText();

		if (xText != null) {
			XTextCursor xTextCursor = xText.createTextCursor();
			xTextCursor.gotoStart(false);
			xTextCursor.gotoEnd(true);
			xTextCursor.setString(s);
		}
	}

	/** Method to get the XTextDocument interface.
	*
	* @return the XTextDocument interface of the current document
	*
	*/    
	public XTextDocument getTextDocument()
	{
		XTextDocument xTextDocument = null;
		XModel xModel = null;

		if ((getFrame() != null) && (getFrame().getController() != null))
			xModel = getFrame().getController().getModel();

		if (xModel != null) {
			xTextDocument =  (XTextDocument) UnoRuntime.queryInterface(
				XTextDocument.class, xModel);
		}
		return xTextDocument;        
	}

	/** Method which gets the text currently selected in the Writer document
	 *
	 * @return the currently selected text as a String 
	 */
	public String getSelection()
	{
		XTextRange xTextRange = null;
		String s = "";

		try {
			xTextRange = queryCurrentPosition();
			if (xTextRange != null)
				s = xTextRange.getString();
		}
		catch (Exception e) {
			e.printStackTrace();
		}
		return s;
	}

	/** Adds a ChangeListener, event is fired when the text selection changes
	 *
	 * @param l ChangeListener to be added
	 */    
	public void addSelectionChangeListener(ChangeListener l)
	{
		if (mSelectionChangeListenerList == null)
			mSelectionChangeListenerList = new EventListenerList();

		mSelectionChangeListenerList.add(ChangeListener.class , l);

		if (mXSelectionChangeListener == null)
			postInitListenersEvent();
	}

	/** Removes a previously added ChangeListener
	 *
	 * @param l ChangeListener to be removed
	 */    
	public void removeSelectionChangeListener(ChangeListener l)
	{
		if (mSelectionChangeListenerList == null)
			return;

		mSelectionChangeListenerList.remove(ChangeListener.class, l);

		if (mSelectionChangeListenerList.getListenerCount() == 0)
		{
			removeXSelectionChangeListener();
			mSelectionChangeListenerList = null;
		}
	}

	/**
	 * Loads a document referenced by a URL.
	 *
	 * @param url The document's URL string. 
	 * @exception java.io.IOException if the document loading process has 
	 *	failed.
	 */
	public synchronized void load(String url)
		throws java.io.IOException
	{
		// remove XSelectionChangeListener for previous document
		removeXSelectionChangeListener();

		// now call parent load method
		super.load(url);

		// and queue a call to add a XSelectionChangeListener to the new document
		postInitListenersEvent();
	}

	/** Bean is always focusTraversable
	 *
	 * @return boolean, always true as the bean is focusTraversable
	 */    
	public boolean isFocusTraversable()
	{
		return true;
	}

	/* Get the XText interface of the current document */
	private XText getXText()
	{
		XTextDocument xTextDocument = getTextDocument();
		XText xText = null;

		if (xTextDocument != null) {
			xText = (XText) xTextDocument.getText();
		}
		return xText;
	}

	/** Returns the first selected TextRange in the TextDocument,
	 * or the cursor position if no text position is selected.
	 *
	 * @return current cursor position
	 * @throws Exception element could not be retrieved
	 */
	private XTextRange queryCurrentPosition()
		throws java.lang.Exception
	{
		XTextRange xTextRange = null;

		try {
			XIndexAccess xIndexAccess = getSelectionIndex();
			if(xIndexAccess != null) {
				XInterface xInterfaceText = (XInterface)UnoRuntime.queryInterface(
                    XInterface.class, xIndexAccess.getByIndex((int)0));
				xTextRange = (XTextRange)UnoRuntime.queryInterface(
					XTextRange.class, xInterfaceText);
			}
		}
		catch( java.lang.Exception e ) {
			e.printStackTrace(System.err);
		}
		return xTextRange;
	}

	/** Returns a container of all selected text positions in the document,
	 * at least the current cursor position
	 *
	 * @return Indexed Access to container
	 * @throws Exception No SelectionSupplier available
	 */    
	private XIndexAccess getSelectionIndex()
		throws java.lang.Exception
	{
		XIndexAccess xIndexAccess = null;
		XSelectionSupplier xSelectionSupplier = getSelectionSupplier();

		if (xSelectionSupplier == null)
			throw new Exception("queryCurrentPosition: xSelectionSupplier == null");

		try {
			XInterface xInterface = (XInterface) UnoRuntime.queryInterface(
                XInterface.class,xSelectionSupplier.getSelection());
			xIndexAccess = (XIndexAccess)UnoRuntime.queryInterface(
				XIndexAccess.class,xInterface);
		}
		catch( java.lang.Exception e ) {
			e.printStackTrace(System.err);
		}
		return xIndexAccess;
	}

	/** Fires StateChanged Event to Listeners
	 *
	 * @param event StateChanged event
	 */    
	private void fireStateChangedEvent(EventListenerList list, ChangeEvent event)
	{
		Object listenerList[] = list.getListenerList();

		for (int i = listenerList.length - 2; i >= 0; i -= 2) {
			if (listenerList[i] == ChangeListener.class) {
				((ChangeListener) listenerList[i+1]).stateChanged(event);
			}
		}
	}

	private void postInitListenersEvent()
	{
		EventQueue eq = java.awt.Toolkit.getDefaultToolkit().getSystemEventQueue();
		eq.postEvent(new AddSelectionChangeEvent(this, AWTEvent.RESERVED_ID_MAX+1));
	}

	/** Creates an XSelectionChangeListener and adds it to the active document
	 */
	private void initListeners()
	{
		XSelectionSupplier xSelectionSupplier = null;

		if (getFrame() == null) return;

		if (mXSelectionChangeListener == null) {
			try {
				mXSelectionChangeListener = new CSelectionChangeListener();
				getSelectionSupplier().addSelectionChangeListener(
					mXSelectionChangeListener);
			}
			catch (Exception e) {
				mXSelectionChangeListener = null;
				e.printStackTrace(System.err);
			}
		}
	}

	private void removeXSelectionChangeListener()
	{
		if (mXSelectionChangeListener != null) {
			try {
				getSelectionSupplier().removeSelectionChangeListener(
					mXSelectionChangeListener);
				mXSelectionChangeListener.disposing(null);
				mXSelectionChangeListener = null;
			}
			catch (java.lang.Exception e) {
				e.printStackTrace(System.err);
			}            
		}              
	}

	/**
	 * Closes the connection.
	 */
	public synchronized void closeConnection()
	{
		removeXSelectionChangeListener();
		super.closeConnection();
	}

	/** Inner Class implements java.awt.ActiveEvent
	 * creates API SelectionChangeListener
	 */    
	private class AddSelectionChangeEvent extends AWTEvent implements ActiveEvent
	{
		/** Constructor of inner class
		 * @param source event source
		 * @param id event id
		 */        
		public AddSelectionChangeEvent(Object source, int id)
		{
			super (source, id);
		}

		/** Creates a Listener */        
		public void dispatch ()
		{
			initListeners();
		}
	}

	private class CSelectionChangeListener implements XSelectionChangeListener
	{
		/** Listener is about to be destroyed
		 * @param aEvent Event
 		 */    
		public void disposing(com.sun.star.lang.EventObject o)
		{
		}

		/** SelectionChanged
		 * @param aEvent Event
		 */    
		public void selectionChanged(com.sun.star.lang.EventObject o)
		{
			fireStateChangedEvent(mSelectionChangeListenerList, new ChangeEvent(o));
		}
	}
}
