/*************************************************************************
 *
 *  $RCSfile: imapclnt.hxx,v $
 *
 *  $Revision: 1.1.1.1 $
 *
 *  last change: $Author: hr $ $Date: 2000/09/18 16:31:15 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef INET_IMAPCLNT_HXX
#define INET_IMAPCLNT_HXX

#ifndef _DATETIME_HXX
#include <tools/datetime.hxx>
#endif
#ifndef _TOOLS_DEBUG_HXX
#include <tools/debug.hxx>
#endif
#ifndef _ERRCODE_HXX
#include <tools/errcode.hxx>
#endif
#ifndef _REF_HXX
#include <tools/ref.hxx>
#endif
#ifndef _STRING_HXX
#include <tools/string.hxx>
#endif

class INetCoreNewsMessage;
class INetIMAPClient_Impl;
class INetIMAPMessageNumberSet;
class SvStream;

//============================================================================
//
//  Client Command Arguments
//
//============================================================================

/** Some US-ASCII character and string constants used by the IMAP4 protocol.
 */
#define INET_IMAP_LIST_WILDCARD_NON_SEPARATOR '%'
#define INET_IMAP_LIST_WILDCARD_SEPARATOR '*'
#define INET_IMAP_CANONIC_NIL "NIL"
#define INET_IMAP_CANONIC_INBOX_NAME "INBOX"

//============================================================================
enum INetIMAPStatusAttributes
{
	INET_IMAP_STATUS_ATTRIBUTE_MESSAGES = 0x01,
	INET_IMAP_STATUS_ATTRIBUTE_RECENT = 0x02,
	INET_IMAP_STATUS_ATTRIBUTE_UIDNEXT = 0x04,
	INET_IMAP_STATUS_ATTRIBUTE_UIDVALIDITY = 0x08,
	INET_IMAP_STATUS_ATTRIBUTE_UNSEEN = 0x10
};

//============================================================================
class INetIMAPMessageNumberSetBase
{
public:
	virtual INetIMAPMessageNumberSet * clone() const = 0;
};

//============================================================================
class INetIMAPMessageNumberSet: public INetIMAPMessageNumberSetBase
{
	enum { HALF_OPEN = 0xFFFFFFFF };

	struct Range
	{
		Range * m_pNext;
		sal_uInt32 m_nLowerBound;
		sal_uInt32 m_nUpperBound;
	};

	Range * m_pRanges;

	virtual void add_Impl(sal_uInt32 nLowerBound, sal_uInt32 nUpperBound);

public:
	INetIMAPMessageNumberSet(): m_pRanges(0) {}

	virtual ~INetIMAPMessageNumberSet();

	virtual INetIMAPMessageNumberSet * clone() const;

	inline void add(sal_uInt32 nMessageNumber);

	inline void add(sal_uInt32 nLowerBound, sal_uInt32 nUpperBound);

	inline void addHalfOpen(sal_uInt32 nLowerBound);

	bool isEmpty() { return !m_pRanges; }

	sal_uInt32 getRangeCount() const;

	void getRange(sal_uInt32 nIndex, bool & rHalfOpen,
				  sal_uInt32 & rLowerBound, sal_uInt32 & rUpperBound) const;

	ByteString toString() const;
};

inline void INetIMAPMessageNumberSet::add(sal_uInt32 nMessageNumber)
{
	DBG_ASSERT(nMessageNumber != 0 && nMessageNumber != HALF_OPEN,
			   "INetIMAPMessageNumberSet::add(): Bad range");
	add_Impl(nMessageNumber, nMessageNumber);
}

inline void INetIMAPMessageNumberSet::add(sal_uInt32 nLowerBound,
										  sal_uInt32 nUpperBound)
{
	DBG_ASSERT(nLowerBound != 0 && nLowerBound <= nUpperBound
			   && nUpperBound != HALF_OPEN,
			   "INetIMAPMessageNumberSet::add(): Bad range");
	add_Impl(nLowerBound, nUpperBound);
}

inline void INetIMAPMessageNumberSet::addHalfOpen(sal_uInt32 nLowerBound)
{
	DBG_ASSERT(nLowerBound != 0 && nLowerBound != HALF_OPEN,
			   "INetIMAPMessageNumberSet::addHalfOpen(): Bad range");
	add_Impl(nLowerBound, HALF_OPEN);
}

//============================================================================
class INetIMAPSearchKey
{
public:
	virtual ~INetIMAPSearchKey();

	virtual void appendCommandArguments(INetIMAPClient_Impl & rClient) const
		= 0;
};

//============================================================================
class INetIMAPSearchKeyList: private List
{
public:
	inline INetIMAPSearchKeyList(INetIMAPSearchKey * pKey);

	virtual ~INetIMAPSearchKeyList();

	inline void append(INetIMAPSearchKey * pKey);

	sal_uInt32 getCount() const { return Count(); }

	INetIMAPSearchKey const & get(sal_uInt32 nIndex) const
	{ return *static_cast< INetIMAPSearchKey * >(GetObject(nIndex)); }
};

inline INetIMAPSearchKeyList::INetIMAPSearchKeyList(INetIMAPSearchKey * pKey)
{
	DBG_ASSERT(pKey, "INetIMAPSearchKeyList::INetIMAPSearchKeyList(): Null");
	Insert(pKey, LIST_APPEND);
}

inline void INetIMAPSearchKeyList::append(INetIMAPSearchKey * pKey)
{
	DBG_ASSERT(pKey, "INetIMAPSearchKeyList::append(): Null");
	Insert(pKey, LIST_APPEND);
}

//============================================================================
class INetIMAPSimpleSearchKey: public INetIMAPSearchKey
{
public:
	enum Criterium
	{
		CRITERIUM_ALL,
		CRITERIUM_ANSWERED,
		CRITERIUM_DELETED,
		CRITERIUM_DRAFT,
		CRITERIUM_FLAGGED,
		CRITERIUM_NEW,
		CRITERIUM_OLD,
		CRITERIUM_RECENT,
		CRITERIUM_SEEN,
		CRITERIUM_UNANSWERED,
		CRITERIUM_UNDELETED,
		CRITERIUM_UNDRAFT,
		CRITERIUM_UNFLAGGED,
		CRITERIUM_UNSEEN
	};

private:
	Criterium m_eCriterium;

public:
	INetIMAPSimpleSearchKey(Criterium eTheCriterium):
		m_eCriterium(eTheCriterium) {}

	virtual void appendCommandArguments(INetIMAPClient_Impl & rClient) const;
};

//============================================================================
class INetIMAPTextSearchKey: public INetIMAPSearchKey
{
public:
	enum Property
	{
		PROPERTY_BCC,
		PROPERTY_BODY,
		PROPERTY_CC,
		PROPERTY_FROM,
		PROPERTY_SUBJECT,
		PROPERTY_TEXT,
		PROPERTY_TO
	};

private:
	ByteString m_aText;
	Property m_eProperty;

public:
	INetIMAPTextSearchKey(Property eTheProperty, ByteString const & rTheText):
		m_eProperty(eTheProperty), m_aText(rTheText) {}

	virtual void appendCommandArguments(INetIMAPClient_Impl & rClient) const;
};

//============================================================================
class INetIMAPRFC822SizeSearchKey: public INetIMAPSearchKey
{
public:
	enum Conditional
	{
		CONDITIONAL_LARGER,
		CONDITIONAL_SMALLER
	};

private:
	sal_uInt32 m_nSize;
	Conditional m_eConditional;

public:
	INetIMAPRFC822SizeSearchKey(Conditional eTheConditional,
								sal_uInt32 nTheSize):
		m_eConditional(eTheConditional), m_nSize(nTheSize) {}

	virtual void appendCommandArguments(INetIMAPClient_Impl & rClient) const;
};

//============================================================================
class INetIMAPDateSearchKey: public INetIMAPSearchKey
{
public:
	enum Conditional
	{
		CONDITIONAL_BEFORE,
		CONDITIONAL_ON,
		CONDITIONAL_SENTBEFORE,
		CONDITIONAL_SENTON,
		CONDITIONAL_SENTSINCE,
		CONDITIONAL_SINCE
	};

private:
	Date m_aDate;
	Conditional m_eConditional;

public:
	INetIMAPDateSearchKey(Conditional eTheConditional, Date const & rTheDate):
		m_eConditional(eTheConditional), m_aDate(rTheDate) {}

	virtual void appendCommandArguments(INetIMAPClient_Impl & rClient) const;
};

//============================================================================
class INetIMAPKeywordSearchKey: public INetIMAPSearchKey
{
public:
	enum Conditional
	{
		CONDITIONAL_KEYWORD,
		CONDITIONAL_UNKEYWORD
	};

private:
	ByteString m_aKeyword;
	Conditional m_eConditional;

public:
	INetIMAPKeywordSearchKey(Conditional eTheConditional,
							 ByteString const & rTheKeyword):
		m_eConditional(eTheConditional), m_aKeyword(rTheKeyword) {}

	virtual void appendCommandArguments(INetIMAPClient_Impl & rClient) const;
};

//============================================================================
class INetIMAPHeaderSearchKey: public INetIMAPSearchKey
{
	ByteString m_aFieldName;
	ByteString m_aText;

public:
	INetIMAPHeaderSearchKey(ByteString const & rTheFieldName,
							ByteString const & rTheText):
		m_aFieldName(rTheFieldName), m_aText(rTheText) {}

	virtual void appendCommandArguments(INetIMAPClient_Impl & rClient) const;
};

//============================================================================
class INetIMAPSetSearchKey: public INetIMAPSearchKey
{
	INetIMAPMessageNumberSet * m_pSet;

public:
	inline INetIMAPSetSearchKey(INetIMAPMessageNumberSet * pTheSet);

	virtual ~INetIMAPSetSearchKey();

	virtual void appendCommandArguments(INetIMAPClient_Impl & rClient) const;
};

inline INetIMAPSetSearchKey::INetIMAPSetSearchKey(INetIMAPMessageNumberSet *
												      pTheSet):
	m_pSet(pTheSet)
{
	DBG_ASSERT(m_pSet, "INetIMAPSetSearchKey::INetIMAPSetSearchKey(): Null");
}

//============================================================================
class INetIMAPListSearchKey: public INetIMAPSearchKey
{
	INetIMAPSearchKeyList m_aList;

public:
	INetIMAPListSearchKey(INetIMAPSearchKey * pKey): m_aList(pKey) {}

	void append(INetIMAPSearchKey * pKey) { m_aList.append(pKey); }

	virtual void appendCommandArguments(INetIMAPClient_Impl & rClient) const;
};

//============================================================================
class INetIMAPNegationSearchKey: public INetIMAPSearchKey
{
	INetIMAPSearchKey * m_pKey;

public:
	inline INetIMAPNegationSearchKey(INetIMAPSearchKey * pTheKey);

	virtual ~INetIMAPNegationSearchKey();

	virtual void appendCommandArguments(INetIMAPClient_Impl & rClient) const;
};

inline
INetIMAPNegationSearchKey::INetIMAPNegationSearchKey(INetIMAPSearchKey *
													     pTheKey):
	m_pKey(pTheKey)
{
	DBG_ASSERT(m_pKey,
			   "INetIMAPNegationSearchKey::INetIMAPNegationSearchKey():"
			       " Null");
}

//============================================================================
class INetIMAPDisjunctionSearchKey: public INetIMAPSearchKey
{
	INetIMAPSearchKey * m_pKey1;
	INetIMAPSearchKey * m_pKey2;

public:
	inline INetIMAPDisjunctionSearchKey(INetIMAPSearchKey * pTheKey1,
										INetIMAPSearchKey * pTheKey2);

	virtual ~INetIMAPDisjunctionSearchKey();

	virtual void appendCommandArguments(INetIMAPClient_Impl & rClient) const;
};

inline
INetIMAPDisjunctionSearchKey::INetIMAPDisjunctionSearchKey(INetIMAPSearchKey *
														       pTheKey1,
														   INetIMAPSearchKey *
														       pTheKey2):
	m_pKey1(pTheKey1),
	m_pKey2(pTheKey2)
{
	DBG_ASSERT(m_pKey1 && m_pKey2,
			   "INetIMAPDisjunctionSearchKey::INetIMAPDisjunctionSearchKey():"
			       " Null");
}

//============================================================================
class INetIMAPSectionNumberSequence: private List
{
public:
	void append(sal_uInt32 nSectionNumber)
	{ Insert(reinterpret_cast< void * >(nSectionNumber), LIST_APPEND); }

	sal_uInt32 getCount() const { return Count(); }

	sal_uInt32 get(sal_uInt32 nIndex) const
	{ return reinterpret_cast< sal_uInt32 >(GetObject(nIndex)); }
};

//============================================================================
class INetIMAPHeaderFieldList: private List
{
	bool m_bNegated;

public:
	INetIMAPHeaderFieldList(): m_bNegated(false) {}

	virtual ~INetIMAPHeaderFieldList();

	void setNegated() { m_bNegated = true; }

	virtual void append(ByteString const & rHeaderField);

	sal_uInt32 getCount() const { return Count(); }

	ByteString const & get(sal_uInt32 nIndex) const
	{ return *static_cast< ByteString * >(GetObject(nIndex)); }

	ByteString toString() const;
};

//============================================================================
class INetIMAPHeaderFieldListList: private List
{
public:
	virtual ~INetIMAPHeaderFieldListList();

	void append(INetIMAPHeaderFieldList * pList)
	{ Insert(pList, LIST_APPEND); }

	sal_uInt32 getCount() const { return Count(); }

	INetIMAPHeaderFieldList const & get(sal_uInt32 nIndex) const
	{ return *static_cast< INetIMAPHeaderFieldList * >(GetObject(nIndex)); }
};

//============================================================================
class INetIMAPBodySectionDescriptor
{
public:
	/** @ATT  RFC822, RFC822_HEADER, and RFC822_TEXT are only valid with
		INetIMAPFetchResponseBodySection, not with
		INetIMAPArgumentBodySection.
	 */
	enum SectionText
	{
		SECTION_TEXT_NO,
		SECTION_TEXT_HEADER,
		SECTION_TEXT_HEADER_FIELDS,
		SECTION_TEXT_HEADER_FIELDS_NOT,
		SECTION_TEXT_TEXT,
		SECTION_TEXT_MIME,
		RFC822,
		RFC822_HEADER,
		RFC822_TEXT
	};

private:
	INetIMAPSectionNumberSequence m_aSectionNumberSequence;
	SectionText m_eSectionText;
	INetIMAPHeaderFieldList m_aHeaderFields;

public:
	INetIMAPBodySectionDescriptor(): m_eSectionText(SECTION_TEXT_NO) {}

	virtual void appendSectionNumber(sal_uInt32 nSectionNumber);

	virtual void appendSectionText(SectionText eTheSectionText);

	virtual void appendHeaderField(ByteString const & rHeaderField);

	INetIMAPSectionNumberSequence const & getSectionNumberSequence() const
	{ return m_aSectionNumberSequence; }

	SectionText getSectionText() const { return m_eSectionText; }

	INetIMAPHeaderFieldList const & getHeaderFields() const
	{ return m_aHeaderFields; }
};

//============================================================================
class INetIMAPArgumentBodySection: public INetIMAPBodySectionDescriptor
{
	sal_uInt32 m_nPartialOffset;
	sal_uInt32 m_nPartialLength;
	bool m_bPeek;
	bool m_bPartial;

public:
	INetIMAPArgumentBodySection(): m_bPeek(false), m_bPartial(false) {}

	void setPeek() { m_bPeek = true; }

	inline void setPartial(sal_uInt32 nThePartialOffset,
						   sal_uInt32 nThePartialLength);

	ByteString toString() const;
};

inline void INetIMAPArgumentBodySection::setPartial(sal_uInt32
													    nThePartialOffset,
													sal_uInt32
													    nThePartialLength)
{
	DBG_ASSERT(nThePartialLength,
			   "INetIMAPArgumentBodySection::setPartial(): Invalid length");
	m_bPartial = true;
	m_nPartialOffset = nThePartialOffset;
	m_nPartialLength = nThePartialLength;
}

//============================================================================
class INetIMAPArgumentBodySectionList: private List
{
public:
	virtual ~INetIMAPArgumentBodySectionList();

	inline void append(INetIMAPArgumentBodySection * pSection);

	sal_uInt32 getCount() const { return Count(); }

	inline INetIMAPArgumentBodySection const & get(sal_uInt32 nIndex) const;
};

inline
void INetIMAPArgumentBodySectionList::append(INetIMAPArgumentBodySection *
											     pSection)
{
	DBG_ASSERT(pSection, "INetIMAPArgumentBodySectionList::append(): Null");
	Insert(pSection, LIST_APPEND);
}

inline INetIMAPArgumentBodySection const &
INetIMAPArgumentBodySectionList::get(sal_uInt32 nIndex) const
{
	return *static_cast< INetIMAPArgumentBodySection * >(GetObject(nIndex));
}

//============================================================================
enum INetIMAPFlags
{
	INET_IMAP_FLAG_ANSWERED = 0x01,
	INET_IMAP_FLAG_FLAGGED = 0x02,
	INET_IMAP_FLAG_DELETED = 0x04,
	INET_IMAP_FLAG_SEEN = 0x08,
	INET_IMAP_FLAG_DRAFT = 0x10
};

//============================================================================
class INetIMAPFlagKeywordList: private List
{
public:
	virtual ~INetIMAPFlagKeywordList();

	void append(ByteString const & rFlagKeyword);

	sal_uInt32 getCount() const { return Count(); }

	ByteString const & get(sal_uInt32 nIndex) const
	{ return *static_cast< ByteString * >(GetObject(nIndex)); }
};

//============================================================================
//
//  Client
//
//============================================================================

class INetIMAPClient: public SvRefBase
{
public:
	enum FetchAttributes
	{
		FETCH_BODY = 0x001,
		FETCH_BODYSTRUCTURE = 0x002,
		FETCH_ENVELOPE = 0x004,
		FETCH_FLAGS = 0x008,
		FETCH_INTERNALDATE = 0x010,
		FETCH_RFC822 = 0x020,
		FETCH_RFC822_HEADER = 0x040,
		FETCH_RFC822_PEEK = 0x080, // obsolete
		FETCH_RFC822_SIZE = 0x100,
		FETCH_RFC822_TEXT = 0x200,
		FETCH_RFC822_TEXT_PEEK = 0x400, // obsolete
		FETCH_UID = 0x800,
		FETCH_ALL = FETCH_ENVELOPE | FETCH_FLAGS | FETCH_INTERNALDATE
		                | FETCH_RFC822_SIZE,
		FETCH_FULL = FETCH_BODY | FETCH_ENVELOPE | FETCH_FLAGS
		                 | FETCH_INTERNALDATE | FETCH_RFC822_SIZE,
		FETCH_FAST = FETCH_FLAGS | FETCH_INTERNALDATE | FETCH_RFC822_SIZE
	};

	enum StoreAttribute
	{
		STORE_FLAGS,
		STORE_FLAGS_SILENT,
		STORE_ADD_FLAGS,
		STORE_ADD_FLAGS_SILENT,
		STORE_REMOVE_FLAGS,
		STORE_REMOVE_FLAGS_SILENT
	};

	enum Capabilities
	{
		CAPABILITY_IMAP4 = 0x001,
		CAPABILITY_IMAP4REV1 = 0x002,
		CAPABILITY_ACL = 0x004,
		CAPABILITY_QUOTA = 0x008,
		CAPABILITY_LITERAL_PLUS = 0x010,
		CAPABILITY_IDLE = 0x020,
		CAPABILITY_MAILBOX_REFERRALS = 0x040,
		CAPABILITY_LOGIN_REFERRALS = 0x080,
		CAPABILITY_NAMESPACE = 0x100,
		CAPABILITY_UIDPLUS = 0x200
	};

protected:
	virtual ~INetIMAPClient();

public:
	virtual bool hasOpenConnection() const = 0;

	virtual bool hasClosedConnection() const = 0;

	virtual ErrCode openConnection(UniString const & rHost, sal_uInt16 nPort,
								   UniString const & rSocksProxyHost,
								   sal_uInt16 nSocksProxyPort,
								   Link const & rTheUnilateralCallback,
								   void * pTheUnilateralResponseData,
								   Link const & rConnectCallback,
								   void * pConnectResponseData) = 0;

	virtual ErrCode abortConnection() = 0;

	virtual ErrCode cancelCommand() = 0;

	virtual ErrCode commandCapability(Link const & rCallback,
									  void * pResponseData) = 0;

	virtual ErrCode commandNoOp(Link const & rCallback, void * pResponseData)
		= 0;

	virtual ErrCode commandLogOut(Link const & rCallback,
								  void * pResponseData) = 0;

	virtual ErrCode commandLogIn(Link const & rCallback, void * pResponseData,
								 UniString const & rUserID,
								 UniString const & rPassword) = 0;

	virtual ErrCode commandSelect(Link const & rCallback,
								  void * pResponseData,
								  ByteString const & rMailboxName) = 0;

	virtual ErrCode commandExamine(Link const & rCallback,
								   void * pResponseData,
								   ByteString const & rMailboxName) = 0;

	virtual ErrCode commandCreate(Link const & rCallback,
								  void * pResponseData,
								  ByteString const & rMailboxName) = 0;

	virtual ErrCode commandDelete(Link const & rCallback,
								  void * pResponseData,
								  ByteString const & rMailboxName) = 0;

	virtual ErrCode commandRename(Link const & rCallback,
								  void * pResponseData,
								  ByteString const & rOldMailboxName,
								  ByteString const & rNewMailboxName) = 0;

	virtual ErrCode commandSubscribe(Link const & rCallback,
									 void * pResponseData,
									 ByteString const & rMailboxName) = 0;

	virtual ErrCode commandUnSubscribe(Link const & rCallback,
									   void * pResponseData,
									   ByteString const & rMailboxName) = 0;

	virtual ErrCode commandList(Link const & rCallback, void * pResponseData,
								ByteString const & rReference,
								ByteString const & rPattern) = 0;

	virtual ErrCode commandLSub(Link const & rCallback, void * pResponseData,
								ByteString const & rReference,
								ByteString const & rPattern) = 0;

	virtual ErrCode commandStatus(Link const & rCallback,
								  void * pResponseData,
								  ByteString const & rMailboxName,
								  INetIMAPStatusAttributes eAttributes) = 0;

	virtual ErrCode commandAppend(Link const & rCallback,
								  void * pResponseData,
								  ByteString const & rMailboxName,
								  ByteString const & rRFC822Header,
								  ByteString const & rMediaType,
								  SvStream * pBody) = 0;

	virtual ErrCode commandCheck(Link const & rCallback,
								 void * pResponseData) = 0;

	virtual ErrCode commandClose(Link const & rCallback,
								 void * pResponseData) = 0;

	virtual ErrCode commandExpunge(Link const & rCallback,
								   void * pResponseData) = 0;

	virtual ErrCode commandSearch(Link const & rCallback,
								  void * pResponseData, bool bUIDCommand,
								  rtl_TextEncoding eCharset,
								  INetIMAPSearchKeyList const & rKeys) = 0;

	virtual ErrCode commandFetch(Link const & rCallback, void * pResponseData,
								 bool bUIDCommand,
								 INetIMAPMessageNumberSet const &
								     rMessageNumberSet,
								 FetchAttributes eAttributes,
								 INetIMAPArgumentBodySectionList const *
								     pBodySections,
								 INetIMAPHeaderFieldListList const *
								     pRFC822HeaderLines,
								 Link const & rStreamCallback) = 0;

	virtual ErrCode commandStore(Link const & rCallback, void * pResponseData,
								 bool bUIDCommand,
								 INetIMAPMessageNumberSet const &
								     rMessageNumberSet,
								 StoreAttribute eAttribute,
								 INetIMAPFlags eFlags,
								 INetIMAPFlagKeywordList const *
								     pFlagKeywords) = 0;

	virtual ErrCode commandCopy(Link const & rCallback, void * pResponseData,
								bool bUIDCommand,
								INetIMAPMessageNumberSet const &
								    rMessageNumberSet,
								ByteString const & rMailboxName) = 0;

	virtual ErrCode commandNamespace(Link const & rCallback,
									 void * pResponseData) = 0;

	virtual bool isAuthenticated() const = 0;

	virtual bool hasDeterminedCapabilities() const = 0;

	virtual Capabilities getCapabilities() const = 0;

	virtual ByteString const & getSelectedMailbox() const = 0;

	virtual INetIMAPMessageNumberSet * createMessageNumberSet() const;

	virtual INetIMAPSearchKeyList * createSearchKeyList(INetIMAPSearchKey *
														    pKey) const;

	virtual INetIMAPSimpleSearchKey *
	createSimpleSearchKey(INetIMAPSimpleSearchKey::Criterium eCriterium)
		const;

	virtual INetIMAPTextSearchKey *
	createTextSearchKey(INetIMAPTextSearchKey::Property eProperty,
						ByteString const & rText) const;

	virtual INetIMAPRFC822SizeSearchKey *
	createRFC822SizeSearchKey(INetIMAPRFC822SizeSearchKey::Conditional
							      eConditional,
							  sal_uInt32 nSize) const;

	virtual INetIMAPDateSearchKey *
	createDateSearchKey(INetIMAPDateSearchKey::Conditional eConditional,
						Date const & rDate) const;

	virtual INetIMAPKeywordSearchKey *
	createKeywordSearchKey(INetIMAPKeywordSearchKey::Conditional
						    eConditional,
						   ByteString const & rKeyword) const;

	virtual INetIMAPHeaderSearchKey *
	createHeaderSearchKey(ByteString const & rFieldName,
						  ByteString const & rText) const;

	virtual INetIMAPSetSearchKey *
	createSetSearchKey(INetIMAPMessageNumberSet * pSet) const;

	virtual INetIMAPListSearchKey *
	createListSearchKey(INetIMAPSearchKey * pKey) const;

	virtual INetIMAPNegationSearchKey *
	createNegationSearchKey(INetIMAPSearchKey * pKey) const;

	virtual INetIMAPDisjunctionSearchKey *
	createDisjunctionSearchKey(INetIMAPSearchKey * pKey1,
							   INetIMAPSearchKey * pKey2) const;

	virtual INetIMAPArgumentBodySection * createArgumentBodySection() const;

	virtual INetIMAPArgumentBodySectionList * createArgumentBodySectionList()
		const;

	virtual INetIMAPHeaderFieldList * createHeaderFieldList() const;

	virtual INetIMAPHeaderFieldListList * createHeaderFieldListList() const;

	virtual ErrCode commandAppend(Link const & rCallback,
								  void * pResponseData,
								  ByteString const & rMailboxName,
								  SvStream * pMessage) = 0;
};

//============================================================================
SV_DECL_IMPL_REF(INetIMAPClient)

//============================================================================
extern "C" INetIMAPClient * __LOADONCALLAPI NewINetImapMailer();

//============================================================================
//
//  Client Responses
//
//============================================================================

class INetIMAPResponse
{
public:
	enum Type
	{
		TYPE_CLOSED,
		TYPE_ERROR,
		TYPE_OPENING,
		TYPE_INVALID,
		TYPE_CONTINUE_CODE,
		TYPE_STATE,
		TYPE_FLAGS,
		TYPE_LIST,
		TYPE_MAILBOX,
		TYPE_SEARCH,
		TYPE_STATUS,
		TYPE_EXISTS,
		TYPE_RECENT,
		TYPE_EXPUNGE,
		TYPE_FETCH,
		TYPE_CAPABILITY,
		TYPE_NAMESPACE
	};

private:
	INetIMAPClient const * m_pClient;
	void * m_pData;

public:
	inline INetIMAPResponse();

	virtual ~INetIMAPResponse();

	virtual Type getType() const = 0;

	inline void setCallbackData(INetIMAPClient const & rTheClient,
								void * pTheData);

	inline INetIMAPClient const & getClient() const;

	inline void * getData() const;
};

inline INetIMAPResponse::INetIMAPResponse()
{
#if defined DBG_UTIL
	m_pClient = 0;
#endif // DBG_UTIL
}

inline void INetIMAPResponse::setCallbackData(INetIMAPClient const &
											      rTheClient,
											  void * pTheData)
{
	m_pClient = &rTheClient;
	m_pData = pTheData;
}

inline INetIMAPClient const & INetIMAPResponse::getClient() const
{
	DBG_ASSERT(m_pClient, "INetIMAPResponse::getIMAP(): No callback data");
	return *m_pClient;
}

inline void * INetIMAPResponse::getData() const
{
	DBG_ASSERT(m_pClient, "INetIMAPResponse::getIMAP(): No callback data");
	return m_pData;
}

//============================================================================
class INetIMAPClosedResponse: public INetIMAPResponse
{
public:
	virtual Type getType() const;
};

//============================================================================
class INetIMAPErrorResponse: public INetIMAPResponse
{
	ErrCode m_nError;

public:
	INetIMAPErrorResponse(ErrCode nTheError): m_nError(nTheError) {}

	virtual Type getType() const;

	ErrCode getError() const { return m_nError; }
};

//============================================================================
class INetIMAPOpeningResponse: public INetIMAPResponse
{
public:
	enum Status
	{
		STATUS_NAME_RESOLUTION_PENDING,
		STATUS_NAME_RESOLUTION_FAILED,
		STATUS_NAME_RESOLUTION_DONE,
		STATUS_CONNECT_PENDING,
		STATUS_CONNECT_FAILED,
		STATUS_CONNECTED,
		STATUS_READ_FAILED
	};

private:
	Status m_eStatus;

public:
	INetIMAPOpeningResponse(Status eTheStatus): m_eStatus(eTheStatus) {}

	virtual Type getType() const;

	Status getStatus() const { return m_eStatus; }
};

//============================================================================
class INetIMAPTextResponse: public INetIMAPResponse
{
public:
	virtual UniString getText() const = 0;
};

//============================================================================
class INetIMAPInvalidResponse: public INetIMAPTextResponse
{
	UniString m_aLine;

public:
	INetIMAPInvalidResponse(UniString const & rTheLine): m_aLine(rTheLine) {}

	virtual Type getType() const;

	virtual UniString getText() const;
};

//============================================================================
class INetIMAPCodeResponse: public INetIMAPTextResponse
{
public:
	enum Code
	{
		CODE_NONE,
		CODE_ALERT,
		CODE_PARSE,
		CODE_PERMANENTFLAGS,
		CODE_READ_ONLY,
		CODE_READ_WRITE,
		CODE_TRYCREATE,
		CODE_UIDVALIDITY,
		CODE_UNSEEN
	};

private:
	UniString m_aText;
	INetIMAPFlagKeywordList const * m_pPermanentFlagKeywords;
	sal_uInt32 m_nNumber;
	Code m_eCode;
	INetIMAPFlags m_ePermanentFlags;
	bool m_bPermanentNewFlagKeywords;

public:
	INetIMAPCodeResponse(Code eTheCode, INetIMAPFlags eThePermanentFlags,
						 INetIMAPFlagKeywordList const &
						     rThePermanentFlagKeywords,
						 bool bThePermanentNewFlagKeywords,
						 sal_uInt32 nTheNumber, UniString const & rTheText);

	virtual UniString getText() const;

	Code getCode() const { return m_eCode; }

	inline INetIMAPFlags getPermanentFlags() const;

	inline INetIMAPFlagKeywordList const & getPermanentFlagKeywords() const;

	inline bool isPermanentNewFlagKeywords() const;

	inline sal_uInt32 getNumber() const;
};

inline INetIMAPFlags INetIMAPCodeResponse::getPermanentFlags() const
{
	DBG_ASSERT(m_eCode == CODE_PERMANENTFLAGS,
			   "INetIMAPCodeResponse::getPermanentFlags(): None");
	return m_ePermanentFlags;
}

inline INetIMAPFlagKeywordList const &
INetIMAPCodeResponse::getPermanentFlagKeywords() const
{
	DBG_ASSERT(m_eCode == CODE_PERMANENTFLAGS && m_pPermanentFlagKeywords,
			   "INetIMAPCodeResponse::getPermanentFlagKeywords(): None");
	return *m_pPermanentFlagKeywords;
}

inline bool INetIMAPCodeResponse::isPermanentNewFlagKeywords() const
{
	DBG_ASSERT(m_eCode == CODE_PERMANENTFLAGS,
			   "INetIMAPCodeResponse::isPermanentNewFlagKeywords(): None");
	return m_bPermanentNewFlagKeywords;
}

inline sal_uInt32 INetIMAPCodeResponse::getNumber() const
{
	DBG_ASSERT(m_eCode == CODE_UIDVALIDITY || m_eCode == CODE_UNSEEN,
			   "INetIMAPCodeResponse::getNumber(): None");
	return m_nNumber;
}

//============================================================================
class INetIMAPContinueCodeResponse: public INetIMAPCodeResponse
{
public:
	inline INetIMAPContinueCodeResponse(Code eCode,
										INetIMAPFlags ePermanentFlags,
										INetIMAPFlagKeywordList const &
										    rPermanentFlagKeywords,
										bool bPermanentNewFlagKeywords,
										sal_uInt32 nNumber,
										UniString const & rText);

	virtual Type getType() const;
};

inline INetIMAPContinueCodeResponse::INetIMAPContinueCodeResponse(
	       Code eCode, INetIMAPFlags ePermanentFlags,
		   INetIMAPFlagKeywordList const & rPermanentFlagKeywords,
		   bool bPermanentNewFlagKeywords, sal_uInt32 nNumber,
		   UniString const & rText):
	INetIMAPCodeResponse(eCode, ePermanentFlags, rPermanentFlagKeywords,
						 bPermanentNewFlagKeywords, nNumber, rText)
{}

//============================================================================
class INetIMAPStateResponse: public INetIMAPCodeResponse
{
public:
	enum State
	{
		STATE_OK,
		STATE_NO,
		STATE_BAD,
		STATE_PREAUTH,
		STATE_BYE
	};

private:
	State m_eState;
	bool m_bFinal;

public:
	inline INetIMAPStateResponse(bool bTheFinal, State eTheState, Code eCode,
								 INetIMAPFlags ePermanentFlags,
								 INetIMAPFlagKeywordList const &
								     rPermanentFlagKeywords,
								 bool bPermanentNewFlagKeywords,
								 sal_uInt32 nNumber, UniString const & rText);

	virtual Type getType() const;

	bool isFinal() const { return m_bFinal; }

	State getState() const { return m_eState; }
};

inline
INetIMAPStateResponse::INetIMAPStateResponse(bool bTheFinal, State eTheState,
											 Code eCode,
											 INetIMAPFlags ePermanentFlags,
											 INetIMAPFlagKeywordList const &
											     rPermanentFlagKeywords,
											 bool bPermanentNewFlagKeywords,
											 sal_uInt32 nNumber,
											 UniString const & rText):
	INetIMAPCodeResponse(eCode, ePermanentFlags, rPermanentFlagKeywords,
						 bPermanentNewFlagKeywords, nNumber, rText),
	m_bFinal(bTheFinal),
	m_eState(eTheState)
{}

//============================================================================
class INetIMAPFlagsResponse: public INetIMAPResponse
{
	INetIMAPFlagKeywordList m_aFlagKeywords;
	INetIMAPFlags m_eFlags;

public:
	INetIMAPFlagsResponse(): m_eFlags(INetIMAPFlags(0)) {}

	virtual Type getType() const;

	void addFlags(INetIMAPFlags eTheFlags)
	{ m_eFlags = INetIMAPFlags(m_eFlags | eTheFlags); }

	INetIMAPFlagKeywordList & getFlagKeywords() { return m_aFlagKeywords; }

	INetIMAPFlags getFlags() const { return m_eFlags; }

	INetIMAPFlagKeywordList const & getFlagKeywords() const
	{ return m_aFlagKeywords; }
};

//============================================================================
class INetIMAPListResponseMailbox
{
public:
	enum Flags
	{
		FLAG_MARKED = 0x1,
		FLAG_NOINFERIORS = 0x2,
		FLAG_NOSELECT = 0x4,
		FLAG_UNMARKED = 0x8
	};

private:
	ByteString m_aMailboxName;
	Flags m_eFlags;
	sal_Char m_cHierarchySeparator;
	bool m_bMailboxNameNIL;

public:
	INetIMAPListResponseMailbox():
		m_eFlags(Flags(0)), m_cHierarchySeparator(0) {}

	void addFlags(Flags eTheFlags) { m_eFlags = Flags(m_eFlags | eTheFlags); }

	void setHierarchySeparator(sal_Char cTheHierarchySeparator)
	{ m_cHierarchySeparator = cTheHierarchySeparator; }

	inline void setMailboxName(ByteString const & rTheMailboxName,
							   bool bTheMailboxNameNIL);

	inline bool appendMailboxName(ByteString const & rSuffix);

	Flags getFlags() const { return m_eFlags; }

	sal_Char getHierarchySeparator() const { return m_cHierarchySeparator; }

	ByteString const & getMailboxName() const { return m_aMailboxName; }

	bool isMailboxNameNIL() const { return m_bMailboxNameNIL; }
};

inline void
INetIMAPListResponseMailbox::setMailboxName(ByteString const &
											    rTheMailboxName,
											bool bTheMailboxNameNIL)
{
	m_aMailboxName = rTheMailboxName;
	m_bMailboxNameNIL = bTheMailboxNameNIL;
}

inline bool INetIMAPListResponseMailbox::appendMailboxName(ByteString const &
														       rSuffix)
{
	if (rSuffix.Len() > STRING_MAXLEN - m_aMailboxName.Len())
		return false;
	m_aMailboxName += rSuffix;
	m_bMailboxNameNIL = false;
	return true;
}

//============================================================================
DECLARE_LIST(INetIMAPListResponseMailboxList, INetIMAPListResponseMailbox *)

//============================================================================
class INetIMAPListResponse: public INetIMAPResponse
{
	INetIMAPListResponseMailboxList m_aMailboxes;

public:
	virtual ~INetIMAPListResponse();

	virtual Type getType() const;

	void appendMailbox(INetIMAPListResponseMailbox * pMailbox)
	{ m_aMailboxes.Insert(pMailbox, LIST_APPEND); }

	sal_uInt32 getMailboxCount() const { return m_aMailboxes.Count(); }

	INetIMAPListResponseMailbox const & getMailbox(sal_uInt32 nIndex) const
	{ return *m_aMailboxes.GetObject(nIndex); }
};

//============================================================================
class INetIMAPMailboxResponse: public INetIMAPResponse
{
	UniString m_aText;

public:
	INetIMAPMailboxResponse(UniString const & rTheText): m_aText(rTheText) {}

	virtual Type getType() const;

	UniString const & getText() const { return m_aText; }
};

//============================================================================
class INetIMAPSearchResponse: public INetIMAPResponse
{
	INetIMAPMessageNumberSet m_aMessageNumberSet;

public:
	virtual Type getType() const;

	void add(sal_uInt32 nMessageNumber)
	{ m_aMessageNumberSet.add(nMessageNumber); }

	INetIMAPMessageNumberSet const & getMessageNumberSet() const
	{ return m_aMessageNumberSet; }
};

//============================================================================
class INetIMAPStatusResponse: public INetIMAPResponse
{
	ByteString m_aMailboxName;
	sal_uInt32 m_nMessagesAttributeValue;
	sal_uInt32 m_nRecentAttributeValue;
	sal_uInt32 m_nUIDNextAttributeValue;
	sal_uInt32 m_nUIDValidityAttributeValue;
	sal_uInt32 m_nUnseenAttributeValue;
	INetIMAPStatusAttributes m_eAttributesPresent;

public:
	INetIMAPStatusResponse():
		m_eAttributesPresent(INetIMAPStatusAttributes(0)) {}

	virtual Type getType() const;

	void setMailboxName(ByteString const & rTheMailboxName)
	{ m_aMailboxName = rTheMailboxName; }

	inline bool appendMailboxName(ByteString const & rSuffix);

	inline void setMessagesAttribute(sal_uInt32 nValue);

	inline void setRecentAttribute(sal_uInt32 nValue);

	inline void setUIDNextAttribute(sal_uInt32 nValue);

	inline void setUIDValidityAttribute(sal_uInt32 nValue);

	inline void setUnseenAttribute(sal_uInt32 nValue);

	ByteString const & getMailboxName() const { return m_aMailboxName; }

	bool hasAttributes(INetIMAPStatusAttributes eAttributes) const
	{ return (m_eAttributesPresent & eAttributes) == eAttributes; }

	inline sal_uInt32 getMessagesAttributeValue() const;

	inline sal_uInt32 getRecentAttributeValue() const;

	inline sal_uInt32 getUIDNextAttributeValue() const;

	inline sal_uInt32 getUIDValidityAttributeValue() const;

	inline sal_uInt32 getUnseenAttributeValue() const;
};

inline bool INetIMAPStatusResponse::appendMailboxName(ByteString const &
													      rSuffix)
{
	if (rSuffix.Len() > STRING_MAXLEN - m_aMailboxName.Len())
		return false;
	m_aMailboxName += rSuffix;
	return true;
}

inline void INetIMAPStatusResponse::setMessagesAttribute(sal_uInt32 nValue)
{
	m_eAttributesPresent
		= INetIMAPStatusAttributes(m_eAttributesPresent
								       | INET_IMAP_STATUS_ATTRIBUTE_MESSAGES);
	m_nMessagesAttributeValue = nValue;
}

inline void INetIMAPStatusResponse::setRecentAttribute(sal_uInt32 nValue)
{
	m_eAttributesPresent
		= INetIMAPStatusAttributes(m_eAttributesPresent
								       | INET_IMAP_STATUS_ATTRIBUTE_RECENT);
	m_nRecentAttributeValue = nValue;
}

inline void INetIMAPStatusResponse::setUIDNextAttribute(sal_uInt32 nValue)
{
	m_eAttributesPresent
		= INetIMAPStatusAttributes(m_eAttributesPresent
								       | INET_IMAP_STATUS_ATTRIBUTE_UIDNEXT);
	m_nUIDNextAttributeValue = nValue;
}

inline void INetIMAPStatusResponse::setUIDValidityAttribute(sal_uInt32 nValue)
{
	m_eAttributesPresent = INetIMAPStatusAttributes(
		                       m_eAttributesPresent
							       | INET_IMAP_STATUS_ATTRIBUTE_UIDVALIDITY);
	m_nUIDValidityAttributeValue = nValue;
}

inline void INetIMAPStatusResponse::setUnseenAttribute(sal_uInt32 nValue)
{
	m_eAttributesPresent
		= INetIMAPStatusAttributes(m_eAttributesPresent
								       | INET_IMAP_STATUS_ATTRIBUTE_UNSEEN);
	m_nUnseenAttributeValue = nValue;
}

inline sal_uInt32 INetIMAPStatusResponse::getMessagesAttributeValue() const
{
	DBG_ASSERT(m_eAttributesPresent & INET_IMAP_STATUS_ATTRIBUTE_MESSAGES,
			   "INetIMAPStatusResponse::getMessagesAttributeValue(): None");
	return m_nMessagesAttributeValue;
}

inline sal_uInt32 INetIMAPStatusResponse::getRecentAttributeValue() const
{
	DBG_ASSERT(m_eAttributesPresent & INET_IMAP_STATUS_ATTRIBUTE_RECENT,
			   "INetIMAPStatusResponse::getRecentAttributeValue(): None");
	return m_nRecentAttributeValue;
}

inline sal_uInt32 INetIMAPStatusResponse::getUIDNextAttributeValue() const
{
	DBG_ASSERT(m_eAttributesPresent & INET_IMAP_STATUS_ATTRIBUTE_UIDNEXT,
			   "INetIMAPStatusResponse::getUIDNextAttributeValue(): None");
	return m_nUIDNextAttributeValue;
}

inline sal_uInt32 INetIMAPStatusResponse::getUIDValidityAttributeValue() const
{
	DBG_ASSERT(m_eAttributesPresent & INET_IMAP_STATUS_ATTRIBUTE_UIDVALIDITY,
			   "INetIMAPStatusResponse::getUIDValidityAttributeValue():"
			       " None");
	return m_nUIDValidityAttributeValue;
}

inline sal_uInt32 INetIMAPStatusResponse::getUnseenAttributeValue() const
{
	DBG_ASSERT(m_eAttributesPresent & INET_IMAP_STATUS_ATTRIBUTE_UNSEEN,
			   "INetIMAPStatusResponse::getUnseenAttributeValue(): None");
	return m_nUnseenAttributeValue;
}

//============================================================================
class INetIMAPExistsResponse: public INetIMAPResponse
{
	sal_uInt32 m_nMessageCount;

public:
	INetIMAPExistsResponse(sal_uInt32 nTheMessageCount):
		m_nMessageCount(nTheMessageCount) {}

	virtual Type getType() const;

	sal_uInt32 getMessageCount() const { return m_nMessageCount; }
};

//============================================================================
class INetIMAPRecentResponse: public INetIMAPResponse
{
	sal_uInt32 m_nRecentMessageCount;

public:
	INetIMAPRecentResponse(sal_uInt32 nTheRecentMessageCount):
		m_nRecentMessageCount(nTheRecentMessageCount) {}

	virtual Type getType() const;

	sal_uInt32 getRecentMessageCount() const { return m_nRecentMessageCount; }
};

//============================================================================
class INetIMAPExpungeResponse: public INetIMAPResponse
{
	sal_uInt32 m_nMessageSequenceNumber;

public:
	INetIMAPExpungeResponse(sal_uInt32 nTheMessageSequenceNumber):
		m_nMessageSequenceNumber(nTheMessageSequenceNumber) {}

	virtual Type getType() const;

	sal_uInt32 getMessageSequenceNumber() const
	{ return m_nMessageSequenceNumber; }
};

//============================================================================
class INetIMAPFetchResponseBodySection: public INetIMAPBodySectionDescriptor
{
	INetCoreNewsMessage * m_pData;
	sal_uInt32 m_nPartialOffset;
	bool m_bPartial;

public:
	INetIMAPFetchResponseBodySection(): m_bPartial(false), m_pData(0) {}

	virtual ~INetIMAPFetchResponseBodySection();

	inline void setPartial(sal_uInt32 nThePartialOffset);

	void setData(INetCoreNewsMessage * pTheData) { m_pData = pTheData; }

	bool isPartial() const { return m_bPartial; }

	INetCoreNewsMessage const * getData() const { return m_pData; }
};

inline void
INetIMAPFetchResponseBodySection::setPartial(sal_uInt32 nThePartialOffset)
{
	m_bPartial = true;
	m_nPartialOffset = nThePartialOffset;
}

//============================================================================
class INetIMAPFetchResponseBodySectionList: private List
{
public:
	virtual ~INetIMAPFetchResponseBodySectionList();

	inline void append(INetIMAPFetchResponseBodySection * pSection);

	sal_uInt32 getCount() const { return Count(); }

	inline INetIMAPFetchResponseBodySection const & get(sal_uInt32 nIndex)
		const;
};

inline void INetIMAPFetchResponseBodySectionList::append(
	            INetIMAPFetchResponseBodySection * pSection)
{
	DBG_ASSERT(pSection,
			   "INetIMAPFetchResponseBodySectionList::append(): Null");
	Insert(pSection, LIST_APPEND);
}

inline INetIMAPFetchResponseBodySection const &
INetIMAPFetchResponseBodySectionList::get(sal_uInt32 nIndex) const
{
	return
		*static_cast< INetIMAPFetchResponseBodySection * >(GetObject(nIndex));
}

//============================================================================
class INetIMAPFetchResponse: public INetIMAPResponse
{
public:
	enum Parts
	{
		PART_ENVELOPE = 0x001,
		PART_FLAGS = 0x002,
		PART_INTERNALDATE = 0x004,
		PART_RFC822 = 0x008,
		PART_RFC822_HEADER = 0x010,
		PART_RFC822_TEXT = 0x020,
		PART_RFC822_SIZE = 0x040,
		PART_BODY = 0x080,
		PART_BODYSTRUCTURE = 0x100,
		PART_UID = 0x200
	};

	enum EnvelopeParts
	{
		ENVELOPE_DATE = 0x001,
		ENVELOPE_SUBJECT = 0x002,
		ENVELOPE_FROM = 0x004,
		ENVELOPE_SENDER = 0x008,
		ENVELOPE_REPLY_TO = 0x010,
		ENVELOPE_TO = 0x020,
		ENVELOPE_CC = 0x040,
		ENVELOPE_BCC = 0x080,
		ENVELOPE_IN_REPLY_TO = 0x100,
		ENVELOPE_MESSAGE_ID = 0x200
	};

	enum Flags
	{
		FLAG_ANSWERED = 0x01,
		FLAG_FLAGGED = 0x02,
		FLAG_DELETED = 0x04,
		FLAG_SEEN = 0x08,
		FLAG_DRAFT = 0x10,
		FLAG_RECENT = 0x20
	};

private:
	DateTime m_aEnvelopeDate;
	ByteString m_aEnvelopeSubject;
	ByteString m_aEnvelopeFrom;
	ByteString m_aEnvelopeSender;
	ByteString m_aEnvelopeReplyTo;
	ByteString m_aEnvelopeTo;
	ByteString m_aEnvelopeCC;
	ByteString m_aEnvelopeBCC;
	ByteString m_aEnvelopeInReplyTo;
	ByteString m_aEnvelopeMessageID;
	INetIMAPFlagKeywordList m_aFlagKeywords;
	DateTime m_aInternalDate;
	INetIMAPFetchResponseBodySectionList m_aBodySections;
	sal_uInt32 m_nMessageSequenceNumber;
	sal_uInt32 m_nRFC822Size;
	sal_uInt32 m_nUID;
	Parts m_ePresentParts;
	EnvelopeParts m_ePresentEnvelopeParts;
	Flags m_eFlags;

public:
	inline INetIMAPFetchResponse(sal_uInt32 nTheMessageSequenceNumber);

	virtual Type getType() const;

	inline void setEnvelope();

	inline void setEnvelopeDate(DateTime const & rTheEnvelopeDate);

	inline void setEnvelopeSubject(ByteString const & rTheEnvelopeSubject);

	inline void setEnvelopeFrom(ByteString const & rTheEnvelopeFrom);

	inline void setEnvelopeSender(ByteString const & rTheEnvelopeSender);

	inline void setEnvelopeReplyTo(ByteString const & rTheEnvelopeReplyTo);

	inline void setEnvelopeTo(ByteString const & rTheEnvelopeTo);

	inline void setEnvelopeCC(ByteString const & rTheEnvelopeCC);

	inline void setEnvelopeBCC(ByteString const & rTheEnvelopeBCC);

	inline void setEnvelopeInReplyTo(ByteString const &
									     rTheEnvelopeInReplyTo);

	inline void setEnvelopeMessageID(ByteString const &
									     rTheEnvelopeMessageID);

	inline void setFlags();

	inline void addFlags(Flags eTheFlags);

	inline INetIMAPFlagKeywordList & getFlagKeywords();

	inline void setInternalDate(DateTime const & rTheInternalDate);

	inline void setRFC822Size(sal_uInt32 nTheRFC822Size);

	void appendBodySection(INetIMAPFetchResponseBodySection * pSection)
	{ m_aBodySections.append(pSection); }

	inline void setUID(sal_uInt32 nTheUID);

	sal_uInt32 getMessageSequenceNumber() const
	{ return m_nMessageSequenceNumber; }

	Parts getPresentParts() const { return m_ePresentParts; }

	inline EnvelopeParts getPresentEnvelopeParts() const;

	inline DateTime const & getEnvelopeDate() const;

	inline ByteString const & getEnvelopeSubject() const;

	inline ByteString const & getEnvelopeFrom() const;

	inline ByteString const & getEnvelopeSender() const;

	inline ByteString const & getEnvelopeReplyTo() const;

	inline ByteString const & getEnvelopeTo() const;

	inline ByteString const & getEnvelopeCC() const;

	inline ByteString const & getEnvelopeBCC() const;

	inline ByteString const & getEnvelopeInReplyTo() const;

	inline ByteString const & getEnvelopeMessageID() const;

	inline Flags getFlags() const;

	inline INetIMAPFlagKeywordList const & getFlagKeywords() const;

	inline DateTime const & getInternalDate() const;

	inline sal_uInt32 getRFC822Size() const;

	INetIMAPFetchResponseBodySectionList const & getBodySections() const
	{ return m_aBodySections; }

	inline sal_uInt32 getUID() const;
};

inline
INetIMAPFetchResponse::INetIMAPFetchResponse(sal_uInt32
											     nTheMessageSequenceNumber):
	m_nMessageSequenceNumber(nTheMessageSequenceNumber),
	m_ePresentParts(Parts(0)),
	m_aEnvelopeDate(0, 0),
	m_aInternalDate(0, 0)
{}

inline void INetIMAPFetchResponse::setEnvelope()
{
	m_ePresentParts = Parts(m_ePresentParts | PART_ENVELOPE);
	m_ePresentEnvelopeParts = EnvelopeParts(0);
}

inline void INetIMAPFetchResponse::setEnvelopeDate(DateTime const &
												       rTheEnvelopeDate)
{
	DBG_ASSERT(m_ePresentParts & PART_ENVELOPE,
			   "INetIMAPFetchResponse::setEnvelopeDate(): None");
	m_ePresentEnvelopeParts
		= EnvelopeParts(m_ePresentEnvelopeParts | ENVELOPE_DATE);
	m_aEnvelopeDate = rTheEnvelopeDate;
}

inline void INetIMAPFetchResponse::setEnvelopeSubject(ByteString const &
													      rTheEnvelopeSubject)
{
	DBG_ASSERT(m_ePresentParts & PART_ENVELOPE,
			   "INetIMAPFetchResponse::setEnvelopeSubject(): None");
	m_ePresentEnvelopeParts
		= EnvelopeParts(m_ePresentEnvelopeParts | ENVELOPE_SUBJECT);
	m_aEnvelopeSubject = rTheEnvelopeSubject;
}

inline void INetIMAPFetchResponse::setEnvelopeFrom(ByteString const &
												       rTheEnvelopeFrom)
{
	DBG_ASSERT(m_ePresentParts & PART_ENVELOPE,
			   "INetIMAPFetchResponse::setEnvelopeFrom(): None");
	m_ePresentEnvelopeParts
		= EnvelopeParts(m_ePresentEnvelopeParts | ENVELOPE_FROM);
	m_aEnvelopeFrom = rTheEnvelopeFrom;
}

inline void INetIMAPFetchResponse::setEnvelopeSender(ByteString const &
													     rTheEnvelopeSender)
{
	DBG_ASSERT(m_ePresentParts & PART_ENVELOPE,
			   "INetIMAPFetchResponse::setEnvelopeSender(): None");
	m_ePresentEnvelopeParts
		= EnvelopeParts(m_ePresentEnvelopeParts | ENVELOPE_SENDER);
	m_aEnvelopeSender = rTheEnvelopeSender;
}

inline void INetIMAPFetchResponse::setEnvelopeReplyTo(ByteString const &
													      rTheEnvelopeReplyTo)
{
	DBG_ASSERT(m_ePresentParts & PART_ENVELOPE,
			   "INetIMAPFetchResponse::setEnvelopeReplyTo(): None");
	m_ePresentEnvelopeParts
		= EnvelopeParts(m_ePresentEnvelopeParts | ENVELOPE_REPLY_TO);
	m_aEnvelopeReplyTo = rTheEnvelopeReplyTo;
}

inline void INetIMAPFetchResponse::setEnvelopeTo(ByteString const &
												     rTheEnvelopeTo)
{
	DBG_ASSERT(m_ePresentParts & PART_ENVELOPE,
			   "INetIMAPFetchResponse::setEnvelopeTo(): None");
	m_ePresentEnvelopeParts
		= EnvelopeParts(m_ePresentEnvelopeParts | ENVELOPE_TO);
	m_aEnvelopeTo = rTheEnvelopeTo;
}

inline void INetIMAPFetchResponse::setEnvelopeCC(ByteString const &
												     rTheEnvelopeCC)
{
	DBG_ASSERT(m_ePresentParts & PART_ENVELOPE,
			   "INetIMAPFetchResponse::setEnvelopeCC(): None");
	m_ePresentEnvelopeParts
		= EnvelopeParts(m_ePresentEnvelopeParts | ENVELOPE_CC);
	m_aEnvelopeCC = rTheEnvelopeCC;
}

inline void INetIMAPFetchResponse::setEnvelopeBCC(ByteString const &
												      rTheEnvelopeBCC)
{
	DBG_ASSERT(m_ePresentParts & PART_ENVELOPE,
			   "INetIMAPFetchResponse::setEnvelopeBCC(): None");
	m_ePresentEnvelopeParts
		= EnvelopeParts(m_ePresentEnvelopeParts | ENVELOPE_BCC);
	m_aEnvelopeBCC = rTheEnvelopeBCC;
}

inline
void INetIMAPFetchResponse::setEnvelopeInReplyTo(ByteString const &
												     rTheEnvelopeInReplyTo)
{
	DBG_ASSERT(m_ePresentParts & PART_ENVELOPE,
			   "INetIMAPFetchResponse::setEnvelopeInReplyTo(): None");
	m_ePresentEnvelopeParts
		= EnvelopeParts(m_ePresentEnvelopeParts | ENVELOPE_IN_REPLY_TO);
	m_aEnvelopeInReplyTo = rTheEnvelopeInReplyTo;
}

inline
void INetIMAPFetchResponse::setEnvelopeMessageID(ByteString const &
												     rTheEnvelopeMessageID)
{
	DBG_ASSERT(m_ePresentParts & PART_ENVELOPE,
			   "INetIMAPFetchResponse::setEnvelopeMessageID(): None");
	m_ePresentEnvelopeParts
		= EnvelopeParts(m_ePresentEnvelopeParts | ENVELOPE_MESSAGE_ID);
	m_aEnvelopeMessageID = rTheEnvelopeMessageID;
}

inline void INetIMAPFetchResponse::setFlags()
{
	m_ePresentParts = Parts(m_ePresentParts | PART_FLAGS);
	m_eFlags = Flags(0);
}

inline void INetIMAPFetchResponse::addFlags(Flags eTheFlags)
{
	DBG_ASSERT(m_ePresentParts & PART_FLAGS,
			   "INetIMAPFetchResponse::addFlags(): None");
	m_eFlags = Flags(m_eFlags | eTheFlags);
}

inline INetIMAPFlagKeywordList & INetIMAPFetchResponse::getFlagKeywords()
{
	DBG_ASSERT(m_ePresentParts & PART_FLAGS,
			   "INetIMAPFetchResponse::getFlagKeywords(): None");
	return m_aFlagKeywords;
}

inline void INetIMAPFetchResponse::setInternalDate(DateTime const &
												       rTheInternalDate)
{
	m_ePresentParts = Parts(m_ePresentParts | PART_INTERNALDATE);
	m_aInternalDate = rTheInternalDate;
}

inline void INetIMAPFetchResponse::setRFC822Size(sal_uInt32 nTheRFC822Size)
{
	m_ePresentParts = Parts(m_ePresentParts | PART_RFC822_SIZE);
	m_nRFC822Size = nTheRFC822Size;
}

inline void INetIMAPFetchResponse::setUID(sal_uInt32 nTheUID)
{
	m_ePresentParts = Parts(m_ePresentParts | PART_UID);
	m_nUID = nTheUID;
}

inline INetIMAPFetchResponse::EnvelopeParts
INetIMAPFetchResponse::getPresentEnvelopeParts() const
{
	DBG_ASSERT(m_ePresentParts & PART_ENVELOPE,
			   "INetIMAPFetchResponse::getPresentEnvelopeParts(): None");
	return m_ePresentEnvelopeParts;
}

inline DateTime const & INetIMAPFetchResponse::getEnvelopeDate() const
{
	DBG_ASSERT(m_ePresentParts & PART_ENVELOPE
			   && m_ePresentEnvelopeParts & ENVELOPE_DATE,
			   "INetIMAPFetchResponse::getEnvelopeDate(): None");
	return m_aEnvelopeDate;
}

inline ByteString const & INetIMAPFetchResponse::getEnvelopeSubject() const
{
	DBG_ASSERT(m_ePresentParts & PART_ENVELOPE
			   && m_ePresentEnvelopeParts & ENVELOPE_SUBJECT,
			   "INetIMAPFetchResponse::getEnvelopeSubject(): None");
	return m_aEnvelopeSubject;
}

inline ByteString const & INetIMAPFetchResponse::getEnvelopeFrom() const
{
	DBG_ASSERT(m_ePresentParts & PART_ENVELOPE
			   && m_ePresentEnvelopeParts & ENVELOPE_FROM,
			   "INetIMAPFetchResponse::getEnvelopeFrom(): None");
	return m_aEnvelopeFrom;
}

inline ByteString const & INetIMAPFetchResponse::getEnvelopeSender() const
{
	DBG_ASSERT(m_ePresentParts & PART_ENVELOPE
			   && m_ePresentEnvelopeParts & ENVELOPE_SENDER,
			   "INetIMAPFetchResponse::getEnvelopeSender(): None");
	return m_aEnvelopeSender;
}

inline ByteString const & INetIMAPFetchResponse::getEnvelopeReplyTo() const
{
	DBG_ASSERT(m_ePresentParts & PART_ENVELOPE
			   && m_ePresentEnvelopeParts & ENVELOPE_REPLY_TO,
			   "INetIMAPFetchResponse::getEnvelopeReplyTo(): None");
	return m_aEnvelopeReplyTo;
}

inline ByteString const & INetIMAPFetchResponse::getEnvelopeTo() const
{
	DBG_ASSERT(m_ePresentParts & PART_ENVELOPE
			   && m_ePresentEnvelopeParts & ENVELOPE_TO,
			   "INetIMAPFetchResponse::getEnvelopeTo(): None");
	return m_aEnvelopeTo;
}

inline ByteString const & INetIMAPFetchResponse::getEnvelopeCC() const
{
	DBG_ASSERT(m_ePresentParts & PART_ENVELOPE
			   && m_ePresentEnvelopeParts & ENVELOPE_CC,
			   "INetIMAPFetchResponse::getEnvelopeCC(): None");
	return m_aEnvelopeCC;
}

inline ByteString const & INetIMAPFetchResponse::getEnvelopeBCC() const
{
	DBG_ASSERT(m_ePresentParts & PART_ENVELOPE
			   && m_ePresentEnvelopeParts & ENVELOPE_BCC,
			   "INetIMAPFetchResponse::getEnvelopeBCC(): None");
	return m_aEnvelopeBCC;
}

inline ByteString const & INetIMAPFetchResponse::getEnvelopeInReplyTo() const
{
	DBG_ASSERT(m_ePresentParts & PART_ENVELOPE
			   && m_ePresentEnvelopeParts & ENVELOPE_TO,
			   "INetIMAPFetchResponse::getEnvelopeTo(): None");
	return m_aEnvelopeTo;
}

inline ByteString const & INetIMAPFetchResponse::getEnvelopeMessageID() const
{
	DBG_ASSERT(m_ePresentParts & PART_ENVELOPE
			   && m_ePresentEnvelopeParts & ENVELOPE_MESSAGE_ID,
			   "INetIMAPFetchResponse::getEnvelopeMessageID(): None");
	return m_aEnvelopeMessageID;
}

inline INetIMAPFetchResponse::Flags INetIMAPFetchResponse::getFlags() const
{
	DBG_ASSERT(m_ePresentParts & PART_FLAGS,
			   "INetIMAPFetchResponse::getFlags(): None");
	return m_eFlags;
}

inline
INetIMAPFlagKeywordList const & INetIMAPFetchResponse::getFlagKeywords() const
{
	DBG_ASSERT(m_ePresentParts & PART_FLAGS,
			   "INetIMAPFetchResponse::getFlagKeywords(): None");
	return m_aFlagKeywords;
}

inline DateTime const & INetIMAPFetchResponse::getInternalDate() const
{
	DBG_ASSERT(m_ePresentParts & PART_INTERNALDATE,
			   "INetIMAPFetchResponse::getInternalDate(): None");
	return m_aInternalDate;
}

inline sal_uInt32 INetIMAPFetchResponse::getRFC822Size() const
{
	DBG_ASSERT(m_ePresentParts & PART_RFC822_SIZE,
			   "INetIMAPFetchResponse::getRFC822Size(): None");
	return m_nRFC822Size;
}

inline sal_uInt32 INetIMAPFetchResponse::getUID() const
{
	DBG_ASSERT(m_ePresentParts & PART_UID,
			   "INetIMAPFetchResponse::getUID(): None");
	return m_nUID;
}

//============================================================================
class INetIMAPCapabilityResponse: public INetIMAPResponse
{
	INetIMAPClient::Capabilities m_eCapabilities;

public:
	INetIMAPCapabilityResponse():
		m_eCapabilities(INetIMAPClient::Capabilities(0)) {}

	virtual Type getType() const;

	inline void addCapabilities(INetIMAPClient::Capabilities
								    eTheCapabilities);

	INetIMAPClient::Capabilities getCapabilities() const
	{ return m_eCapabilities; }
};

inline void
INetIMAPCapabilityResponse::addCapabilities(INetIMAPClient::Capabilities
											    eTheCapabilities)
{
	m_eCapabilities
		= INetIMAPClient::Capabilities(m_eCapabilities | eTheCapabilities);
}

//============================================================================
class INetIMAPNamespace
{
	ByteString m_aPrefix;
	sal_Char m_cHierarchySeparator;

public:
	inline INetIMAPNamespace(ByteString const & rThePrefix,
							 sal_Char cTheHierarchySeparator);

	ByteString const & getPrefix() const { return m_aPrefix; }

	sal_Char getHierarchySeparator() const { return m_cHierarchySeparator; }
};

inline INetIMAPNamespace::INetIMAPNamespace(ByteString const & rThePrefix,
											sal_Char cTheHierarchySeparator):
	m_aPrefix(rThePrefix),
	m_cHierarchySeparator(cTheHierarchySeparator)
{}

//============================================================================
class INetIMAPNamespaceList: private List
{
public:
	virtual ~INetIMAPNamespaceList();

	void add(INetIMAPNamespace * pNamespace)
	{ Insert(pNamespace, LIST_APPEND); }

	sal_uInt32 getCount() const { return Count(); }

	INetIMAPNamespace const & get(sal_uInt32 nIndex) const
	{ return *static_cast< INetIMAPNamespace * >(GetObject(nIndex)); }
};

//============================================================================
class INetIMAPNamespaceResponse: public INetIMAPResponse
{
public:
	enum NamespaceKind
	{
		NAMESPACE_PERSONAL,
		NAMESPACE_OTHER_USERS,
		NAMESPACE_SHARED
	};

private:
	enum { NAMESPACE_TYPE_COUNT = NAMESPACE_SHARED + 1 };

	INetIMAPNamespaceList * m_pNamespaces[NAMESPACE_TYPE_COUNT];

public:
	inline INetIMAPNamespaceResponse();

	virtual ~INetIMAPNamespaceResponse();

	virtual Type getType() const;

	void addNamespace(NamespaceKind eKind, INetIMAPNamespace * pNamespace);

	inline INetIMAPNamespaceList const * getNamespaces(NamespaceKind eKind)
		const;
};

inline INetIMAPNamespaceResponse::INetIMAPNamespaceResponse()
{
	m_pNamespaces[NAMESPACE_PERSONAL] = 0;
	m_pNamespaces[NAMESPACE_OTHER_USERS] = 0;
	m_pNamespaces[NAMESPACE_SHARED] = 0;
}

inline INetIMAPNamespaceList const *
INetIMAPNamespaceResponse::getNamespaces(NamespaceKind eKind) const
{
	DBG_ASSERT(eKind < NAMESPACE_TYPE_COUNT,
			   "INetIMAPNamespaceResponse::getNamespaces(): Invalid kind");
	return m_pNamespaces[eKind];
}

#endif // INET_IMAPCLNT_HXX

