/* 
   WebDAV Properties manipulation
   Copyright (C) 2000, Joe Orton <joe@orton.demon.co.uk>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA

*/

#ifndef DAV_PROPS_H
#define DAV_PROPS_H

#include <http_request.h>
#include <dav_207.h>

BEGIN_NEON_DECLS

struct dav_propfind_handler_s;
typedef struct dav_propfind_handler_s dav_propfind_handler;

/* Callback types:
 * The return value is a cookie which is passed to end_resource again.
 * This can also be retrieved from element handlers using the 
 * dav_propfind_get_current_resource function. */
typedef void *(*dav_pf_start_resource)(void *userdata, const char *href);
typedef void (*dav_pf_end_resource)(void *userdata, void *resource,
				    const char *status_line, 
				    const http_status *status,
				    const char *description);
/* TODO: dav_pf_{start|end}_resource
 *    == dav_207_{start|end}_response.
 * Nuke dav_pf_*?
 */

/* Retrieve the current resource cookie of the given handler */
void *dav_propfind_get_current_resource(dav_propfind_handler *handler);

/* Create a PROPFIND handler, for the given URI.
 * Depth must be one of DAV_DEPTH_*.
 */
dav_propfind_handler *
dav_propfind_create(http_session *sess, const char *uri, int depth);

/* Set the start_resource and end_resource callbacks for the handler */
void dav_propfind_set_resource_handlers(dav_propfind_handler *handler,
					dav_pf_start_resource start_res,
					dav_pf_end_resource end_res);

hip_xml_parser *dav_propfind_get_parser(dav_propfind_handler *handler);

/* dav_propfind_allprop and dav_propfind_named will call the callbacks in 
 * this order:
 *   start_resource: Start a resource, href = "/foo/bar/"
 *     check_context, start_element, end_element: A property
 *     check_context, start_element, end_element: Another property
 *   end_resource: End the resource
 *   start_resource: Start a resource, href = "/foo/bar/another"
 *     check_context... etc.
 *  
 * Note that if properties have sub-elements, then these may be passed to the
 * element callbacks too.
 */

/* Find all properties. Returns HTTP_*. */
int dav_propfind_allprop(dav_propfind_handler *handler, void *userdata);

/* Find only named properties. Returns HTTP_*. */
int dav_propfind_named(dav_propfind_handler *handler, 
		       const dav_propname *names, void *userdata);

/* Destroy a propfind handler after use. */
void dav_propfind_destroy(dav_propfind_handler *handler);

/* A PROPPATCH request may include any number of operations. Pass an
 * array of these operations to dav_proppatch, with the last item
 * having the name element being NULL.  If the type is propset, the
 * property of the given name is set to the new value.  If the type is
 * propremove, the property of the given name is deleted, and the
 * value is ignored.  */
typedef struct {
    const dav_propname *name;
    enum {
	dav_propset,
	dav_propremove
    } type;
    const char *value;
} dav_proppatch_operation;

int dav_proppatch(http_session *sess, const char *uri,
		  const dav_proppatch_operation *items);

END_NEON_DECLS

#endif /* DAV_PROPS_H */
