#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <iomanip.h>
#include <fstream.h>
#include <assert.h>

#include "error.h"
#include "config_impl.hh"
#include "error_messages.hh"
#include "file_util.hh"
#include "itemize.hh"
#include "getdata.hh"

static const PspellModule a_module = PspellModule();

typedef PspellNotifier * PspellNotifierPtr;

PspellConfigImpl::PspellConfigImpl(const char * name,
				   const PspellKeyInfo * mainbegin, 
				   const PspellKeyInfo * mainend)
  : name_(name)
  , data_(new_pspell_string_map()),
    attached_(0)
{
  kmi.main_begin = mainbegin;
  kmi.main_end   = mainend;
  kmi.extra_begin = 0;
  kmi.extra_end   = 0;
  kmi.modules_begin = &a_module;
  kmi.modules_end   = &a_module;
  notifier_list = new PspellNotifierPtr[1];
  notifier_list[0] = 0;
}

PspellConfigImpl::~PspellConfigImpl() {
  delete data_;
  delete[] notifier_list;
}

PspellConfigImpl::PspellConfigImpl(const PspellConfigImpl & other) 
  : name_(other.name_), attached_(0), kmi(other.kmi) 
{
  data_ = other.data_->clone();
  notifier_list = new PspellNotifierPtr[1];
  notifier_list[0] = 0;
}

PspellConfigImpl & PspellConfigImpl::operator= (const PspellConfigImpl & other)
{
  delete data_;
  delete[] notifier_list;
  attached_ = 0;
  kmi = other.kmi;
  data_->assign(other.data_);
  notifier_list = new PspellNotifierPtr[1];
  notifier_list[0] = 0;
  return *this;
}

PspellConfig * PspellConfigImpl::clone() {
  return new PspellConfigImpl(*this);
}

bool PspellConfigImpl::assign(const PspellConfig * other) {
  *this = *(const PspellConfigImpl *)(other);
  return true;
}

void PspellConfigImpl::set_modules(const PspellModule * modbegin, 
				   const PspellModule * modend)
{
  kmi.modules_begin = modbegin;
  kmi.modules_end   = modend;
}

void PspellConfigImpl::set_extra(const PspellKeyInfo * begin, 
				 const PspellKeyInfo * end) 
{
  kmi.extra_begin = begin;
  kmi.extra_end   = end;
}

//
// Notifier methods
//

PspellNotifierEmulation * PspellConfigImpl::notifiers() const 
{
  return new PspellNotifierEmulation(notifier_list);
}
  
bool PspellConfigImpl::add_notifier(PspellNotifier * n) 
{
  PspellNotifier * * i = notifier_list;

  while (*i != 0 && *i != n)
    ++i;

  if (*i != 0) {
    
    return false;
    
  } else {

    PspellNotifier * * temp = notifier_list;
    size_t old_size = i - temp;
    notifier_list = new PspellNotifierPtr[old_size + 2];
    unsigned int j = 0;
    for (; j != old_size; ++j)
      notifier_list[j] = temp[j];
    notifier_list[j] = n;
    notifier_list[j+1] = 0;
    return true;

  }
}

bool PspellConfigImpl::remove_notifier(const PspellNotifier * n) 
{
  PspellNotifier * * i = notifier_list;
  while (*i != 0 && *i != n)
    ++i;
  if (*i == 0) {
    return false;
  } else {
    PspellNotifier * * temp = notifier_list;
    size_t old_size = i - temp;
    notifier_list = new PspellNotifierPtr[old_size];
    unsigned j = 0;
    for (; j != old_size - 1; ++j) {
      if (temp[j] != n)
	notifier_list[j] = temp[j];
    }
    notifier_list[j] = 0;
    return true;
  }
  return true;
}

bool PspellConfigImpl::replace_notifier(const PspellNotifier * o, 
					PspellNotifier * n) 
{
  PspellNotifier * * i = notifier_list;
  while (*i != 0 && *i != n)
    ++i;
  if (*i == 0) {
    return false;
  } else {
    *i = n;
    return true;
  }
}

bool PspellConfigImpl::have(const char * key) const 
{
  return data_->have(key); 
}

//
// retrive methods
//

int PspellConfigImpl::retrieve_bool(const char * key)
{
  const char * str = retrieve(key);
  return str == 0 ? -1 : retrieve(key)[0] == 't';
}

int PspellConfigImpl::retrieve_int(const char * key)
{
  const char * str = retrieve(key);
  if (str == 0) return -1;
  int i;
  sscanf(str, "%i", &i);
  return i;
}

const char * PspellConfigImpl::retrieve(const char * key)
{
  error_.reset_error();
  const char * value = data_->lookup(key);
  if (value != 0) 
    return value;
  else 
    return get_default(key);
}

bool PspellConfigImpl::retrieve_list(const char * key, 
				     PspellMutableContainer & m)
{
  get_default(key); // sets temp_str to default;
  if (error_number() != 0) return false;
  const char * value = data_->lookup(key);
  if (value != 0) {
    temp_str += ',';
    temp_str += data_->lookup(key);
  }
  itemize(temp_str.c_str(), m);
  return true;
}

static const PspellKeyInfo * find(const char * key, 
				  const PspellKeyInfo * i, 
				  const PspellKeyInfo * end) 
{
  while (i != end) {
    if (strcmp(key, i->name) == 0)
      return i;
    ++i;
  }
  return i;
}

static const PspellModule * find(const char * key, 
				 const PspellModule * i, 
				 const PspellModule * end) 
{
  while (i != end) {
    if (strcmp(key, i->name) == 0) 
      return i;
    ++i;
  }
  return i;
}

const char * PspellConfigImpl::base_name(const char * name) 
{
  const char * c = strchr(name, '-');
  unsigned int p = c ? c - name : -1;
  if ((p == 3 && (strncmp(name, "add",p) == 0 
		  || strncmp(name, "rem",p) == 0))
      || (p == 4 && strncmp(name, "dont",p) == 0)) 
    return name + p + 1;
  else
    return name;
}

const PspellKeyInfo * PspellConfigImpl::keyinfo(const char * key)
{
  error_.reset_error();
  const PspellKeyInfo * i;
  
  i = ::find(key, kmi.main_begin, kmi.main_end);
  if (i != kmi.main_end) return i;
  
  i = ::find(key, kmi.extra_begin, kmi.extra_end);
  if (i != kmi.extra_end) return i;
  
  const char * h = strchr(key, '-');

  if (h == 0) 
    {error_.set_error(unknown_key, key); return 0;}

  PspellString k(key,h-key);
  const PspellModule * j = ::find(k.c_str(), 
				  kmi.modules_begin, 
				  kmi.modules_end);
  if (j == kmi.modules_end)
    {error_.set_error(unknown_key, key); return 0;}
  
  i = ::find(key, j->begin, j->end);
  if (i != j->end) return i;
  
  error_.set_error(unknown_key, key); return 0;
}

const char * PspellConfigImpl::get_default(const char * key)
{
  error_.reset_error();

  const PspellKeyInfo * ki = keyinfo(key);
  if (error_number() != 0) return 0;

  temp_str = "";
  bool   in_replace = false;
  PspellString final_str;
  PspellString replace;
  for(const char * i = ki->def; *i; ++i) {

    if (!in_replace) {

      if (*i == '<') {
	in_replace = true;
      } else {
	final_str += *i;
      }

    } else { // in_replace
      
      if (*i == '/' || *i == ':' || *i == '|') {
	char sep = *i;
	PspellString second;
	++i;
	while (*i != '\0' && *i != '>') second += *i++;
	if (sep == '/') {
	  PspellString s1 = retrieve(replace.c_str());
	  PspellString s2 = retrieve(second.c_str());
	  final_str += add_possible_dir(s1.c_str(), s2.c_str());
	} else if (sep == ':') {
	  PspellString s1 = retrieve(replace.c_str());
	  final_str += add_possible_dir(s1.c_str(), second.c_str());
	} else { // sep == '|'
	  assert(replace[0] == '$');
	  const char * env = getenv(replace.c_str()+1);
	  final_str += env ? env : second;
	}
	replace = "";
	in_replace = false;

      } else if (*i == '>') {

	final_str += retrieve(replace.c_str());
	replace = "";
	in_replace = false;

      } else {

	replace += *i;

      }

    }
      
  }
  temp_str = final_str;
  return temp_str.c_str();
}


#define notify_all(ki, value, fun)        \
  do {                                    \
    PspellNotifier * * i = notifier_list; \
    while (*i != 0) {                     \
      (*i)->fun(ki,value);                \
      ++i;                                \
    }                                     \
  } while (false)


class PspellNotifyListBlockChange : public PspellMutableContainer 
{
  const PspellKeyInfo * key_info;
  PspellNotifier * * notifier_list;
public:
  PspellNotifyListBlockChange(const PspellKeyInfo * ki, PspellNotifier * * n);
  bool add(const char *);
  bool remove(const char *);
  void clear();
};

PspellNotifyListBlockChange::
PspellNotifyListBlockChange(const PspellKeyInfo * ki, PspellNotifier * * n)
  : key_info(ki), notifier_list(n) {}

bool PspellNotifyListBlockChange::add(const char * v) {
  notify_all(key_info, v, item_added);
  return true;
}

bool PspellNotifyListBlockChange::remove(const char * v) {
  notify_all(key_info, v, item_removed);
  return true;
}

void PspellNotifyListBlockChange::clear() {
  notify_all(key_info, 0, all_removed);
}

bool PspellConfigImpl::replace(const char * k, const char * value) {
  if (strcmp(value,"<default>") == 0)
    return remove(k);

  error_.reset_error();
  
  const char * key;
  const char * i = strchr(k, '-');
  int p = (i == 0 ? -1 : i - k);
  if ((p == 3 && (strncmp(k, "add",p) == 0 
		  || strncmp(k, "rem",p) == 0))
      || (p == 4 && strncmp(k, "dont",p) == 0))
    {
      key = k + p + 1;
      if (strncmp(key, "all-", 4) == 0) {
	key = key + 4;
	p = 7;
      }
    } else {
      key = k;
      p = 0;
    }
  
  const PspellKeyInfo * ki = keyinfo(key);

  if (error_number() != 0) return false;

  if (ki->otherdata[1] && attached_) {
    error_.set_error(cant_change_value, key);
    return false;
  }
  
  assert(ki->def != 0); // if null this key should never have values
                        // directly added to it

  int num;
  switch (ki->type) {
    
  case PspellKeyInfoBool:
    
    if (p == 4 || (p == 0 && strcmp(value,"false") == 0)) {

      data_->replace(key, "false");
      notify_all(ki, false, item_updated);
      return true;

    } else if (p != 0) {

      error_.set_error(unknown_key,  k);
      return false;

    } else if (value[0] == '\0' || strcmp(value,"true") == 0) {

      data_->replace(key, "true");
      notify_all(ki, true, item_updated);
      return true;

    } else {

      error_.set_error(bad_value, key, value, "either \"true\" or \"false\"");
      return false;

    }
    break;
      
  case PspellKeyInfoString:
      
    if (p == 0) {

      data_->replace(key,value);
      notify_all(ki, value, item_updated);
      return true;
      
    } else {
      
      error_.set_error(unknown_key,  key);
      return false;
      
    }
    break;
      
  case PspellKeyInfoInt:

    if (p == 0 && sscanf(value, "%i", &num) == 1 && num >= 0) {

      data_->replace(key,value);
      notify_all(ki, num, item_updated);
      return true;

    } else if (p != 0) {

      error_.set_error(unknown_key, key);
      return false;

    } else {

      error_.set_error(bad_value, key, value, "a positive integer");
      return false;

    }
    break;

  case PspellKeyInfoList:

    char a;
    if (p == 0) {
      error_.set_error(list_set, key); 
      return false;
    } else if (p == 7) {        // prefix must be "rem-all-"
      if (value[0] != '\0') {
	error_.set_error(bad_value, k, value, "nothing");
	return false;
      }
      a = '!';
    } else if (k[0] == 'a') { // prefix must be "add-"
      a = '+';
    } else {                  // prefix must be "rem-"
      a = '-';
    }

    if (a != '!') {
      i = data_->lookup(key);
      if (i == 0) i = "";
      PspellString s = i;
      s += ',';
      s += a;
      s += value;
      data_->replace(key, s.c_str());
    } else {
      data_->replace(key, "!");
    }

    switch (a) {
    case '!': 
      notify_all(ki, value, all_removed);  
      break;
    case '+': 
      notify_all(ki, value, item_added);
      break;
    case '-': 
      notify_all(ki, value, item_removed);
      break;
    }

    break;
  }

  return true;
}

bool PspellConfigImpl::remove (const char * key) {
  error_.reset_error();

  const PspellKeyInfo * ki = keyinfo(key);

  if (error_number() != 0) return false;

  if (ki->otherdata[1] && attached_) {
    error_.set_error(cant_change_value, key);
    return false;
  }
  
  assert(ki->def != 0); // if null this key should never have values
                        // directly added to it

  bool success = data_->remove(key);

  switch (ki->type) {

  case PspellKeyInfoString:

    notify_all(ki, retrieve(key), item_updated);
    break;
    
  case PspellKeyInfoBool:

    notify_all(ki, retrieve_bool(key), item_updated);
    break;

  case PspellKeyInfoInt:

    notify_all(ki, retrieve_int(key), item_updated);
    break;

  case PspellKeyInfoList:
    
    PspellNotifyListBlockChange n(ki, notifier_list);
    retrieve_list(key, n);
    break;
  }

  return success;
}

PspellStringPairEmulation * PspellConfigImpl::elements() 
{
  return data_->elements();
}


/////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////

class PspellPossibleElementsEmul : public PspellKeyInfoEmulation
{
private:
  bool include_extra;
  const PspellConfigImpl * cd;
  const PspellKeyInfo * i;
  const PspellModule  * m;
public:
  PspellPossibleElementsEmul(const PspellConfigImpl * d, bool ic)
    : include_extra(ic), cd(d), i(d->kmi.main_begin), m(0) {}

  PspellKeyInfoEmulation * clone() const {
    return new PspellPossibleElementsEmul(*this);
  }

  void assign(const PspellKeyInfoEmulation * other) {
    *this = *(const PspellPossibleElementsEmul *)(other);
  }

  const PspellKeyInfo * next() {
    if (i == cd->kmi.main_end) {
      if (include_extra)
	i = cd->kmi.extra_begin;
      else
	i = cd->kmi.extra_end;
    }
      
    if (i == cd->kmi.extra_end) {
      m = cd->kmi.modules_begin;
      if (m == cd->kmi.modules_end) return 0;
      else i = m->begin;
    }

    if (m == 0)
      return i++;

    if (m == cd->kmi.modules_end)
      return 0;

    if (i == m->end) {
      ++m;
      if (m == cd->kmi.modules_end) return 0;
      else i = m->begin;
    }

    return i++;
  }

  bool at_end() const {
    return (m == cd->kmi.modules_end);
  }
};

PspellKeyInfoEmulation *
PspellConfigImpl::possible_elements(bool include_extra)
{
  return new PspellPossibleElementsEmul(this, include_extra);
}

class PspellListDump : public PspellMutableContainer 
{
  ostream * out;
  const char * name;
public:
  PspellListDump(ostream & o, const char * n) 
    : out(&o), name(n) {}
  bool add(const char * d) {
    *out << "add-" << name << " " << d << "\n";
    return true;
  }
  bool remove(const char * d) {
    *out << "rem-" << name << " " << d << "\n";
    return true;
  }
  void clear() {*out << "rem-all-" << name << "\n";} 
};

void PspellConfigImpl::write_to_stream(ostream & out, 
				       bool include_extra) 
{
  ios::fmtflags f = out.flags();
  out.setf(ios::left);
  //out << "## " << name() << "\n\n";
  PspellKeyInfoEmulation * els = possible_elements(include_extra);
  const PspellKeyInfo * i;
  while ((i = els->next()) != 0) {
    if (i->desc == 0) continue;
    out << "# " 
	<< (i->type ==  PspellKeyInfoList ? "add|rem-" : "")
	<< i->name << " descrip: " 
	<< (i->def == 0 ? "(action option) " : "")
	<< i->desc << "\n";
    if (i->def != 0) {
      out << "# " << i->name << " default: " << i->def << "\n";
      const char * value = data_->lookup(i->name);
      if (i->type != PspellKeyInfoList) {
	out << "# " << i->name << " current: " << retrieve(i->name) << "\n";
	if (value != 0) 
	  out << setw(15) << i->name << " " << value << "\n";
      } else {
	if (value != 0) {
	  PspellListDump ld(out, i->name);
	  itemize(value, ld);
	}
      }
    }
    out << "\n\n";
  }
  out.flags(f);
  delete els;
}

bool PspellConfigImpl::read_in(PspellGetLine & in) 
{
  PspellString key,value;
  while (getdata_pair(in, key, value)) {
    if (!replace(key.c_str(), value.c_str())) {
      return false;
    }
  }
  return true;
}

bool PspellConfigImpl::read_in_stream(istream & in) 
{
  PspellGetLineFromStream gl(&in);
  return read_in(gl);
}

bool PspellConfigImpl::read_in_file(const char * file) {
  ifstream in(file);
  if (in.rdstate() & ios::badbit) {
    error_.set_error(cant_read_file, file);
    return false;
  }
  PspellGetLineFromStream gl(&in);
  return read_in(gl);
}

bool PspellConfigImpl::read_in_string(const char * str) {
  PspellGetLineFromString gl(str);
  return read_in(gl);
}

void PspellConfigImpl::merge(PspellConfigImpl & other) {
  PspellKeyInfoEmulation * els = possible_elements();
  bool diff_name = strcmp(name(), other.name()) != 0;
  const PspellKeyInfo * k;
  const PspellKeyInfo * other_k;
  const char * other_name;
  const char * this_value;
  const char * other_value;
  while ( (k = els->next()) != 0) {
    if (diff_name && k->otherdata[0] == 'p'
	&& strncmp(k->name, other.name_.c_str(), other.name_.size())
	&& k->name[other.name_.size()] == '_')
      other_name = k->name + other.name_.size();
    else
      other_name = k->name;

    other_k = other.keyinfo(other_name);
    if (diff_name && other_k && other_k->otherdata[0] == 'r') continue;
    // the other key is a prefix key so skip it
    // when this is a prefix key than this key
    // would be prefix_
    
    if (other_k != 0 && 
	strcmp(k->def, other_k->def) == 0 
	&& !other.have(other_name))
      continue;
    
    other_value = other.retrieve(other_name);
    if (other_value == 0) continue;
    // if other_value == 0 then this key does not exist in the other
    // table.
    if (strcmp(other_value, "(default)") == 0) continue;
    this_value = retrieve(k->name);
    if (strcmp(this_value, other_value) == 0 && 
	!other.have(k->name)) continue;
    // if the two values match there is no need to insert it into the
    // table unless the other value is specificly set
    if (k->type != PspellKeyInfoList) {
      data_->replace(k->name, other_value);
    } else {
      PspellString new_value;
      if (other_value[0] != '!') {
	new_value  = this_value;
	new_value += ',';
      }
      new_value += other_value;
      data_->replace(k->name, new_value.c_str());
    }
  }
  delete els;
}

static const PspellKeyInfo config_keys[] = {
  {"language-tag", PspellKeyInfoString, "en", "language code"}
  , {"encoding",   PspellKeyInfoString, "iso8859-1", "encoding to expect data to be in"}
  , {"spelling",   PspellKeyInfoString, "", "spelling for languages with more than one"}
  , {"jargon",     PspellKeyInfoString, "", "extra information for the word list"}
  , {"ignore",     PspellKeyInfoInt,    "(default)", "ignore words <= n chars"}
  , {"sug-mode",   PspellKeyInfoString, "(default)", "suggestion mode"}
  , {"run-together",     PspellKeyInfoBool, "(default)", "consider run-together words legal"}
  , {"personal",      PspellKeyInfoString, "(default)", "personal word list to use"}
  , {"ignore-repl", PspellKeyInfoBool  , "(default)", "ignore commands to store replacement pairs"}
  , {"save-repl",   PspellKeyInfoBool  , "(default)", "save replacement pairs on save all"}
  , {"repl",        PspellKeyInfoString, "(default)", "replacements list file name"}
  , {"word-list-path", PspellKeyInfoList, DATADIR, 
       "Search path for word list information files"}
  , {"module-search-order", PspellKeyInfoList, "aspell,ispell", ""}
  , {"master",        PspellKeyInfoString, "", 0}
  , {"master-flags",  PspellKeyInfoString, "", 0}
  , {"module",        PspellKeyInfoString, "", 0}
  , {"data-dir",      PspellKeyInfoString, DATADIR, "", "r"}
};

const PspellKeyInfo * pspell_config_impl_keys_begin = config_keys;
const PspellKeyInfo * pspell_config_impl_keys_end   
  = config_keys + sizeof(config_keys)/sizeof(PspellKeyInfo);


PspellConfig * new_pspell_config() {
  return new PspellConfigImpl("pspell",
			      pspell_config_impl_keys_begin, 
			      pspell_config_impl_keys_end);
}
