/*************************************************************************
 *
 *  $RCSfile: cntstg.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: mhu $ $Date: 2001/07/25 13:04:54 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#ifndef _RTL_CRC_H_
#include <rtl/crc.h>
#endif
#ifndef _VOS_TIMER_HXX_
#include <vos/timer.hxx>
#endif
#ifndef _STORE_STORE_HXX_
#include <store/store.hxx>
#endif
#ifndef _SOLAR_H
#include <tools/solar.h>
#endif
#ifndef _TOOLS_DEBUG_HXX
#include <tools/debug.hxx>
#endif
#ifndef _FSYS_HXX
#include <tools/fsys.hxx>
#endif
#ifndef _BIGINT_HXX
#include <tools/bigint.hxx>
#endif

#ifndef _CNTSTG_HXX
#include "cntstg.hxx"
#endif

#ifndef _CNTSTORE_HXX
#include "cntstore.hxx"
#endif
#ifndef _CHAOS_STORITEM_HXX
#include "storitem.hxx"
#endif
#ifndef _CHAOS_STORSTRM_HXX
#include "storstrm.hxx"
#endif
#ifndef _CNTIDS_HRC
#include "cntids.hrc"	// for ERRCODE_CHAOS_*
#endif

using namespace chaos;

#define LONGNAMES_STREAM_VERSION 2
#define LONGNAMES_STREAM 		 ".longnames"
#define SHORTNAME_PREFIX 		 ".shortname:"
#define SHORTNAME_PREFIX_LEN	 11

#define AUTO_CLOSE_VIEW_STORAGES
#define VIEW_STORAGE_CLOSE_DELAY 2000

namespace chaos
{

/*========================================================================
 *
 * helper function mapStoreToToolsError.
 *
 *======================================================================*/

static ErrCode mapStoreToToolsError( const storeError eError )
{
	ErrCode nError = ERRCODE_IO_UNKNOWN;

	switch ( eError )
	{
		case store_E_None:
			nError = ERRCODE_NONE;
			break;
		case store_E_AccessViolation:
			nError = ERRCODE_IO_ACCESSDENIED;
			break;
		case store_E_LockingViolation:
			nError = ERRCODE_IO_LOCKVIOLATION;
			break;
		case store_E_CantSeek:
			nError = ERRCODE_IO_CANTSEEK;
			break;
		case store_E_CantRead:
			nError = ERRCODE_IO_CANTREAD;
			break;
		case store_E_CantWrite:
			nError = ERRCODE_IO_CANTWRITE;
			break;
		case store_E_InvalidAccess:
			nError = ERRCODE_IO_INVALIDACCESS;
			break;
		case store_E_InvalidHandle:
			nError = SVSTREAM_INVALID_HANDLE;
			break;
		case store_E_InvalidParameter:
			nError = ERRCODE_IO_INVALIDPARAMETER;
			break;
		case store_E_InvalidChecksum:
			nError = ERRCODE_IO_BADCRC;
			break;
		case store_E_AlreadyExists:
			nError = ERRCODE_IO_ALREADYEXISTS;
			break;
		case store_E_NotExists:
			nError = ERRCODE_IO_NOTEXISTS;
			break;
		case store_E_NotDirectory:
			nError = ERRCODE_IO_NOTADIRECTORY;
			break;
		case store_E_NotFile:
			nError = ERRCODE_IO_NOTAFILE;
			break;
// Not really an error. Returned by iter::next() to indicate EOF.
//		case store_E_NoMoreFiles:
//			nError = ;
//			break;
		case store_E_NameTooLong:
			nError = ERRCODE_IO_NAMETOOLONG;
			break;
		case store_E_OutOfMemory:
			nError = ERRCODE_IO_OUTOFMEMORY;
			break;
		case store_E_OutOfSpace:
			nError = ERRCODE_IO_OUTOFSPACE;
			break;
		case store_E_Pending:
			nError = ERRCODE_IO_PENDING;
			break;
		case store_E_WrongFormat:
			nError = ERRCODE_IO_WRONGFORMAT;
			break;
		case store_E_WrongVersion:
			nError = ERRCODE_IO_WRONGVERSION;
			break;
		case store_E_Unknown:
			nError = ERRCODE_IO_UNKNOWN;
			break;
		default:
			DBG_ERROR( "mapStoreToToolsError - unknown storeError" );
			nError = ERRCODE_IO_UNKNOWN;
			break;
	}

	return nError;
}

/*========================================================================
 *
 * helper function mapToolsToStoreStreamMode.
 *
 *======================================================================*/

storeAccessMode mapToolsToStoreStreamMode( StreamMode eStreamMode )
{
	storeAccessMode	eMode = store_AccessReadOnly;

	if ( ( eStreamMode & STREAM_WRITE ) &&
		 !( eStreamMode & STREAM_READ ) &&
		 !( eStreamMode & STREAM_NOCREATE ) )
		eMode = store_AccessCreate;
	else if ( ( eStreamMode & STREAM_WRITE ) &&
		 	  ( eStreamMode & STREAM_READ ) &&
		 	  !( eStreamMode & STREAM_NOCREATE ) )
		eMode = store_AccessReadCreate;
	else if ( ( eStreamMode & STREAM_WRITE ) &&
		 	  ( eStreamMode & STREAM_NOCREATE ) )
		eMode = store_AccessReadWrite;
	else if ( !( eStreamMode & STREAM_WRITE ) &&
		 	  ( eStreamMode & STREAM_READ ) )
		eMode = store_AccessReadOnly;
	else
		DBG_ERROR( "mapToolsToStoreStreamMode - Unknown mode!" );

	return eMode;
}

/*========================================================================
 *
 * class CntLongName*
 *
 *======================================================================*/
struct CntLongNameEntry
{
	String aLongName;
	String aShortName;
	BOOL   bPersist;

	CntLongNameEntry( const String& rLongName,
				      const String& rShortName,
					  BOOL  bStore )
	: aLongName( rLongName ), aShortName( rShortName ), bPersist( bStore ) {}
};

DECLARE_LIST( CntLongNamesList, CntLongNameEntry* )

/*========================================================================
 *
 * class CntDirDestroyer_Impl
 *
 *======================================================================*/
class CntDirDestroyer_Impl : public store::OStoreDirectory::traveller
{
	typedef store::OStoreDirectory::iterator iter;

	String            	   m_aFullName;
	store::OStoreFile 	   m_aStoreFile;
	ErrCode           	   m_nError;

public:
	CntDirDestroyer_Impl( const store::OStoreFile& rStore,
						  const String& rFullName )
	: m_aStoreFile( rStore ), m_aFullName( rFullName ),
	  m_nError( ERRCODE_NONE ) {}

	ErrCode getError() const { return m_nError; }

	virtual sal_Bool visit( const iter& it );
};

//===========================================================================
//
// class CntStorageCloser_Impl
//
//===========================================================================
class CntStorageCloser_Impl : public vos::OTimer
{
	CntLazyRootStorage* m_pStorage;

protected:
	virtual ~CntStorageCloser_Impl();
	virtual void SAL_CALL onShot();

public:
	CntStorageCloser_Impl( const vos::TTimeValue& rTime,
						   CntLazyRootStorage* pStorage  );
};

//===========================================================================
//
// class CntStoreLockBytes_Impl
//
//===========================================================================
class CntStoreLockBytes_Impl : public SvLockBytes
{
	store::OStoreStream m_aStream;

public:
	CntStoreLockBytes_Impl( const store::OStoreStream& rStream )
	: m_aStream( rStream ) {}
	virtual ~CntStoreLockBytes_Impl();

	// SvLockBytes overloads.
	virtual ErrCode ReadAt( ULONG nPos, void* pBuffer,
							ULONG nCount, ULONG* pRead ) const;

	virtual ErrCode WriteAt( ULONG nPos, const void* pBuffer,
							 ULONG nCount, ULONG* pWritten );

	virtual ErrCode Flush() const;

	virtual ErrCode SetSize( ULONG nSize );

	virtual ErrCode LockRegion( ULONG, ULONG, LockType );

	virtual ErrCode UnlockRegion( ULONG, ULONG, LockType );

	virtual ErrCode Stat( SvLockBytesStat* pStat, SvLockBytesStatFlag ) const;
};

} // namespace chaos

//============================================================================
//
// CntDirDestroyer_Impl Implementation.
//
//============================================================================

//virtual
sal_Bool CntDirDestroyer_Impl::visit( const iter& it )
{
	String aDirName( it.m_pszName );

	if ( it.m_nAttrib & CNTSTORE_ATTRIB_ISDIR )
	{
		// Create subdirectory.
		store::OStoreDirectory aDir;
		storeError eError = aDir.create( m_aStoreFile,
										 m_aFullName,
										 aDirName,
										 store_AccessReadWrite );
		aDirName += '/';

		if ( eError == store_E_None )
		{
			String aSubPath( m_aFullName );
			aSubPath += aDirName;

			CntDirDestroyer_Impl aSubDestroyer( m_aStoreFile, aSubPath );
			aDir.travel( aSubDestroyer );

			m_nError = aSubDestroyer.getError();
			if ( m_nError != ERRCODE_NONE )
				return 0;
		}

		eError = m_aStoreFile.remove( m_aFullName, aDirName );
		m_nError = mapStoreToToolsError( eError );
	}
	else
	{
		storeError eError = m_aStoreFile.remove( m_aFullName, aDirName );
		m_nError = mapStoreToToolsError( eError );
	}

	if ( m_nError == ERRCODE_IO_NOTEXISTS )
		m_nError = ERRCODE_NONE;

	return ( m_nError == ERRCODE_NONE );
}

//============================================================================
//
// CntStorage Implementation.
//
//============================================================================

//-----------------------------------------------------------------------------
CntStorage::CntStorage()
: m_pLongNamesList( NULL ),
  m_pParent( NULL ),
  m_pStore( NULL ),
  m_nFlags( 0 )
{
}

//-----------------------------------------------------------------------------
// virtual
CntStorage::~CntStorage()
{
	if ( m_pLongNamesList )
	{
		ULONG nCount = m_pLongNamesList->Count();
		for ( ULONG n = 0; n < nCount; ++n )
			delete m_pLongNamesList->GetObject( n );

		delete m_pLongNamesList;
	}

	delete m_pStore;
}

//-----------------------------------------------------------------------------
// virtual
ErrCode CntStorage::initialize( CntStorage* pParent,
								const String& rURL,
								USHORT nInitFlags /* = 0 */ )
{
	if ( !pParent )
	{
    	DBG_ERROR( "CntStorage::initialize - No parent!" );
		return ERRCODE_IO_INVALIDPARAMETER;
	}

    if ( !rURL.Len() )
	{
    	DBG_ERROR( "CntStorage::initialize - Invalid URL!" );
		return ERRCODE_IO_INVALIDPARAMETER;
	}

	vos::OGuard aGuard( m_aMutex );

	m_pParent = pParent;
	m_nFlags  = nInitFlags;

	// Get directory name from rURL.
   	String aName = rURL.GetToken( 1, '#' );

	// Name must be at least '/' +  an additional char.
   	if ( aName.Len() < 2 )
   	{
    	DBG_ERROR( "CntStorage::initialize - Invalid URL!" );
		return ERRCODE_IO_INVALIDPARAMETER;
   	}

	// Name must start with '/'.
   	if ( aName.GetChar( 0 ) != '/' )
   	{
    	DBG_ERROR( "CntStorage::initialize - Invalid URL!" );
		return ERRCODE_IO_INVALIDPARAMETER;
   	}

	if ( aName.GetChar( aName.Len() - 1 ) == '/' )
		aName.Erase( aName.Len() - 1 ); // cut trailing '/'

	xub_StrLen nTokens = aName.GetTokenCount( '/' );

   	if ( nTokens < 2 )
   	{
    	DBG_ERROR( "CntStorage::initialize - Invalid URL!" );
		return ERRCODE_IO_INVALIDPARAMETER;
   	}

	m_aName = aName.GetToken( nTokens - 1, '/' );

	m_aPath = pParent->m_aFullName;

	m_aFullName = m_aPath;
	m_aFullName += m_aName;
	m_aFullName += '/';

	return ERRCODE_NONE;
}

//-----------------------------------------------------------------------------
// virtual
ErrCode CntStorage::initStorage( BOOL bCreate )
{
	vos::OGuard aGuard( m_aMutex );

	// Make sure parent chain is initialized, too.
	ErrCode nError = m_pParent->initStorage( bCreate );
	if ( nError != ERRCODE_NONE )
		return nError;

	// Note: nError == ERRCODE_NONE && !m_pParent->m_pStore => storage file
	//       does not exist and we called pParent->initStorage( FALSE ) to
	//       indicate we do not want it created.
	if ( !m_pParent->m_pStore )
		return nError;

	if ( !m_pStore )
		m_pStore = new store::OStoreFile( *m_pParent->m_pStore );

	storeError eError = store_E_None;

	if ( bCreate )
	{
		store::OStoreDirectory aDir;
		eError = aDir.create( *m_pStore,
							  m_aPath,
							  m_aName,
							  store_AccessReadCreate );
	}
	else
	{
		eError = m_pStore->attrib( m_aPath, m_aName, 0, 0 );

		if ( eError == store_E_NotExists )
			return ERRCODE_NONE;
	}

	return mapStoreToToolsError( eError );
}

//-----------------------------------------------------------------------------
// virtual
ErrCode	CntStorage::closeStorage()
{
	vos::OGuard aGuard( m_aMutex );

	delete m_pStore;
	m_pStore = NULL;

	return ERRCODE_NONE;
}

//-----------------------------------------------------------------------------
// virtual
ErrCode CntStorage::destroyStorage()
{
	vos::OGuard aGuard( m_aMutex );

	ErrCode nError = initStorage( FALSE );
	if ( nError != ERRCODE_NONE )
		return nError;

	// Note: nError == ERRCODE_NONE && !m_pStore => storage file does not
	//       exist and we called initStorage( FALSE ) to indicate we do not
	//       want it created.
	if ( !m_pStore )
		return nError;

	store::OStoreDirectory aDir;
	storeError eError = aDir.create( *m_pStore,
									 m_aPath,
									 m_aName,
									 store_AccessReadWrite );
	if ( eError != store_E_None )
		return mapStoreToToolsError( eError );

	String aDirName( m_aName );
	aDirName += '/';

	// Destroy sub contents.
	CntDirDestroyer_Impl aDestroyer( *m_pStore, m_aFullName );
	aDir.travel( aDestroyer );

	nError = aDestroyer.getError();
	if ( nError == ERRCODE_NONE )
	{
		// Remove my directory.
		eError = m_pStore->remove( m_aPath, aDirName );
		nError = mapStoreToToolsError( eError );

		// Ignore "not exists" error.
		if ( nError == ERRCODE_IO_NOTEXISTS )
			nError = ERRCODE_NONE;
	}

	if ( nError == ERRCODE_NONE )
	{
		delete m_pStore;
		m_pStore = NULL;
		m_aName.Erase();
	}
	return nError;
}

//-----------------------------------------------------------------------------
CntStoreItemSet* CntStorage::openItemSet( const String& rName,
									 	  SfxItemPool& rPool,
                                  	 	  StreamMode eOpenMode,
									 	  const USHORT* pWhichPairTable )
{
    if( !rName.Len() )
		return NULL;

	BOOL bWriteAccess = !!( eOpenMode & STREAM_WRITE );

	vos::OGuard aGuard( m_aMutex );

	ErrCode nError = initStorage( bWriteAccess );
	if ( nError != ERRCODE_NONE )
		return NULL;

	// Note: nError == ERRCODE_NONE && !m_pStore => storage file does not
	//       exist and we called initStorage( FALSE ) to indicate we do not
	//       want it created.
	if ( !m_pStore )
		return NULL;

	eOpenMode &= ~STREAM_SHARE_DENYREAD;

	CntStoreItemSet* pItemSet = NULL;

	String aName( getShortName( rName, bWriteAccess ) );
	storeAccessMode eMode = mapToolsToStoreStreamMode( eOpenMode );

	store::OStoreStream aStream;
	storeError eError = aStream.create(
							*m_pStore, m_aFullName, aName, eMode );

	if ( eError == store_E_None )
	{
		SvLockBytesRef xLockBytes( new CntStoreLockBytes_Impl( aStream ) );
		nError = CntStoreItemSet::createItemSet(
	 				pItemSet, xLockBytes, eOpenMode, rPool, pWhichPairTable );
	}

   	return pItemSet;
}

//-----------------------------------------------------------------------------
SvStream* CntStorage::openStream( const String& rName, StreamMode eOpenMode )
{
    if( !rName.Len() )
		return NULL;

	BOOL bWriteAccess = !!( eOpenMode & STREAM_WRITE );

	vos::OGuard aGuard( m_aMutex );

	ErrCode nError = initStorage( bWriteAccess );
	if ( nError != ERRCODE_NONE )
		return NULL;

	// Note: nError == ERRCODE_NONE && !m_pStore => storage file does not
	//       exist and we called initStorage( FALSE ) to indicate we do not
	//       want it created.
	if ( !m_pStore )
		return NULL;

	eOpenMode &= ~STREAM_SHARE_DENYREAD;

	String aName( getShortName( rName, bWriteAccess ) );

	storeAccessMode eMode = mapToolsToStoreStreamMode( eOpenMode );

	store::OStoreStream aStream;
	storeError eError = aStream.create(
							*m_pStore, m_aFullName, aName, eMode );

	if ( eError == store_E_None )
	{
		SvLockBytesRef xLockBytes( new CntStoreLockBytes_Impl( aStream ) );
		CntStoreStream *pStrm = new CntStoreStream( xLockBytes, eOpenMode );
		pStrm->SetBufferSize( 4096 );
		pStrm->SetNumberFormatInt( NUMBERFORMAT_INT_LITTLEENDIAN );
		return pStrm;
	}

    return NULL;
}

//-----------------------------------------------------------------------------
ErrCode CntStorage::rename( const String& rName, const String& rNewName )
{
    if( !rName.Len() )
		return ERRCODE_IO_INVALIDPARAMETER;

    if( !rNewName.Len() )
		return ERRCODE_IO_INVALIDPARAMETER;

	vos::OGuard aGuard( m_aMutex );

	ErrCode nError = initStorage( FALSE );
	if ( nError != ERRCODE_NONE )
		return nError;

	// Note: nError == ERRCODE_NONE && !m_pStore => storage file does not
	//       exist and we called initStorage( FALSE ) to indicate we do not
	//       want it created.
	if ( !m_pStore )
		return nError;

	String aName( getShortName( rName, FALSE ) );
	String aNewName( getShortName( rNewName, TRUE ) );

	storeError eError = m_pStore->rename(
								m_aFullName, aName, m_aFullName, aNewName );
	nError = mapStoreToToolsError( eError );

	if ( nError == ERRCODE_NONE )
		removeName( aName );

	return nError;
}

//-----------------------------------------------------------------------------
ErrCode CntStorage::remove( const String& rName )
{
    if( !rName.Len() )
		return ERRCODE_IO_INVALIDPARAMETER;

	vos::OGuard aGuard( m_aMutex );

	ErrCode nError = initStorage( FALSE );
	if ( nError != ERRCODE_NONE )
		return nError;

	// Note: nError == ERRCODE_NONE && !m_pStore => storage file does not
	//       exist and we called initStorage( FALSE ) to indicate we do not
	//       want it created.
	if ( !m_pStore )
		return nError;

	String aName( getShortName( rName, FALSE ) );

	storeError eError = m_pStore->remove( m_aFullName, aName );
	nError = mapStoreToToolsError( eError );

	if ( nError == ERRCODE_NONE )
		removeName( aName );

	return nError;
}

//-----------------------------------------------------------------------------
ErrCode CntStorage::link( const String& rName, const String& rNewName )
{
    if( !rName.Len() )
		return ERRCODE_IO_INVALIDPARAMETER;

    if( !rNewName.Len() )
		return ERRCODE_IO_INVALIDPARAMETER;

	vos::OGuard aGuard( m_aMutex );

	ErrCode nError = initStorage( FALSE );
	if ( nError != ERRCODE_NONE )
		return nError;

	// Note: nError == ERRCODE_NONE && !m_pStore => storage file does not
	//       exist and we called initStorage( FALSE ) to indicate we do not
	//       want it created.
	if ( !m_pStore )
		return nError;

	String aName   ( getShortName( rName, FALSE ) );
	String aNewName( getShortName( rNewName, TRUE ) );

	storeError eError = m_pStore->link(
								m_aFullName, aNewName, m_aFullName, aName );
	return mapStoreToToolsError( eError );
}

//-----------------------------------------------------------------------------
ErrCode CntStorage::attrib( const String& rName,
							UINT32 nClearMask,
							UINT32 nSetMask,
							UINT32& nAttrib )
{
    if( !rName.Len() )
		return ERRCODE_IO_INVALIDPARAMETER;

	vos::OGuard aGuard( m_aMutex );

	ErrCode nError = initStorage( nSetMask != 0 );
	if ( nError != ERRCODE_NONE )
		return nError;

	// Note: nError == ERRCODE_NONE && !m_pStore => storage file does not
	//       exist and we called initStorage( FALSE ) to indicate we do not
	//       want it created.
	if ( !m_pStore )
		return nError;

	String aName( getShortName( rName,
								!!( nSetMask & CNTDIRENTRY_ATTRIB_CREATE ) ) );

	storeError eError = store_E_None;
	if ( nSetMask & CNTDIRENTRY_ATTRIB_CREATE )
	{
		nSetMask &= ~CNTDIRENTRY_ATTRIB_CREATE;

		store::OStoreStream aStream;
		eError = aStream.create( *m_pStore,
								 m_aFullName,
								 aName,
								 store_AccessReadCreate );
		if ( eError != store_E_None )
			return mapStoreToToolsError( eError );
	}

	eError = m_pStore->attrib(
						m_aFullName, aName, nClearMask, nSetMask, nAttrib );
	return mapStoreToToolsError( eError );
}

//-----------------------------------------------------------------------------

static const String aEmpty_Impl;

//-----------------------------------------------------------------------------
const String CntStorage::iter( CntStorageIterator& rIter )
{
	vos::OGuard aGuard( m_aMutex );

	ErrCode nError = initStorage( FALSE );
	if ( nError != ERRCODE_NONE )
	{
		rIter.m_eState = CntStorageIterator::STATE_EOF;
		return aEmpty_Impl;
	}

	// Note: nError == ERRCODE_NONE && !m_pStore => storage file does not
	//       exist and we called initStorage( FALSE ) to indicate we do not
	//       want it created.
	if ( !m_pStore )
	{
		rIter.m_eState = CntStorageIterator::STATE_EOF;
		return aEmpty_Impl;
	}

	if ( rIter.eof() )
		return aEmpty_Impl;

	BOOL bMatches = FALSE;
	if ( rIter.m_eState == CntStorageIterator::STATE_FIRST )
	{
		String aName;
		if ( m_aPath.Len() )
			aName += m_aName;

		storeError eError = rIter.m_aStore.create( *m_pStore,
												   m_aPath,
												   aName,
												   store_AccessReadOnly );
		if ( eError != store_E_None )
		{
			rIter.m_eState = CntStorageIterator::STATE_EOF;
			return aEmpty_Impl;
		}

		if ( rIter.m_aStore.first( *rIter) != store_E_None )
		{
			rIter.m_eState = CntStorageIterator::STATE_EOF;
			return aEmpty_Impl;
		}

		rIter.m_eState = CntStorageIterator::STATE_NEXT;
		bMatches = rIter.matches();
	}

	while ( !bMatches )
	{
		if ( rIter.m_aStore.next( *rIter ) != store_E_None )
		{
			rIter.m_eState = CntStorageIterator::STATE_EOF;
			return aEmpty_Impl;
		}

		bMatches = rIter.matches();
	}

	return String( getLongName( (*rIter).m_pszName ) );
}

//-----------------------------------------------------------------------------
// static
String CntStorage::makeShortName( const String &rLongName )
{
	String aShortName( rLongName );
	if ( STORE_MAXIMUM_NAMESIZE <= rLongName.Len() )
	{
		aShortName.Erase( STORE_MAXIMUM_NAMESIZE - 12 );
		aShortName += '~';
		aShortName += rtl_crc32( 0, rLongName.GetBuffer(), rLongName.Len() );
	}
	return aShortName;
}

//-----------------------------------------------------------------------------
CntLongNamesList* CntStorage::getLongNamesList()
{
	if ( !m_pLongNamesList )
	{
		m_pLongNamesList = new CntLongNamesList;

		SvStream* pStream = openStream( UniString::CreateFromAscii(
											RTL_CONSTASCII_STRINGPARAM(
												LONGNAMES_STREAM ) ),
										STREAM_STD_READ );
		if ( pStream )
		{
			ULONG nVersion = LONGNAMES_STREAM_VERSION;
			ULONG nCount   = 0;
			*pStream >> nVersion;
			*pStream >> nCount;

			if ( ( nVersion > 0 ) && ( nVersion <= LONGNAMES_STREAM_VERSION ) )
			{
				for ( ULONG n = 0; n < nCount; ++n )
				{
					UniString aLongName;
					SfxPoolItem::readUnicodeString(
									*pStream, aLongName, nVersion >= 2 );

					UniString aShortName;
					SfxPoolItem::readUnicodeString(
									*pStream, aShortName, nVersion >= 2 );

					m_pLongNamesList->Insert(
						new CntLongNameEntry( aLongName, aShortName, TRUE ),
						LIST_APPEND );
				}
			}

			delete pStream;
		}
	}
	return m_pLongNamesList;
}

//-----------------------------------------------------------------------------
void CntStorage::saveLongNamesList()
{
	if ( m_pLongNamesList )
	{
		const UniString aStreamName( UniString::CreateFromAscii(
										RTL_CONSTASCII_STRINGPARAM(
											LONGNAMES_STREAM ) ) );
		// destroy stream
		remove( aStreamName );

		ULONG nTotalCount = m_pLongNamesList->Count();
		if ( !nTotalCount )
			return;

		UINT32 nAttrib = 0;
		attrib( aStreamName,
				0,
				CNTDIRENTRY_ATTRIB_HIDDEN | CNTDIRENTRY_ATTRIB_CREATE,
				nAttrib );
		SvStream* pStream =	openStream(
				aStreamName, STREAM_READWRITE | STREAM_SHARE_DENYWRITE );
		if ( pStream )
		{
			ULONG nVersion      = LONGNAMES_STREAM_VERSION;
			ULONG nPersistCount = nTotalCount;

			for ( ULONG i = 0; i < nTotalCount; ++i )
			{
				CntLongNameEntry* pEntry = m_pLongNamesList->GetObject( i );
				if ( !pEntry->bPersist )
					nPersistCount--;
			}

			if ( nPersistCount )
			{
				*pStream << nVersion;
				*pStream << nPersistCount;

				for ( ULONG n = 0; n < nTotalCount; ++n )
				{
					CntLongNameEntry* pEntry = m_pLongNamesList->GetObject( n );
					if ( pEntry->bPersist )
					{
						SfxPoolItem::writeUnicodeString(
										*pStream, pEntry->aLongName );
						SfxPoolItem::writeUnicodeString(
										*pStream, pEntry->aShortName );
					}
				}
			}

			delete pStream;
		}
	}
}

//-----------------------------------------------------------------------------
const String& CntStorage::getLongName( const String& rShortName )
{
	getLongNamesList();

	if ( rShortName.EqualsAscii( SHORTNAME_PREFIX, 0, SHORTNAME_PREFIX_LEN ) )
	{
		ULONG nCount = m_pLongNamesList->Count();
		for ( ULONG n = 0; n < nCount; ++n )
		{
			CntLongNameEntry* pEntry = m_pLongNamesList->GetObject( n );
			if ( pEntry->aShortName == rShortName )
				return pEntry->aLongName;
		}
	}

	return rShortName;
}

//-----------------------------------------------------------------------------
String CntStorage::getShortName( const String& rLongName, BOOL bStore )
{
	if ( rLongName.Len() < STORE_MAXIMUM_NAMESIZE )
		return String( rLongName );

	getLongNamesList();

	ULONG nCount = m_pLongNamesList->Count();
	for ( ULONG n = 0; n < nCount; ++n )
	{
		CntLongNameEntry* pEntry = m_pLongNamesList->GetObject( n );
		if ( pEntry->aLongName == rLongName )
		{
			BOOL bWasStore = pEntry->bPersist;

			pEntry->bPersist |= bStore;

			if ( !bWasStore && pEntry->bPersist )
				saveLongNamesList();

			return pEntry->aShortName;
		}
	}

	// Create short name.
	String aLongName( UniString::CreateFromAscii(
						RTL_CONSTASCII_STRINGPARAM( SHORTNAME_PREFIX ) ) );
	aLongName += rLongName;

	String aShortName( makeShortName( aLongName ) );

	m_pLongNamesList->Insert(
				new CntLongNameEntry( rLongName, aShortName, bStore ),
				LIST_APPEND );

	if ( bStore )
		saveLongNamesList();

	return aShortName;
}

//-----------------------------------------------------------------------------
void CntStorage::removeName( const String& rName )
{
	getLongNamesList();

	ULONG nCount = m_pLongNamesList->Count();
	for ( ULONG n = 0; n < nCount; ++n )
	{
		CntLongNameEntry* pEntry = m_pLongNamesList->GetObject( n );
		if ( ( pEntry->aLongName == rName ) ||
			 ( pEntry->aShortName == rName ) )
		{
			m_pLongNamesList->Remove( pEntry );
			delete pEntry;
			saveLongNamesList();
			return;
		}
	}
}

//============================================================================
//
// CntRootStorage Implementation.
//
//============================================================================

CntRootStorage::CntRootStorage()
{
}

//-----------------------------------------------------------------------------
//virtual
CntRootStorage::~CntRootStorage()
{
}

//-----------------------------------------------------------------------------
// virtual
ErrCode CntRootStorage::initialize( CntStorage* pParent,
									const String& rURL,
									USHORT nInitFlags /* = 0 */ )
{
    if( !rURL.Len() )
		return ERRCODE_IO_INVALIDPARAMETER;

	vos::OGuard aGuard( m_aMutex );

	m_nFlags = nInitFlags;

	// Note: root storages never have a parent!
	m_pParent = NULL;

	m_aName = DirEntry( rURL, FSYS_STYLE_URL ).GetFull( FSYS_STYLE_HOST );
	m_aPath.Erase();
	m_aFullName = '/';

	return ERRCODE_NONE;
}

//-----------------------------------------------------------------------------
//virtual
ErrCode CntRootStorage::initStorage( BOOL bCreate )
{
	vos::OGuard aGuard( m_aMutex );

	storeError eError = store_E_None;

	if ( !m_pStore )
	{
 		if ( !m_aName.Len() )
		{
			DBG_ERROR( "CntRootStorage::initStorage - No name!" );
			return ERRCODE_IO_ACCESSDENIED;
		}

		store::OStoreFile aStoreFile;
		eError = aStoreFile.create( m_aName, store_AccessReadWrite );
		if ( eError == store_E_NotExists )
		{
			if ( !bCreate )
			{
				// File does not exist and we don't want to create it.
				return ERRCODE_NONE;
			}

			// Note: This creates the storage file, if it does not exist.
			eError = aStoreFile.create( m_aName, store_AccessReadCreate );
			if ( eError != store_E_None )
				return mapStoreToToolsError( eError );

			// Root directory must be created explicitely!
			store::OStoreDirectory aRootDir;
			eError = aRootDir.create( aStoreFile,
									  rtl::OUString(),
									  rtl::OUString(),
									  store_AccessReadCreate );
		}

		if ( eError == store_E_LockingViolation )
		{
			DBG_WARNING( "Storage file locked! Opening read-only!" );
			eError = aStoreFile.create( m_aName, store_AccessReadOnly );
		}

		if ( eError == store_E_WrongFormat )
		{
			// Damaged storage file.
			// nError = m_pRoot->recover()
		}

		if ( m_nFlags & STG_INIT_FLAG_BACKUP )
		{
			DirEntry aOrig( m_aName );
			DirEntry aBak( aOrig );
			aBak.SetExtension( UniString::CreateFromAscii(
									RTL_CONSTASCII_STRINGPARAM( "bak" ) ) );
			if ( eError != store_E_None )
			{
				// Try backup of storage file.

            	if ( aBak.Exists() )
				{
					DirEntry aTmp( aOrig );
					aTmp.SetExtension( UniString::CreateFromAscii(
										RTL_CONSTASCII_STRINGPARAM( "kso" ) ) );
					aTmp.Kill();

					if ( aOrig.MoveTo( aTmp ) == ERRCODE_NONE )
					{
						ErrCode nError = mapStoreToToolsError( eError );

						if ( aBak.CopyTo( aOrig, FSYS_ACTION_COPYFILE )
					     	 == ERRCODE_NONE )
						{
							// Retry.
							m_nFlags &= ~STG_INIT_FLAG_BACKUP;
							nError = initStorage( bCreate );

							if ( nError == ERRCODE_NONE )
							{
								// Success.
								aTmp.Kill();
								return ERRCODE_NONE;
							}
						}

						// Error.
						aTmp.MoveTo( aOrig );
						return nError;
					}
            	}
			}
			else
			{
				// Create backup of 'proper' storage file.

				aBak.Kill();
				aOrig.CopyTo( aBak, FSYS_ACTION_COPYFILE );
			}
		}

		DBG_ASSERT( ( eError == store_E_None ) ||
					( eError == store_E_LockingViolation ) ||
					( eError == store_E_WrongVersion ),
		   			"CntRootStorage::initStorage - error!" );

		if ( eError == store_E_None )
			m_pStore = new store::OStoreFile( aStoreFile );
	}

	return mapStoreToToolsError( eError );
}

//-----------------------------------------------------------------------------
// virtual
ErrCode	CntRootStorage::closeStorage()
{
	vos::OGuard aGuard( m_aMutex );

	if ( !m_pStore )
	{
		// Already closed.
		return ERRCODE_NONE;
	}

	sal_uInt32 nRefs = 0;
	storeError eError = m_pStore->getRefererCount( nRefs );

	if ( eError != store_E_None )
		return mapStoreToToolsError( eError );

	if ( nRefs )
	{
		// Other clients are locking the file!!!
		return ERRCODE_IO_ACCESSDENIED;
	}

	delete m_pStore;
	m_pStore = NULL;

	return ERRCODE_NONE;
}

//-----------------------------------------------------------------------------
//virtual
ErrCode CntRootStorage::destroyStorage()
{
	vos::OGuard aGuard( m_aMutex );

	ErrCode nError = initStorage( FALSE );
	if ( nError != ERRCODE_NONE )
		return nError;

	// Note: nError == ERRCODE_NONE && !m_pStore => storage file does not
	//       exist and we called initStorage( FALSE ) to indicate we do not
	//       want it created.
	if ( !m_pStore )
		return nError;

	// Close storage file
	nError = closeStorage();

	if ( nError == ERRCODE_NONE )
	{
		// Destroy storage file
		DirEntry aFile( m_aName );
		nError = aFile.Kill();

		m_aName.Erase();
	}

	DBG_ASSERT( nError == ERRCODE_NONE,
				"CntRootStorage::destroyStorage - Error!" );

	return nError;
}

//-----------------------------------------------------------------------------
ErrCode CntRootStorage::flush()
{
	vos::OGuard aGuard( m_aMutex );

	ErrCode nError = ERRCODE_NONE;

	if ( m_pStore )
		nError = m_pStore->flush();

	return nError;
}

//-----------------------------------------------------------------------------
ErrCode CntRootStorage::size( UINT32& rSize )
{
	vos::OGuard aGuard( m_aMutex );

	ErrCode nError = initStorage( FALSE );
	if ( nError != ERRCODE_NONE )
		return nError;

	// Note: nError == ERRCODE_NONE && !m_pStore => storage file does not
	//       exist and we called initStorage( FALSE ) to indicate we do not
	//       want it created.
	if ( !m_pStore )
		return nError;

	return m_pStore->getSize( rSize );
}

//-----------------------------------------------------------------------------
ErrCode CntRootStorage::rebuild()
{
	vos::OGuard aGuard( m_aMutex );

	ErrCode nError = initStorage( FALSE );
	if ( nError != ERRCODE_NONE )
		return nError;

	// Note: nError == ERRCODE_NONE && !m_pStore => storage file does not
	//       exist and we called initStorage( FALSE ) to indicate we do not
	//       want it created.
	if ( !m_pStore )
		return nError;

	// First, check for sufficient disk space...

	UINT32  nSize;
	nError = size( nSize );
	if ( nError == ERRCODE_NONE )
	{
		BigInt nFreeBytes, nTotalBytes;
		nError = FileStat::QueryDiskSpace( m_aName, nFreeBytes, nTotalBytes );
		if ( nError == ERRCODE_NONE )
		{
			if ( nFreeBytes < BigInt( nSize ) )
			{
				// Not enough disk space for making a backup...
				return ERRCODE_CHAOS_REORGANIZE_NO_DISKSPACE;
			}
		}
	}

	nError = closeStorage();
	if ( nError != ERRCODE_NONE )
	{
		nError = ERRCODE_CHAOS_REORGANIZE_FILE_LOCKED;
	   	return nError;
	}

	DirEntry aSrc( m_aName );
	DirEntry aDst( aSrc );

	aDst.SetExtension( UniString::CreateFromAscii(
									RTL_CONSTASCII_STRINGPARAM( "bak" ) ) );

	storeError eError = store_rebuildFile(
								rtl::OUString( aSrc.GetFull() ).pData,
								rtl::OUString( aDst.GetFull() ).pData );
	nError = mapStoreToToolsError( eError );

	if ( nError != ERRCODE_NONE )
	{
		DBG_ERROR( "CntRootStorage::rebuild - rebuild failed!" );
	   	return nError;
	}

	nError = aSrc.Kill();
	if ( nError != ERRCODE_NONE )
	{
		DBG_ERROR( "CntRootStorage::rebuild - old file not killed!" );
	   	return nError;
	}

	nError = aDst.MoveTo( aSrc );
	if ( nError != ERRCODE_NONE )
	{
		DBG_ERROR( "CntRootStorage::rebuild - New file not renamed!" );
	   	return nError;
	}

	return ERRCODE_NONE;
}

//-----------------------------------------------------------------------------
// static
ErrCode CntRootStorage::create( const String& rURL )
{
	vos::OGuard aGuard(	vos::OMutex::getGlobalMutex() );

	// Note: This creates the storage file, if it does not exist.
	store::OStoreFile aStoreFile;
	storeError eError = aStoreFile.create( DirEntry( rURL, FSYS_STYLE_URL ).
												GetFull( FSYS_STYLE_HOST ),
	  										store_AccessReadCreate );
	if ( eError == store_E_None )
	{
		// Root directory must be created explicitely!
		store::OStoreDirectory aRootDir;
		eError = aRootDir.create( aStoreFile,
								  rtl::OUString(),
								  rtl::OUString(),
								  store_AccessReadCreate );
	}
	return mapStoreToToolsError( eError );
}

//============================================================================
//
// CntLazyRootStorage Implementation.
//
//============================================================================

CntLazyRootStorage::CntLazyRootStorage()
: m_nLocks( 0 )
{
}

//-----------------------------------------------------------------------------
// virtual
CntLazyRootStorage::~CntLazyRootStorage()
{
}

//-----------------------------------------------------------------------------
// virtual
ErrCode CntLazyRootStorage::closeStorage()
{
	vos::OGuard aGuard( m_aMutex );

	// Remove locks.
	while ( isStorageLocked() )
		releaseStorage_Impl( FALSE );

	return CntRootStorage::closeStorage();
}

//-----------------------------------------------------------------------------
// virtual
ErrCode CntLazyRootStorage::destroyStorage()
{
	vos::OGuard aGuard( m_aMutex );

	// Remove locks.
	while ( isStorageLocked() )
		releaseStorage_Impl( FALSE );

	return CntRootStorage::destroyStorage();
}

//-----------------------------------------------------------------------------
oslInterlockedCount CntLazyRootStorage::acquireStorage()
{
	vos::OGuard aGuard( m_aMutex );

	if ( m_xCloseTimer.isValid() && m_xCloseTimer->isTicking() )
		m_xCloseTimer->stop();

	return osl_incrementInterlockedCount( &m_nLocks );
}

//-----------------------------------------------------------------------------
oslInterlockedCount CntLazyRootStorage::releaseStorage()
{
	vos::OGuard aGuard( m_aMutex );
	return releaseStorage_Impl( TRUE );
}

//-----------------------------------------------------------------------------
oslInterlockedCount CntLazyRootStorage::releaseStorage_Impl( BOOL bAutoClose )
{
	if ( osl_decrementInterlockedCount( &m_nLocks ) == 0 )
	{
		BOOL bStop = TRUE;

#ifndef AUTO_CLOSE_VIEW_STORAGES
		bAutoClose = FALSE;
#endif
		if ( bAutoClose )
		{
			if ( m_pStore )
			{
				if ( !m_xCloseTimer.isValid() )
				{
					m_xCloseTimer = new CntStorageCloser_Impl(
											VIEW_STORAGE_CLOSE_DELAY, this );
				}
				else
				{
					m_xCloseTimer->stop();
					m_xCloseTimer->setRemainingTime( VIEW_STORAGE_CLOSE_DELAY );
				}

				m_xCloseTimer->start();
				bStop = FALSE;
			}
		}

		if ( bStop && m_xCloseTimer.isValid() && m_xCloseTimer->isTicking() )
			m_xCloseTimer->stop();
	}

	return m_nLocks;
}

//===========================================================================
//
// CntStorageCloser_Impl Implementation.
//
//===========================================================================

CntStorageCloser_Impl::CntStorageCloser_Impl(
				const vos::TTimeValue& rTime, CntLazyRootStorage* pStorage )
: m_pStorage( pStorage )
{
	setRemainingTime( rTime );
}

//-----------------------------------------------------------------------------
//virtual
CntStorageCloser_Impl::~CntStorageCloser_Impl()
{
}

//-----------------------------------------------------------------------------
//virtual
void SAL_CALL CntStorageCloser_Impl::onShot()
{
	m_pStorage->CntRootStorage::closeStorage();
}

//===========================================================================
//
// CntStoreLockBytes_Impl Implementation.
//
//===========================================================================

// virtual
CntStoreLockBytes_Impl::~CntStoreLockBytes_Impl()
{
}

//-----------------------------------------------------------------------------
// virtual
ErrCode CntStoreLockBytes_Impl::ReadAt(
			ULONG nPos, void* pBuffer, ULONG nCount, ULONG* pRead ) const
{
	CntStoreLockBytes_Impl* pThis
				= const_cast< CntStoreLockBytes_Impl * >( this );
	return mapStoreToToolsError(
				pThis->m_aStream.readAt( nPos, pBuffer, nCount, *pRead ) );
}

//-----------------------------------------------------------------------------
// virtual
ErrCode CntStoreLockBytes_Impl::WriteAt(
			ULONG nPos, const void* pBuffer, ULONG nCount, ULONG* pWritten )
{
	return mapStoreToToolsError(
				m_aStream.writeAt( nPos, pBuffer, nCount, *pWritten ) );
}

//-----------------------------------------------------------------------------
// virtual
ErrCode CntStoreLockBytes_Impl::Flush() const
{
	return mapStoreToToolsError( m_aStream.flush() );
}

//-----------------------------------------------------------------------------
// virtual
ErrCode CntStoreLockBytes_Impl::SetSize( ULONG nSize )
{
	return mapStoreToToolsError( m_aStream.setSize( nSize ) );
}

//-----------------------------------------------------------------------------
// virtual
ErrCode CntStoreLockBytes_Impl::LockRegion( ULONG, ULONG, LockType )
{
	return ERRCODE_NONE;
}

//-----------------------------------------------------------------------------
// virtual
ErrCode CntStoreLockBytes_Impl::UnlockRegion( ULONG, ULONG, LockType )
{
	return ERRCODE_NONE;
}

//-----------------------------------------------------------------------------
// virtual
ErrCode CntStoreLockBytes_Impl::Stat(
			SvLockBytesStat* pStat, SvLockBytesStatFlag ) const
{
	if ( !pStat )
		return ERRCODE_IO_INVALIDPARAMETER;

	return mapStoreToToolsError( m_aStream.getSize( pStat->nSize ) );
}

