/*************************************************************************
 *
 *  $RCSfile: binaryreader.cxx,v $
 *
 *  $Revision: 1.16 $
 *
 *  last change: $Author: jb $ $Date: 2001/11/09 12:18:46 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include "binaryreader.hxx"

#ifndef CONFIGMGR_TREE_NODEFACTORY_HXX
#include "treenodefactory.hxx"
#endif
#ifndef CONFIGMGR_BINARYTYPE_HXX
#include "binarytype.hxx"
#endif
#ifndef CONFIGMGR_CONFIGURATION_ATTRIBUTES_HXX_
#include "attributes.hxx"
#endif
#ifndef CONFIGMGR_SIMPLETYPEHELPER_HXX
#include "simpletypehelper.hxx"
#endif
#ifndef CONFIGMGR_CONFIGPATH_HXX_
#include "configpath.hxx"
#endif
#ifndef CONFIGMGR_TRIVIALBUFFEREDFILE_HXX
#include "trivialbufferedfile.hxx"
#endif
#ifndef _CONFIGMGR_FILEHELPER_HXX_
#include "filehelper.hxx"
#endif
#ifndef CONFIGMGR_BINARYBASEREADER_HXX
#include "binarybasereader.hxx"
#endif

#ifndef _CONFIGMGR_TRACER_HXX_
#include "tracer.hxx"
#endif

#ifndef _COM_SUN_STAR_IO_IOEXCEPTION_HPP_
#include <com/sun/star/io/IOException.hpp>
#endif
#ifndef _COM_SUN_STAR_LANG_XMULTISERVICEFACTORY_HPP_
#include <com/sun/star/lang/XMultiServiceFactory.hpp>
#endif
#ifndef _COM_SUN_STAR_UNO_ANY_HXX_
#include <com/sun/star/uno/Any.hxx>
#endif

#ifndef _OSL_FILE_HXX_
#include <osl/file.hxx>
#endif
#ifndef _RTL_STRING_HXX_
#include <rtl/string.hxx>
#endif
#ifndef _RTL_USTRING_HXX_
#include <rtl/ustring.hxx>
#endif


#ifndef INCLUDED_ALGORITHM
#include <algorithm>
#define INCLUDED_ALGORITHM
#endif

#define ASCII(x) rtl::OUString::createFromAscii(x)

// -----------------------------------------------------------------------------
namespace configmgr
{
	namespace css = com::sun::star;
	namespace io = css::io;
	namespace uno = css::uno;
	namespace lang = css::lang;

    using namespace configuration;
// --------------------------------- Prototypes ---------------------------------
uno::Type getSequenceElementType(uno::Type const& rSequenceType); // from typeconverter.cxx

// -----------------------------------------------------------------------------
NodeType::Enum readNodeType(OBinaryBaseReader &_aReader)
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	NodeType::Enum eType;
	sal_Int8 nValue;
	_aReader.read(nValue);
	sal_uInt8 nRightValue = nValue;
	switch(nRightValue /* & (0xff ^ NodeType::vtyp_mask) */ )
	{
	case NodeType::stop:
		eType = NodeType::stop;
		break;
		
	case NodeType::value:
		eType = NodeType::value;
		break;
		
	case NodeType::group:
		eType = NodeType::group;
		break;
		
	case NodeType::ntyp_set:
		eType = NodeType::ntyp_set;
		break;
		
	default:
		OSL_ENSURE(false, "readNodeType(): unknown or unhandled type");
		eType = NodeType::stop;
	}
	return eType;
	// if (nValue & NodeType::group) return NodeType::group;
}

// -----------------------------------------------------------------------------
ValueType::Enum readValueType(OBinaryBaseReader& _aReader)
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	ValueType::Enum eType;
	sal_Int8 nValue;
	_aReader.read(nValue);
	bool bIsNull = false;
	if (nValue & 0x10)
	{
		nValue = nValue & 0xef;
		bIsNull = true;
	}
	switch(nValue /* & (0xff ^ NodeType::vtyp_mask) */ )
	{
	case ValueType::val_any:
		eType = ValueType::val_any;
		break;
		
	case ValueType::val_string:
		eType = ValueType::val_string;
		break;
		
	case ValueType::val_boolean:
		eType = ValueType::val_boolean;
		break;
		
	case ValueType::val_int8:
		eType = ValueType::val_int8;
		break;
		
	case ValueType::val_int16:
		eType = ValueType::val_int16;
		break;
		
	case ValueType::val_int32:
		eType = ValueType::val_int32;
		break;
		
	case ValueType::val_int64:
		eType = ValueType::val_int64;
		break;
		
	case ValueType::val_double:
		eType = ValueType::val_double;
		break;
		
	case ValueType::val_binary:
		eType = ValueType::val_binary;
		break;
		
	case ValueType::seq_any:
		eType = ValueType::seq_any;
		break;
		
	case ValueType::seq_string:
		eType = ValueType::seq_string;
		break;
		
	case ValueType::seq_boolean:
		eType = ValueType::seq_boolean;
		break;
		
	case ValueType::seq_int8:
		eType = ValueType::seq_int8;
		break;
		
	case ValueType::seq_int16:
		eType = ValueType::seq_int16;
		break;
		
	case ValueType::seq_int32:
		eType = ValueType::seq_int32;
		break;
		
	case ValueType::seq_int64:
		eType = ValueType::seq_int64;
		break;
		
	case ValueType::seq_double:
		eType = ValueType::seq_double;
		break;
		
	case ValueType::seq_binary:
		eType = ValueType::seq_binary;
		break;
		
	default:
		OSL_ENSURE(false, "unknown or unhandled type");
		eType = ValueType::val_any;
	}

	if (bIsNull)
	{
		eType = ValueType::Enum(eType | ValueType::val_is_null);
	}
	return eType;

}

// -----------------------------------------------------------------------------

void readString(OBinaryBaseReader& _aReader, rtl::OUString &_aString)
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	_aReader.read (_aString);
}

// -----------------------------------------------------------------------------
void readAttributes(OBinaryBaseReader& _aReader, configuration::Attributes &_aAttributes)
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	sal_Int8 nValue;
	_aReader.read(nValue);

    _aAttributes.setState( node::State(nValue & 0x03) );

	_aAttributes.bWritable	  = nValue & 0x04 ? true : false;
	_aAttributes.bFinalized	  = nValue & 0x08 ? true : false;

	_aAttributes.bNullable	  = nValue & 0x10 ? true : false;
	_aAttributes.bLocalized	  = nValue & 0x20 ? true : false;

	_aAttributes.bNotified	  = nValue & 0x40 ? true : false;
	_aAttributes.bConstrained = nValue & 0x80 ? true : false;
}
// -----------------------------------------------------------------------------
void readHeader(OBinaryBaseReader& _aReader, rtl::OUString &_aName, configuration::Attributes &_aAttributes)
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	readString(_aReader, _aName);
	readAttributes(_aReader, _aAttributes);
}
// -----------------------------------------------------------------------------

void readGroup(OBinaryBaseReader& _aReader, rtl::OUString &_aName, configuration::Attributes &_aAttributes)
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	// ValueType::Enum eType = readValueType(_aReader);
	// OSL_ENSURE(eType == ValueType::val_string, "no string.");

	readHeader(_aReader, _aName, _aAttributes);
}
	
void readSet(OBinaryBaseReader& _aReader, rtl::OUString &_aName, configuration::Attributes &_aAttributes,
				 rtl::OUString &_sInstanceName, rtl::OUString &_sInstanceModule)
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	readHeader(_aReader, _aName, _aAttributes);
	readString(_aReader, _sInstanceName);
	readString(_aReader, _sInstanceModule);	
}
	
// -----------------------------------------------------------------------------
template <class T>
static
inline
void readAsAny(OBinaryBaseReader& _aReader, uno::Any & _aValue, uno::Type& _aType, T& _aVar)
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	_aReader.read(_aVar);
	_aValue <<= _aVar;
	_aType = _aValue.getValueType();
}

// -----------------------------------------------------------------------------
uno::Type convertValueTypeToType(ValueType::Enum _eValueType)
{
	uno::Type aType;

	ValueType::Enum eValueType = _eValueType;
	/*
	if (_eValueType & ValueType::sequence)
	{
		eValueType = ValueType::Enum(eValueType & ValueType::seq_mask);
	}
	*/
	if (_eValueType & ValueType::val_is_null)
	{
		eValueType = ValueType::Enum(eValueType & 0xef);
	}

	switch(eValueType)
	{
		// --------------- Simple types ---------------

	case ValueType::val_boolean:
		aType = SimpleTypeHelper::getBooleanType();
		break;
	case ValueType::val_int8:
		aType = SimpleTypeHelper::getByteType();
		break;
	case ValueType::val_int16:
		aType = SimpleTypeHelper::getShortType();
		break;
	case ValueType::val_int32:
		aType = SimpleTypeHelper::getIntType();
		break;
	case ValueType::val_int64:
		aType = SimpleTypeHelper::getLongType();
		break;
	case ValueType::val_double:
		aType = SimpleTypeHelper::getDoubleType();
		break;
	case ValueType::val_string:
		aType = SimpleTypeHelper::getStringType();
		break;

		// ------------ Sequences ------------

	case ValueType::seq_int8:
	case ValueType::seq_binary:
		aType = ::getCppuType(static_cast<uno::Sequence<sal_Int8> const*>(0));
		break;
	case ValueType::seq_int16:
		aType = ::getCppuType(static_cast<uno::Sequence<sal_Int16> const*>(0));
		break;
	case ValueType::seq_int32:
		aType = ::getCppuType(static_cast<uno::Sequence<sal_Int32> const*>(0));
		break;
	case ValueType::seq_int64:
		aType = ::getCppuType(static_cast<uno::Sequence<sal_Int64> const*>(0));
		break;
	case ValueType::seq_double:
		aType = ::getCppuType(static_cast<uno::Sequence<double> const*>(0));
		break;
	case ValueType::seq_string:
		aType = ::getCppuType(static_cast<uno::Sequence<rtl::OUString> const*>(0));
		break;
	case ValueType::seq_boolean:
		aType = ::getCppuType(static_cast<uno::Sequence<sal_Bool> const*>(0));
		break;
		/*
		  case ValueType::seq_any:
		  aType = ::getCppuType(static_cast<uno::Sequence<uno::Any> const*>(0));
		  break;
		*/
	default:
	{
		::rtl::OString aStr("Wrong typeclass! ");
		// aStr += rtl::OString::valueOf((sal_Int32)aClass);
		OSL_ENSURE(0,aStr.getStr());
	}
	}
	
	return aType;
}



// -----------------------------------------------------------------------------
class OBinaryReadSequence
{
	uno::Any&       m_aValue;					 // init by ctor-param
	ValueType::Enum m_eElementType;				 // init in ctor, real type from the sequence

public:
	OBinaryReadSequence(uno::Any& _aAny, ValueType::Enum const& _eType)
		:m_aValue(_aAny)
	{
		OSL_ENSURE(ValueType::Enum(_eType & ValueType::sequence) == ValueType::sequence, "Error: Type must be a Sequence");
		m_eElementType = ValueType::Enum(_eType & (0xff ^ ValueType::sequence));
	}

	void read(OBinaryBaseReader& _aReader) SAL_THROW( (io::IOException, uno::RuntimeException) )
	{
		readSequenceValue(_aReader, m_aValue, convertValueTypeToType(m_eElementType));
	}
};

// -----------------------------------------------------------------------------
void readValue(OBinaryBaseReader& _aReader, rtl::OUString &_aName, configuration::Attributes &_aAttributes,
			   /*OUT*/uno::Any& _aValue, /*OUT*/uno::Type& _aType)
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	readHeader(_aReader, _aName, _aAttributes);
	
	ValueType::Enum eType = readValueType(_aReader);
	if (eType & ValueType::val_is_null)
	{
		_aType = convertValueTypeToType(eType);
	}
	else
	{
		switch(eType)
		{
		case ValueType::val_any:
		{
			uno::Any aAny;
			OSL_ENSURE(false, "Any type not supported yet.");
			break;
		}
		case ValueType::val_string:
		{
			OUString aStr;
			readAsAny(_aReader, _aValue, _aType, aStr);
			break;
		}
		case ValueType::val_boolean:
		{
			sal_Bool nValue;
			readAsAny(_aReader, _aValue, _aType, nValue);
			break;
		}
		case ValueType::val_int8:
		{
			sal_Int8 nValue;
			readAsAny(_aReader, _aValue, _aType, nValue);
			break;
		}
		case ValueType::val_int16:
		{
			sal_Int16 nValue;
			readAsAny(_aReader, _aValue, _aType, nValue);
			break;
		}
		case ValueType::val_int32:
		{
			sal_Int32 nValue;
			readAsAny(_aReader, _aValue, _aType, nValue);
			break;
		}
		case ValueType::val_int64:
		{
			sal_Int64 nValue;
			readAsAny(_aReader, _aValue, _aType, nValue);
			break;
		}
		case ValueType::val_double:
		{
			double nValue;
			readAsAny(_aReader, _aValue, _aType, nValue);
			break;
		}
		case ValueType::val_binary:
		{
			sal_Int8 nValue;
			readAsAny(_aReader, _aValue, _aType, nValue);
			break;
		}
		case ValueType::seq_any:
		case ValueType::seq_string:
		case ValueType::seq_boolean:
		case ValueType::seq_int8:
		case ValueType::seq_int16:
		case ValueType::seq_int32:
		case ValueType::seq_int64:
		case ValueType::seq_double:
		{
			// OSL_ENSURE(false, "try to read sequence.");
		}
		case ValueType::seq_binary:
		{
			OBinaryReadSequence aSequenceReader(_aValue, eType);
			aSequenceReader.read(_aReader);
			break;
		}
	
		default:
			OSL_ENSURE(false, "readValue(): no known ValueType");
		}
	}
}
// -----------------------------------------------------------------------------

void readFileHeader(OBinaryBaseReader& _aReader, sal_Int32 &_nMagic, sal_Int32 &_nVersion,
					sal_Int32 &_nDateStamp)
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	_aReader.read(_nMagic);
	_aReader.read(_nVersion);
	_aReader.read(_nDateStamp);
}

// -----------------------------------------------------------------------------
// Helper class for binary read. It's provide functions to filter the reads
// -----------------------------------------------------------------------------
class OFilter
{
	vector<OUString> m_aFilterList;
public:
	enum Enum
	{
		INDEX_OUT_OF_BOUND,
		VALUE_IN_FILTER,
		VALUE_NOT_FOUND,
		VALUE_IS_LAST_AND_RIGHT
	};
	
	OFilter(AbsolutePath const& _aFilter)
		{
			for (AbsolutePath::Iterator it = _aFilter.begin();
				 it != _aFilter.end();
				 ++it)
			{
				rtl::OUString aName = it->getName().toString();
				m_aFilterList.push_back(aName);
			}
		}
	OFilter::Enum isInFilter(rtl::OUString const& _aName, sal_Int32 _nIndex)
		{
			sal_Int32 nSize = m_aFilterList.size();
			if (_nIndex >= nSize || _nIndex < 0)
			{
				return INDEX_OUT_OF_BOUND;
			}
			if (_nIndex + 1 == nSize && m_aFilterList[_nIndex].equals(_aName))
			{
				return VALUE_IS_LAST_AND_RIGHT;
			}
			if (m_aFilterList[_nIndex].equals(_aName))
			{
				return VALUE_IN_FILTER;
			}
			return VALUE_NOT_FOUND;
		}
};

// -----------------------------------------------------------------------------
// This class really creates the Subtree
// -----------------------------------------------------------------------------
class OCreateTree
{
	OBinaryBaseReader& m_aReader;
	OTreeNodeFactory&  m_aFactory;

    std::auto_ptr<OFilter> m_pFilter;
	sal_Int32          m_nDepth;

	void createInnerLevel(std::auto_ptr<ISubtree>& _pSubtree, std::auto_ptr<ISubtree>& _pNewNode)
	    SAL_THROW( (io::IOException, uno::RuntimeException) );

	void createLevel(std::auto_ptr<ISubtree>& _pSubtree)
	    SAL_THROW( (io::IOException, uno::RuntimeException) );

	OCreateTree(OCreateTree const&);
	OCreateTree& operator=(OCreateTree const&);

public:
	OCreateTree(OBinaryBaseReader& _aReader, OTreeNodeFactory& _aFactory)
			:m_aReader(_aReader),
			 m_aFactory(_aFactory),
			 m_nDepth(0),
			 m_pFilter(NULL)
	{
	}

	void createLevel(std::auto_ptr<ISubtree>& _pSubtree, AbsolutePath const& _aFilter)
	    SAL_THROW( (io::IOException, uno::RuntimeException) )
	{
		m_pFilter.reset( new OFilter(_aFilter) );
		createLevel(_pSubtree);
	}
	
	~OCreateTree()
	{
	}
};

// -----------------------------------------------------------------------------

void OCreateTree::createInnerLevel(std::auto_ptr<ISubtree>& _pSubtree, std::auto_ptr<ISubtree>& _pNewNode)
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	++m_nDepth;

	createLevel(_pNewNode);

	if (_pNewNode.get())
	{	
		if (_pSubtree.get())
		{
			_pSubtree->addChild(std::auto_ptr<INode>(_pNewNode.release()));
		}
		else
		{
			// should be the first one, so only replace it.
			_pSubtree = _pNewNode;
		}
	}	
	--m_nDepth;
}

// -----------------------------------------------------------------------------
void OCreateTree::createLevel(std::auto_ptr<ISubtree>& _pSubtree)
	SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	NodeType::Enum eType = readNodeType(m_aReader);
	
	while(eType != NodeType::stop)
	{
		rtl::OUString aName;
		configuration::Attributes aAttributes;
		switch(eType)
		{
		case NodeType::group:
		{
			readGroup(m_aReader, aName, aAttributes);

			OInputMark aMark(&m_aReader);

			OFilter::Enum eResult = m_pFilter->isInFilter(aName, m_nDepth);
			switch(eResult)
			{
			case OFilter::VALUE_IN_FILTER:
			{
				std::auto_ptr<ISubtree> pNullNode;
				createInnerLevel(_pSubtree, pNullNode);
				break;
			}
			case OFilter::VALUE_IS_LAST_AND_RIGHT:
			case OFilter::INDEX_OUT_OF_BOUND:
			{
				std::auto_ptr<ISubtree> pNewNode = m_aFactory.createGroupNode(aName, aAttributes);
				createInnerLevel(_pSubtree, pNewNode);
				break;
			}
			default:
				aMark.jumpToMark();
			}
			
			/* BUG!!!???
			   // the position isn't what we expect and after read the new position is null.
			   
			  sal_uInt64 nThisPos;
			  osl::FileBase::RC eError = m_aReader.getFile()->getPos(nThisPos);
			  if (eError != osl_File_E_None)
			  {
			  OUString aErrStr = ASCII("read Position: ");
			  aErrStr += FileHelper::createOSLErrorString(eError);
			  volatile int dummy = 0;
			  }
			*/
			break;
		}
		case NodeType::ntyp_set:
		{
			// read a set, createAllSubLevels and concat it to the actual subtree
			rtl::OUString sInstanceName, sInstanceModule;
			readSet(m_aReader, aName, aAttributes, sInstanceName, sInstanceModule);

			OInputMark aMark(&m_aReader);

			OFilter::Enum eResult = m_pFilter->isInFilter(aName, m_nDepth);
			switch(eResult)
			{
			case OFilter::VALUE_IN_FILTER:
			{
				std::auto_ptr<ISubtree> pNullNode;
				createInnerLevel(_pSubtree, pNullNode);
				break;
			}
			case OFilter::VALUE_IS_LAST_AND_RIGHT:
			case OFilter::INDEX_OUT_OF_BOUND:
			{
				std::auto_ptr<ISubtree> pNewNode = m_aFactory.createSetNode(aName, sInstanceName, sInstanceModule, aAttributes);
				createInnerLevel(_pSubtree, pNewNode);
				break;
			}
			default:
				aMark.jumpToMark();
			}
			break;
		}
		case NodeType::value:
		{
			// read a Value, and concat it to the actual subtree
			uno::Any aValue;
			uno::Type aType;
			readValue(m_aReader, aName, aAttributes, aValue, aType);

//?#ifdef DEBUG
//?			OFilter::Enum eResult = m_pFilter->isInFilter(aName, m_nDepth);
//?#endif
			std::auto_ptr<ValueNode> pNewNode;
			if (aValue.hasValue())
			{
				pNewNode = m_aFactory.createValueNode( aName, aValue, aAttributes );
			}
			else
			{
				pNewNode =  m_aFactory.createNullValueNode(aName, aType, aAttributes );
			}
			if (_pSubtree.get())
			{
				_pSubtree->addChild(std::auto_ptr<INode>(pNewNode.release()));
			}
			else
			{
				// it has shown that this assertion is wrong, if the Subtree
				// isn't set, there is a reason why, maybe, we are in a deep filter
				// had until here, all things would filtered out.
				// so this assertion will removed next.
//?				OSL_ENSURE(false, "Wrong, we need an initialised Subtree");
				// throw IllegalValue...();
			}
			break;
		}
		default:
			OSL_ENSURE(false, "binary session: Format error.");
		}
		eType = readNodeType(m_aReader);
	}
}
// -----------------------------------------------------------------------------

std::auto_ptr<ISubtree> read (
	AbsolutePath const& _aFilter, rtl::OUString const& _aFilename, 
	uno::Reference< lang::XMultiServiceFactory > const& _xServiceProvider
  )
  SAL_THROW( (io::IOException, uno::RuntimeException) )
{
	std::auto_ptr<ISubtree> pSubtree;

    OBinaryBaseReader aReader(_aFilename);
	aReader.open();

	sal_Int32 nMagic, nVersion, nDateStamp;
	readFileHeader(aReader, nMagic, nVersion, nDateStamp);
    if (CFG_BINARY_VERSION != nVersion || CFG_BINARY_SUBTREE_MAGIC != nMagic)
    {
        throw io::IOException(ASCII("read binary: version or magic mismatch."), NULL);
    }
	OTreeNodeFactory aFactory; // = this->getNodeFactory();

	OCreateTree aCreate(aReader, aFactory);
	aCreate.createLevel(pSubtree, _aFilter);

	sal_Int32 nMagic2 = 0;
	aReader.read(nMagic2);
	aReader.dispose();

	if (nMagic != nMagic2)
	{
		throw io::IOException(ASCII("read binary: the magic values are not equal, binary file is corrupt."), NULL);
	}

    return pSubtree;
}

} // namespace configmgr
