/*************************************************************************
 *
 *  $RCSfile: synccoll.cxx,v $
 *
 *  $Revision: 1.1.1.1 $
 *
 *  last change: $Author: hr $ $Date: 2000/09/18 16:16:53 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#include "syncbase.hxx"
#include "synccont.hxx"
#include "synccoll.hxx"

// -----------------
// - SyncCollector -
// -----------------

SyncCollector::SyncCollector( const REF( XMultiServiceFactory )& rxFact ) :
	mxFact		( rxFact ),
	mpSyncBases	( NULL ),
	mbLocked	( sal_False )	
{
}

// -----------------------------------------------------------------------------

SyncCollector::~SyncCollector()
{
	for( sal_uInt32 i = 0, nCount = ( GetSyncSchemeCount() << 1 ); i < nCount; i++ )
		delete mpSyncBases[ i ];

	delete[] mpSyncBases;
}

// -----------------------------------------------------------------------------

Any SAL_CALL SyncCollector::queryInterface( const Type & rType ) throw (RuntimeException)
{
	const Any aRet( ::cppu::queryInterface( rType,
											static_cast< XContentEventListener* >( this ), 
											static_cast< XSyncCollector* >( this ),
											static_cast< XContentCreationListener* >( this ) ) );

	return( aRet.hasValue() ? aRet : OWeakObject::queryInterface( rType ) );
}

// -----------------------------------------------------------------------------

void SAL_CALL SyncCollector::acquire() throw( RuntimeException )
{
	OWeakObject::acquire();
}

// -----------------------------------------------------------------------------

void SAL_CALL SyncCollector::release() throw( RuntimeException )
{
	OWeakObject::release();
}

// -----------------------------------------------------------------------------

void SAL_CALL SyncCollector::disposing( const EventObject& rSource ) throw( RuntimeException )
{
}

// -----------------------------------------------------------------------------

void SAL_CALL SyncCollector::contentEvent( const ContentEvent& rEvt ) throw( RuntimeException )
{
	if( !IsLocked() )
	{
		sal_uInt32 nSyncEvent;

		// map content action to sync event
		switch( rEvt.Action )
		{
			case( ContentAction::DELETED ):
				nSyncEvent = SyncEvent::DELETED;
			break;

			// !!!
			default:
				nSyncEvent = SyncEvent::CREATED;
			break;
		}

		// is it an important event
		if( ( SyncEvent::NONE != nSyncEvent ) && rEvt.Content.is() ) 
		{
			REF( XContentIdentifier)	xCntId( rEvt.Content->getIdentifier() );
			const OUString				aSchemeName( xCntId->getContentProviderScheme() );

			// is scheme registered
			if( HasSyncBase( aSchemeName ) )
			{
				const OUString				aId( ToRel( xCntId->getContentIdentifier(), GetSyncScheme( aSchemeName ) ) );
				SyncBase&					rDynBase = GetSyncBase( aSchemeName, sal_True );
				SyncBase&					rStatBase = GetSyncBase( aSchemeName, sal_False );
				const SyncBaseElement*		pOldElem = rDynBase.GetElement( aId );
				SyncBaseElement*			pNewElem = NULL;

				if( !pOldElem )
					pOldElem = rStatBase.GetElement( aId );

				if( !pOldElem )
				{
					const sal_uInt32 nUID = SyncBaseElement::GenerateUID();
					const sal_uInt32 nType = SyncContent::TypeToTypeId( rEvt.Content->getContentType().getStr() );

					if( nType != SyncType::NONE )
						pNewElem = new SyncBaseElement( aId, nType, nUID, 1, nSyncEvent );
				}
				else
				{
					pNewElem = new SyncBaseElement( *pOldElem );
					pNewElem->SetEvent( nSyncEvent );
				}

				if( pNewElem )
				{
					if( pOldElem )
						rStatBase.RemoveElement( pOldElem->GetContentIdentifier() );

					rDynBase.PutElement( *pNewElem );
					delete pNewElem;
				}
			}
		}
	}
}

// -----------------------------------------------------------------------------

sal_Bool SAL_CALL SyncCollector::initialize( const REF( XContentProviderManager )& rxMgr, 
											 const SEQ( SyncScheme )& rSyncSchemes,
											 const OUString& rBaseFileName )
{
	const sal_uInt32	nSyncSchemeCount = rSyncSchemes.getLength();
	sal_Bool			bRet;

	mxProvMgr = rxMgr;

	if( mxProvMgr.is() && nSyncSchemeCount )
	{
		maSyncSchemes = rSyncSchemes;
		mpSyncBases	= new PSYNCBASE[ GetSyncSchemeCount() << 1 ];

		for( sal_uInt32 i = 0, nCount = GetSyncSchemeCount(); i < nCount; i++ )
		{
			const OUString	aName( maSyncSchemes[ i ].Scheme );
			OUString		aDynaName( aName ); aDynaName += ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM( "_dyna" ) );
			OUString		aStatName( aName ); aStatName += ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM( "_stat" ) );

			mpSyncBases[ i << 1 ] = new SyncBase( rBaseFileName, aDynaName, BASE_FLAGS_AUTOFLUSH );
			mpSyncBases[ ( i << 1 ) + 1 ] = new SyncBase( rBaseFileName, aStatName, BASE_FLAGS_AUTOFLUSH );
		}

		bRet = sal_True;
	}
	else
		bRet = sal_False;

	return bRet;
}

// -----------------------------------------------------------------------------

SEQ( SyncScheme ) SAL_CALL SyncCollector::getRegisteredSyncSchemes()
{
	return maSyncSchemes;
}

// -----------------------------------------------------------------------------

SEQ( SyncElement ) SAL_CALL SyncCollector::getSyncSequence( const OUString& rSchemeName, const SyncInfo& rSyncInfo )
{
	SEQ( SyncElement )	aElemSeq( 0 );
	SyncBase&			rBase = GetSyncBase( rSchemeName, sal_True );
	SyncCursor*			pCursor = rBase.GetCursor();

	if( pCursor )
	{
		static const sal_uInt32	nResize = 1024;
		const sal_uInt32		nFilterCount = rSyncInfo.SyncFilter.getLength();
		sal_uInt32				i = 0;

		if( nFilterCount )
		{
			// get filtered sequence
			for( const SyncBaseElement* pElem = pCursor->First(); pElem; pElem = pCursor->Next() )
			{
				const FilterData& rFilter = rSyncInfo.SyncFilter;

				for( sal_uInt32 n = 0; n < nFilterCount; n++ )
				{
					if( pElem->Matches( rFilter[ n ] ) )
					{
						if( i == aElemSeq.getLength() )
							aElemSeq.realloc( aElemSeq.getLength() + nResize );

						aElemSeq[ i++ ] = *pElem;
					}
				}
			}
		}
		else
		{
			// get full sequence
			for( const SyncBaseElement* pElem = pCursor->First(); pElem; pElem = pCursor->Next() )
			{
				if( i == aElemSeq.getLength() )
					aElemSeq.realloc( aElemSeq.getLength() + nResize );

				aElemSeq[ i++ ] = *pElem;
			}
		}

		delete pCursor;
		aElemSeq.realloc( i );
	}

	return aElemSeq;
}

// -----------------------------------------------------------------------------

void SAL_CALL SyncCollector::beginSynchronize()
{
	DBG_ASSERT( !mbLocked, "already locked" );
	mbLocked = sal_True;
}

// -----------------------------------------------------------------------------

void SAL_CALL SyncCollector::endSynchronize()
{
	DBG_ASSERT( mbLocked, "not locked" );
	mbLocked = sal_False;
}

// -----------------------------------------------------------------------------

sal_Bool SAL_CALL SyncCollector::getCorrespondingElement( const SyncScheme& rSyncSchemeIn,
														  const SyncElement& rSyncElementIn, 
														  SyncElement& rSyncElementOut )
{
	const SyncScheme&		rSyncSchemeOut = GetSyncScheme( rSyncSchemeIn.Scheme );
	const SyncBaseElement*	pCorrElem = GetSyncBase( rSyncSchemeOut.Scheme, sal_True ).GetElement( rSyncElementIn.ContentIdentifier );
	sal_Bool				bRet;

	if( !pCorrElem )
		pCorrElem = GetSyncBase( rSyncSchemeOut.Scheme, sal_False ).GetElement( rSyncElementIn.ContentIdentifier );

	if( pCorrElem )
	{
		rSyncElementOut = *pCorrElem;
		delete pCorrElem;
		bRet = sal_True;
	}
	else
	{
		bRet = sal_False;
		rSyncElementOut.ContentIdentifier = OUString();
		rSyncElementOut.Type = SyncType::NONE;
		rSyncElementOut.UID = rSyncElementOut.Generation = rSyncElementOut.Event = rSyncElementOut.Action = 0;
	}

	return bRet;
}

// -----------------------------------------------------------------------------

sal_Bool SAL_CALL SyncCollector::executeElementCommand( const OUString& rSchemeName,
														const SyncElement& rElem,
														const OUString& rCommand, 
														Any& rAny )
{
	DBG_ASSERT( HasSyncBase( rSchemeName ), "SyncCollector::executeElementCommand: scheme not registered" );
	
	const SyncScheme&	rScheme = GetSyncScheme( rSchemeName );
	OUString			aId( rScheme.Root ); 
	SyncContent*		pCnt = SyncContent::Create( mxFact, mxProvMgr, aId += rElem.ContentIdentifier, rElem.Type, sal_True );
	sal_Bool			bRet = sal_False;

	if( pCnt->IsValid() )
	{
		if( ( pCnt->GetTypeId() == SyncType::FOLDER || pCnt->GetTypeId() == SyncType::VFS_FOLDER ) &&
			( rCommand.equalsIgnoreCase( ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM( "open" ) ) ) || 
			  rCommand.equalsIgnoreCase( ::rtl::OUString( RTL_CONSTASCII_USTRINGPARAM( "insert" ) ) ) ) )
		{
			// do nothing
		}
		else
			pCnt->ExecuteCommand( rCommand, rAny );

		bRet = sal_True;
	}

	SyncContent::Destroy( pCnt );

	return bRet;
}

// -----------------------------------------------------------------------------

void SAL_CALL SyncCollector::elementSynchronized( const OUString& rSchemeName,
												  const SyncElement& rElem,
												  sal_uInt32 nSyncAction )
{
	SyncBase& rDynBase = GetSyncBase( rSchemeName, sal_True );
	SyncBase& rStatBase = GetSyncBase( rSchemeName, sal_False );

	rDynBase.RemoveElement( rElem.ContentIdentifier );

	if( ( nSyncAction & SyncAction::COPY_TO_SERVER ) || 
		( nSyncAction & SyncAction::COPY_TO_CLIENT ) )
	{
		rStatBase.PutElement( rElem );
	}
	else if( ( nSyncAction & SyncAction::REMOVE_FROM_SERVER ) || 
			 ( nSyncAction & SyncAction::REMOVE_FROM_CLIENT ) )
	{
		rStatBase.RemoveElement( rElem.ContentIdentifier );
	}
}

// -----------------------------------------------------------------------------

void SAL_CALL SyncCollector::notify( const REF( XContent )& rxCnt, sal_uInt32 nEvent )
{
	ContentEvent aEvt;

	aEvt.Content = rxCnt;
	aEvt.Action = nEvent;

	contentEvent( aEvt );
}

// -----------------------------------------------------------------------------

void SAL_CALL SyncCollector::contentCreationEvent( const ContentCreationEvent& rEvt )
{
	rEvt.Content->addContentEventListener( this );
}

// -----------------------------------------------------------------------------

const SyncScheme& SyncCollector::GetSyncScheme( const OUString& rSchemeName ) const
{
	sal_uInt32	nPos, nCount;
	sal_Bool	bFound = sal_False;

	for( nPos = 0, nCount = GetSyncSchemeCount(); ( nPos < nCount ) && !bFound; nPos++ )
		if( rSchemeName == maSyncSchemes[ nPos ].Scheme ) 
			bFound = sal_True;

	DBG_ASSERT( bFound, "Invalid SchemeName" );

	return( maSyncSchemes[ nPos - 1 ] );
}

// -----------------------------------------------------------------------------

sal_Bool SyncCollector::HasSyncBase( const OUString& rSchemeName ) const
{
	sal_uInt32	nPos, nCount;
	sal_Bool	bFound = sal_False;

	for( nPos = 0, nCount = GetSyncSchemeCount(); ( nPos < nCount ) && !bFound; nPos++ )
		if( rSchemeName == maSyncSchemes[ nPos ].Scheme ) 
			bFound = sal_True;

	return bFound;
}

// -----------------------------------------------------------------------------

SyncBase& SyncCollector::GetSyncBase( const OUString& rSchemeName, sal_Bool bDynamic ) const
{
	DBG_ASSERT( HasSyncBase( rSchemeName ), "Invalid/Not registered SchemeName" );

	sal_uInt32	nPos, nCount;
	sal_Bool	bFound = sal_False;

	for( nPos = 0, nCount = GetSyncSchemeCount(); ( nPos < nCount ) && !bFound; nPos++ )
		if( rSchemeName == maSyncSchemes[ nPos ].Scheme ) 
			bFound = sal_True;

	return( *( mpSyncBases[ ( ( nPos - 1 ) << 1 ) + ( bDynamic ? 0 : 1 ) ] ) );
}
