/*************************************************************************
 *
 *  $RCSfile: filtercachedata.cxx,v $
 *
 *  $Revision: 1.15.8.1 $
 *
 *  last change: $Author: mh $ $Date: 2002/04/03 13:36:15 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/


//_________________________________________________________________________________________________________________
//	my own includes
//_________________________________________________________________________________________________________________

#ifndef __FRAMEWORK_CLASSES_FILTERCACHEDATA_HXX_
#include <classes/filtercachedata.hxx>
#endif

#ifndef __FRAMEWORK_GENERAL_H_
#include <general.h>
#endif

#ifndef __FRAMEWORK_MACROS_DEBUG_HXX_
#include <macros/debug.hxx>
#endif

//_________________________________________________________________________________________________________________
//	interface includes
//_________________________________________________________________________________________________________________

//_________________________________________________________________________________________________________________
//	other includes
//_________________________________________________________________________________________________________________

#ifndef _UTL_CONFIGMGR_HXX_
#include <unotools/configmgr.hxx>
#endif

#ifndef _UTL_CONFIGITEM_HXX_
#include <unotools/configitem.hxx>
#endif

#ifndef _UTL_CONFIGPATHES_HXX_
#include <unotools/configpathes.hxx>
#endif

#ifndef _RTL_USTRBUF_HXX_
#include <rtl/ustrbuf.hxx>
#endif

#ifndef _RTL_URI_HXX_
#include <rtl/uri.hxx>
#endif

//_________________________________________________________________________________________________________________
//	namespace
//_________________________________________________________________________________________________________________

namespace framework{

//_________________________________________________________________________________________________________________
//  non exported const
//_________________________________________________________________________________________________________________

//_________________________________________________________________________________________________________________
//  non exported definitions
//_________________________________________________________________________________________________________________

/*-****************************************************************************************************//**
    @short      standard constructor
    @descr      This will open an configuration access to given path ... but no values
                are readed! Because user will do that by explicit calling of read-method. He give
                us a pointer to his container, and we fill it.
                Writing works in the same way. We use given data container to do that.

    @seealso    baseclass ConfigItem
    @seealso    method read()
    @seealso    method write()

    @param      "sPath", path to configuration file and root node in this file
    @return     -

    @onerror    -
*//*-*****************************************************************************************************/
FilterCFGAccess::FilterCFGAccess( const ::rtl::OUString& sPath   ,
                                        sal_Int32        nVersion,
                                        sal_Int16        nMode   )
    : ::utl::ConfigItem ( sPath, nMode )
    , m_nVersion        ( nVersion     )
{
    // Check incoming parameter first.
    LOG_ASSERT2( implcp_ctor( sPath, nVersion, nMode ), "FilterCFGAccess::ctor", "Invalid parameter detected!" )

    // Use given path to set internal "working mode".
    // It's neccessary to handle read/write operations in a different way.
    // Some items are not handled for additional configuration file!
    if( sPath == PACKAGENAME_TYPEDETECTION_STANDARD )
    {
        m_ePackage = E_STANDARD;
    }
    else
    if( sPath == PACKAGENAME_TYPEDETECTION_ADDITIONAL )
    {
        m_ePackage = E_ADDITIONAL;
    }

    css::uno::Any aProperty = ::utl::ConfigManager::GetConfigManager()->GetDirectConfigProperty( ::utl::ConfigManager::PRODUCTNAME );
    sal_Bool      bState    = (aProperty >>= m_sProductName)                                                                    ;
    if(
        ( bState                     == sal_False )    ||
        ( m_sProductName.getLength() <  1         )
      )
    {
        LOG_ERROR( "FilterCFGAccess::read()", "Couldn't get ProductName from configuration! Use \"StarOffice\" as fallback ..." )
        m_sProductName = PRODUCTNAME_FALLBACK;
    }

    m_bActivateOpenofficePatch = m_sProductName.equalsIgnoreAsciiCaseAscii("openoffice.org");

    // Initialize right sub key count for current set file version.
    // It's neccessary to read right structure from configuration and get enough memory for that operations!
    impl_initKeyCounts();
}

/*-****************************************************************************************************//**
    @short      standard destructor
    @descr      We do nothing here!

    @seealso    -

    @param      -
    @return     -

    @onerror    -
*//*-*****************************************************************************************************/
FilterCFGAccess::~FilterCFGAccess()
{
}

/*-****************************************************************************************************//**
    @short      read configuration data into given data container
    @descr      We use given data container to fill it with readed values from configuration.
                Reading values depend from opened path! Because - our filter configuration is
                tiled into two parts ... standard and additional filter. Format of both files
                is always the same ... but some items are not set in additional file.
                e.g. default detector and loader
                This informations are set in standard file only!

    @seealso    -

    @param      "rData", pointer to data container for filling
    @return     Filled data container.

    @onerror    We do nothing! (But we show some assertions.)
*//*-*****************************************************************************************************/
void FilterCFGAccess::read( DataContainer& rData )
{
    // Safe impossible cases!
    LOG_ASSERT2( implcp_read( rData ), "FilterCFGAccess::read()", "Invalid parameter detected!" )

    // Clear container first!
    rData.free();

    // Get current set locale from configuration.
    // We read all localized values for all locales from our configuration ...
    // but sometimes we must give our owners one value for current set locale!
    // This value is neccessary for later "impl_load...()" calls too!!!
    css::uno::Any   aProperty = ::utl::ConfigManager::GetConfigManager()->GetDirectConfigProperty( ::utl::ConfigManager::LOCALE );
    sal_Bool        bState    = (aProperty >>= rData.m_sLocale)                                                                    ;
    if(
        ( bState                      == sal_False )    ||
        ( rData.m_sLocale.getLength() <  1         )
      )
    {
        LOG_ERROR( "FilterCFGAccess::read()", "Couldn't get Locale of configuration! Localized values will not handled correctly. But I try it by using \"en-US\" as fallback ..." )
        rData.m_sLocale = LOCALE_FALLBACK;
    }

    // Read values from configuration. Look for right package and version handling too!
    switch( m_ePackage )
    {
        case E_STANDARD     :   {
                                    impl_loadTypes      ( rData );
                                    impl_loadFilters    ( rData );
                                    impl_loadDetectors  ( rData );
                                    impl_loadLoaders    ( rData );
                                    impl_loadDefaults   ( rData );
                                    if( m_nVersion >= 5 )
                                    {
                                        impl_loadContentHandlers( rData );
                                    }
                                    if( m_nVersion >= 7 )
                                    {
                                        impl_loadProtocolHandlers( rData );
                                    }
                                }
                                break;
        case E_ADDITIONAL   :   {
                                    impl_loadTypes      ( rData );
                                    impl_loadFilters    ( rData );
                                }
                                break;
    }
}

/*-****************************************************************************************************//**
    @short      write configuration data from given data container to configuration
    @descr      We use given data container to write his values to configuration.
                Writing values depend from opened path! Because - our filter configuration is
                tiled into two parts ... standard and additional filter. Format of both files
                is always the same ... but some items are not set in additional file.
                e.g. default detector and loader
                This informations are set in standard file only!

    @seealso    -

    @param      "rData", pointer to data container
    @return     -

    @onerror    We do nothing! (But we show some assertions.)
*//*-*****************************************************************************************************/
void FilterCFGAccess::write( DataContainer& rData )
{
    // Safe impossible cases!
    LOG_ASSERT2( implcp_write( rData ), "FilterCFGAccess::write()", "Invalid parameter detected!" )

	// Work only if something was changed!
    if( rData.m_bIsModified == sal_True )
	{
        // look for right package and version handling too.
        switch( m_ePackage )
        {
            case E_STANDARD     :   {
                                        impl_saveTypes  ( rData );
                                        impl_saveFilters( rData );
                                        /*TODO ... not implemented yet!
                                        impl_saveDetectors( rData );
                                        impl_saveLoaders  ( rData );
                                        if( m_nVersion >= 5 )
                                        {
                                            impl_saveContentHandlers( rData );
                                        }
                                        if( m_nVersion >= 7 )
                                        {
                                            impl_saveProtocolHandlers( rData );
                                        }
                                        */
                                    }
                                    break;
            case E_ADDITIONAL   :   {
                                        impl_saveTypes  ( rData );
                                        impl_saveFilters( rData );
                                    }
                                    break;
        }

        // Don't forget to reset modified flag ... otherwise
        // we will called again and again and ...
        rData.m_bIsModified = sal_False;
	}
}

/*-************************************************************************************************************//**
    @short      encode/decode set names
    @descr      Our configuration support special characters of set names by a special encoding/decoding.
                These two methods do that for our values.

                encode:     <name>                     =>      <templatetype>["<name>"]
                decode:     <templatetype>["<name>"]   =>      <name>

                - with <templatetype> = {Type|Filter|DetectService|FrameLoader|ContentHandler}
                  depends from given eType!
                - with <name> valid html-encoded! "&" => "&amp;" ...!

    @seealso    -

    @param      "sName", source for convertion
    @param      "eType", specify template
    @return     Converted value or untouched one, if there is nothing to do.

    @onerror    -
*//*-*************************************************************************************************************/
/*
::rtl::OUString FilterCFGAccess::encodeSetName( const ::rtl::OUString& sName, ETemplateType eType )
{
    ::rtl::OUStringBuffer  sSource     ( sName )          ;
    ::rtl::OUStringBuffer  sDestination( 256   )          ;
    sal_Int32              nCount      = sName.getLength();
    sal_Int32              nPosition   = 0                ;

    // add <templatename>
    switch( eType )
    {
        case E_TEMPLATE_TYPE             :   sDestination.append( TEMPLATE_TYPE              );
                                    break;
        case E_TEMPLATE_FILTER           :   sDestination.append( TEMPLATE_FILTER            );
                                    break;
        case E_TEMPLATE_DETECTSERVICE    :   sDestination.append( TEMPLATE_DETECTSERVICE     );
                                    break;
        case E_TEMPLATE_FRAMELOADER      :   sDestination.append( TEMPLATE_FRAMELOADER       );
                                    break;
        case E_TEMPLATE_CONTENTHANDLER   :   sDestination.append( TEMPLATE_CONTENTHANDLER    );
                                    break;
    }

    // add ["
    sDestination.append( CFG_ENCODING_OPEN );

    // add name (html encoded!)
    for( nPosition=0; nPosition<nCount; ++nPosition )
	{
        sal_Unicode cSign = sSource.charAt(nPosition);
        switch( cSign )
		{
            // code &, ", ', <, > ...
            case '&' :  sDestination.appendAscii( "&amp;"   );
						break;
            case '<' :  sDestination.appendAscii( "&lt;"    );
						break;
            case '>' :  sDestination.appendAscii( "&gt;"    );
						break;
            case '\'':  sDestination.appendAscii( "&rsquo;" );
						break;
            case '\"':  sDestination.appendAscii( "&quot;"  );
						break;
			// copy all other letters
            default :   sDestination.append( cSign );
						break;
		}
	}

    // add "]
    sDestination.append( CFG_ENCODING_CLOSE );

    // return encoded name
    return sDestination.makeStringAndClear();
}

//*****************************************************************************************************************
::rtl::OUString FilterCFGAccess::decodeSetName( const ::rtl::OUString& sName, ETemplateType eType )
{
    LOG_ERROR( "FilterCFGAccess::decodeSetName()", "Not implemented yet!" )
    return sName;
}
*/

/*-************************************************************************************************************//**
    @short      build own formated string of type/filter properties with right encoding of forbidden signs
    @descr      To make our configuration file smaller and guarant a faster acces - we don't handle all properties
                seperatly. We write an own formated string for some of them.
                These two methods build this string for types/filters ... look at "parse..." methods too.
                They analyze this string at reading and fill right type/filter structures.

    @seealso    method decodeTypeData()
    @seealso    method decodeFilterData()

    @param      "aType"  , source for convertion
    @param      "aFilter", source for convertion
    @return     Own formated string of item properties.

    @onerror    -
*//*-*************************************************************************************************************/
::rtl::OUString FilterCFGAccess::encodeTypeData( const FileType& aType )
{
    ::rtl::OUStringBuffer sData( 1000 );
    ::rtl::OUString       sValue       ;

    // Preferred
    if( aType.bPreferred == sal_True )
    {
        sData.appendAscii( "1" );
    }
    else
    {
        sData.appendAscii( "0" );
    }
    sData.append( PROPERTY_SEPERATOR                    );

    // MediaType
    sValue = ::rtl::Uri::encode( aType.sMediaType, rtl_UriCharClassUnoParamValue, rtl_UriEncodeIgnoreEscapes, RTL_TEXTENCODING_UTF8 );
    sData.append( sValue                                );
    sData.append( PROPERTY_SEPERATOR                    );

    // ClipboardFormat
    sValue = ::rtl::Uri::encode( aType.sClipboardFormat, rtl_UriCharClassUnoParamValue, rtl_UriEncodeIgnoreEscapes, RTL_TEXTENCODING_UTF8 );
    sData.append( sValue                                );
    sData.append( PROPERTY_SEPERATOR                    );

    // URLPattern
    sData.append( encodeStringList( aType.lURLPattern ) );
    sData.append( PROPERTY_SEPERATOR                    );

    // Extensions
    sData.append( encodeStringList( aType.lExtensions ) );
    sData.append( PROPERTY_SEPERATOR                    );

    // DocumentIconID
    sData.append( (sal_Int32)aType.nDocumentIconID      );
    sData.append( PROPERTY_SEPERATOR                    );

    return sData.makeStringAndClear();
}

//*****************************************************************************************************************
void FilterCFGAccess::decodeTypeData( const ::rtl::OUString& sData, FileType& aType )
{
    sal_Int32       nIndex  = 0;
    ::rtl::OUString sToken     ;
    sal_Int32       nToken  = 0;

    do
    {
        sToken = sData.getToken( 0, PROPERTY_SEPERATOR, nIndex );
        switch( nToken )
        {
            // Preferred
            case 0:     {
                            aType.bPreferred = sal_False;
                            if( sToken.toInt32() == 1 )
                            {
                                aType.bPreferred = sal_True;
                            }
                        }
                        break;
            // MediaType
            case 1:     aType.sMediaType       = ::rtl::Uri::decode( sToken, rtl_UriDecodeWithCharset, RTL_TEXTENCODING_UTF8 );
                        break;
            // ClipboardFormat
            case 2:     aType.sClipboardFormat = ::rtl::Uri::decode( sToken, rtl_UriDecodeWithCharset, RTL_TEXTENCODING_UTF8 );
                        break;
            // URLPattern
            case 3:     aType.lURLPattern      = decodeStringList( sToken );
                        break;
            // Extensions
            case 4:     aType.lExtensions      = decodeStringList( sToken );
                        break;
            // DocumentIconID
            case 5:     aType.nDocumentIconID  = sToken.toInt32();
                        break;
        }
        ++nToken;
    }
    while( nIndex >= 0 );
}

//*****************************************************************************************************************
::rtl::OUString FilterCFGAccess::encodeFilterData( const Filter& aFilter )
{
    ::rtl::OUStringBuffer sData ( 1000 );
    ::rtl::OUString       sValue        ;

    // Order
    sData.append( (sal_Int32)aFilter.nOrder                );
    sData.append( PROPERTY_SEPERATOR                       );

    // Type
    sValue = ::rtl::Uri::encode( aFilter.sType, rtl_UriCharClassUnoParamValue, rtl_UriEncodeIgnoreEscapes, RTL_TEXTENCODING_UTF8 );
    sData.append( sValue                                   );
    sData.append( PROPERTY_SEPERATOR                       );

    // DocumentService
    sValue = ::rtl::Uri::encode( aFilter.sDocumentService, rtl_UriCharClassUnoParamValue, rtl_UriEncodeIgnoreEscapes, RTL_TEXTENCODING_UTF8 );
    sData.append( sValue                                   );
    sData.append( PROPERTY_SEPERATOR                       );

    // FilterService
    sValue = ::rtl::Uri::encode( aFilter.sFilterService, rtl_UriCharClassUnoParamValue, rtl_UriEncodeIgnoreEscapes, RTL_TEXTENCODING_UTF8 );
    sData.append( sValue                                   );
    sData.append( PROPERTY_SEPERATOR                       );

    // Flags
    sData.append( (sal_Int32)aFilter.nFlags                );
    sData.append( PROPERTY_SEPERATOR                       );

    // UserData
    sData.append( encodeStringList( aFilter.lUserData )    );
    sData.append( PROPERTY_SEPERATOR                       );

    // FileFormatVersion
    sData.append( (sal_Int32)aFilter.nFileFormatVersion    );
    sData.append( PROPERTY_SEPERATOR                       );

    // TemplateName
    sValue = ::rtl::Uri::encode( aFilter.sTemplateName, rtl_UriCharClassUnoParamValue, rtl_UriEncodeIgnoreEscapes, RTL_TEXTENCODING_UTF8 );
    sData.append( sValue                                   );
    sData.append( PROPERTY_SEPERATOR                       );

    return sData.makeStringAndClear();
}

//*****************************************************************************************************************
void FilterCFGAccess::decodeFilterData( const ::rtl::OUString& sData, Filter& aFilter )
{
    sal_Int32       nIndex  = 0;
    ::rtl::OUString sToken     ;
    sal_Int32       nToken  = 0;

    do
    {
        sToken = sData.getToken( 0, PROPERTY_SEPERATOR, nIndex );
        switch( nToken )
        {
            // Order
            case 0:     aFilter.nOrder              = sToken.toInt32();
                        break;
            // Type
            case 1:     aFilter.sType               = ::rtl::Uri::decode( sToken, rtl_UriDecodeWithCharset, RTL_TEXTENCODING_UTF8 );
                        break;
            // DocumentService
            case 2:     aFilter.sDocumentService    = ::rtl::Uri::decode( sToken, rtl_UriDecodeWithCharset, RTL_TEXTENCODING_UTF8 );
                        break;
            // FilterService
            case 3:     aFilter.sFilterService      = ::rtl::Uri::decode( sToken, rtl_UriDecodeWithCharset, RTL_TEXTENCODING_UTF8 );
                        break;
            // Flags
            case 4:     aFilter.nFlags              = sToken.toInt32();
                        break;
            // UserData
            case 5:     aFilter.lUserData           = decodeStringList( sToken );
                        break;
            // FileFormatVersion
            case 6:     aFilter.nFileFormatVersion  = sToken.toInt32();
                        break;
            // TemplateName
            case 7:     aFilter.sTemplateName       = ::rtl::Uri::decode( sToken, rtl_UriDecodeWithCharset, RTL_TEXTENCODING_UTF8 );
                        break;
        }
        ++nToken;
    }
    while( nIndex >= 0 );
}

//*****************************************************************************************************************
::rtl::OUString FilterCFGAccess::encodeStringList( const StringList& lList )
{
    ::rtl::OUStringBuffer sData( 1000 );
    ::rtl::OUString       sValue       ;

    ConstStringListIterator pItem = lList.begin();
    while( pItem != lList.end() )
    {
        sValue = ::rtl::Uri::encode( *pItem, rtl_UriCharClassUnoParamValue, rtl_UriEncodeIgnoreEscapes, RTL_TEXTENCODING_UTF8 );
        sData.append( sValue );
        ++pItem;
        if( pItem != lList.end() )
        {
            sData.append( LIST_SEPERATOR );
        }
    }

    return sData.makeStringAndClear();
}

//*****************************************************************************************************************
StringList FilterCFGAccess::decodeStringList( const ::rtl::OUString& sValue )
{
    StringList      lList      ;
    sal_Int32       nIndex  = 0;
    ::rtl::OUString sToken     ;

    do
    {
        sToken = sValue.getToken( 0, LIST_SEPERATOR, nIndex );
        if( sToken.getLength() > 0 )
        {
            lList.push_back( ::rtl::Uri::decode( sToken, rtl_UriDecodeWithCharset, RTL_TEXTENCODING_UTF8 ) );
        }
    }
    while( nIndex >= 0 );

    return lList;
}

/*-************************************************************************************************************//**
    @short      set right key count of set nodes depending from config file version
    @descr      Our file "TypeDetection.xml" exist in different versions! So we can support reading/writing of older
                ones. It's neccessary for our own tool "xml2xcd". Some set nodes changed here key count often.
                So we must react in a right way. We set our internal count variables and use it at read/write access
                on our configuration services.

    @attention  If count is set wrong - nothing will work. Warn programmer or user!
                ... but some entries doesn't depend from this version handling ... They are fix ...

    @seealso    method impl_load...()
    @seealso    method impl_save...()

    @param      -
    @return     -

    @onerror    Warn programmer ... it must work!
*//*-*************************************************************************************************************/
void FilterCFGAccess::impl_initKeyCounts()
{
    // set fix values
    m_nKeyCountDetectors        = 1;
    m_nKeyCountLoaders          = 2;
    m_nKeyCountContentHandlers  = 1;
    m_nKeyCountProtocolHandlers = 1;

    // set variable values ...
    // ... for types
    if( m_nVersion <= 2 )
        m_nKeyCountTypes = 7 ;
    else
        m_nKeyCountTypes = 2 ;

    // ... for filters
    if( m_nVersion == 1 )
        m_nKeyCountFilters = 9 ;
    else
    if( m_nVersion == 2 )
        m_nKeyCountFilters = 10;
    else
        m_nKeyCountFilters = 3 ;
}

/*-************************************************************************************************************//**
    @short      methods to load all configuration values
    @descr      Call this methods to read configuration values from configuration services into given data container.

    @seealso    method impl_save...()

    @param      "rData", reference to data container for filling
    @return     -

    @onerror    -
*//*-*************************************************************************************************************/
void FilterCFGAccess::impl_loadTypes( DataContainer& rData )
{
	//-------------------------------------------------------------------------------------------------------------
	// We must handle a dynamic set of keys!
    // structure version 1/2:       /Types
    //                                  /<entry_01>
    //                                      /Preferred              [boolean    ]
    //                                      /UIName                 [string     ] localized!
    //                                      /MediaType              [string     ]
    //                                      /ClipboardFormat        [string     ]
    //                                      /URLPattern             [stringlist ]
    //                                      /Extensions             [stringlist ]
    //                                      /DocumentIconID         [integer    ]
    //                                  /<entry_02>
    //                                      /...
    // structure version 3:         /Types
    //                                  /<entry_01>
    //                                      /UIName                 [string     ] localized!
    //                                      /Data                   [string     ] own format ... {Preferred,MediaType,ClipboardFormat,URLPattern,Extensions,DocumentIconID}
	//
	// a)	Get names of all current existing type nodes in configuration.
	// b)	Reserve memory for list of all type pathes - his names and properties!
	// c)	Fill list of all pathes.
	// d)	Get values for this list.
	// e)	Fill internal caches from these list.
    //-------------------------------------------------------------------------------------------------------------

    RTL_LOGFILE_CONTEXT( aMeasure, "framework (as96863) ::FilterCFGAccess::impl_loadTypes" );

	// a)
    css::uno::Sequence< ::rtl::OUString > lNodeNames;
    if( m_nVersion <= 5 )
        lNodeNames = GetNodeNames( SUBLIST_TYPES, ::utl::CONFIG_NAME_LOCAL_NAME ); // without encoding!
    else
        lNodeNames = GetNodeNames( SUBLIST_TYPES, ::utl::CONFIG_NAME_LOCAL_PATH ); // encoded!
	// b)
    sal_uInt32                            nNodeCount      = lNodeNames.getLength()       ;
    css::uno::Sequence< ::rtl::OUString > lPropertyNames  ( nNodeCount*m_nKeyCountTypes );
    sal_uInt32                            nNodeName       = 0                            ;
    sal_uInt32                            nProperty       = 0                            ;
    ::rtl::OUString                       sPath                                          ;

    // c)
	for( nNodeName=0; nNodeName<nNodeCount; ++nNodeName )
	{
        sPath  = SUBLIST_TYPES          ;
        sPath += CFG_PATH_SEPERATOR     ;
        sPath += lNodeNames[nNodeName]  ;
        sPath += CFG_PATH_SEPERATOR     ;

        if( m_nVersion<=2  )
        {
            lPropertyNames[nProperty] = sPath + SUBKEY_PREFERRED        ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_UINAME           ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_MEDIATYPE        ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_CLIPBOARDFORMAT  ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_URLPATTERN       ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_EXTENSIONS       ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_DOCUMENTICONID   ;
            ++nProperty;
        }
        else if( m_nVersion>=3  )
        {
            lPropertyNames[nProperty] = sPath + SUBKEY_UINAME           ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_DATA             ;
            ++nProperty;
        }
	}

	// d)
    css::uno::Sequence< css::uno::Any > lPropertyValues = GetProperties( lPropertyNames );

	// Safe impossible cases.
	// We have asked for ALL our subtree keys and we would get all values.
	// It's important for next loop and our index using!
    LOG_ASSERT2( lPropertyNames.getLength()!=lPropertyValues.getLength(), "FilterCFGAccess::impl_loadTypes()", "Miss some configuration values of type set!" )

	// e)
    FileType                              aType                   ;
    css::uno::Sequence< ::rtl::OUString > lTempList               ;
    ::rtl::OUString                       sTempString             ;
                                          nProperty    = 0        ;
	for( nNodeName=0; nNodeName<nNodeCount; ++nNodeName )
	{
        if( m_nVersion<=2 )
        {
            lPropertyValues[nProperty]  >>= aType.bPreferred;
            ++nProperty;
            DataContainer::extractLocalizedStrings( rData.m_sLocale, lPropertyValues[nProperty], aType.lUINames );
            ++nProperty;
            lPropertyValues[nProperty]  >>= aType.sMediaType;
            ++nProperty;
            lPropertyValues[nProperty]  >>= aType.sClipboardFormat;
            ++nProperty;
            lPropertyValues[nProperty]  >>= lTempList;
            DataContainer::convertStringSequenceToVector( lTempList, aType.lURLPattern );
            ++nProperty;
            lPropertyValues[nProperty]  >>= lTempList;
            DataContainer::convertStringSequenceToVector( lTempList, aType.lExtensions );
            DataContainer::correctExtensions( aType.lExtensions );
            ++nProperty;
            lPropertyValues[nProperty]  >>= aType.nDocumentIconID;
            ++nProperty;
        }
        else if( m_nVersion>=3 )
        {
            DataContainer::extractLocalizedStrings( rData.m_sLocale, lPropertyValues[nProperty], aType.lUINames );
            ++nProperty;
            lPropertyValues[nProperty]  >>= sTempString;
            FilterCFGAccess::decodeTypeData( sTempString, aType );
            ++nProperty;
        }

        if( m_nVersion<=5 )
            aType.sName = lNodeNames[nNodeName];
        else
            aType.sName = ::utl::extractFirstFromConfigurationPath( lNodeNames[nNodeName] );
        rData.addType( aType, sal_False );
	}
}

//*****************************************************************************************************************
//	private method
//*****************************************************************************************************************
void FilterCFGAccess::impl_loadFilters( DataContainer& rData )
{
	//-------------------------------------------------------------------------------------------------------------
	// We must handle a dynamic set of keys!
    // structure version 1/2:       /Filters
    //                                  /<filtername_01>
    //                                      /Installed          [boolean    ]
    //                                      /Order              [integer    ] => only for version=2!
    //                                      /Type               [string     ]
    //                                      /UIName             [string     ]
    //                                      /DocumentService    [string     ]
    //                                      /FilterService      [string     ]
    //                                      /Flags              [integer    ]
    //                                      /UserData           [stringlist ]
    //                                      /FileFormatVersion  [integer    ]
    //                                      /TemplateName       [string     ]
    //                                  /<filtername_02>
    //                                      /...
    // structure version 3:         /Filters
    //                                  /<filtername_01> = old format encoded!
    //                                      /Installed          [boolean    ]
    //                                      /UIName             [string     ]
    //                                      /Data               [string     ] own format {Order,Type,DocumentService,FilterService,Flags,UserData, FileFormatVersion,TemplateName}
    //                                  /<filtername_02>
    //                                      /...
	//
	// a)	Get names of all current existing filter nodes in configuration.
	// b)	Reserve memory for list of all filter pathes - his names and properties!
	// c)	Fill list of all pathes.
	// d)	Get values for this list.
	// e)	Fill internal caches from these list.
	//-------------------------------------------------------------------------------------------------------------

    RTL_LOGFILE_CONTEXT( aMeasure, "framework (as96863) ::FilterCFGAccess::impl_loadFilters" );

	// a)
    css::uno::Sequence< ::rtl::OUString > lNodeNames;
    if( m_nVersion <= 5 )
        lNodeNames = GetNodeNames( SUBLIST_FILTERS, ::utl::CONFIG_NAME_LOCAL_NAME ); // without encoding!
    else
        lNodeNames = GetNodeNames( SUBLIST_FILTERS, ::utl::CONFIG_NAME_LOCAL_PATH ); // encoded!
	// b)
    sal_uInt32                              nNodeCount      = lNodeNames.getLength()         ;
    css::uno::Sequence< ::rtl::OUString >   lPropertyNames  ( nNodeCount*m_nKeyCountFilters );
    sal_uInt32                              nNodeName       = 0                              ;
    sal_uInt32                              nProperty       = 0                              ;
    ::rtl::OUString                         sPath                                            ;

    // c)
	for( nNodeName=0; nNodeName<nNodeCount; ++nNodeName )
	{
        sPath  = SUBLIST_FILTERS        ;
        sPath += CFG_PATH_SEPERATOR     ;
        sPath += lNodeNames[nNodeName]  ;
        sPath += CFG_PATH_SEPERATOR     ;

        if( m_nVersion<=2 )
        {
            lPropertyNames[nProperty] = sPath + SUBKEY_INSTALLED            ;
            ++nProperty;
            if( m_nVersion==2 )
            {
                lPropertyNames[nProperty] = sPath + SUBKEY_ORDER            ;
                ++nProperty;
            }
            lPropertyNames[nProperty] = sPath + SUBKEY_TYPE                 ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_UINAME               ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_DOCUMENTSERVICE      ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_FILTERSERVICE        ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_FLAGS                ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_USERDATA             ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_FILEFORMATVERSION    ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_TEMPLATENAME         ;
            ++nProperty;
        }
        else if( m_nVersion>=3 )
        {
            lPropertyNames[nProperty] = sPath + SUBKEY_INSTALLED            ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_UINAME               ;
            ++nProperty;
            lPropertyNames[nProperty] = sPath + SUBKEY_DATA                 ;
            ++nProperty;
        }
	}

	// d)
    css::uno::Sequence< css::uno::Any > lPropertyValues = GetProperties( lPropertyNames );

	// Safe impossible cases.
	// We have asked for ALL our subtree keys and we would get all values.
	// It's important for next loop and our index using!
    LOG_ASSERT2( lPropertyNames.getLength()!=lPropertyValues.getLength(), "FilterCFGAccess::impl_loadFilters()", "Miss some configuration values of filter set!" )

	// e)
    Filter                                 aFilter                 ;
    css::uno::Sequence< ::rtl::OUString >  lTempList               ;
    sal_Bool                               bInstalled              ;
    ::rtl::OUString                        sTempString             ;
                                           nProperty    = 0        ;
	for( nNodeName=0; nNodeName<nNodeCount; ++nNodeName )
	{
		sal_Bool bCheck = ( lPropertyValues[nProperty] >>= bInstalled);
        LOG_ASSERT2( bCheck==sal_False, "FilterCFGAccess::impl_loadFilters()", "Invalid property type for installed flag detected!" )
		++nProperty;

		// Ignore not installed filter!
        // Attention: Wrong packed "ANY"s are ignored too!
		if(
            ( bInstalled == sal_False ) ||
            ( bCheck     == sal_False )
          )
		{
			// If this filter was ignored .. we must step to next one in our lPropertyNames and lPropertyValues lists!
            // Because ... every name in list 1 match with a value in list 2.
            nProperty += m_nKeyCountFilters;
            // -1 ... because "Installed" already readed!
            --nProperty;
		}
		else
		{
			// Attention: If you change anything in this section ...
			// change "nProperty += ..." too! (see before!)
            // To find out, if somewhere forget this ... check right count in debug version!
            #ifdef ENABLE_ASSERTIONS
            sal_uInt32 nCheckPropCount = nProperty;
            #endif

            if( m_nVersion<=2 )
            {
                if( m_nVersion==2 )
                {
                    lPropertyValues[nProperty]  >>= aFilter.nOrder          ;
                    ++nProperty;
                }
                lPropertyValues[nProperty]  >>= aFilter.sType               ;
                ++nProperty;
                DataContainer::extractLocalizedStrings( rData.m_sLocale, lPropertyValues[nProperty], aFilter.lUINames );
                ++nProperty;
                lPropertyValues[nProperty]  >>= aFilter.sDocumentService    ;
                ++nProperty;
                lPropertyValues[nProperty]  >>= aFilter.sFilterService      ;
                ++nProperty;
                lPropertyValues[nProperty]  >>= aFilter.nFlags              ;
                ++nProperty;
                lPropertyValues[nProperty]  >>= lTempList                   ;
                DataContainer::convertStringSequenceToVector( lTempList, aFilter.lUserData );
                ++nProperty;
                lPropertyValues[nProperty]  >>= aFilter.nFileFormatVersion  ;
                ++nProperty;
                lPropertyValues[nProperty]  >>= aFilter.sTemplateName       ;
                ++nProperty;
            }
            else if( m_nVersion>=3 )
            {
                DataContainer::extractLocalizedStrings( rData.m_sLocale, lPropertyValues[nProperty], aFilter.lUINames );
                ++nProperty;
                lPropertyValues[nProperty]  >>= sTempString;
                FilterCFGAccess::decodeFilterData( sTempString, aFilter );
                ++nProperty;
            }

            LOG_ASSERT2( (sal_uInt32)(nProperty-nCheckPropCount)!=(sal_uInt32)(m_nKeyCountFilters-1), "FilterCFGAccess::impl_loadFilters()", "Mismatch between key count of filters and count of ignored filter properties detected. Follow readed filters would be wrong!" )

            if( m_nVersion<=5 )
                aFilter.sName = lNodeNames[nNodeName];
            else
                aFilter.sName = ::utl::extractFirstFromConfigurationPath( lNodeNames[nNodeName] );
            impl_setProductName( aFilter.lUINames );
            rData.addFilter( aFilter, sal_False );
		}
	}
}

//*****************************************************************************************************************
//	private method
//*****************************************************************************************************************
void FilterCFGAccess::impl_loadDetectors( DataContainer& rData )
{
	//-------------------------------------------------------------------------------------------------------------
	// We must handle a dynamic set of keys!
	// structure:		/DetectServices
	//						/<entry_01>
	//							/Types				[stringlist]
	//						/<entry_02>
	//							/...
	//
	// a)	Get names of all current existing detector nodes in configuration.
	// b)	Reserve memory for list of all detector pathes - his names and properties!
	// c)	Fill list of all pathes.
	// d)	Get values for this list.
	// e)	Fill internal caches from these list.
	//-------------------------------------------------------------------------------------------------------------

    RTL_LOGFILE_CONTEXT( aMeasure, "framework (as96863) ::FilterCFGAccess::impl_loadDetectors" );

	// a)
    css::uno::Sequence< ::rtl::OUString > lNodeNames;
    if( m_nVersion <= 5 )
        lNodeNames = GetNodeNames( SUBLIST_DETECTSERVICES, ::utl::CONFIG_NAME_LOCAL_NAME ); // without encoding!
    else
        lNodeNames = GetNodeNames( SUBLIST_DETECTSERVICES, ::utl::CONFIG_NAME_LOCAL_PATH ); // encoded!
	// b)
    sal_uInt32                              nNodeCount      = lNodeNames.getLength()           ;
    css::uno::Sequence< ::rtl::OUString >   lPropertyNames  ( nNodeCount*m_nKeyCountDetectors );
    sal_uInt32                              nNodeName       = 0                                ;
    sal_uInt32                              nProperty       = 0                                ;
    ::rtl::OUString                         sPath                                              ;

	// c)
	for( nNodeName=0; nNodeName<nNodeCount; ++nNodeName )
	{
        sPath  = SUBLIST_DETECTSERVICES ;
        sPath += CFG_PATH_SEPERATOR     ;
        sPath += lNodeNames[nNodeName]  ;
        sPath += CFG_PATH_SEPERATOR     ;

        lPropertyNames[nProperty] = sPath + SUBKEY_TYPES;
		++nProperty;
	}

	// d)
    css::uno::Sequence< css::uno::Any > lPropertyValues = GetProperties( lPropertyNames );

	// Safe impossible cases.
	// We have asked for ALL our subtree keys and we would get all values.
	// It's important for next loop and our index using!
	LOG_ASSERT2( lPropertyNames.getLength()!=lPropertyValues.getLength(), "DetectorCache::impl_loadDetectors()", "Miss some configuration values of detector set!" )

	// e)
    Detector                              aDetector     ;
    css::uno::Sequence< ::rtl::OUString > lTempList     ;
                                          nProperty = 0 ;
	for( nNodeName=0; nNodeName<nNodeCount; ++nNodeName )
	{
		lPropertyValues[nProperty]	>>=	lTempList;
        DataContainer::convertStringSequenceToVector( lTempList, aDetector.lTypes );
		++nProperty;
        if( m_nVersion<=5 )
            aDetector.sName = lNodeNames[nNodeName];
        else
            aDetector.sName = ::utl::extractFirstFromConfigurationPath( lNodeNames[nNodeName] );
        rData.addDetector( aDetector, sal_False );
	}
}

//*****************************************************************************************************************
//	private method
//*****************************************************************************************************************
void FilterCFGAccess::impl_loadLoaders( DataContainer& rData )
{
	//-------------------------------------------------------------------------------------------------------------
	// We must handle a dynamic set of keys!
    // structure:       /FrameLoaders
	//						/<entry_01>
	//							/UIName		[string		]
	//							/Types		[stringlist	]
	//						/<entry_02>
	//							/...
	//
	// a)	Get names of all current existing loader nodes in configuration.
	// b)	Reserve memory for list of all loader pathes - his names and properties!
	// c)	Fill list of all pathes.
	// d)	Get values for this list.
	// e)	Fill internal caches from these list.
	//-------------------------------------------------------------------------------------------------------------

    RTL_LOGFILE_CONTEXT( aMeasure, "framework (as96863) ::FilterCFGAccess::impl_loadLoaders" );

	// a)
    css::uno::Sequence< ::rtl::OUString > lNodeNames;
    if( m_nVersion <= 5 )
        lNodeNames = GetNodeNames( SUBLIST_FRAMELOADERS, ::utl::CONFIG_NAME_LOCAL_NAME ); // without encoding!
    else
        lNodeNames = GetNodeNames( SUBLIST_FRAMELOADERS, ::utl::CONFIG_NAME_LOCAL_PATH ); // encoded!
	// b)
    sal_uInt32                              nNodeCount      = lNodeNames.getLength()         ;
    css::uno::Sequence< ::rtl::OUString >   lPropertyNames  ( nNodeCount*m_nKeyCountLoaders );
    sal_uInt32                              nNodeName       = 0                              ;
    sal_uInt32                              nProperty       = 0                              ;
    ::rtl::OUString                         sPath                                            ;

	// c)
	for( nNodeName=0; nNodeName<nNodeCount; ++nNodeName )
	{
        sPath  = SUBLIST_FRAMELOADERS   ;
        sPath += CFG_PATH_SEPERATOR     ;
        sPath += lNodeNames[nNodeName]  ;
        sPath += CFG_PATH_SEPERATOR     ;

        lPropertyNames[nProperty] = sPath + SUBKEY_UINAME;
        ++nProperty;
        lPropertyNames[nProperty] = sPath + SUBKEY_TYPES ;
		++nProperty;
	}

	// d)
    css::uno::Sequence< css::uno::Any > lPropertyValues = GetProperties( lPropertyNames );

	// Safe impossible cases.
	// We have asked for ALL our subtree keys and we would get all values.
	// It's important for next loop and our index using!
	LOG_ASSERT2( lPropertyNames.getLength()!=lPropertyValues.getLength(), "LoaderCache::impl_loadLoaders()", "Miss some configuration values of loader set!" )

	// e)
    Loader                                aLoader        ;
    css::uno::Sequence< ::rtl::OUString > lTempList      ;
    ::rtl::OUString                       sUIName        ;
                                          nProperty  = 0 ;
	for( nNodeName=0; nNodeName<nNodeCount; ++nNodeName )
	{
        DataContainer::extractLocalizedStrings( rData.m_sLocale, lPropertyValues[nProperty], aLoader.lUINames );
		++nProperty;
		lPropertyValues[nProperty]	>>=	lTempList;
        DataContainer::convertStringSequenceToVector( lTempList, aLoader.lTypes );
		++nProperty;
        if( m_nVersion<=5 )
            aLoader.sName = lNodeNames[nNodeName];
        else
            aLoader.sName = ::utl::extractFirstFromConfigurationPath( lNodeNames[nNodeName] );
        rData.addLoader( aLoader, sal_False );
	}
}

//*****************************************************************************************************************
//	private method
//*****************************************************************************************************************
void FilterCFGAccess::impl_loadContentHandlers( DataContainer& rData )
{
	//-------------------------------------------------------------------------------------------------------------
	// We must handle a dynamic set of keys!
    // structure:       /ContentHandlers
	//						/<entry_01>
	//							/Types		[stringlist	]
	//						/<entry_02>
	//							/...
	//
    // a)   Get names of all current existing handler nodes in configuration.
    // b)   Reserve memory for list of all handler pathes - his names and properties!
	// c)	Fill list of all pathes.
	// d)	Get values for this list.
	// e)	Fill internal caches from these list.
	//-------------------------------------------------------------------------------------------------------------

    RTL_LOGFILE_CONTEXT( aMeasure, "framework (as96863) ::FilterCFGAccess::impl_loadContentHandlers" );

	// a)
    css::uno::Sequence< ::rtl::OUString > lNodeNames;
    if( m_nVersion <= 5 )
        lNodeNames = GetNodeNames( SUBLIST_CONTENTHANDLERS, ::utl::CONFIG_NAME_LOCAL_NAME ); // without encoding!
    else
        lNodeNames = GetNodeNames( SUBLIST_CONTENTHANDLERS, ::utl::CONFIG_NAME_LOCAL_PATH ); // encoded!
	// b)
    sal_uInt32                              nNodeCount      = lNodeNames.getLength()                 ;
    css::uno::Sequence< ::rtl::OUString >   lPropertyNames  ( nNodeCount*m_nKeyCountContentHandlers );
    sal_uInt32                              nNodeName       = 0                                      ;
    sal_uInt32                              nProperty       = 0                                      ;
    ::rtl::OUString                         sPath                                                    ;

	// c)
	for( nNodeName=0; nNodeName<nNodeCount; ++nNodeName )
	{
        sPath  = SUBLIST_CONTENTHANDLERS;
        sPath += CFG_PATH_SEPERATOR     ;
        sPath += lNodeNames[nNodeName]  ;
        sPath += CFG_PATH_SEPERATOR     ;

        lPropertyNames[nProperty] = sPath + SUBKEY_TYPES;
		++nProperty;
	}

	// d)
    css::uno::Sequence< css::uno::Any > lPropertyValues = GetProperties( lPropertyNames );

	// Safe impossible cases.
	// We have asked for ALL our subtree keys and we would get all values.
	// It's important for next loop and our index using!
    LOG_ASSERT2( lPropertyNames.getLength()!=lPropertyValues.getLength(), "HandlerCache::impl_loadContentHandlers()", "Miss some configuration values of content handler set!" )

	// e)
    ContentHandler                        aHandler       ;
    css::uno::Sequence< ::rtl::OUString > lTempList      ;
                                          nProperty  = 0 ;
	for( nNodeName=0; nNodeName<nNodeCount; ++nNodeName )
	{
		lPropertyValues[nProperty]	>>=	lTempList;
        DataContainer::convertStringSequenceToVector( lTempList, aHandler.lTypes );
		++nProperty;
        if( m_nVersion<=5 )
            aHandler.sName = lNodeNames[nNodeName];
        else
            aHandler.sName = ::utl::extractFirstFromConfigurationPath( lNodeNames[nNodeName] );
        rData.addContentHandler( aHandler, sal_False );
	}
}

//*****************************************************************************************************************
//	private method
//*****************************************************************************************************************
void FilterCFGAccess::impl_loadProtocolHandlers( DataContainer& rData )
{
	//-------------------------------------------------------------------------------------------------------------
	// We must handle a dynamic set of keys!
    // structure:       /ProtocolHandlers
	//						/<entry_01>
    //                          /Protocols      [stringlist ]
	//						/<entry_02>
	//							/...
	//
    // a)   Get names of all current existing handler nodes in configuration. (encoded!)
    // b)   Reserve memory for list of all handler pathes - his names and properties!
    // c)   Fill list of all pathes by building relative pathes.
    // d)   Get values for this list from configuration.
    // e)   Fill internal caches from these value list.
	//-------------------------------------------------------------------------------------------------------------

    // measure time of reading!
    RTL_LOGFILE_CONTEXT( aMeasure, "framework (as96863) ::FilterCFGAccess::impl_loadProtocolHandlers" );

	// a)
    css::uno::Sequence< ::rtl::OUString > lNodeNames;
    if( m_nVersion <= 5 )
        lNodeNames = GetNodeNames( SUBLIST_PROTOCOLHANDLERS, ::utl::CONFIG_NAME_LOCAL_NAME ); // without encoding!
    else
        lNodeNames = GetNodeNames( SUBLIST_PROTOCOLHANDLERS, ::utl::CONFIG_NAME_LOCAL_PATH ); // encoded!
	// b)
    sal_uInt32                              nNodeCount      = lNodeNames.getLength()                  ;
    css::uno::Sequence< ::rtl::OUString >   lPropertyNames  ( nNodeCount*m_nKeyCountProtocolHandlers );
    sal_uInt32                              nNodeName       = 0                                       ;
    sal_uInt32                              nProperty       = 0                                       ;
    ::rtl::OUString                         sPath                                                     ;

	// c)
	for( nNodeName=0; nNodeName<nNodeCount; ++nNodeName )
	{
        sPath  = SUBLIST_PROTOCOLHANDLERS   ;
        sPath += CFG_PATH_SEPERATOR         ;
        sPath += lNodeNames[nNodeName]      ;
        sPath += CFG_PATH_SEPERATOR         ;

        lPropertyNames[nProperty] = sPath + SUBKEY_PROTOCOLS;
		++nProperty;
	}

	// d)
    css::uno::Sequence< css::uno::Any > lPropertyValues = GetProperties( lPropertyNames );

	// Safe impossible cases.
	// We have asked for ALL our subtree keys and we would get all values.
	// It's important for next loop and our index using!
    LOG_ASSERT2( lPropertyNames.getLength()!=lPropertyValues.getLength(), "HandlerCache::impl_loadProtocolHandlers()", "Miss some configuration values of protocol handler set!" )

	// e)
    ProtocolHandler                       aHandler       ;
    css::uno::Sequence< ::rtl::OUString > lTempList      ;
                                          nProperty  = 0 ;
	for( nNodeName=0; nNodeName<nNodeCount; ++nNodeName )
	{
        lPropertyValues[nProperty] >>= lTempList;
        DataContainer::convertStringSequenceToVector( lTempList, aHandler.lProtocols );
		++nProperty;
        if( m_nVersion<=5 )
            aHandler.sName = lNodeNames[nNodeName];
        else
            aHandler.sName = ::utl::extractFirstFromConfigurationPath( lNodeNames[nNodeName] );
        rData.addProtocolHandler( aHandler, sal_False );
	}
}

//*****************************************************************************************************************
//	private method
//*****************************************************************************************************************
void FilterCFGAccess::impl_loadDefaults( DataContainer& rData )
{
	//-------------------------------------------------------------------------------------------------------------
	// We must handle follow structure!
	// structure:		/Defaults
	//						/DetectService
	//							/<name>
	//						/FrameLoader
	//							/<name>
	//
	// a)	Build list with pathes to configuration nodes.
	// b)	Get values for this list.
	// c)	Fill internal caches from these list.
	//-------------------------------------------------------------------------------------------------------------

    RTL_LOGFILE_CONTEXT( aMeasure, "framework (as96863) ::FilterCFGAccess::impl_loadDefaults" );

	// a)
    css::uno::Sequence< ::rtl::OUString > lPropertyNames( 2 );
    ::rtl::OUString                       sPath              ;

    sPath  = SUBLIST_DEFAULTS  ;
    sPath += CFG_PATH_SEPERATOR;

    lPropertyNames[0] = sPath + SUBKEY_DEFAULTDETECTOR;
    lPropertyNames[1] = sPath + SUBKEY_GENERICLOADER  ;

	// b)
    css::uno::Sequence< css::uno::Any > lPropertyValues = GetProperties( lPropertyNames );

	// Safe impossible cases.
	// We have asked for ALL our subtree keys and we would get all values.
	// It's important for next loop and our index using!
	LOG_ASSERT2( lPropertyNames.getLength()!=lPropertyValues.getLength(), "LoaderCache::impl_loadDefaults()", "Miss some configuration values of default set!" )

	// c)
    lPropertyValues[0]  >>= rData.m_aDefaultDetector.sName;
    lPropertyValues[1]  >>= rData.m_aGenericLoader.sName  ;

	// Follow properties are fix!
    DataContainer::setLocalelizedString( rData.m_aGenericLoader.lUINames, LOCALE_FALLBACK, UINAME_GENERICLOADER );
    rData.m_aGenericLoader.lTypes.push_back  ( TYPELIST_GENERICLOADER   );
    rData.m_aDefaultDetector.lTypes.push_back( TYPELIST_DEFAULTDETECTOR );
}

void FilterCFGAccess::impl_removeNodes(        StringList&        rChangesList  ,
                                         const ::rtl::OUString&   sTemplateType ,
                                         const ::rtl::OUString&   sSetName      )
{
    css::uno::Sequence< ::rtl::OUString > lRemovedElements;
    DataContainer::convertStringVectorToSequence( rChangesList, lRemovedElements );
    if( m_nVersion>=6 )
    {
        sal_Int32 nCount = lRemovedElements.getLength();
        for( sal_Int32 nItem=0; nItem<nCount; ++nItem )
        {
            lRemovedElements[nItem] = ::utl::wrapConfigurationElementName( lRemovedElements[nItem], sTemplateType );
        }
    }
    ClearNodeElements( sSetName, lRemovedElements );
    rChangesList.free();
}

//*****************************************************************************************************************
//	private method
//*****************************************************************************************************************
void FilterCFGAccess::impl_saveTypes( DataContainer& rData )
{
	//-------------------------------------------------------------------------------------------------------------
	// We must handle a dynamic set of keys!
    // structure: ... see impl_loadTypes() for further informations ...
	//
    // a)   Step over all cache entries which are set to MODIFIED!
	// b)	Build list of all properties of a type node.
	// c)	Set it on configuration.
	//-------------------------------------------------------------------------------------------------------------

    RTL_LOGFILE_CONTEXT( aMeasure, "framework (as96863) ::FilterCFGAccess::impl_saveTypes" );

    // Collect names of all removed elements and use list
    // to remove it realy.
    if( rData.m_aTypeCache.lRemovedItems.size() > 0 )
    {
        impl_removeNodes( rData.m_aTypeCache.lRemovedItems, TEMPLATENAME_TYPE, SUBLIST_TYPES );
    }

    // Collect names of all added AND changed elements (there is no difference in handling!)
    // ... build a list of properties for every set node
    // ... add values to this list
    // ... and write changes to configuration.
    // Use one list - don't handle every set node seperat!
    if  (
            ( rData.m_aTypeCache.lAddedItems.size  () > 0 ) ||
            ( rData.m_aTypeCache.lChangedItems.size() > 0 )
        )
    {
        css::uno::Sequence< ::rtl::OUString >   lTempList       ;
        ::rtl::OUString                         sBasePath       ;
        ::rtl::OUString                         sFullPath       ;
        sal_Int32                               nStep       = 0 ;

        sBasePath  = SUBLIST_TYPES       ;   //  "Types"
        sBasePath += CFG_PATH_SEPERATOR  ;   //  "Types/"

        // Reserve enough memory for complete list.
        // = count of added AND changed elements * count of properties for every set node
        sal_Int32 nSize;
        nSize  = rData.m_aTypeCache.lAddedItems.size()  ;
        nSize += rData.m_aTypeCache.lChangedItems.size();
        nSize *= m_nKeyCountTypes                                      ;
        css::uno::Sequence< css::beans::PropertyValue > lChangedElements( nSize );

        // Add "added" types to list.
        for( ConstStringListIterator pAddedType=rData.m_aTypeCache.lAddedItems.begin(); pAddedType!=rData.m_aTypeCache.lAddedItems.end(); ++pAddedType )
        {
            sFullPath = sBasePath;
            if( m_nVersion<=5 )
                sFullPath += *pAddedType; // without encoding
            else
                sFullPath += ::utl::wrapConfigurationElementName( *pAddedType, TEMPLATENAME_TYPE ); // encoded
            sFullPath += CFG_PATH_SEPERATOR;

            if( m_nVersion<=2 )
            {
                // "Types/<typename>/Preferred"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_PREFERRED                                                               ;
                lChangedElements[nStep].Value    <<= rData.m_aTypeCache[*pAddedType].bPreferred                                                 ;
                ++nStep;
                // "Types/<typename>/UIName"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_UINAME                                                                  ;
                DataContainer::packLocalizedStrings( rData.m_sLocale, lChangedElements[nStep].Value, rData.m_aTypeCache[*pAddedType].lUINames ) ;
                ++nStep;
                // "Types/<typename>/MediaType"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_MEDIATYPE                                                               ;
                lChangedElements[nStep].Value    <<= rData.m_aTypeCache[*pAddedType].sMediaType                                                 ;
                ++nStep;
                // "Types/<typename>/ClipboardFormat"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_CLIPBOARDFORMAT                                                         ;
                lChangedElements[nStep].Value    <<= rData.m_aTypeCache[*pAddedType].sClipboardFormat                                           ;
                ++nStep;
                // "Types/<typename>/URLPattern"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_URLPATTERN                                                              ;
                DataContainer::convertStringVectorToSequence( rData.m_aTypeCache[*pAddedType].lURLPattern, lTempList )                          ;
                lChangedElements[nStep].Value    <<= lTempList                                                                                  ;
                ++nStep;
                // "Types/<typename>/Extensions"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_EXTENSIONS                                                              ;
                DataContainer::convertStringVectorToSequence( rData.m_aTypeCache[*pAddedType].lExtensions, lTempList )                          ;
                lChangedElements[nStep].Value    <<= lTempList                                                                                  ;
                ++nStep;
                // "Types/<typename>/DocumentIconID"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_DOCUMENTICONID                                                          ;
                lChangedElements[nStep].Value    <<= rData.m_aTypeCache[*pAddedType].nDocumentIconID                                            ;
                ++nStep;
            }
            else if( m_nVersion>=3 )
            {
                // "Types/<typename>/UIName"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_UINAME                                                                  ;
                DataContainer::packLocalizedStrings( rData.m_sLocale, lChangedElements[nStep].Value, rData.m_aTypeCache[*pAddedType].lUINames ) ;
                // "Types/<typename>/Data"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_DATA                                                                    ;
                lChangedElements[nStep].Value    <<= FilterCFGAccess::encodeTypeData( rData.m_aTypeCache[*pAddedType] )                         ;
                ++nStep;
            }
        }

        // Attention: Don't change "nStep" here!

        // Add "changed" types to list.
        for( ConstStringListIterator pChangedType=rData.m_aTypeCache.lChangedItems.begin(); pChangedType!=rData.m_aTypeCache.lChangedItems.end(); ++pChangedType )
        {
            sFullPath = sBasePath;
            if( m_nVersion<=5 )
                sFullPath += *pChangedType;
            else
                sFullPath += ::utl::wrapConfigurationElementName( *pChangedType, TEMPLATENAME_TYPE );
            sFullPath += CFG_PATH_SEPERATOR;

            if( m_nVersion<=2 )
            {
                // "Types/<typename>/Preferred"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_PREFERRED                                                                   ;
                lChangedElements[nStep].Value    <<= rData.m_aTypeCache[*pChangedType].bPreferred                                                   ;
                ++nStep;
                // "Types/<typename>/UIName"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_UINAME                                                                      ;
                DataContainer::packLocalizedStrings( rData.m_sLocale, lChangedElements[nStep].Value, rData.m_aTypeCache[*pChangedType].lUINames )   ;
                ++nStep;
                // "Types/<typename>/MediaType"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_MEDIATYPE                                                                   ;
                lChangedElements[nStep].Value    <<= rData.m_aTypeCache[*pChangedType].sMediaType                                                   ;
                ++nStep;
                // "Types/<typename>/ClipboardFormat"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_CLIPBOARDFORMAT                                                             ;
                lChangedElements[nStep].Value    <<= rData.m_aTypeCache[*pChangedType].sClipboardFormat                                             ;
                ++nStep;
                // "Types/<typename>/URLPattern"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_URLPATTERN                                                                  ;
                DataContainer::convertStringVectorToSequence( rData.m_aTypeCache[*pChangedType].lURLPattern, lTempList )                            ;
                lChangedElements[nStep].Value    <<= lTempList                                                                                      ;
                ++nStep;
                // "Types/<typename>/Extensions"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_EXTENSIONS                                                                  ;
                DataContainer::convertStringVectorToSequence( rData.m_aTypeCache[*pChangedType].lExtensions, lTempList )                            ;
                lChangedElements[nStep].Value    <<= lTempList                                                                                      ;
                ++nStep;
                // "Types/<typename>/DocumentIconID"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_DOCUMENTICONID                                                              ;
                lChangedElements[nStep].Value    <<= rData.m_aTypeCache[*pChangedType].nDocumentIconID                                              ;
                ++nStep;
            }
            else if( m_nVersion>=3 )
            {
                // "Types/<typename>/UIName"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_UINAME                                                                      ;
                DataContainer::packLocalizedStrings( rData.m_sLocale, lChangedElements[nStep].Value, rData.m_aTypeCache[*pChangedType].lUINames )   ;
                ++nStep;
                // "Types/<typename>/Data"
                lChangedElements[nStep].Name     =   sFullPath + SUBKEY_DATA                                                                        ;
                lChangedElements[nStep].Value    <<= FilterCFGAccess::encodeTypeData( rData.m_aTypeCache[*pChangedType] )                           ;
                ++nStep;
            }
        }

        // Commit complete list of changes.
        SetSetProperties( SUBLIST_TYPES, lChangedElements );

        // Don't forget to free "changes lists". Otherwise these changes will bewritten every time ...
        rData.m_aTypeCache.lAddedItems.free();
        rData.m_aTypeCache.lChangedItems.free();
	}
}

//*****************************************************************************************************************
void FilterCFGAccess::impl_saveFilters( DataContainer& rData )
{
	//-------------------------------------------------------------------------------------------------------------
	// We must handle a dynamic set of keys!
    // structure: ... see impl_loadFilters() for further informations ...
	//
	// a)	Step over all cache entries.
	// b)	Build list of all properties of a filter node.
	// c)	Set it on configuration.
	//-------------------------------------------------------------------------------------------------------------

    RTL_LOGFILE_CONTEXT( aMeasure, "framework (as96863) ::FilterCFGAccess::impl_saveFilters" );

    // Collect names of all removed elements and use list
    // to remove it realy.
    if( rData.m_aFilterCache.lRemovedItems.size() > 0 )
    {
        impl_removeNodes( rData.m_aFilterCache.lRemovedItems, TEMPLATENAME_FILTER, SUBLIST_FILTERS );
    }

    // Collect names of all added AND changed elements (there is no difference in handling!)
    // ... build a list of properties for every set node
    // ... add values to this list
    // ... and write changes to configuration.
    // Use one list - don't handle every set node seperat!
    if  (
            ( rData.m_aFilterCache.lAddedItems.size  () > 0 ) ||
            ( rData.m_aFilterCache.lChangedItems.size() > 0 )
        )
    {
        css::uno::Sequence< ::rtl::OUString >   lTempList       ;
        ::rtl::OUString                         sBasePath       ;
        ::rtl::OUString                         sFullPath       ;
        sal_Int32                               nStep       = 0 ;

        sBasePath  = SUBLIST_FILTERS    ;   //  "Filters"
        sBasePath += CFG_PATH_SEPERATOR ;   //  "Filters/"

        // Reserve enough memory for complete list.
        // = count of added AND changed elements * count of properties for every set node
        sal_Int32 nSize;
        nSize  = rData.m_aFilterCache.lAddedItems.size()  ;
        nSize += rData.m_aFilterCache.lChangedItems.size();
        nSize *= m_nKeyCountFilters                       ;
        css::uno::Sequence< css::beans::PropertyValue > lChangedElements( nSize );

        // Add "added" filters to list.
        for( ConstStringListIterator pAddedFilter=rData.m_aFilterCache.lAddedItems.begin(); pAddedFilter!=rData.m_aFilterCache.lAddedItems.end(); ++pAddedFilter )
        {
            sFullPath = sBasePath;
            if( m_nVersion<=5 )
                sFullPath += *pAddedFilter;
            else
                sFullPath += ::utl::wrapConfigurationElementName( *pAddedFilter, TEMPLATENAME_FILTER );
            sFullPath += CFG_PATH_SEPERATOR;

            StringHash lUINames = rData.m_aFilterCache[*pAddedFilter].lUINames;
            impl_resetProductName( lUINames );

            if( m_nVersion<=2 )
            {
                // "Filters/<filtername>/Installed"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_INSTALLED                                                               ;
                lChangedElements[nStep].Value   <<= sal_True                                                                                   ;
                ++nStep;
                // "Filters/<filtername>/Order"
                if( m_nVersion==2 )
                {
                    lChangedElements[nStep].Name      = sFullPath + SUBKEY_ORDER                                                               ;
                    lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pAddedFilter].nOrder                                             ;
                    ++nStep;
                }
                // "Filters/<filtername>/Type"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_TYPE                                                                    ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pAddedFilter].sType                                                  ;
                ++nStep;
                // "Filters/<filtername>/UIName"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_UINAME                                                                  ;
                DataContainer::packLocalizedStrings( rData.m_sLocale, lChangedElements[nStep].Value, lUINames );
                ++nStep;
                // "Filters/<filtername>/DocumentService"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_DOCUMENTSERVICE                                                         ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pAddedFilter].sDocumentService                                       ;
                ++nStep;
                // "Filters/<filtername>/FilterService"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_FILTERSERVICE                                                           ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pAddedFilter].sFilterService                                         ;
                ++nStep;
                // "Filters/<filtername>/Flags"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_FLAGS                                                                   ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pAddedFilter].nFlags                                                 ;
                ++nStep;
                // "Filters/<filtername>/UserData"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_USERDATA                                                                ;
                DataContainer::convertStringVectorToSequence( rData.m_aFilterCache[*pAddedFilter].lUserData, lTempList )                       ;
                lChangedElements[nStep].Value   <<= lTempList                                                                                  ;
                ++nStep;
                // "Filters/<filtername>/FileFormatVersion"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_FILEFORMATVERSION                                                       ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pAddedFilter].nFileFormatVersion                                     ;
                ++nStep;
                // "Filters/<filtername>/TemplateName"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_TEMPLATENAME                                                            ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pAddedFilter].sTemplateName                                          ;
                ++nStep;
            }
            else if( m_nVersion>=3 )
            {
                // "Filters/<filtername>/Installed"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_INSTALLED                                                               ;
                lChangedElements[nStep].Value   <<= sal_True                                                                                   ;
                ++nStep;
                // "Filters/<filtername>/UIName"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_UINAME                                                                  ;
                DataContainer::packLocalizedStrings( rData.m_sLocale, lChangedElements[nStep].Value, lUINames );
                ++nStep;
                // "Filters/<filtername>/Data"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_DATA                                                                    ;
                lChangedElements[nStep].Value   <<= FilterCFGAccess::encodeFilterData( rData.m_aFilterCache[*pAddedFilter] )                   ;
                ++nStep;
            }
        }

        // Attention: Don't change "nStep" here!

        // Add "changed" filters to list.
        for( ConstStringListIterator pChangedFilter=rData.m_aFilterCache.lChangedItems.begin(); pChangedFilter!=rData.m_aFilterCache.lChangedItems.end(); ++pChangedFilter )
        {
            sFullPath  = sBasePath;
            if( m_nVersion<=5 )
                sFullPath += *pChangedFilter;
            else
                sFullPath += ::utl::wrapConfigurationElementName( *pChangedFilter, TEMPLATENAME_FILTER );
            sFullPath += CFG_PATH_SEPERATOR;

            StringHash lUINames = rData.m_aFilterCache[*pChangedFilter].lUINames;
            impl_resetProductName( lUINames );

            if( m_nVersion<=2 )
            {
                // "Filters/<filtername>/Installed"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_INSTALLED                                                                   ;
                lChangedElements[nStep].Value   <<= sal_True                                                                                       ;
                ++nStep;
                // "Filters/<filtername>/Order"
                if( m_nVersion==2 )
                {
                    lChangedElements[nStep].Name      = sFullPath + SUBKEY_ORDER                                                                   ;
                    lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pChangedFilter].nOrder                                               ;
                    ++nStep;
                }
                // "Filters/<filtername>/Type"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_TYPE                                                                        ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pChangedFilter].sType                                                    ;
                ++nStep;
                // "Filters/<filtername>/UIName"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_UINAME                                                                      ;
                DataContainer::packLocalizedStrings( rData.m_sLocale, lChangedElements[nStep].Value, lUINames )  ;
                ++nStep;
                // "Filters/<filtername>/DocumentService"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_DOCUMENTSERVICE                                                             ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pChangedFilter].sDocumentService                                         ;
                ++nStep;
                // "Filters/<filtername>/FilterService"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_FILTERSERVICE                                                               ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pChangedFilter].sFilterService                                           ;
                ++nStep;
                // "Filters/<filtername>/Flags"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_FLAGS                                                                       ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pChangedFilter].nFlags                                                   ;
                ++nStep;
                // "Filters/<filtername>/UserData"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_USERDATA                                                                    ;
                DataContainer::convertStringVectorToSequence( rData.m_aFilterCache[*pChangedFilter].lUserData, lTempList )                         ;
                lChangedElements[nStep].Value   <<= lTempList                                                                                      ;
                ++nStep;
                // "Filters/<filtername>/FileFormatVersion"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_FILEFORMATVERSION                                                           ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pChangedFilter].nFileFormatVersion                                       ;
                ++nStep;
                // "Filters/<filtername>/TemplateName"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_TEMPLATENAME                                                                ;
                lChangedElements[nStep].Value   <<= rData.m_aFilterCache[*pChangedFilter].sTemplateName                                            ;
                ++nStep;
            }
            else if( m_nVersion>=3 )
            {
                // "Filters/<filtername>/Installed"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_INSTALLED                                                                   ;
                lChangedElements[nStep].Value   <<= sal_True                                                                                       ;
                ++nStep;
                // "Filters/<filtername>/UIName"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_UINAME                                                                      ;
                DataContainer::packLocalizedStrings( rData.m_sLocale, lChangedElements[nStep].Value, lUINames )  ;
                ++nStep;
                // "Filters/<filtername>/Data"
                lChangedElements[nStep].Name      = sFullPath + SUBKEY_DATA                                                                        ;
                lChangedElements[nStep].Value   <<= FilterCFGAccess::encodeFilterData( rData.m_aFilterCache[*pChangedFilter] )                     ;
                ++nStep;
            }
        }

        // Commit complete list of changes.
        SetSetProperties( SUBLIST_FILTERS, lChangedElements );

        // Don't forget to free "changes lists". Otherwise these changes will bewritten every time ...
        rData.m_aFilterCache.lAddedItems.free();
        rData.m_aFilterCache.lChangedItems.free();
	}
}

//*****************************************************************************************************************
void FilterCFGAccess::impl_saveDetectors( DataContainer& rData )
{
	//-------------------------------------------------------------------------------------------------------------
	// We must handle a dynamic set of keys!
	// structure:		/DetectServices
	//						/<entry_01>
	//							/Types				[stringlist]
	//						/<entry_02>
	//							/...
	//
	// a)	Step over all cache entries.
	// b)	Build list of all properties of a detector node.
	// c)	Set it on configuration.
	//-------------------------------------------------------------------------------------------------------------

    LOG_WARNING( "FilterCFGAccess::impl_saveDetectors()", "Not implemented yet!" )
}

//*****************************************************************************************************************
void FilterCFGAccess::impl_saveLoaders( DataContainer& rData )
{
	//-------------------------------------------------------------------------------------------------------------
	// We must handle a dynamic set of keys!
	// structure:		/FrameLoader
	//						/<entry_01>
	//							/UIName		[string		]
	//							/Types		[stringlist	]
	//						/<entry_02>
	//							/...
	//
	// a)	Step over all cache entries.
	// b)	Build list of all properties of a loader node.
	// c)	Set it on configuration.
	//-------------------------------------------------------------------------------------------------------------

    LOG_WARNING( "FilterCFGAccess::impl_saveLoaders()", "Not implemented yet!" )
}

//*****************************************************************************************************************
void FilterCFGAccess::impl_saveContentHandlers( DataContainer& rData )
{
	//-------------------------------------------------------------------------------------------------------------
	// We must handle a dynamic set of keys!
    // structure:       /ContentHandler
	//						/<entry_01>
	//							/Types		[stringlist	]
	//						/<entry_02>
	//							/...
	//
	// a)	Step over all cache entries.
    // b)   Build list of all properties of a handler node.
	// c)	Set it on configuration.
	//-------------------------------------------------------------------------------------------------------------

    LOG_WARNING( "FilterCFGAccess::impl_saveContentHandlers()", "Not implemented yet!" )
}

//*****************************************************************************************************************
//  private method
//      (1) replace %PRODUCTNAME% with right value from configuration
//      (2) by the way: if product name is "openoffice.org" it tries to find "6.0" and replace it with "1.0"
//*****************************************************************************************************************
void FilterCFGAccess::impl_setProductName( StringHash& lValues )
{
    sal_Bool         bPatchCandidate = sal_False; // disable patchcode!
    ::rtl::OUString* pValue          = NULL;
    for( StringHashIterator pItem=lValues.begin(); pItem!=lValues.end(); ++pItem )
    {
        pValue = &(pItem->second);
        sal_Int32 nIndex = pValue->indexOf( PRODUCTNAME_VARIABLE );
        // some filters include productname more then ones!
        while( nIndex != -1 )
        {
            bPatchCandidate=m_bActivateOpenofficePatch; // enable it for existing "productname" AND value of that="openoffice.org" only!
            *pValue = pValue->replaceAt( nIndex, PRODUCTNAME_VARLENGTH, m_sProductName );
            nIndex  = pValue->indexOf( PRODUCTNAME_VARIABLE, nIndex );
        }
        // special patch for openoffice
        // If filter uiname includes the productname - it must include "6.0" too
        // Replace it with "1.0"!
        // For better performance we indicates this mode one times only during getting the product name initialy.
        if(bPatchCandidate==sal_True)
        {
            nIndex = pValue->indexOf( PRODUCTPATCH_ORGVERSION );
            while( nIndex != -1 )
            {
                *pValue = pValue->replaceAt( nIndex, PRODUCTPATCH_VARLENGTH, PRODUCTPATCH_NEWVERSION );
                nIndex = pValue->indexOf( PRODUCTPATCH_ORGVERSION, nIndex );
            }
        }
    }
}

//*****************************************************************************************************************
//  private method
//*****************************************************************************************************************
void FilterCFGAccess::impl_resetProductName( StringHash& lValues )
{
    /**
     * temp. disabled till another solution exist for that.
     * We change filter ui names at loading time and save this changes ones
     * back to the configuration. Next startup willn't force such replacements ...
     */

    /**
    ::rtl::OUString* pValue  = NULL;
    sal_Int32        nLength = m_sProductName.getLength();
    for( StringHashIterator pItem=lValues.begin(); pItem!=lValues.end(); ++pItem )
    {
        pValue = &(pItem->second);
        sal_Int32 nIndex = pValue->indexOf( m_sProductName );
        if( nIndex != -1 )
        {
            *pValue = pValue->replaceAt( nIndex, nLength, PRODUCTNAME_VARIABLE );
            if(m_bActivateOpenofficePatch)
            {
                nIndex = pValue->indexOf( PRODUCTPATCH_NEWVERSION );
                if( nIndex != -1 )
                {
                    *pValue = pValue->replaceAt( nIndex, PRODUCTPATCH_VARLENGTH, PRODUCTPATCH_ORGVERSION );
                }
            }
        }
    }
    */
}

/*-************************************************************************************************************//**
    @short      filter all "*." from an extension
    @descr      We search for extensions without wildcards at runtime!
                Our configuration contains sometimes extensions WITH wildcards ..
                We should correct it at runtime.
                Algorithm:
                    - Step over all strings in list - search for "*.<extension>" and convert it to "<extension>".
                    - If first signs are not "*." ignore item and do nothing!
                    - But make all extensions lowercase too!

    @seealso    search methods of class FilterCFGAccess

    @param      "lExtensions", list of extensions to correct
    @return     -

    @onerror    No error should occure.
*//*-*************************************************************************************************************/
void DataContainer::correctExtensions( StringList& lExtensions )
{
	sal_Int32 nPosition;
	for( StringListIterator pExtension=lExtensions.begin(); pExtension!=lExtensions.end(); ++pExtension )
	{
		nPosition = pExtension->indexOf( DECLARE_ASCII("*.") );
		if( nPosition != -1 )
		{
			nPosition += 2;
			*pExtension = pExtension->copy( nPosition, pExtension->getLength()-nPosition );
		}
        *pExtension = pExtension->toAsciiLowerCase();
	}
}

/*-************************************************************************************************************//**
    @short      extract localized strings from given configuration item or reverse
    @descr      It exist to methods to get a localized value from our configuration.
                a) cfg returns one value for current set locale
                b) cfg returns a list of values for all possible locales
                These method detect it and fills our internal "localelized" hash automaticly or
                convert internal structures to external ones.

    @seealso    baseclass ConfigItem
    @seealso    method impl_setLocalelizedString()
    @seealso    method impl_getLocalelizedString()

    @param      "pLocale"   , is used by a) only to insert value at right hash position
    @param      "aCFGValue" , contains a) [OUString] ... or b) [Sequence< PropertyValue >]
    @param      "lLocales"  , internal description of localized values!
    @return     List of sorted locales with values.

    @onerror    No error should occure.
*//*-*************************************************************************************************************/

//*****************************************************************************************************************
void DataContainer::extractLocalizedStrings( const ::rtl::OUString& sCurrentLocale, const css::uno::Any& aCFGValue, StringHash& lLocales )
{
	// Algorithm:
	//	a	)	free current hash (we return an empty hash if method failed!)
	//	b	)	detect type of Any-value
	//	c.1	)	if( type = string )
	//				insert it for right locale "pCurrentLocale" in hash
	//	c.2)	or if( type = seq< prop > )
	//				copy sequence to hash (ignore "pCurrentLocale"!)

	// a)
	lLocales.free();
	// b)
    css::uno::Type aType = aCFGValue.getValueType();
	// c.1)
    if( aType == ::getCppuType( (const ::rtl::OUString*)NULL ) )
	{
        ::rtl::OUString sValue;
		aCFGValue >>= sValue;
        setLocalelizedString( lLocales, sCurrentLocale, sValue );
	}
	else
	// c.2)
    if( aType == ::getCppuType( (const css::uno::Sequence< css::beans::PropertyValue >*)NULL ) )
	{
        css::uno::Sequence< css::beans::PropertyValue > lLocaleValues;
		aCFGValue >>= lLocaleValues;
        convertSequenceToStringHash( lLocaleValues, lLocales );
	}
	#ifdef ENABLE_ASSERTIONS
	else
	{
        LOG_ERROR( "FilterCFGAccess::impl_extractLocalizedStrings()", "Unknown cfg-type detected! Ignore item ..." )
	}
	#endif
}

//*****************************************************************************************************************
void DataContainer::packLocalizedStrings( const ::rtl::OUString& sCurrentLocale, css::uno::Any& aCFGValue, const StringHash& lLocales )
{
	// Our cache works in "ALL LOCALES" mode only!
	// We can ignore "pCurrentLocale" here ...
	// but if someone change that ... he should have a look at this method too!
    css::uno::Sequence< css::beans::PropertyValue > lLocaleValues;
    convertStringHashToSequence( lLocales, lLocaleValues );
	aCFGValue <<= lLocaleValues;
}

/*-************************************************************************************************************//**
    @short      return right value from list of all localized ones
    @descr      We hold a list of all e.g. UINames for every set item.
                If user whish to get current UIName for current set locale ... you can call
                this method to get this information.

    @attention  We have optimized our TypeDetection.xml ... So we don't save all localized values, if
                there nothing exist, which should be loclialized (e.g. "Writer 6.0")!
                In this case we must implement a fallback if no value exist for current set locale.
                Our fallback is "en-US" everytime - because only this locale exit in xml file!

    @seealso    method impl_extractLocalizedStrings()

    @param      "lLocales"  , list of all localized values
    @param      "sLocale"   , current set locale
    @param      "sValue"    , new localized value to set in list for current locale
    @return     Matching UIName for current locale from list.

    @onerror    No error should occure.
*//*-*************************************************************************************************************/

//*****************************************************************************************************************
::rtl::OUString DataContainer::getLocalelizedString( const StringHash& lLocales, const ::rtl::OUString& sLocale )
{
    ::rtl::OUString         sValue                             ;
    ConstStringHashIterator pItem   = lLocales.find( sLocale ) ;
	if( pItem != lLocales.end() )
	{
		sValue = pItem->second;
	}
    else
    {
        pItem = lLocales.find( LOCALE_FALLBACK );
        if( pItem != lLocales.end() )
        {
    		sValue = pItem->second;
        }
    }
    LOG_ASSERT2( sValue.getLength()<1, "DataContainer::getLocalizedString()", "Couldn't find any value for this locale!" )
	return sValue;
}

//*****************************************************************************************************************
void DataContainer::setLocalelizedString( StringHash& lLocales, const ::rtl::OUString& sLocale, const ::rtl::OUString& sValue )
{
    // Follow case can occure:
    //  a) given locale is the fallback one                                                                         => we can set this value directly
    //  b) given locale is different from fallback & but fallback not exist in list                                 => set new value for given locale ... but don't touch add it as fallback
    //                                                                                                                 May be we override it at pushing item into our internal structures.
    //                                                                                                                 e.g.: i  ) creation of empty filter item
    //                                                                                                                       ii ) copy properties set by user to this item
    //                                                                                                                       iii) set item at internal structures
    //                                                                                                                       At ii) no fallback value exist ... but if we set it we must merge data in step iii)!
    //  c) given locale is different from fallback & fallback exist & new value is different from fallback value    => set value for given locale only
    //  d) given locale is different from fallback & fallback exist & new value is same like fallback value         => do nothing! use fallback

    // a)
    if( sLocale == LOCALE_FALLBACK )
    {
        lLocales[sLocale] = sValue;
    }
    else
    {
        ConstStringHashIterator pFallback = lLocales.find( LOCALE_FALLBACK );
        // b)
        if( pFallback == lLocales.end() )
        {
            lLocales[sLocale] = sValue;
        }
        else
        {
            // c)
            if( pFallback->second != sValue )
            {
                lLocales[sLocale] = sValue;
            }
            // d!)
        }
    }
}

/*-************************************************************************************************************//**
    @short      clear complete cache REALY
    @descr      STL don't free memory if you call erase() or clear()! The only way to do it realy, is to
                swap an empty container into this one :-( We use special free methods of our container structures
                to do that.

    @seealso    stl

    @param      -
    @return     -

    @onerror    -
*//*-*************************************************************************************************************/
void DataContainer::free()
{
    m_aTypeCache.free                 ();
    m_aFilterCache.free               ();
    m_aDetectorCache.free             ();
    m_aLoaderCache.free               ();
    m_aContentHandlerCache.free       ();
    m_aProtocolHandlerCache.free      ();

    m_aFastFilterCache.free           ();
    m_aFastDetectorCache.free         ();
    m_aFastLoaderCache.free           ();
    m_aFastContentHandlerCache.free   ();
    m_aFastProtocolHandlerCache.free  ();

    m_aPreferredTypesCache.free       ();

    m_aDefaultDetector.free           ();
    m_aGenericLoader.free             ();

    m_sLocale = ::rtl::OUString();
}

/*-************************************************************************************************************//**
    @short      add elements from given container to this one
    @descr      Our filter configuration is splitted into two parts - standard and additional ones.
                We read it at different time; but use it in one container. That's why we must merge
                different versions!

    @attention  It's not possible to merge default detector and loader! This two entries could exist one times only.
                "sLocale" exist one times too.

    @seealso    using in class FilterCFGAccess

    @param      "rData", reference to another container to add his values to this one
    @return     -

    @onerror    -
*//*-*************************************************************************************************************/
void DataContainer::mergeData( const DataContainer& rData )
{
    /*
    ::std::merge( aTypeCache.begin()        , aTypeCache.end()        , rData.m_aTypeCache.begin()        , rData.m_aTypeCache.end()        , aTypeCache         );
    ::std::merge( aFilterCache.begin()      , aFilterCache.end()      , rData.m_aFilterCache.begin()      , rData.m_aFilterCache.end()      , aFilterCache       );
    ::std::merge( aDetectorCache.begin()    , aDetectorCache.end()    , rData.m_aDetectorCache.begin()    , rData.m_aDetectorCache.end()    , aDetectorCache     );
    ::std::merge( aLoaderCache.begin()      , aLoaderCache.end()      , rData.m_aLoaderCache.begin()      , rData.m_aLoaderCache.end()      , aLoaderCache       );
    ::std::merge( aFastFilterCache.begin()  , aFastFilterCache.end()  , rData.m_aFastFilterCache.begin()  , rData.m_aFastFilterCache.end()  , aFastFilterCache   );
    ::std::merge( aFastDetectorCache.begin(), aFastDetectorCache.end(), rData.m_aFastDetectorCache.begin(), rData.m_aFastDetectorCache.end(), aFastDetectorCache );
    ::std::merge( aFastLoaderCache.begin()  , aFastLoaderCache.end()  , rData.m_aFastLoaderCache.begin()  , rData.m_aFastLoaderCache.end()  , aFastLoaderCache   );
    */
    LOG_ERROR( "DataContainer::mergeData()", "Not implemented yet!" )
}

/*-************************************************************************************************************//**
    @short      register new config item in data container
    @descr      This should be used to add a new item to this data container.

    @attention  We differ between impl and normal methods. Normal versions set modified flag additional to impl ones.
                This enable flushing of changed data to configuration. Impl operations could be used to fill
                cache too.

    @seealso    method replace...()
    @seealso    method remove...()

    @param      "aType, aFilter ...", information about new item to set it on internal structures
    @param      "bSetModified"      , regulate registration of new item as realy new element, which should be flushed to file!
    @return     -

    @onerror    Item will be ignored.
*//*-*************************************************************************************************************/

//*****************************************************************************************************************
void DataContainer::addType( const FileType& aType, sal_Bool bSetModified )
{
    // Insert entry in normal type cache.
    m_aTypeCache[ aType.sName ] = aType;
	// If this is a preferred type ...
	// add it to special hash too!
	// Register him for all supported extensions.
	if( aType.bPreferred == sal_True )
	{
        for (   ConstStringListIterator pExtension=m_aTypeCache[ aType.sName ].lExtensions.begin()  ;
                pExtension!=m_aTypeCache[ aType.sName ].lExtensions.end()                           ;
				++pExtension
			)
		{
            m_aPreferredTypesCache[ *pExtension ] = aType.sName;
		}
	}
    // If modification allowed ... add it to list and set flag!
    if( bSetModified == sal_True )
    {
        m_aTypeCache.appendChange( aType.sName, E_ADDED );
        m_bIsModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::addFilter( const Filter& aFilter, sal_Bool bSetModified )
{
    #ifdef ENABLE_WARNINGS
    ConstFilterIterator pFilter = m_aFilterCache.find( aFilter.sName );
    if( pFilter != m_aFilterCache.end() )
    {
        ::rtl::OUStringBuffer _sDoubleFilterMsg( 256 );
        _sDoubleFilterMsg.appendAscii( "Adding of double filter detected! Name=\"" );
        _sDoubleFilterMsg.append     ( aFilter.sName                               );
        _sDoubleFilterMsg.appendAscii( "\"\n"                                      );
        LOG_WARNING( "DataContainer::addFilter()", U2B(_sDoubleFilterMsg.makeStringAndClear()) )
    }
    #endif

	// Do it in our normal filter cache ...
    m_aFilterCache[ aFilter.sName ] = aFilter;
	// ... but don't forget it in our faster performance cache!
    m_aFastFilterCache[ aFilter.sType ].push_back( aFilter.sName );
    // If modification allowed ... add it to list and set flag!
    if( bSetModified == sal_True )
    {
        m_aFilterCache.appendChange( aFilter.sName, E_ADDED );
        m_bIsModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::addDetector( const Detector& aDetector, sal_Bool bSetModified )
{
	// Do it for our normal cache first ...
    m_aDetectorCache[ aDetector.sName ] = aDetector;
	// But don*t forget to add every detector item to our faster performance cache for type-detector assigning too!
	sal_uInt32 nLength = (sal_uInt32)(aDetector.lTypes.size());
	for( sal_uInt32 nType=0; nType<nLength; ++nType )
	{
        m_aFastDetectorCache[ aDetector.lTypes[nType] ].push_back( aDetector.sName );
	}
    // If modification allowed ... add it to list and set flag!
    if( bSetModified == sal_True )
    {
        m_aDetectorCache.appendChange( aDetector.sName, E_ADDED );
        m_bIsModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::addLoader( const Loader& aLoader, sal_Bool bSetModified )
{
	// Put entry in stl-container.
    m_aLoaderCache[ aLoader.sName ] = aLoader;
	// Add every loader item to our faster performance cache for type-loader assigning too!
	sal_uInt32 nLength = (sal_uInt32)(aLoader.lTypes.size());
	for( sal_uInt32 nType=0; nType<nLength; ++nType )
	{
        m_aFastLoaderCache[ aLoader.lTypes[nType] ].push_back( aLoader.sName );
	}
    // If modification allowed ... add it to list and set flag!
    if( bSetModified == sal_True )
    {
        m_aLoaderCache.appendChange( aLoader.sName, E_ADDED );
        m_bIsModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::addContentHandler( const ContentHandler& aHandler, sal_Bool bSetModified )
{
	// Put entry in stl-container.
    m_aContentHandlerCache[ aHandler.sName ] = aHandler;
    // Add every handler item to our faster performance cache for type-Handler assignment too!
    sal_uInt32 nLength = (sal_uInt32)(aHandler.lTypes.size());
	for( sal_uInt32 nType=0; nType<nLength; ++nType )
	{
        m_aFastContentHandlerCache[ aHandler.lTypes[nType] ].push_back( aHandler.sName );
	}
    // If modification allowed ... add it to list and set flag!
    if( bSetModified == sal_True )
    {
        m_aContentHandlerCache.appendChange( aHandler.sName, E_ADDED );
        m_bIsModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::addProtocolHandler( const ProtocolHandler& aHandler, sal_Bool bSetModified )
{
	// Put entry in stl-container.
    m_aProtocolHandlerCache[ aHandler.sName ] = aHandler;
    // Add every handler item to our faster performance cache for protocl pattern assignment too!
    sal_uInt32 nLength = (sal_uInt32)(aHandler.lProtocols.size());
    for( sal_uInt32 nProtocol=0; nProtocol<nLength; ++nProtocol )
	{
        m_aFastProtocolHandlerCache[ aHandler.lProtocols[nProtocol] ].push_back( aHandler.sName );
	}
    // If modification allowed ... add it to list and set flag!
    if( bSetModified == sal_True )
    {
        m_aProtocolHandlerCache.appendChange( aHandler.sName, E_ADDED );
        m_bIsModified = sal_True;
    }
}

/*-************************************************************************************************************//**
    @short      replace existing config item in data container
    @descr      This should be used to change a item of this data container.

    @attention  Call combination of remove/add by using flag "sal_False" ...
                because a "sal_True" will trigger two flush operations!
                One change call should be enough ...

    @seealso    method add...()
    @seealso    method remove...()

    @param      "aType, aFilter ...", information about changed item to set it on internal structures
    @param      "bSetModified"      , regulate registration of changed item as "realy" changed element, which should be flushed to file!
    @return     -

    @onerror    Item will be ignored.
*//*-*************************************************************************************************************/

//*****************************************************************************************************************
void DataContainer::replaceType( const FileType& aType, sal_Bool bSetModified )
{
    removeType( aType.sName, sal_False );
    addType   ( aType      , sal_False );
    if( bSetModified == sal_True )
    {
        m_aTypeCache.appendChange( aType.sName, E_CHANGED );
        m_bIsModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::replaceFilter( const Filter& aFilter, sal_Bool bSetModified )
{
    removeFilter( aFilter.sName, sal_False );
    addFilter   ( aFilter      , sal_False );
    if( bSetModified == sal_True )
    {
        m_aFilterCache.appendChange( aFilter.sName, E_CHANGED );
        m_bIsModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::replaceDetector( const Detector& aDetector, sal_Bool bSetModified )
{
    removeDetector( aDetector.sName, sal_False );
    addDetector   ( aDetector      , sal_False );
    if( bSetModified == sal_True )
    {
        m_aDetectorCache.appendChange( aDetector.sName, E_CHANGED );
        m_bIsModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::replaceLoader( const Loader& aLoader, sal_Bool bSetModified )
{
    removeLoader( aLoader.sName, sal_False );
    addLoader   ( aLoader      , sal_False );
    if( bSetModified == sal_True )
    {
        m_aLoaderCache.appendChange( aLoader.sName, E_CHANGED );
        m_bIsModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::replaceContentHandler( const ContentHandler& aHandler, sal_Bool bSetModified )
{
    removeContentHandler( aHandler.sName, sal_False );
    addContentHandler   ( aHandler      , sal_False );
    if( bSetModified == sal_True )
    {
        m_aContentHandlerCache.appendChange( aHandler.sName, E_CHANGED );
        m_bIsModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::replaceProtocolHandler( const ProtocolHandler& aHandler, sal_Bool bSetModified )
{
    removeProtocolHandler( aHandler.sName, sal_False );
    addProtocolHandler   ( aHandler      , sal_False );
    if( bSetModified == sal_True )
    {
        m_aProtocolHandlerCache.appendChange( aHandler.sName, E_CHANGED );
        m_bIsModified = sal_True;
    }
}

/*-************************************************************************************************************//**
    @short      remove existing config item from data container
    @descr      This should be used to remove an item of this data container.

    @attention  We differ between impl and normal methods. Normal versions set modified flag additional to impl ones.
                This enable flushing of changed data to configuration.

    @seealso    method add...()
    @seealso    method replace...()

    @param      "aType, aFilter ...", information about changed item to set it on internal structures
    @param      "bSetModified"      , regulate registration of removed item as "realy" deleted element, which should be flushed to file!
    @return     -

    @onerror    Item will be ignored.
*//*-*************************************************************************************************************/

//*****************************************************************************************************************
void DataContainer::removeType( const ::rtl::OUString& sName, sal_Bool bSetModified )
{
    // Remove it from our normal cache.
    m_aTypeCache.erase( sName );
    if( bSetModified == sal_True )
    {
        m_aTypeCache.appendChange( sName, E_REMOVED );
        m_bIsModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::removeFilter( const ::rtl::OUString& sName, sal_Bool bSetModified )
{
	// Delete it in performance cache first ...
    // BUT ...
    // first we must get type which filter has registered!
    // It's neccessary to find these filter in our performance cache which is reverse to our normal filter cache!!!
    // Otherwise we have no chance to find it there ...
    ::rtl::OUString sType = m_aFilterCache[ sName ].sType  ;
    StringList&     rList = m_aFastFilterCache[ sType ]    ;
    rList.erase( ::std::find( rList.begin(), rList.end(), sName ) );
	// then in our normal cache.
    m_aFilterCache.erase( sName );
    if( bSetModified == sal_True )
    {
        m_aFilterCache.appendChange( sName, E_REMOVED );
        m_bIsModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::removeDetector( const ::rtl::OUString& sName, sal_Bool bSetModified )
{
	// Delete it in performance cache first ...
    // BUT ...
    // first we must get type which detector has registered!
    // It's neccessary to find these detector in our performance cache which is reverse to our normal detector cache!!!
    // Otherwise we have no chance to find it there ...
    for( ConstStringListIterator pType=m_aDetectorCache[ sName ].lTypes.begin(); pType!=m_aDetectorCache[ sName ].lTypes.end(); ++pType )
    {
        StringList rDetectors = m_aFastDetectorCache[ *pType ];
        rDetectors.erase( ::std::find( rDetectors.begin(), rDetectors.end(), sName ) );
    }
	// then in our normal cache.
    m_aDetectorCache.erase( sName );
    if( bSetModified == sal_True )
    {
        m_aDetectorCache.appendChange( sName, E_REMOVED );
        m_bIsModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::removeLoader( const ::rtl::OUString& sName, sal_Bool bSetModified )
{
	// Delete it in performance cache first ...
    // BUT ...
    // first we must get type which loader has registered!
    // It's neccessary to find these loader in our performance cache which is reverse to our normal loader cache!!!
    // Otherwise we have no chance to find it there ...
    for( ConstStringListIterator pType=m_aLoaderCache[ sName ].lTypes.begin(); pType!=m_aLoaderCache[ sName ].lTypes.end(); ++pType )
    {
        StringList rLoaders = m_aFastLoaderCache[ *pType ];
        rLoaders.erase( ::std::find( rLoaders.begin(), rLoaders.end(), sName ) );
    }
	// then in our normal cache.
    m_aLoaderCache.erase( sName );
    if( bSetModified == sal_True )
    {
        m_aLoaderCache.appendChange( sName, E_REMOVED );
        m_bIsModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::removeContentHandler( const ::rtl::OUString& sName, sal_Bool bSetModified )
{
	// Delete it in performance cache first ...
    // BUT ...
    // first we must get type which handler has registered!
    // It's neccessary to find these handler in our performance cache which is reverse to our normal handler cache!!!
    // Otherwise we have no chance to find it there ...
    for( ConstStringListIterator pType=m_aContentHandlerCache[ sName ].lTypes.begin(); pType!=m_aContentHandlerCache[ sName ].lTypes.end(); ++pType )
    {
        StringList rHandlers = m_aFastContentHandlerCache[ *pType ];
        rHandlers.erase( ::std::find( rHandlers.begin(), rHandlers.end(), sName ) );
    }
	// then in our normal cache.
    m_aContentHandlerCache.erase( sName );
    if( bSetModified == sal_True )
    {
        m_aContentHandlerCache.appendChange( sName, E_REMOVED );
        m_bIsModified = sal_True;
    }
}

//*****************************************************************************************************************
void DataContainer::removeProtocolHandler( const ::rtl::OUString& sName, sal_Bool bSetModified )
{
	// Delete it in performance cache first ...
    // BUT ...
    // first we must get protocol which handler has registered!
    // It's neccessary to find these handler in our performance cache which is reverse to our normal handler cache!!!
    // Otherwise we have no chance to find it there ...
    for( ConstStringListIterator pProtocol=m_aProtocolHandlerCache[ sName ].lProtocols.begin(); pProtocol!=m_aProtocolHandlerCache[ sName ].lProtocols.end(); ++pProtocol )
    {
        StringList rHandlers = m_aFastProtocolHandlerCache[ *pProtocol ];
        rHandlers.erase( ::std::find( rHandlers.begin(), rHandlers.end(), sName ) );
    }
	// then in our normal cache.
    m_aProtocolHandlerCache.erase( sName );
    if( bSetModified == sal_True )
    {
        m_aProtocolHandlerCache.appendChange( sName, E_REMOVED );
        m_bIsModified = sal_True;
    }
}

//*****************************************************************************************************************
//	static private method
//*****************************************************************************************************************
void DataContainer::convertStringSequenceToVector( const css::uno::Sequence< ::rtl::OUString >& lSource, StringList& lDestination )
{
	// Copy values from one list to another one ...
    // but look for empty entries! I think - they are superflous and dangerous.
	// Skip it.

	lDestination.free();
    sal_uInt32         nCount = lSource.getLength();
    ::rtl::OUString    sValue                      ;
	for( sal_uInt32 nPosition=0; nPosition<nCount; ++nPosition )
	{
		sValue = lSource[nPosition];
        LOG_ASSERT2( sValue.getLength()<1, "DataContainer::convertStringSequenceToVector()", "Empty value detected! Don't use leading or leaving seperators ..." )
		if( sValue.getLength() > 0 )
		{
			lDestination.push_back( sValue );
		}
	}
}

//*****************************************************************************************************************
//	static private method
//*****************************************************************************************************************
void DataContainer::convertStringVectorToSequence( const StringList& lSource, css::uno::Sequence< ::rtl::OUString >& lDestination )
{
	// Copy values from one list to another one ...
	// but look for empty entries! I think - thea are superflous and dangerous.
	// Skip it.

	lDestination.realloc( lSource.size() );
    sal_uInt32         nPosition = 0   ;
    ::rtl::OUString    sValue          ;
	for( ConstStringListIterator pIterator=lSource.begin(); pIterator!=lSource.end(); ++pIterator )
	{
		sValue = *pIterator;
        LOG_ASSERT2( sValue.getLength()<1, "DataContainer::convertStringVectorToSequence()", "Empty value detected! Don't use leading or leaving seperators ..." )
		if( sValue.getLength() > 0 )
		{
            lDestination[ nPosition ] = sValue;
        }
		++nPosition;
	}
}

//*****************************************************************************************************************
//	static private method
//*****************************************************************************************************************
void DataContainer::convertFileTypeToPropertySequence( const FileType& lSource, css::uno::Sequence< css::beans::PropertyValue >& lDestination, const ::rtl::OUString& sCurrentLocale )
{
    css::uno::Sequence< ::rtl::OUString >           lPattern    ;
    css::uno::Sequence< ::rtl::OUString >           lExtensions ;

    convertStringVectorToSequence( lSource.lURLPattern, lPattern    );
    convertStringVectorToSequence( lSource.lExtensions, lExtensions );

    lDestination.realloc( 7 );

    lDestination[0].Name    =   PROPERTY_PREFERRED                                        ;
    lDestination[0].Value   <<= lSource.bPreferred                                        ;

    lDestination[1].Name    =   PROPERTY_UINAME                                           ;
    lDestination[1].Value   <<= getLocalelizedString( lSource.lUINames, sCurrentLocale )  ;

    lDestination[2].Name    =   PROPERTY_MEDIATYPE                                        ;
    lDestination[2].Value   <<= lSource.sMediaType                                        ;

    lDestination[3].Name    =   PROPERTY_CLIPBOARDFORMAT                                  ;
    lDestination[3].Value   <<= lSource.sClipboardFormat                                  ;

    lDestination[4].Name    =   PROPERTY_URLPATTERN                                       ;
    lDestination[4].Value   <<= lPattern                                                  ;

    lDestination[5].Name    =   PROPERTY_EXTENSIONS                                       ;
    lDestination[5].Value   <<= lExtensions                                               ;

    lDestination[6].Name    =   PROPERTY_DOCUMENTICONID                                   ;
    lDestination[6].Value   <<= lSource.nDocumentIconID                                   ;
/*OBSOLETE
    css::uno::Sequence< css::beans::PropertyValue > lUINames    ;
    convertStringHashToSequence  ( lSource.lUINames   , lUINames    );
    lDestination[7].Name    =   PROPERTY_UINAMES                                          ;
    lDestination[7].Value   <<= lUINames                                                  ;
*/
}

//*****************************************************************************************************************
//	static private method
//*****************************************************************************************************************
void DataContainer::convertFilterToPropertySequence( const Filter& lSource, css::uno::Sequence< css::beans::PropertyValue >& lDestination, const ::rtl::OUString& sCurrentLocale )
{
    css::uno::Sequence< ::rtl::OUString > lUserData;

    lDestination.realloc( 9 );

    lDestination[0].Name    =   PROPERTY_TYPE                                              ;
    lDestination[0].Value   <<= lSource.sType                                              ;

    convertStringVectorToSequence( lSource.lUserData, lUserData );
    lDestination[1].Name    =   PROPERTY_UINAME                                            ;
    lDestination[1].Value   <<= getLocalelizedString( lSource.lUINames, sCurrentLocale )   ;

    lDestination[2].Name    =   PROPERTY_DOCUMENTSERVICE                                   ;
    lDestination[2].Value   <<= lSource.sDocumentService                                   ;

    lDestination[3].Name    =   PROPERTY_FILTERSERVICE                                     ;
    lDestination[3].Value   <<= lSource.sFilterService                                     ;

    lDestination[4].Name    =   PROPERTY_FLAGS                                             ;
    lDestination[4].Value   <<= lSource.nFlags                                             ;

    lDestination[5].Name    =   PROPERTY_USERDATA                                          ;
    lDestination[5].Value   <<= lUserData                                                  ;

    lDestination[6].Name    =   PROPERTY_FILEFORMATVERSION                                 ;
    lDestination[6].Value   <<= lSource.nFileFormatVersion                                 ;

    lDestination[7].Name    =   PROPERTY_TEMPLATENAME                                      ;
    lDestination[7].Value   <<= lSource.sTemplateName                                      ;

    lDestination[8].Name    =   PROPERTY_ORDER                                             ;
    lDestination[8].Value   <<= lSource.nOrder                                             ;

/*OBSOLETE
    css::uno::Sequence< css::beans::PropertyValue > lUINames ;
    convertStringHashToSequence  ( lSource.lUINames , lUINames  );
    lDestination[8].Name    =   PROPERTY_UINAMES                                           ;
    lDestination[8].Value   <<= lUINames                                                   ;
*/
}

//*****************************************************************************************************************
//	static private method
//*****************************************************************************************************************
void DataContainer::convertPropertySequenceToFilter( const css::uno::Sequence< css::beans::PropertyValue >& lSource, Filter& lDestination, const ::rtl::OUString& sCurrentLocale )
{
	// First clear all entries of destination struct!
	lDestination.free();
	// Then copy existing properties from source to destination.
	// Non existing properties will have a default value in destination then!
	sal_Int32 nCount = lSource.getLength();
	for( sal_Int32 nProperty=0; nProperty<nCount; ++nProperty )
	{
		//_________________________________________________________________________________________________________
		if( lSource[nProperty].Name == PROPERTY_TYPE )
		{
			lSource[nProperty].Value >>= lDestination.sType;
		}
		else
		//_________________________________________________________________________________________________________
		if( lSource[nProperty].Name == PROPERTY_UINAME )
		{
            ::rtl::OUString sUIName;
			lSource[nProperty].Value >>= sUIName;
            setLocalelizedString( lDestination.lUINames, sCurrentLocale, sUIName );
		}
		else
/*OBSOLETE
        //_________________________________________________________________________________________________________
		if( lSource[nProperty].Name == PROPERTY_UINAMES )
		{
            css::uno::Sequence< css::beans::PropertyValue > lUINames;
			lSource[nProperty].Value >>= lUINames;
            convertSequenceToStringHash( lUINames, lDestination.lUINames );
		}
        else*/
		//_________________________________________________________________________________________________________
		if( lSource[nProperty].Name == PROPERTY_DOCUMENTSERVICE )
		{
			lSource[nProperty].Value >>= lDestination.sDocumentService;
		}
		else
		//_________________________________________________________________________________________________________
		if( lSource[nProperty].Name == PROPERTY_FILTERSERVICE )
		{
			lSource[nProperty].Value >>= lDestination.sFilterService;
		}
		else
		//_________________________________________________________________________________________________________
		if( lSource[nProperty].Name == PROPERTY_FLAGS )
		{
			lSource[nProperty].Value >>= lDestination.nFlags;
		}
		else
		//_________________________________________________________________________________________________________
		if( lSource[nProperty].Name == PROPERTY_USERDATA )
		{
            css::uno::Sequence< ::rtl::OUString > lTempData;
			lSource[nProperty].Value >>= lTempData;
            convertStringSequenceToVector( lTempData, lDestination.lUserData );
		}
		else
		//_________________________________________________________________________________________________________
		if( lSource[nProperty].Name == PROPERTY_FILEFORMATVERSION )
		{
			lSource[nProperty].Value >>= lDestination.nFileFormatVersion;
		}
		else
		//_________________________________________________________________________________________________________
		if( lSource[nProperty].Name == PROPERTY_TEMPLATENAME )
		{
			lSource[nProperty].Value >>= lDestination.sTemplateName;
		}
		else
		//_________________________________________________________________________________________________________
		if( lSource[nProperty].Name == PROPERTY_ORDER )
		{
			lSource[nProperty].Value >>= lDestination.nOrder;
		}
	}
}

//*****************************************************************************************************************
//	static private method
//*****************************************************************************************************************
void DataContainer::convertLoaderToPropertySequence( const Loader& lSource, css::uno::Sequence< css::beans::PropertyValue >& lDestination, const ::rtl::OUString& sCurrentLocale )
{
    css::uno::Sequence< ::rtl::OUString >           lTypes  ;
    convertStringVectorToSequence( lSource.lTypes  , lTypes   );

    lDestination.realloc( 2 );

    lDestination[0].Name    =   PROPERTY_TYPES                                             ;
    lDestination[0].Value   <<= lTypes                                                     ;

    lDestination[1].Name    =   PROPERTY_UINAME                                            ;
    lDestination[1].Value   <<= getLocalelizedString( lSource.lUINames, sCurrentLocale )   ;

/*OBSOLETE
    css::uno::Sequence< css::beans::PropertyValue > lUINames;
    convertStringHashToSequence  ( lSource.lUINames, lUINames );
    lDestination[2].Name    =   PROPERTY_UINAMES                                           ;
    lDestination[2].Value   <<= lUINames                                                   ;
*/
}

//*****************************************************************************************************************
//	static private method
//*****************************************************************************************************************
void DataContainer::convertDetectorToPropertySequence( const Detector& lSource, css::uno::Sequence< css::beans::PropertyValue >& lDestination )
{
    css::uno::Sequence< ::rtl::OUString > lTypes;
    convertStringVectorToSequence( lSource.lTypes, lTypes );

    lDestination.realloc( 1 );

    lDestination[0].Name    =   PROPERTY_TYPES ;
    lDestination[0].Value   <<= lTypes         ;
}

//*****************************************************************************************************************
//	static private method
//*****************************************************************************************************************
void DataContainer::convertContentHandlerToPropertySequence( const ContentHandler&                                  aSource      ,
                                                                   css::uno::Sequence< css::beans::PropertyValue >& lDestination )
{
    css::uno::Sequence< ::rtl::OUString > lTypes;
    convertStringVectorToSequence( aSource.lTypes, lTypes );

    lDestination.realloc( 1 );

    lDestination[0].Name    = PROPERTY_TYPES ;
    lDestination[0].Value <<= lTypes         ;
}

//*****************************************************************************************************************
//	static private method
//*****************************************************************************************************************
void DataContainer::convertProtocolHandlerToPropertySequence( const ProtocolHandler&                                 aSource      ,
                                                                    css::uno::Sequence< css::beans::PropertyValue >& lDestination )
{
    css::uno::Sequence< ::rtl::OUString > lProtocols;
    convertStringVectorToSequence( aSource.lProtocols, lProtocols );

    lDestination.realloc( 1 );

    lDestination[0].Name    = PROPERTY_PROTOCOLS ;
    lDestination[0].Value <<= lProtocols         ;
}

//*****************************************************************************************************************
//	static private method
//*****************************************************************************************************************
void DataContainer::convertStringHashToSequence( const StringHash& lSource, css::uno::Sequence< css::beans::PropertyValue >& lDestination )
{
	sal_Int32 nItem = 0;
	lDestination.realloc( lSource.size() );
	for( ConstStringHashIterator pItem=lSource.begin(); pItem!=lSource.end(); ++pItem )
	{
		lDestination[ nItem ].Name	=	pItem->first	;
		lDestination[ nItem ].Value	<<=	pItem->second	;
		++nItem;
	}
}

//*****************************************************************************************************************
//	static private method
//*****************************************************************************************************************
void DataContainer::convertSequenceToStringHash( const css::uno::Sequence< css::beans::PropertyValue >& lSource, StringHash& lDestination )
{
	lDestination.free();
	sal_Int32 nCount = lSource.getLength();
	for( sal_Int32 nItem=0; nItem<nCount; ++nItem )
	{
		lSource[nItem].Value >>= lDestination[ lSource[nItem].Name ];

        ::rtl::OUString sLocale = lSource[nItem].Name;
        ::rtl::OUString sValue  = lDestination[ lSource[nItem].Name ];
	}
}

//*****************************************************************************************************************
// We accept right package names and valid versions only. Otherwise we would work on ANYTHING ... but not
// on right data set!
sal_Bool FilterCFGAccess::implcp_ctor( const ::rtl::OUString& sPath   ,
                                             sal_Int32        nVersion,
                                             sal_Int16        nMode   )
{
    return(
            ( &sPath   == NULL                        ) ||
            ( nVersion >  DEFAULT_FILTERCACHE_VERSION ) ||
            ( nVersion <  1                           ) ||
            (
                ( sPath != PACKAGENAME_TYPEDETECTION_STANDARD   )   &&
                ( sPath != PACKAGENAME_TYPEDETECTION_ADDITIONAL )
            )
          );
}

//*****************************************************************************************************************
// Check valid c++ reference only!
sal_Bool FilterCFGAccess::implcp_read( const DataContainer& rData )
{
    return(
            ( &rData == NULL )
          );
}

//*****************************************************************************************************************
// Check valid c++ reference only!
sal_Bool FilterCFGAccess::implcp_write( const DataContainer& rData )
{
    return(
            ( &rData == NULL )
          );
}

} // namespace framework
