/*************************************************************************
 *
 *  $RCSfile: jobexecutor.cxx,v $
 *
 *  $Revision: 1.2 $
 *
 *  last change: $Author: fs $ $Date: 2001/12/20 13:51:24 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

//_________________________________________________________________________________________________________________
//	my own includes
//_________________________________________________________________________________________________________________

#ifndef __FRAMEWORK_SERVICES_JOBEXECUTOR_HXX_
#include <services/jobexecutor.hxx>
#endif

#ifndef __FRAMEWORK_THREADHELP_TRANSACTIONGUARD_HXX_
#include <threadhelp/transactionguard.hxx>
#endif

#ifndef __FRAMEWORK_SERVICES_H_
#include <services.h>
#endif

//_________________________________________________________________________________________________________________
//	interface includes
//_________________________________________________________________________________________________________________

#ifndef _COM_SUN_STAR_REGISTRY_XSIMPLEREGISTRY_HPP_
#include <com/sun/star/registry/XSimpleRegistry.hpp>
#endif

#ifndef _COM_SUN_STAR_TASK_XJOB_HPP_
#include <com/sun/star/task/XJob.hpp>
#endif

#ifndef _COM_SUN_STAR_LANG_ILLEGALARGUMENTEXCEPTION_HPP_
#include <com/sun/star/lang/IllegalArgumentException.hpp>
#endif

//_________________________________________________________________________________________________________________
//	includes of other projects
//_________________________________________________________________________________________________________________

#ifndef UNOTOOLS_CONFIGPATHES_HXX_INCLUDED
#include <unotools/configpathes.hxx>
#endif

#ifndef _SV_SVAPP_HXX
#include <vcl/svapp.hxx>
#endif

//_________________________________________________________________________________________________________________
//	namespace
//_________________________________________________________________________________________________________________

namespace framework{

//_________________________________________________________________________________________________________________
//	non exported const
//_________________________________________________________________________________________________________________

#define JOBREGISTRY             DECLARE_ASCII("Office.Jobs/")

#define JOBSET                  DECLARE_ASCII("Jobs"        )
#define EVENTSET                DECLARE_ASCII("Events"      )

#define PATHSEPERATOR           DECLARE_ASCII("/"           )

#define JOBPROP_SERVICE         DECLARE_ASCII("Service"     )
#define JOBPROP_ASYNC           DECLARE_ASCII("Async"       )
#define JOBPROP_ARGUMENTS       DECLARE_ASCII("Arguments"   )
#define JOBPROP_ADMINTIME       DECLARE_ASCII("AdminTime"   )
#define JOBPROP_USERTIME        DECLARE_ASCII("UserTime"    )

#define EVENTPROP_JOBLIST       DECLARE_ASCII("Joblist"     )

//_________________________________________________________________________________________________________________
//	non exported definitions
//_________________________________________________________________________________________________________________

//_________________________________________________________________________________________________________________
//	declarations
//_________________________________________________________________________________________________________________

/*-****************************************************************************************************************
    @descr  initialize object with default start values
            We should mark "job" as "sleeping" ... not "working"! => bWork=false
            It's not neccessary to set defaults on all members. Most of them has own defaults ... or
            start value isnt realy important!
****************************************************************************************************************-*/
Job::Job()
{
    bWork = sal_False;
}

/*-****************************************************************************************************************
    @descr  append new argument to job specific argument list
            Should be used on reading configuration ... to implement easy way to fill
            these list. Changing of these list should be a part of specific job.
            We support writing back of whole list to cfg only ... not changing of some items of that.
            See class JobCFG too!
****************************************************************************************************************-*/
void Job::appendArgument( const ::rtl::OUString& sName  ,
                          const css::uno::Any&   aValue )
{
    sal_Int32 nCount = lArguments.getLength();
    lArguments.realloc( nCount+1 );
    lArguments[nCount].Name  = sName ;
    lArguments[nCount].Value = aValue;
}

/*-****************************************************************************************************************
    @descr  check, if an adminstrator has set newer time stamp in configuration then these job was finished last time
            We use it at reading of configuration to decide, which jobs must be cached or not.
            We don't delete finished jobs realy - we set user time similar to admin time. So job will be ignored
            next read - till admin change that!
            If a time stamp is corrupted or not exist ... then we ignore the job completly!

    FEATURE

    May be it can be usefull to suppress this job for ever by user by setting a special property of job.
    e.g. "UserDeinstalled [boolean]"
    These value must be set by user only and let us ignore normal time stamp mechanism ...
****************************************************************************************************************-*/
sal_Bool Job::mustBeActivated( const ::rtl::OUString& sAdmin ,
                               const ::rtl::OUString& sUser  )
{
    DateTime aAdminStamp;
    DateTime aUserStamp ;
    return  (
                ( convertString2TimeStamp( sAdmin, aAdminStamp ) == sal_True   ) &&
                ( convertString2TimeStamp( sUser , aUserStamp  ) == sal_True   ) &&
                ( aAdminStamp                                    >  aUserStamp )
            );
}

/*-****************************************************************************************************************
    @descr  helper to convert saved time stamp of configuration into internal format
            Configuration can't save such time informations. So we convert it from/to normal strings ...
            Format: "<day>.<month>.<year>/<hour>:<min>:<sec>"
            e.g.  : "1.11.2001/13:45:16"
****************************************************************************************************************-*/
sal_Bool Job::convertString2TimeStamp( const ::rtl::OUString& sString ,
                                             DateTime&        rStamp  )
{
    sal_Bool  bValid = sal_False;
    sal_Int32 nIndex = 0        ;

    sal_uInt16 nDay = (sal_uInt16)(sString.getToken( 0, (sal_Unicode)'.', nIndex ).toInt32());
    if( nIndex>0 )
    {
        sal_uInt16 nMonth = (sal_uInt16)(sString.getToken( 0, (sal_Unicode)'.', nIndex ).toInt32());
        if( nIndex>0 )
        {
            sal_uInt16 nYear = (sal_uInt16)(sString.getToken( 0, (sal_Unicode)'/', nIndex ).toInt32());
            if( nIndex>0 )
            {
                sal_uInt32 nHour = sString.getToken( 0, (sal_Unicode)':', nIndex ).toInt32();
                if( nIndex>0 )
                {
                    sal_uInt32 nMin = sString.getToken( 0, (sal_Unicode)':', nIndex ).toInt32();
                    if( nIndex>0 && nIndex<sString.getLength() )
                    {
                        sal_uInt32 nSec = sString.copy( nIndex, sString.getLength()-nIndex ).toInt32();

                        Date aDate( nDay , nMonth, nYear );
                        Time aTime( nHour, nMin  , nSec  );
                        rStamp = DateTime( aDate, aTime );
                        bValid = sal_True;
                    }
                }
            }
        }
    }
    return bValid;
}

/*-****************************************************************************************************************
    @descr  helper to convert given time stamp from internal format to string, which can be written to configuration!
            see impl_String2TimeStamp() too ...
            Format: "<day>.<month>.<year>/<hour>:<min>:<sec>"
            e.g.  : "1.11.2001/13:45:16"
****************************************************************************************************************-*/
::rtl::OUString Job::convertTimeStamp2String( const DateTime& aStamp )
{
    ::rtl::OUStringBuffer sBuffer(25);

    sBuffer.append( (sal_Int32)aStamp.GetDay()   );
    sBuffer.append( (sal_Unicode)'.'             );
    sBuffer.append( (sal_Int32)aStamp.GetMonth() );
    sBuffer.append( (sal_Unicode)'.'             );
    sBuffer.append( (sal_Int32)aStamp.GetYear()  );
    sBuffer.append( (sal_Unicode)'/'             );
    sBuffer.append( (sal_Int32)aStamp.GetHour()  );
    sBuffer.append( (sal_Unicode)':'             );
    sBuffer.append( (sal_Int32)aStamp.GetMin()   );
    sBuffer.append( (sal_Unicode)':'             );
    sBuffer.append( (sal_Int32)aStamp.GetSec()   );

    return sBuffer.makeStringAndClear();
}

// Only way to initialize static class member realy ...
DynamicConfigItem*  JobCFG::m_pConfigAccess = NULL;
sal_Int32           JobCFG::m_nRefCount     = 0   ;

/*-****************************************************************************************************************
    @descr  Create new access to configration. We open cfg for one times only - by creating m_pConfigAccess.
            Creation will be regulated by another static member m_nRefCount.
****************************************************************************************************************-*/
JobCFG::JobCFG()
{
    ResetableGuard aLock( m_aLock );
    if( m_nRefCount==0 )
        m_pConfigAccess = new DynamicConfigItem( JOBREGISTRY );
    ++m_nRefCount;
}

/*-****************************************************************************************************************
    @descr  Opening of configuration was done by our ctor automaticly ... so we must close it in dtor - if
            last instance of this class will gone (means ref count decreased to 0).
****************************************************************************************************************-*/
JobCFG::~JobCFG()
{
    ResetableGuard aLock( m_aLock );
    if( m_nRefCount==1 )
    {
        delete m_pConfigAccess;
        m_pConfigAccess = NULL;
    }
    --m_nRefCount;
}

/*-****************************************************************************************************************
    @descr  If an instance of this class exist (otherwise this method couldnt be called !...)
            user of it can use it to read current job informations from it. He give us references to
            his list's ... and we will fill it.
****************************************************************************************************************-*/
void JobCFG::readAll( JobHash&   aJobCache   ,
                      EventHash& aEventCache )
{
    ResetableGuard aLock( m_aLock );
    // Don't change order of reading methods ... because
    // second one (impl_readEvents()) eliminate superflous job registrations for events.
    // That means: A job will be registered for an event ... but was ignored in impl_readJobs() ...
    // because he was finished or deregistered before!
    impl_readJobSet  ( aJobCache              );
    impl_readEventSet( aEventCache, aJobCache );
}

/*-****************************************************************************************************************
    @descr  Change user time stamp of specified job (by given name) to current date & time.
            We get right time automaticly and write it to cfg of these job.
            It's an easy access for user - he give us the job name - we do the rest automaticly.
****************************************************************************************************************-*/
void JobCFG::setUserTimeOnJob( const ::rtl::OUString& sJob )
{
    ResetableGuard aLock( m_aLock );
    css::uno::Sequence< css::beans::PropertyValue > lProperties(1);
    lProperties[0].Name    = JOBSET                                       ;
    lProperties[0].Name   += PATHSEPERATOR                                ;
    lProperties[0].Name   += ::utl::wrapConfigurationElementName(sJob)    ;
    lProperties[0].Name   += PATHSEPERATOR                                ;
    lProperties[0].Name   += JOBPROP_USERTIME                             ;

    DateTime aCurrentStamp; // default ctor set current date & time on instanciated object automaticly!
    lProperties[0].Value <<= Job::convertTimeStamp2String( aCurrentStamp );

    m_pConfigAccess->SetSetProperties( JOBSET, lProperties );
}

/*-****************************************************************************************************************
    @descr  Write given arguments to right configuration entry specified by "sJob".
****************************************************************************************************************-*/
void JobCFG::saveJobArguments( const ::rtl::OUString&                              sJob       ,
                               const css::uno::Sequence< css::beans::NamedValue >& lArguments )
{
    ResetableGuard aLock( m_aLock );
    sal_Int32                                           nCount     = lArguments.getLength();
    css::uno::Sequence< css::beans::PropertyValue >     lProperties( nCount );
    ::rtl::OUString                                     sBasePath  = JOBSET                                     ;
                                                        sBasePath += PATHSEPERATOR                              ;
                                                        sBasePath += ::utl::wrapConfigurationElementName(sJob)  ;
                                                        sBasePath += PATHSEPERATOR                              ;
                                                        sBasePath += JOBPROP_ARGUMENTS                          ;
                                                        sBasePath += PATHSEPERATOR                              ;
    for( sal_Int32 nStep=0; nStep<nCount; ++nStep )
    {
        lProperties[nStep].Name   = sBasePath              ;
        lProperties[nStep].Name  += lArguments[nStep].Name ;
        lProperties[nStep].Value  = lArguments[nStep].Value;
    }
    m_pConfigAccess->ReplaceSetProperties( JOBSET, lProperties );
}

/*-****************************************************************************************************************
    @descr      read set with job informations from configuration
                We clear our internal cache first and try to get the whole job set of cfg file.
                But we don't cache all of them. We look on set time stamps too, to decide if
                job can be started or not. Unused jobs will not be cached.

    @threadsafe No - because they are called from our ctor. Normaly we mustn't use any mutex opr lock there.
                Otherwise - caller of this methods must guard it ...
****************************************************************************************************************-*/
void JobCFG::impl_readJobSet( JobHash& aJobCache )
{
    // Before we start reading of configuration we should clear our internal cache ...
    aJobCache.clear();

    // Get all names of currently existing job entries of configuration set
    // and reserve enough memory for expandation to list of all subproperties!
    // This is true for fix count of properties ... but we support optioanl argument sets too.
    // So sometimes we must reallocate some memory during next lines too ...
    css::uno::Sequence< ::rtl::OUString > lNames      = m_pConfigAccess->GetNodeNames( JOBSET );
    css::uno::Sequence< ::rtl::OUString > lProperties ( lNames.getLength()*4 );

    // Step over name list and expand it with well known sub property names.
    // So we build follow structure:
    //      e.g.        Jobs/<job_1>/AdminTime
    //                  Jobs/<job_1>/UserTime
    //                  Jobs/<job_1>/Service
    //                  Jobs/<job_1>/Async
    //                  Jobs/<job_1>/Arguments/<argument1>
    //                  ...
    //                  Jobs/<job_1>/Arguments/<argumentn>
    //                  ...
    //                  Jobs/<job_n>/AdminTime
    //                  Jobs/<job_n>/UserTime
    //                  ...
    sal_Int32       nCount  = lNames.getLength();
    sal_Int32       nSource = 0;
    sal_Int32       nTarget = 0;
    ::rtl::OUString sPath   ;
    for( nSource=0; nSource<nCount; ++nSource )
    {
        sPath  = JOBSET         ;
        sPath += PATHSEPERATOR  ;
        sPath += lNames[nSource];
        sPath += PATHSEPERATOR  ; // => sPath="Jobs/job1/"

        lProperties[nTarget]  = sPath            ;
        lProperties[nTarget] += JOBPROP_ADMINTIME;
        ++nTarget;

        lProperties[nTarget]  = sPath            ;
        lProperties[nTarget] += JOBPROP_USERTIME ;
        ++nTarget;

        lProperties[nTarget]  = sPath            ;
        lProperties[nTarget] += JOBPROP_SERVICE  ;
        ++nTarget;

        lProperties[nTarget]  = sPath            ;
        lProperties[nTarget] += JOBPROP_ASYNC    ;
        ++nTarget;

        sPath += JOBPROP_ARGUMENTS; // => sPath="Jobs/job1/Arguments"

        css::uno::Sequence< ::rtl::OUString > lArguments = m_pConfigAccess->GetNodeNames( sPath );
        sal_Int32                             nArgCount  = lArguments.getLength();

        sPath += PATHSEPERATOR; // => sPath="Jobs/job1/Arguments/"

        // ! Reserve enough space for optional detected arguments !
        if( nArgCount>0 )
            lProperties.realloc( lProperties.getLength()+nArgCount );

        for( sal_Int32 nArg=0; nArg<nArgCount; ++nArg )
        {
            lProperties[nTarget]  = sPath           ;
            lProperties[nTarget] += lArguments[nArg];
            ++nTarget;
        }
    }

    // Get all values from configuration for builded name list.
    css::uno::Sequence< css::uno::Any > lValues = m_pConfigAccess->GetProperties( lProperties );

    nCount  = nTarget; // nTarget could be greater then nCount here ... may be we have read some optional arguments ... !!!
    nSource = 0      ;
    nTarget = 0      ;

    ::rtl::OUString sSetName    ;
    ::rtl::OUString sJobName    ;
    ::rtl::OUString sPropName   ;
    ::rtl::OUString sArgName    ;
    ::rtl::OUString sAdminTime  ;
    ::rtl::OUString sUserTime   ;

    // Attention: Follow code doesn't test all property names of name list!
    // We can do that - because we have build it and know ... on some positions
    // must(!) exist expected ones! Because some properties of job structure are fix ... like ["AdminTime","UserTime","Service","Async"].
    // But don't forget optional arguments. Then you MUST check right name of found sub property!!!
    while( nSource<nCount )
    {
        lValues[nSource] >>= sAdminTime;
        ++nSource;
        lValues[nSource] >>= sUserTime;
        ++nSource;

        // Use these values to decide, if a job should be cached or not.
        // Only if AdminTime is greater then UserTime a job will be reactivated.
        // Otherwise it will be ignored.
        if( Job::mustBeActivated( sAdminTime, sUserTime ) == sal_True )
        {
            // Job should be cached.
            // Get name of it ...
            // We can do that without any additional checks ... structure is fix. (seee before!)
            JobCFG::impl_seperatePathEntries( lProperties[nSource], &sSetName, &sJobName );
            lValues[nSource] >>= aJobCache[sJobName].sService;
            ++nSource;
            lValues[nSource] >>= aJobCache[sJobName].bAsync;
            ++nSource;
            // But now optioanl arguments can occure!
            // Loop over all follow entries, if property name is the right one.
            if( nSource>=nCount )
                break;
            JobCFG::impl_seperatePathEntries( lProperties[nSource], &sSetName, &sJobName, &sPropName, &sArgName );
            while( sPropName == JOBPROP_ARGUMENTS )
            {
                aJobCache[sJobName].appendArgument( sArgName, lValues[nSource] );
                // Step to next one and break loop, if end of list is reached!
                ++nSource;
                if( nSource>=nCount )
                    break;
                // Get informations about next valid cfg entry.
                // If it isn't an argument then it was superflous to do that ... we start outer loop again ...
                // But if property name is the right one ... these inner loop willn't end!
                JobCFG::impl_seperatePathEntries( lProperties[nSource], &sSetName, &sJobName, &sPropName, &sArgName );
            }
        }
        else
        {
            // Job should be ignored ... because time stamps are similar.
            // Then we must ignore some following entries of name and value list.
            // Because our structure is fix ... and we must step to next valid block with information
            // about next job!

            // First ignore fix properties ... "Service" & "Async" => 2
            // (AdminTime & UserTime was already readed!)
            nSource += 2;
            // Then step over all optional arguments and ignore it. Use same algorithm like before at reading correct job arguments!
            if( nSource>=nCount )
                break;
            JobCFG::impl_seperatePathEntries( lProperties[nSource], &sSetName, &sJobName, &sPropName, &sArgName );
            while( sPropName == JOBPROP_ARGUMENTS )
            {
                ++nSource;
                if( nSource>=nCount )
                    break;
                JobCFG::impl_seperatePathEntries( lProperties[nSource], &sSetName, &sJobName, &sPropName, &sArgName );
            }
        }
    }
}

/*-****************************************************************************************************************
    @descr      read set with event informations from configuration
                We clear our internal cache first and try to get the whole event set of cfg file.
                But we don't cache all of them. We look for invalid registrations. If a job exist in list of event
                registrations but not in our internal job cache - then we delete it in event cache too.
                That's why this function must be called AFTER impl_readJobSet()!

    @threadsafe No - because they are called from our ctor. Normaly we mustn't use any mutex opr lock there.
                Otherwise - caller of this methods must guard it ...
****************************************************************************************************************-*/
void JobCFG::impl_readEventSet(       EventHash& aEventCache ,
                                const JobHash&   aJobCache   )
{
    // Before we start reading of configuration we should clear our internal cache ...
    aEventCache.clear();

    // Get all names of currently existing event entries of configuration set
    // and reserve enough memory for expandation to list of all subproperties!
    css::uno::Sequence< ::rtl::OUString > lNames     = m_pConfigAccess->GetNodeNames( EVENTSET );
    css::uno::Sequence< ::rtl::OUString > lProperties( lNames.getLength() );

    // Step over name list and expand it with well known sub property names.
    // So we build follow structure:
    //      e.g.        Events/<event_1>/Joblist
    //                  Events/<event_1>/Joblist
    //                  ...
    //                  Events/<event_n>/Joblist
    sal_Int32       nCount  = lNames.getLength();
    sal_Int32       nSource = 0;
    sal_Int32       nTarget = 0;
    ::rtl::OUString sPath   ;
    for( nSource=0; nSource<nCount; ++nSource )
    {
        sPath  = EVENTSET       ;
        sPath += PATHSEPERATOR  ;
        sPath += lNames[nSource];
        sPath += PATHSEPERATOR  ; // => sPath="/Events/event1/"

        lProperties[nSource]  = sPath             ;
        lProperties[nSource] += EVENTPROP_JOBLIST ;
    }

    // Get all values for builded name list.
    css::uno::Sequence< css::uno::Any > lValues = m_pConfigAccess->GetProperties( lProperties );

    ::rtl::OUString                       sSetName  ;
    ::rtl::OUString                       sEventName;
    css::uno::Sequence< ::rtl::OUString > lTemp     ;
    ::std::vector< ::rtl::OUString >      lVector   ;
    sal_Int32                             nTempCount;
    sal_Int32                             nTempStep ;

    nSource = 0;
    while( nSource<nCount )
    {
        // Copy value to temp. sequence.
        // We can't unpack it directly to our needed string vector :-(
        // So we must copy all sequence items to a vector.
        // By the way ... look for superflous job registrations.
        // That means: Jobs are active by right time stamp mechanism only.
        // But this is an information of a job .. not of these event list.
        // So we must check, if jobs of readed "Joblist" are realy exist in our cached aJobCache list ...
        // If not - we should ignore it here too!
        lValues[nSource] >>= lTemp;
        nTempCount = lTemp.getLength();
        for( nTempStep=0; nTempStep<nTempCount; ++nTempStep )
        {
            JobHash::const_iterator pJob = aJobCache.find( lTemp[nTempStep] );
            if( pJob != aJobCache.end() )
                lVector.push_back( lTemp[nTempStep] );
        }

        if( lVector.size() > 0 )
        {
            JobCFG::impl_seperatePathEntries( lProperties[nSource], &sSetName, &sEventName );
            aEventCache[sEventName] = lVector;
        }

        ++nSource;
    }
}

/*-****************************************************************************************************************
    @descr  helper to split any path informations of configuration into his several parts
            We try to split given path and set informations on given pointers ... till
            we detect a NULL pointer. One string reference is neccessary ... more then ones are optional.
            Attention: If there exist not enough parts inside given path ... some return values aren't changed!
****************************************************************************************************************-*/
void JobCFG::impl_seperatePathEntries( const ::rtl::OUString& sPath  ,
                                             ::rtl::OUString* pPart1 ,
                                             ::rtl::OUString* pPart2 ,
                                             ::rtl::OUString* pPart3 ,
                                             ::rtl::OUString* pPart4 )
{
    sal_Int32         nIndex = 0;       // token counter for getToken()
    sal_Int32         nPart  = 1;       // indicates currently used "pPartX" variable
    ::rtl::OUString** ppPart = &pPart1; // pointer to given string pointer of caller!
                                        // NULL breaks follow loop automaticly ...

    while(
            (  ppPart!=NULL )   &&
            ( *ppPart!=NULL )
         )
    {
        // copy next token to caller memory
        **ppPart = sPath.getToken( 0, (sal_Unicode)'/', nIndex );
        // break loop, if no further tokens can be read
        if( nIndex == -1 )
            break;
        // otherwise try to find next user given string reference
        // Loop willn't be started, if next pointer=NULL
        // or we set it to NULL, if no further variables available.
        ++nPart;
        switch( nPart )
        {
            case 2 : ppPart = &pPart2;
                     break;
            case 3 : ppPart = &pPart3;
                     break;
            case 4 : ppPart = &pPart4;
                     break;
            default: ppPart = NULL;
                     break;
        }
    }
}

/*-****************************************************************************************************************
    @descr  standard ctor
            We must  set member m_bInitialized to false here only. So we mark instance as
            "not yet filled from configuration" one. On every interface call we check this member inside method
            implts_provideFilledCaches() to make an late init. So we don't read the whole configuration on startup.
            May be; it's not neccessary - Because nobody use us realy ... but instanciate us!
            So we read our configuration on demand ...
****************************************************************************************************************-*/
JobCache::JobCache()
    : m_bInitialized( sal_False )
{}

/*-****************************************************************************************************************
    @descr  search for jobs, which are registered for given event
            We return a list of job names which are registered for specified event.
            If a job is still running - it will be ignored! Multiple starts
            of same job will not be supported. If we found a "sleeping" one
            we toggle work state to TRUE and set it on return list.
            So we differ everytime between running/sleeping jobs and prevent us against
            multithreaded problems! I HOPE :-)
****************************************************************************************************************-*/
::std::vector< ::rtl::OUString > JobCache::getJobsForWork( const ::rtl::OUString& sEvent )
{
    implts_provideFilledCaches(); // late init of caches done?

    // Must be atomic! Otherwise some jobs can be started twice ... and our implementation
    // isn't prepared for that.
    ResetableGuard aLock( m_aLock );

    ::std::vector< ::rtl::OUString > lNewWorkJobs;
    EventHash::const_iterator        pEvent      = m_lEvents.find( sEvent );
    if( pEvent != m_lEvents.end() )
    {
        for( ::std::vector< ::rtl::OUString >::const_iterator pJobName =pEvent->second.begin();
                                                              pJobName!=pEvent->second.end()  ;
                                                              ++pJobName                      )
        {
            JobHash::iterator pJob = m_lJobs.find( *pJobName );
            // Ignore current working jobs here! They couldnt be started twice ...
            if(
                ( pJob               != m_lJobs.end() )   &&
                ( pJob->second.bWork == sal_False     )
              )
            {
                pJob->second.bWork = sal_True;
                lNewWorkJobs.push_back( *pJobName );
            }
        }
    }
    return lNewWorkJobs;
}

/*-****************************************************************************************************************
    @descr  if a job was finished successfully ... he should be deregistered
            Because we doesnt need it again.
            Attention: Job informations shouldnt be removed from configuration realy ...
            but we delete it from internal caches in memory.
            We support a time stamp mechanism to reactivate it by an administrator level.
            So we must set user time stamp to newer value then admig time. Job will be ignored
            next read till administrator change that.
****************************************************************************************************************-*/
void JobCache::forgetJob( const ::rtl::OUString& sJob )
{
    implts_provideFilledCaches(); // late init of caches done?

    ResetableGuard aLock( m_aLock );

    // delete it from ALL event registrations first!
    // But dont do it in configuration too .. see comment before
    for( EventHash::iterator pEvent =m_lEvents.begin();
                             pEvent!=m_lEvents.end()  ;
                             ++pEvent                 )
    {
        ::std::vector< ::rtl::OUString >::iterator pJobName = pEvent->second.begin();
        while( pJobName != pEvent->second.end() )
        {
            if( *pJobName == sJob )
                pEvent->second.erase( pJobName );
            else
                ++pJobName;
        }
    }

    // delete it from normal job list by set right time stamps on cfg ...
    JobCFG().setUserTimeOnJob( sJob );
    m_lJobs.erase( sJob );
}

/*-****************************************************************************************************************
    @descr  if a job was failed ... he should be suspended till next event start it again.
            Support changes on job specific configuration (means his arguments) too.
            If caller give us a valid pointer to any argument list - we use it.
            But we write full lists of arguments only ... there is no support for some items!
            Attention: Don't change time stamps of job info - because we must support new starting of job ...
****************************************************************************************************************-*/
void JobCache::suspendJob( const ::rtl::OUString&                              sJob       ,
                           const css::uno::Sequence< css::beans::NamedValue >* pArguments )
{
    implts_provideFilledCaches(); // late init of caches done?

    ResetableGuard aLock( m_aLock );

    JobHash::iterator pJob = m_lJobs.find( sJob );
    LOG_ASSERT2( pJob==m_lJobs.end(), "JobCache::suspendJob()", "Algorithm error detected ... unknown job should be suspended !!??" )

    // Don't forget to reset working mode of job!
    pJob->second.bWork = sal_False;
    // Write optional arguments to configuration ...
    if( pArguments != NULL )
    {
        pJob->second.lArguments = *pArguments;
        JobCFG().saveJobArguments( sJob, *pArguments );
    }
}

/*-****************************************************************************************************************
    @descr  at our interface we work with job names only. So we must support an access on job configuration.
            Use this method to get a copy(!) of job specific properties to work on it.
            Give us a name - we fill your given structure.
            We use copy instead of reference - because it's better for multithreading.
****************************************************************************************************************-*/
void JobCache::getJobInfo( const ::rtl::OUString& sJob  ,
                                 Job*             pInfo )
{
    implts_provideFilledCaches(); // late init of caches done?

    ResetableGuard aLock( m_aLock );

    JobHash::const_iterator pJob = m_lJobs.find( sJob );
    LOG_ASSERT2( pJob==m_lJobs.end(), "JobCache::getJobInfo()", "Job was not found in cache ... it couldn't be - It's an algorithm error!" )
    *pInfo = pJob->second; // copy it! don't return reference!
}

/*-****************************************************************************************************************
   @descr   We support feature "load on demand" of configuration. Member m_bInitialized indicates already loaded
            caches. If it is set to FALSE - we open configuration, fill internal cache lists and forget cfg
            access. Opening and closing of cfg access is part of JobCFG class. They use refcount and static access
            point for easy using and for better performance. We instanciate it ... and use it.
****************************************************************************************************************-*/
void JobCache::implts_provideFilledCaches()
{
    ResetableGuard aLock( m_aLock );
    if( m_bInitialized==sal_False )
    {
        JobCFG aCFG;
        aCFG.readAll( m_lJobs, m_lEvents );
        m_bInitialized = sal_True;
    }
}

//*****************************************************************************************************************
//	XInterface, XTypeProvider, XServiceInfo
//*****************************************************************************************************************
DEFINE_XINTERFACE_3                     (   JobExecutor                               ,
                                            OWeakObject                               ,
                                            DIRECT_INTERFACE(css::lang::XTypeProvider),
                                            DIRECT_INTERFACE(css::lang::XServiceInfo ),
                                            DIRECT_INTERFACE(css::task::XJobExecutor )
                                        )

DEFINE_XTYPEPROVIDER_3                  (   JobExecutor                               ,
                                            css::lang::XTypeProvider                  ,
                                            css::lang::XServiceInfo                   ,
                                            css::task::XJobExecutor
                                        )

DEFINE_XSERVICEINFO_ONEINSTANCESERVICE  (   JobExecutor                               ,
                                            ::cppu::OWeakObject                       ,
                                            SERVICENAME_JOBEXECUTOR                   ,
                                            IMPLEMENTATIONNAME_JOBEXECUTOR
                                        )

DEFINE_INIT_SERVICE                     (   JobExecutor,
                                            {
                                                /*Attention
                                                    I think we don't need any mutex or lock here ... because we are called by our own static method impl_createInstance()
                                                    to create a new instance of this class by our own supported service factory.
                                                    see macro DEFINE_XSERVICEINFO_MULTISERVICE and "impl_initService()" for further informations!
                                                */
                                            }
                                        )

/*-****************************************************************************************************//**
	@short		standard constructor to create instance by factory
	@descr		This constructor initialize a new instance of this class by valid factory,
				and will be set valid values on his member and baseclasses.

    @attention  Don't use your own reference during an UNO-Service-ctor! There is no guarantee, that you
                will get over this. (e.g. using of your reference as parameter to initialize some member)
                Do such things in DEFINE_INIT_SERVICE() method, which is called automaticly after your ctor!!!

    @seealso    method DEFINE_INIT_SERVICE()

	@param		"xFactory" is the multi service manager, which create this instance.
				The value must be different from NULL!
	@return		-

	@onerror	ASSERT in debug version or nothing in relaese version.
*//*-*****************************************************************************************************/
JobExecutor::JobExecutor( const css::uno::Reference< css::lang::XMultiServiceFactory >& xFactory )
		//	init baseclasses first!
		//	Attention: Don't change order of initialization!
        :   ThreadHelpBase     ( &Application::GetSolarMutex() )
        ,   ::cppu::OWeakObject(                               )
        //  init member
        ,   m_xFactory         ( xFactory                      )
        ,   m_aJobCache        (                               )
{
    // Check incoming parameter to avoid against wrong initialization.
    LOG_ASSERT2( implcp_ctor( xFactory ), "JobExecutor::JobExecutor()", "Invalid parameter detected!" )
    /* Please have a look on "@attentions" of description before! */
}

/*-****************************************************************************************************//**
	@short		standard destructor
    @descr      -

    @seealso    -

	@param		-
	@return		-

	@onerror	-
*//*-*****************************************************************************************************/
JobExecutor::~JobExecutor()
{
}

/*-****************************************************************************************************//**
    @short      start jobs which are registered for specified event string
    @descr      Any outside code can call us with a string identifier. We search on our event list
                for that and use registered jobs for execution.
                We mustn't know meaning of events. It's part of outside configuration and using.
                We do it only!

                We use some helper for working.

                    A temp. JobCFG instance support easy access to our configuration
                    for whole operation time of this method. He use static data container and a ref count mechanism
                    to support late init feature and to perform.

                    Our member m_aJobCache support methods for easy working with our job registration and configuration.
                    We use it to find right jobs, get her properties and change her confguration after execution.

    @seealso    class JobCFG
    @seealso    class JobCache

    @param      "sEvent", specified event for which jobs could be registered
    @return     -

    @onerror    We do nothing then.
*//*-*****************************************************************************************************/
void SAL_CALL JobExecutor::trigger( const ::rtl::OUString& sEvent ) throw( css::uno::RuntimeException )
{
	/* UNSAFE AREA --------------------------------------------------------------------------------------------- */
    // Check incoming parameter to avoid against wrong calls.
    LOG_ASSERT2( implcp_trigger( sEvent ), "JobExecutor::trigger()", "Invalid parameter detected!" )

	/* SAFE AREA ----------------------------------------------------------------------------------------------- */
    ReadGuard aReadLock( m_aLock );

    // Open and hold alive the configuration access by ref count mechanism!
    // This load configuration on demand if neccessary (and not on startup!)
    // and follow code can work without cehcking, if access already exist ...
    // We can do follow code without that ... but then every call on m_aJobCache() will do it seperatly!
    // That couldn't perform.
    JobCFG aCFGHolder;

    ::std::vector< ::rtl::OUString > lWorkJobs = m_aJobCache.getJobsForWork( sEvent );
    for( ::std::vector< ::rtl::OUString >::iterator pJobName =lWorkJobs.begin();
                                                    pJobName!=lWorkJobs.end()  ;
                                                    ++pJobName                 )
    {
        Job aJob;
        m_aJobCache.getJobInfo( *pJobName, &aJob );
        if( aJob.bAsync == sal_False )
        {
            css::uno::Reference< css::task::XJob > xSyncJob( m_xFactory->createInstance( aJob.sService ), css::uno::UNO_QUERY );
            if( xSyncJob.is() == sal_True )
            {
                try
                {
                    css::uno::Any aResult = xSyncJob->execute( aJob.lArguments );
                    impl_reactForJobResult( *pJobName, aResult );
                }
                catch( css::lang::IllegalArgumentException& eIllegal )
                {
                    LOG_EXCEPTION( "JobExecutor::trigger()", "Why job don't understand his own configuration properties as arguments?", eIllegal.Message )
                    m_aJobCache.forgetJob( *pJobName ); // deactivate this job ... he isn't ready for normal working!
                }
                catch( css::uno::Exception& eUnknown )
                {
                    LOG_EXCEPTION( "JobExecutor::trigger()", "Exception during job execute ...", eUnknown.Message )
                    m_aJobCache.forgetJob( *pJobName ); // deactivate this job ... he isn't ready for normal working!
                }
            }
        }
        else
        {
            /* TODO
                Async jobs not suported yet ...
            */
            LOG_WARNING( "JobExecutor::trigger()", "Async job execution not supported yet! Job will be ignored ..." )
        }
    }
}

/*-****************************************************************************************************//**
    @short      helper to react for return result of job execution
    @descr      We analyze returned protocol information of job execution and decide if a job should be
                dactivated or suspended. Protocol means sequence of items from type beans::NamedValue.
                They make it possible for jobs to use some functionality of our implementation
                without using of call backs!

                Protocol:

                    Name           | Type                  | Description
                    ===============|=======================|==========================================
                    Deactivate     | boolean               | Job wish to be deactivated.
                                   |                       | He should call that after successfull execution,
                                   |                       | or if something was wrong and couldnt be resolved ...
                    SaveArguments  | sequence< NameValue > | Job can save his own special arguments back to configuration
                                   |                       | and mustn't implement it by itself!

    @seealso    method trigger()

    @param      "sJob"    name of job
    @param      "aResult" include returned protocol by job
    @return     -

    @onerror    We do nothing.
*//*-*****************************************************************************************************/
void JobExecutor::impl_reactForJobResult( const ::rtl::OUString& sJob    ,
                                          const css::uno::Any&   aResult )
{
    css::uno::Sequence< css::beans::NamedValue > lProtocol;
    if( aResult >>= lProtocol )
    {
        //  try to find items of protocol to analyze it later
        // Otherwise it's not easy to decide if a property was found before another one ...
        // (use invalid index as "non existing" mark)
        sal_Int32 nPDeactivate    = -1;
        sal_Int32 nPSaveArguments = -1;

        sal_Int32 nCount = lProtocol.getLength();
        for( sal_Int32 nStep=0; nStep<nCount; ++nStep )
        {
            if( lProtocol[nStep].Name.compareToAscii("Deactivate") == 0 )
	                nPDeactivate = nStep;
            else
            if( lProtocol[nStep].Name.compareToAscii("SaveArguments") == 0 )
                nPSaveArguments = nStep;
        }

        // react for "Deactivate"
        // ignore all other arguments if it is set!
        if( nPDeactivate != -1 )
        {
			sal_Bool bDeactivate = sal_False;
			if ( ( lProtocol[ nPDeactivate ].Value >>= bDeactivate ) && bDeactivate )
			{
	            m_aJobCache.forgetJob( sJob );
				return;
			}
        }

        // react for "SaveArguments"
        if( nPSaveArguments != -1 )
        {
            css::uno::Sequence< css::beans::NamedValue > lArguments;
            lProtocol[nPSaveArguments].Value >>= lArguments;
            m_aJobCache.suspendJob( sJob, &lArguments );
            return;
        }
    }
    // job has no request ...
    // That means job wasn't finished successfully.
    // We should suspend it and let him wait for further requests.
    // (don't save arguments here!)
    m_aJobCache.suspendJob( sJob );
}

//_________________________________________________________________________________________________________________
//	debug methods
//_________________________________________________________________________________________________________________


/*-************************************************************************************************************//**
    @short      debug-method to check incoming parameter of some other mehods of this class
    @descr      The following methods are used to check parameters for other methods
                of this class. The return value is directly used for an ASSERT(...).

    @attention  This methods are static and can't use our member directly! It's better for threadsafe code...
                because we call it with references or pointer to check variables ... and must make it safe
                by ourself!

    @seealso    ASSERTs in implementation!

    @param      references to checking variables
    @return     sal_True  ,on invalid parameter
    @return     sal_False ,otherwise

    @onerror    -
*//*-*************************************************************************************************************/

#ifdef ENABLE_ASSERTIONS

//*****************************************************************************************************************
sal_Bool JobExecutor::implcp_ctor( const css::uno::Reference< css::lang::XMultiServiceFactory >& xFactory )
{
	return	(
				( &xFactory		==	NULL		)	||
				( xFactory.is()	==	sal_False	)
			);
}

//*****************************************************************************************************************
sal_Bool JobExecutor::implcp_trigger( const ::rtl::OUString& sEvent )
{
	return	(
                ( &sEvent            == NULL )  ||
                ( sEvent.getLength() <  1    )
			);
}

#endif	// #ifdef ENABLE_ASSERTIONS

}	// namespace framework
